package SDB::Install::Installation::Generic;

use File::Basename;
use SDB::Install::Installation;
use SDB::Install::SysVars qw ($path_separator $isWin);
use SDB::Install::System qw (makedir removeUninstallEntry deltree isAdmin);
use SDB::Install::System::EnvVariable;
use SAPDB::Install::Hostname;
use SDB::Install::Globals qw ($gProductNameClient);
use SAPDB::Install::System::Unix qw(lchown);
use SDB::Install::InstallationEventHandler;
use SDB::Install::DebugUtilities qw (dumpThings);
use strict;

#----------------------------------------------------------------------------

our @ISA = qw (SDB::Install::Installation);

#----------------------------------------------------------------------------

our $installations;

#----------------------------------------------------------------------------

sub new{
    my $self = shift->SUPER::new (@_);
    $self->{'registrypath'} = $self->{'path'}. $path_separator . 'install';
    $self->{'newInstallation'} = (-d $self->{'registrypath'}) ? 0 : 1;
    return $self;
}

#----------------------------------------------------------------------------

sub EnumGenericInstallations{
    my ($compId, $msglst, $nocache) = @_;
    if (defined $installations && !$nocache){
        return $installations;
    }
    my $obj = new __PACKAGE__;
    if (defined $compId) {
        $obj->{_tmpCompId} = $compId;
    }
    $installations = $obj->EnumInstallations ();
    return $installations;
}

#----------------------------------------------------------------------------

sub init{
    my ($self) = @_;
    $self->SUPER::init ();
    $self->{'registrypath'} = $self->{'path'}. $path_separator . 'install';
    $self->{'newInstallation'} = (-d $self->{'registrypath'}) ? 0 : 1;
}



sub getCompId {
	my $compId = $_[0]->SUPER::getCompId();
	if (!defined $compId){
		return $_[0]->{_tmpCompId};
	} else {
		return $compId;
	}
}

#----------------------------------------------------------------------------

sub getRegFileName {
    my ($self) = @_;
    return 'installations.' . $self->getCompId ();
}

#----------------------------------------------------------------------------

sub asString{
	my ($self) = @_;
	return $self->getProductName() . ' ' . $self->GetVersion () . ' ' . $self->{path};
}

#----------------------------------------------------------------------------

# despite its name, this method is also called in case of a 
# software update:
sub initNewInstallation{
    my (
        $self
    ) = @_;
    my $cfg = {};
    if (!$isWin){
		my $uid = $self->getUID ();
		my $gid = $self->getGID ();
		if (defined $uid && defined $gid){
			$cfg->{uid} = $uid;
			$cfg->{gid} = $gid;
			$cfg->{mode} = 0755;
		}
    }
    if (!-d $self->{'path'}) {
        if (!defined makedir($self->{'path'}, $cfg)) {
            $self->AddError("Could not create installation root directory: $self->{'path'}", $cfg);
            return undef;
        }
    }
    if (!-d $self->{'registrypath'}) {
        if (!defined makedir($self->{'registrypath'}, $cfg)) {
            $self->AddError("Could not create install registry directory: $self->{'registrypath'}", $cfg);
            return undef;
        }
    }
    return 1;
}

sub getUID{
	my ($self) = @_;
	my $config = $self->getConfiguration();
	if (defined $config) {
		return $config->get_UID();
	} else {
		return undef;
	}
}

sub getGID{
	my ($self) = @_;
	my $config = $self->getConfiguration();
	if (defined $config) {
		return $config->get_GID();
	} else {
		return undef;
	}
}

#----------------------------------------------------------------------------

sub getProgramPath{
	return $_[0]->{path};
}

sub getDataPath;
*getDataPath = \&getProgramPath;

#----------------------------------------------------------------------------

sub getConfigXmlPath{
	return $_[0]->{path}.$path_separator.'install';
}

#----------------------------------------------------------------------------

sub setConfigParser {
    $_[0]->{configparser} = $_[1];
}

#----------------------------------------------------------------------------

sub getConfigParser {
    if (!defined $_[0]->{configparser}){
        my ($self) = @_;
        require SDB::Install::ConfigXMLParser;
        my $xmlFile = $self->getConfigXmlPath() . $path_separator . 'InstallParams.xml';
        if (!-f $xmlFile){
            $self->AddError ("Configuration file '$xmlFile' not found");
            return undef;
        }
        my $dtdFile = $self->getConfigXmlPath() . $path_separator . 'installcfg.dtd';
        if (!-f $dtdFile){
            $self->AddWarning ("Configuration DTD file '$dtdFile' not found");
            $dtdFile = undef;
        }
        $self->{configparser} = new SDB::Install::ConfigXMLParser ();
        eval{
            $self->{configparser}->parsefile ($xmlFile, $dtdFile);
        };
        if ($@){
            $self->AddError ("Parsing configuration file '$xmlFile' failed: $@");
            return undef;
        }
        my $installType = $self->{configparser}->getProduct()->{InstallType};
        if ($installType) {
            my $instClass = "SDB::Install::Installation::$installType";
            eval ("require $instClass;");
            if (!$@){
                bless ($self, $instClass);
            }
        }
    }
    $_[0]->{configparser}->setInstParamReplaceHash ($_[0]->getReplaceHash());

	return $_[0]->{configparser};
}

sub isUserMode{
    return !isAdmin();
}

#----------------------------------------------------------------------------

sub getReplaceHash{

	my ($self) = @_;
	return {'%(InstallationPath)' => $self->GetInstallationPath ()};
}

#----------------------------------------------------------------------------

sub getProductName{
    my ($self) = @_;
    if (!defined $self->{_product}){
        my $configparser = $self->getConfigParser ();
        if (! defined $configparser){
            return $self->SUPER::getProductName ();
        }
        my $product = $configparser->getProduct ();
        if (!defined $product){
            return $self->SUPER::getProductName ();
        }
        $self->{_product} = $product;
    }
    return $self->{_product}->{ProductName};
}

#----------------------------------------------------------------------------

sub getShortProductName{
    my ($self) = @_;
    if (!defined $self->{_product}){
        my $configparser = $self->getConfigParser ();
        if (!defined $configparser){
            return $self->SUPER::getProductName ();
        }
        my $product = $configparser->getProduct ();
        if (!defined $product){
            return $self->SUPER::getProductName ();
        }
        $self->{_product} = $product;
    }
    return $self->{_product}->{ShortProductName};
}

#----------------------------------------------------------------------------

sub getProductKey{
    my ($self) = @_;
    if (!defined $self->{_product}){
        my $configparser = $self->getConfigParser ();
        if (! defined $configparser){
            return undef;
        }
        my $product = $configparser->getProduct ();
        if (!defined $product){
            return undef;
        }
        $self->{_product} = $product;
    }
    return $self->{_product}->{ProductKey};
}

#----------------------------------------------------------------------------

sub initEventHandler{
    my ($self) = @_;
    my $parser = $self->getConfigParser ();
    if (!defined $parser){
        $self->AddError ("No configParser defined");
        return undef;
    }
    my $handlerClass;
    my $customModules = $parser->getCustomModules ();
    if (defined $customModules){
        $handlerClass = $customModules->{EventHandler};
    }
    if (defined $handlerClass){
        local @INC = @INC;
        push @INC, $self->getCustomModulesDir();
        eval ("require $handlerClass;");
        if ($@){
            $self->AddError ("Cannot initialize custom event handler: $@");
            return undef;
        }
        $self->{eventHandler} = $handlerClass->new ();
    }
    else{
        $self->{eventHandler} = new SDB::Install::InstallationEventHandler ();
    }
    return 1;
}

#----------------------------------------------------------------------------

sub getEventHandler{
    return $_[0]->{eventHandler};
}

#----------------------------------------------------------------------------

sub getCustomModulesDir{
    return $_[0]->getProgramPath().$path_separator.'install'.$path_separator.'CustomModules';
}

#----------------------------------------------------------------------------

sub Uninstall{
	my $self = shift;
	my (undef, $force, $instconfig, $update) = @_;
	my $msg;

    my $configParser = $self->getConfigParser ();

    if (!defined $configParser){
        return undef;
    }
    $self->{configparser}->setInstParamReplaceHash ($self->getReplaceHash());

	# for LCM sdk consumer Data Provisioning:
	my $packages = $self->GetPackages ();
	$msg = $self->AddMessage ('Preparing uninstallation of packages...');
    foreach my $package (reverse @$packages) {
    	$package->setMsgLstContext ([$msg->getSubMsgLst()],1);
        if(!$package->PreemptivePreuninstall()) {
		    $self->AddError('Cannot remove installation: PreemptivePreuninstall step of package '.$package->{data}->{name}.' failed.');
		    return undef;
        }
    }
    $self->AddMessage ('Preparation done.');

	# Initialize custom event handlers

	if (!defined $self->initEventHandler ()){
		return undef;
	}
	my $eventHandler = $self->getEventHandler ();
	$eventHandler->setInstallation ($self);
	$eventHandler->setInstConfig ($instconfig);

    # Execute custom event preUninstall if this is not an update

	if (!$update && !$eventHandler->triggerEvent ('preUninstall', $self->getMsgLst(), $self->getErrMsgLst())){
		return undef;
	}

	# Delete shortcuts

	my $shortcuts = $self->{configparser}->getShortcuts();
	if (defined $shortcuts && @$shortcuts){
		$msg = $self->AddMessage ('Deleting shortcuts');
		$self->SetFormatInfo ($msg, 'h1', 'Deleting shortcuts');
		my $name;
		foreach my $shortcut (@$shortcuts){
			$name = "$shortcut->{ScPath}$path_separator$shortcut->{ScName}";
			if ($isWin) {
				$name = $name .= '.lnk';
			}
			$self->AddSubMessage ($msg, "Deleting shortcut '$name'");
			if (!unlink($name)) {
				$self->AddError ("Cannot delete shortcut '$name'");
			}
			# ignore error if shortcut directory cannot be removed because it is not empty
			rmdir ($shortcut->{ScPath});
		}
	}

	# Unset environment variables

	my $env = $self->{configparser}->getEnvironment();
	if (defined $env && %$env){
		$msg = $self->AddMessage ('Unsetting environment variables');
		$self->SetFormatInfo ($msg, 'h1', 'Unsetting environment variables');
		my $var;
		foreach my $varname (keys %$env){
			$var = new SDB::Install::System::EnvVariable ($varname, $env->{$varname});
			if (!defined $var->unset ()){
				$self->AddError (undef, $var);
			}
			$self->AddSubMsgLst ($msg, $var);
		}
	}

	# Remove config XML file and manifest file

    my $manifestFile = $self->getProgramPath()   . $path_separator . 'manifest';
    my $xmlFile      = $self->getConfigXmlPath() . $path_separator . 'InstallParams.xml';
    my $dtdFile      = $self->getConfigXmlPath() . $path_separator . 'installcfg.dtd';

    foreach my $file ($manifestFile, $xmlFile, $dtdFile){
        if (-f $file){
            if (!unlink($file)) {
                $self->AddError ("Cannot delete file '$file'");
            }
        }
    }

	# Call base class uninstall if this is not an update

	my $rc;
	if (!$update) {
		$rc = $self->SUPER::Uninstall (@_);
		if (!defined $rc) {
			return $rc;
		}
	}

	# Remove files if this is not an update

	if (!$update) {
		my $msglst = new SDB::Install::MsgLst ();
		my $files = $self->{configparser}->getUninstallationEntry()->{UninstFiles};
		if ((defined $files) && @$files){
			$msg = $self->AddMessage ('Deleting files');
			$self->SetFormatInfo ($msg, 'h1', 'Deleting files');
			my $name;
			foreach my $file (@$files){
				if (!unlink ($file)) {
					$msglst->AddError ("Cannot delete file \"$file\"");
				} else {
					$msglst->AddMessage ("File \"$file\" deleted");
				}
			}
			$self->AddSubMsgLst ($msg, $msglst);
		}
	}

	# Remove directories if this is not an update

	if (!$update) {
		my $msglst = new SDB::Install::MsgLst ();
		my $dirs = $self->{configparser}->getUninstallationEntry()->{UninstDirectories};
		if ((defined $dirs) && @$dirs){
			$msg = $self->AddMessage ('Removinging directories');
			$self->SetFormatInfo ($msg, 'h1', 'Removing directories');
			my $name;
			foreach my $dir (@$dirs){
				my $errlst = new SDB::Install::MsgLst();
				if (!deltree ($dir, $errlst)) {
					$msglst->AddError ("Cannot remove directory \"$dir\"", $errlst);
				}
				$self->AddSubMsgLst ($msg, $errlst);
			}
			$self->AddSubMsgLst ($msg, $msglst);
		}
		my $customModulesDir = $self->getCustomModulesDir();
		if (-d $customModulesDir) {
			if (!deltree ($customModulesDir)) {
				$self->AddError ("Cannot remove directory \"" . $customModulesDir . "\": $!");
			} else {
				$self->AddMessage ("Directory \"" . $customModulesDir . "\" removed");
			}
		}
		if (-d $self->getProgramPath()) {
			if (!rmdir ($self->getProgramPath())) {
				$self->AddError ("Cannot remove directory \"" . $self->getProgramPath() . "\": $!");
			} else {
				$self->AddMessage ("Directory \"" . $self->getProgramPath() . "\" removed");
			}
		}
	}

	# Remove uninstall entry

	my $msglst = new SDB::Install::MsgLst ();
	if ((defined $self->{configparser}->getUninstallationEntry()) && (!defined removeUninstallEntry (
		$self->{configparser}->getUninstallationEntry()->{UninstKey},
		0,
		$self->GetInstallationPath (),
		$msglst)))
	{
		$self->AddError ('Cannot remove uninstall entry', $msglst);
	}

	# Execute custom event postUninstall if this is not an update

	if (!$update && !$eventHandler->triggerEvent ('postUninstall', $self->getMsgLst(), $self->getErrMsgLst())){
		return undef;
	}

	# Remove hdblcm status file

	if (!$update){
		my $globalSidDir = dirname($self->GetProgramPath());
		$instconfig->removeHdblcmStatusFile($globalSidDir);
	}

	return $rc;
}

#----------------------------------------------------------------------------

1;
