package SDB::Install::Kit::DatabaseStudio;
use SDB::Install::System qw (copy_tree copy_file deltree exec_program createUninstallEntry);
use SDB::Install::SysVars;
use SDB::Install::Globals qw ($gProductName $gProductNameStudio);
use SDB::Install::EclipseIniFile;
use File::Basename qw (dirname);
use strict;

our @ISA = qw (SDB::Install::Kit);

sub Install{
	my $self = shift;
	$self->{installRc} = 0;
	if (!defined $self->{repository_destination}){
		$self->{installRc} = $self->executeInstall(@_);
		return $self->{installRc} ;
	}
	return $self->copyRepository(@_);
	
}

sub getHanaStudioExe{
    my ($self, $path) = @_;
    my $exe;
    if ($isWin){
        $exe = 'hdbstudio.exe';
    }
    elsif ($isApple){
        $exe = 'Contents/MacOS/hdbstudio';
        if ($path !~ /\.app$/){
            $exe = 'hdbstudio.app/' . $exe;
        }
    }
    else{
        $exe = 'hdbstudio';
    }
    return $path . $path_separator . $exe;
}


sub executeInstall{
	my $self = shift;
	my $str = 'Running Equinox P2 Director';
	
	#add instconfig warnings to the log
	my $warnings    = $self->{installation}->getConfiguration()->getWarningList();
    foreach my $currWng (@$warnings) {
        $self->AddMessage("Note: $currWng");
    }
	
	$self->{append_progress_steps} = [$str];
		
	if (!defined $self->SUPER::Install(@_)){
		return undef;
	}
	my $progress_handler = $self->GetProgressHandler();
	
	my $pp_handler = $self->GetPackageProgressHandler ();
		
	if (defined $pp_handler){
		$pp_handler->InitProgress (5,1);
	}
	my $installation = $self->{installation};
	my $path = $installation->{path};
	my $studioExecutable = $self->getHanaStudioExe ($path);
		 
	if (defined $self->{repository}){
		my $msg = $self->AddProgressMessage ("$str...");
		$self->SetFormatInfo ($msg,'h1','Run Eclipse Equinox Director');
		
		my $featureIdsForInstall = $self->_getFeaturesToInstall();
		my $featureIdsForUninstall = $self->_getFeaturesToUninstall();
		my $submsglst = new SDB::Install::MsgLst;
		my $returnCode = $self->callDirector (
			    $self->{installation}->{path},
				[
				($self->{java_vm} ? ('-vm',$self->{java_vm}) : ()),
				# include uninstall call if in update
				( ( -f $studioExecutable && defined $featureIdsForUninstall )? ('-uninstallIU', $featureIdsForUninstall ) : () ),
        	       '-installIU', $featureIdsForInstall,					
					'-repository', $self->{repository},
                    '-profile', 'hdbstudio',
                    '-destination', "file:$self->{installation}->{path}",					
					'-profileProperties', 'org.eclipse.update.install.features=true',
					'-p2.os',($isWin ? 'win32' : $isApple ? 'macosx' :'linux'),
					'-p2.ws',($isWin ? 'win32' : $isApple ? 'cocoa' : 'gtk'),
					'-p2.arch', ($installerIs64bit ? 'x86_64': 'x86'),
					'-roaming',					
					'-vmargs', "-Declipse.p2.data.area=$path${path_separator}p2"
					],$submsglst);
		if ( !defined  $returnCode || $returnCode != 0 ){
			if($returnCode == 13){ #Fail only if p2 director crashes - error code 13 see Bug 63349 	
				$self->AddError ("Eclipse Equinox Director failed");
				$self->AddSubMsgLst ($msg, $submsglst);
				if (defined $progress_handler){
					$progress_handler->StepFinished (1);
				}
				return undef;
			} else {
				$self->AddMessage ("Eclipse Equinox Director failed", $submsglst);
			}
		}
		$self->AddSubMsgLst ($msg, $submsglst);

		# Configure Java VM in hdbstudio.ini only when such is not contained in the studio repo
		if ($self->{java_vm} && ! $installation->getConfiguration()->_isKitContainingJvm() ) {
			my $java_vm_dir = dirname($self->{java_vm});
			my $submsglst = new SDB::Install::MsgLst;
			my $iniFilePath = $path . $path_separator . 'hdbstudio.ini';
			if (-f $iniFilePath) {
				my $iniFile = new SDB::Install::EclipseIniFile($iniFilePath);
				if (defined $iniFile->{_buffer}) {
					if ($iniFile->setOption(['-vm', $java_vm_dir])) {
						if (!$iniFile->write()) {
							$self->AddError ("Setting option for Java VM not possible: Writing of file $iniFilePath failed", $iniFile);
						}
					} else {
						$self->AddError ("Setting option for Java VM failed", $iniFile);
					}
				} else {
					$self->AddError ("Setting option for Java VM not possible: Parsing of file $iniFilePath failed", $iniFile);
				}
				$self->AddSubMsgLst ($msg, $iniFile);
			} else {
				$self->AddError ("Setting option for Java VM not possible: file $iniFilePath not found", $submsglst);
			}
			$self->AddSubMsgLst ($msg, $submsglst);
		}

		if (defined $pp_handler){
			foreach (0..3){
				$pp_handler->SetProgress ();
			}
		}
	}
	if (defined $progress_handler){
		$progress_handler->StepFinished();
	}

	if ($isWin && !$installation->getConfiguration ()->isUserMode ()){
	    my $hdbuninst = join ($path_separator, $path, 'install', 'hdbuninst.exe');
        if (-f $hdbuninst){
    	    my $errlst = new SDB::Install::MsgLst ();
            my %entry = (
        	    'name' => $gProductNameStudio .  ($self->is64bit () ? ' 64bit' : ' 32bit'),
                'cmd' => $hdbuninst,
                'size' => 148 * 1024,
                'icon' => $studioExecutable,
                'version' => $self->GetVersion ()
            	 );
            if (!defined createUninstallEntry ('HDBSTUDIO', \%entry, 1, $path, $errlst)){
                $self->AddError ("Cannot add uninstall entry", $errlst);
            }
        }
	}
	return 1;
}

sub copyRepository{
	my $self = shift;
	my $src = $self->{archive_dir} . $path_separator . 'core_repository';

	if (!-e $src){
		$src = $self->{archive_dir} . $path_separator . 'repository';
	}

	if (!-e $src){
		$self->AddError ("Repository '$src': $!");
		return undef;
	}

	my @src_statbuf = stat (_);


	my $repo = $self->{repository_destination} . $path_separator . 'repository';
	my $repo_old = $self->{repository_destination} . $path_separator . 'repository_old';

	my $errlst = new SDB::Install::MsgLst ();

	if (-e $repo){
		if (-e $repo_old){
			$self->AddMessage ("Deleting $repo_old");
			if (!deltree ($repo_old,$errlst)){
				$self->AddError ("Cannot remove old repository", $errlst);
				return undef;
			
			}
		}
		$self->AddMessage ("Renaming '$repo' => '$repo_old'");
		if (!rename ($repo, $repo_old)){
			$self->AddError ("Cannot rename old repository ($repo => $repo_old): $!");
			return undef;
		}
	}
	
	#
	# copy repository
	#

	my $msg = $self->AddProgressMessage ("Copying repository to '$self->{repository_destination}'");
	
	my $submsglst = new SDB::Install::MsgLst ();
	
    my $rc = copy_tree ($src,
        $self->{repository_destination}. $path_separator . 'repository',
        $submsglst, \@src_statbuf);
    
    if (!$rc){
        $self->AddError ("Cannot copy repository",$submsglst);
        $self->AddSubMsgLst ($msg, $submsglst);
        return undef;
    }
    my $manifestSrc  = $self->{archive_dir} . $path_separator . 'manifest';
    my $manifestDest = $self->{repository_destination} . $path_separator . 'manifest';
    my $cfg = {};
    $rc = copy_file ($manifestSrc, $manifestDest, $cfg);
    if (!$rc){
        $self->AddError ("Cannot copy repository manifest",$cfg);
        return undef;
    }

	$self->AddSubMsgLst ($msg, $submsglst);
	return 1;
}

sub _getFeaturesToUninstall {
    my ($self) = @_;
    my $featureManager = $self->{installation}->{configuration}->getFeatureManager();
    return $featureManager->getFeaturesToUninstall();    
}

sub _getFeaturesToInstall {
    my ($self) = @_;
    my $featureManager = $self->{installation}->{configuration}->getFeatureManager();
    return $featureManager->getFeaturesToInstall();                  
}

sub is64bit {
	return $installerIs64bit;
}

sub skipInstallationSteps {
	return 0;
}

sub DESTROY{
	my ($self) = @_;

	$self->SUPER::DESTROY ();
	my $startStudioAfterInstallation = $self->{installRc} && $self->{StartStudioAfterInstallation};

	if ($startStudioAfterInstallation){
		my $exe = $self->getHanaStudioExe ($self->{InstallationPath});
		my $cfg = {};
		my $rc = exec_program($exe, undef, $cfg);

		if (!defined $rc || $rc != 0){
			$self->AddError ("Can not start '$gProductNameStudio' after installation", $cfg);
			return undef;
		}
	}
}
1;

