package SDB::Install::Kit::Generic;

use SDB::Install::Kit;
use SDB::Install::System::Shortcut;
use SDB::Install::System::EnvVariable;
use SDB::Install::System qw (makedir exec_program copy_file createUninstallEntry);
use SDB::Install::SysVars qw ($path_separator $isWin);
use SAPDB::Install::System::Win32::API qw (ShellExecute);
use SDB::Install::DebugUtilities qw (dumpThings);
use SDB::Install::InstallationEventHandler;
use SDB::Install::DirectoryWalker;
use Exporter;

use strict;

our @ISA = qw (SDB::Install::Kit Exporter);

sub Install{
	my ($self) = @_;

	my $installation = $self->getInstallation();
    my $instconfig = $installation->getConfiguration();
    my $installationPath = $installation->GetInstallationPath();
    my $replaceHash = $instconfig->getReplaceHash();
    if ($self->isUpdate() && -d $installationPath) {
        $replaceHash->{'%(InstallationPath)'} = $installationPath;
    }
    $instconfig->{configparser}->setInstParamReplaceHash ($replaceHash);
    my $parser = $instconfig->{configparser};
	my ($uid, $gid) = undef;
    if (!$isWin){
		$uid = $installation->getUID ();
		$gid = $installation->getGID ();
    }
    my $fileCfg = {};
    if (!$isWin){
        if (defined $uid && defined $gid){
            $fileCfg->{uid} = $uid;
            $fileCfg->{gid} = $gid;
        }
    }
	my $stepConfigProduct = 'Configuring product';
    my $currProgressSteps = 7;
    my $restProgressSteps = $currProgressSteps;	
	my $p2call            = $parser->getP2Call();
	
    if (defined $p2call && %$p2call) {
    	$self->{append_progress_steps} = [$stepConfigProduct, 'Installing Eclipse repository'];
    	$currProgressSteps = 4;
    	$restProgressSteps = $restProgressSteps - $currProgressSteps;
    }
    else {
    	$p2call = undef;
        $self->{append_progress_steps} = [$stepConfigProduct];
    }

	$installation->setSubstMacros ($parser->getReplaceInFiles ());

    my $phase = $instconfig->getValue ('Phase');
    my $pp_handler = $self->GetPackageProgressHandler ();

    my $progress_handler = $self->GetProgressHandler();
    $instconfig->setMsgLstContext ([$self->getMsgLst()]);
    my $onlinePhase = defined $phase && $phase ne 'offline' ? 1 : 0;
    if (!$onlinePhase && $instconfig->enterExtractSoftwareStep ()){

	    if ($self->isUpdate()) {
		    $installation->Uninstall(0, 0, undef, 1);
	    }


        # execute custom event preInstall
        if (!$self->{eventHandler}->triggerEvent ('preInstall', $self->getMsgLst(), $self->getErrMsgLst())){
            return undef;
        }

        # Save config XML file, manifest, and custom modules for uninstallation


        my @copy_list = (
            # InstallParams.xml
            [$self->getArchiveDir() . $path_separator . 'InstallParams.xml',
             $installation->getConfigXmlPath() . $path_separator . 'InstallParams.xml'],
            # installcfg.dtd
            [$self->getArchiveDir() . $path_separator . 'installcfg.dtd',
             $installation->getConfigXmlPath() . $path_separator . 'installcfg.dtd'],
            # manifest
            [$self->getArchiveDir() . $path_separator . 'manifest',
             $installation->getProgramPath() . $path_separator . 'manifest']
        );
        if(-d $self->getCustomModulesDir()) {
            my $dirWalker = SDB::Install::DirectoryWalker->new(undef, undef, undef,
                                                               undef, undef, undef,
                                                               sub {
                                                                   my $tgt = $installation->getConfigXmlPath().
                                                                                            $path_separator.'CustomModules'.
                                                                                            $path_separator.$_[4];
                                                                   if($_[6]) {
                                                                       if(!defined makedir($tgt, $fileCfg)) {
                                                                           $self->AddError ("Could not create directory '$tgt'.", $fileCfg);
                                                                           return undef;
                                                                       }
                                                                   }
                                                                   else {
                                                                       push @copy_list, [$_[3].$path_separator.$_[4], $tgt];
                                                                   }
                                                                   return 1;
                                                                },              undef, undef,
                                                                1,    # breadth-first
                                                                1,    # dont collect list
                                                                0     # dont follow symlinks
                                                               );
            my $rc = $dirWalker->findAndProcess($self->getCustomModulesDir());
            if(not defined $rc) {
                $self->AddError($dirWalker->getErrorString());
                return undef;
            }
        }
        my $rc;
        my $save_umask;
        if (!$isWin){
            $save_umask = umask (0333);
        }
        foreach my $filesToCopy (@copy_list){
           $fileCfg = {};
           if (!$isWin){
                if (defined $uid && defined $gid){
                    $fileCfg->{uid} = $uid;
                    $fileCfg->{gid} = $gid;
                }
            }
            $fileCfg->{'createdir'} = 1;
            $fileCfg->{'dir_mode'}  = 0755;
            my $msg = $self->getMsgLst()->addMessage ("Copying $filesToCopy->[0] to $filesToCopy->[1]");
            $msg->getSubMsgLst ()->injectIntoConfigHash ($fileCfg);
            $rc = copy_file(@$filesToCopy, $fileCfg, 0);
            if (!defined $rc) {
                $self->AddError ("Cannot copy config file $filesToCopy->[0] to $filesToCopy->[1]", $fileCfg);
                last;
            }
        }
        if (defined $save_umask){
            umask ($save_umask);
        }

	    # call function Install() of super class,
	    # includes extracting of packages and replacing of strings in extracted files
	    # as defined in section ReplaceInFiles of InstallParams.xml

	    if (!defined $rc || !defined $self->SUPER::Install(@_)){
		    return undef;
	    }

	    my $progress_handler = $self->GetProgressHandler();

	    if (defined $pp_handler){
		    $pp_handler->InitProgress ($currProgressSteps,0);
	    }


	    # Installation actions defined in config XML file
	    # Update $pp_handler after each action

	    # Set environment variables

        my $msg;

	    my $env = $parser->getEnvironment();
	    if (defined $env && %$env){
		    $msg = $self->AddMessage ('Setting environment variables');
		    $self->SetFormatInfo ($msg, 'h1', 'Setting environment variables');
		    my $var;
		    foreach my $varname (keys %$env){
			    $var = new SDB::Install::System::EnvVariable ($varname, $env->{$varname});
			    if (!defined $var->set ()){
				    $self->AddError (undef, $var);
				    $progress_handler->StepFinished (1);
			    }
			    $self->AddSubMsgLst ($msg, $var);
		    }
	    }

	    if (defined $pp_handler){
		    $pp_handler->SetProgress ();
	    }

	    # Create directories

	    my $dirs = $parser->getDirectories();
	    if (defined $dirs && @$dirs){
		    $msg = $self->AddMessage ('Creating directories');
		    $self->SetFormatInfo ($msg, 'h1', 'Create directories');

	        my $cfg = {};
	        if (!$isWin){
			    if (defined $uid && defined $gid){
				    $cfg->{uid} = $uid;
				    $cfg->{gid} = $gid;
			    }
	        }
		    foreach my $dir (@$dirs){
			    $self->AddSubMessage ($msg, "Creating directory '$dir'");

			    if (!defined makedir ($dir, $cfg)){
				    $self->AddError ("Cannot create directory '$dir'", $cfg);
				    $progress_handler->StepFinished (1);
				    return undef;
			    }
		    }
	    }

	    if (defined $pp_handler){
		    $pp_handler->SetProgress ();
	    }

	    # Create shortcuts

	    my $shortcuts = $parser->getShortcuts();
	    if (defined $shortcuts && @$shortcuts){
		    $msg = $self->AddMessage ('Creating shortcuts');
		    $self->SetFormatInfo ($msg, 'h1', 'Creating shortcuts');

		    my $name;
		    foreach my $shortcut (@$shortcuts){
			    $name = "$shortcut->{ScPath}$path_separator$shortcut->{ScName}";
			    $self->AddSubMessage ($msg, "Creating shortcut '$name'");
			    my $sc = new SDB::Install::System::Shortcut ($shortcut->{ScPath}, $shortcut->{ScTarget}, $shortcut->{ScName},1);
			    $sc->setIcon ($shortcut->{ScIcon});
			    $sc->setRunAsAdmin ($shortcut->{ScRunAsAdmin});
			    if (!defined $sc->create()){
				    $self->AddError ("Cannot create shortcut", $sc);
				    $progress_handler->StepFinished (1);
				    return undef;
			    }
		    }
	    }

	    if (defined $pp_handler){
		    $pp_handler->SetProgress ();
	    }

	    # Execute commands

	    my $cmds = $parser->getExecuteCommands ();
	    if (defined $cmds && @$cmds){
		    $msg = $self->AddMessage ('Starting external programs');
		    $self->SetFormatInfo ($msg, 'h1', 'Starting external programs');

            my $rc;
            foreach my $cmd (@$cmds){
			    my $cfg = {};
			    if (defined $cmd->{ExecInputStream}) {
				    if ($cmd->{ExecInputStream} eq 'XMLPasswordStream') {
					    my $xmlStream = $instconfig->getXmlPasswordStream();
					    $cfg = {'stdin_buf' => $xmlStream};
				    } else {
					    $cfg = {'stdin_buf' => [$cmd->{ExecInputStream}]};
				    }
			    }

                my $cmdArray;

                if (defined $cmd->{ExecArgs}) {
                    $cmdArray = [split (',', $cmd->{ExecArgs})];
                }
                else{
                    $cmdArray = $cmd->{ExecArg}
                }
			    # open new block within which the environment may be changed
			    {
				    local %ENV = %ENV;
				    if (defined $cmd->{ExecEnv} && %{$cmd->{ExecEnv}}) {
					    foreach my $var (keys %{$cmd->{ExecEnv}}) {
						    $ENV{$var} = $cmd->{ExecEnv}->{$var};
					    }
				    }
                    $rc = exec_program ($cmd->{ExecCommand},$cmdArray,$cfg);
			    }
                if ((!defined $rc || $rc != $cmd->{ReturnCodeOK}) && !$cmd->{IgnoreError}){
                    $self->AddError ('External program failed', $cfg);
                    $self->AddSubMsgLst ($msg, $cfg);
                    $progress_handler->StepFinished (1);
                    return undef;
                }
                $self->AddSubMsgLst ($msg, $cfg);
            }
        }

	    if (defined $pp_handler){
		    $pp_handler->SetProgress ();
	    }

	    # Run P2 Director

        if (defined $p2call) {

            $progress_handler->StepFinished (0);

            if (defined $pp_handler){
                $pp_handler->InitProgress ($restProgressSteps+1, 1);
            }

		    $msg = $self->AddMessage ('Running P2 Director');
		    $self->SetFormatInfo ($msg, 'h1', 'Running P2 Director');
		    my $cfg = new SDB::Install::MsgLst ();
		    my $instroot = $installation->getProgramPath();

		    require SDB::Install::P2Director;
		    my $director = new SDB::Install::P2Director ($instroot);

		    if ($p2call->{JavaVM}) {
			    $director->setJavaVm($p2call->{JavaVM});
		    } else {
			    $director->setJavaVm($instconfig->getJava());
		    }
		    if (defined $p2call->{Repository}) {
			    $director->setRepository($p2call->{Repository});
		    } else {
			    $director->setRepository('file:' . $self->getArchiveDir() . $path_separator . 'repository');
		    }

		    my $rc = $director->install($p2call->{InstUnits}, $cfg);
		    if ($rc != 0) {
			    $self->AddError ('Running P2 Director failed', $director);
			    $self->AddSubMsgLst ($msg, $cfg);
			    $progress_handler->StepFinished (1);
			    return undef;
		    }

		    $self->AddSubMsgLst ($msg, $cfg);
	    }

        if (defined $pp_handler){
            $pp_handler->SetProgress ();
        }

	    # Create uninstallation entry

	    if ($isWin){
		    my $uninstEntry = $parser->getUninstallationEntry();

		    if (defined $uninstEntry){
			    $msg = $self->AddMessage ('Creating uninstallation entry');
			    $self->SetFormatInfo ($msg, 'h1', 'Creating uninstallation entry');
			    my $hdbuninst = join ($path_separator, $installation->getProgramPath(), 'install', 'hdbuninst.exe');

			    if (-f $hdbuninst){
				    $hdbuninst = "\"" . $hdbuninst . "\" --silent --main SDB::Install::App::Gui::Uninstallation::main -path \"" . $installation->getProgramPath() . "\"";
				    my $errlst = new SDB::Install::MsgLst ();
				    my %entry = (
					    'name' => $uninstEntry->{UninstName},
					    'cmd' => $hdbuninst,
					    'size' => $uninstEntry->{UninstSizeKB},
					    'icon' => $uninstEntry->{UninstIcon},
					    'version' => $uninstEntry->{UninstVersion}
				    );
				    if (!defined createUninstallEntry ($uninstEntry->{UninstKey}, \%entry, 0, $installation->getProgramPath(), $errlst)){
					    $self->AddError ("Cannot add uninstall entry", $errlst);
					    return undef;
				    }
			    }
		    }
	    }

	    if (defined $pp_handler){
		    $pp_handler->SetProgress ();
	    }

    } # exctract step

    my $rc = 1;

   $instconfig->enterPostInstallStep ();


    if (!defined $phase || $phase ne 'offline'){
        $rc = $self->_InstallPhaseOnline ($instconfig, $parser, $pp_handler);
    }

    # all done

    $progress_handler->StepFinished (0);
    $instconfig->setMsgLstContext([$self->getMsgLst()]);
    if ($rc && (!defined $phase || $instconfig->isLastPhase ($phase))){
        $instconfig->pers_remove();
    }

    if ($rc && defined $phase && !$instconfig->isLastPhase ($phase)){
        $instconfig->writeHdblcmStatusFile ();
    }
    return $rc;
}

sub _InstallPhaseOnline{
    my ($self, $instconfig, $parser, $pp_handler) = @_;

    # execute custom event postInstall
    if (!$self->{eventHandler}->triggerEvent ('postInstall', $self->getMsgLst(), $self->getErrMsgLst())){
        return undef;
    }

	# Run program after installation

	my $run = $instconfig->{params}->{RunProgramAfterInst}->{value};
	my $cmds = $parser->getRunProgramAfterInstallation();
	if ($run && defined $cmds && $cmds){
		$self->{runProgramAfterInstallation} = $cmds;
		# now the program will be run when DESTROY() is called
	}

	if (defined $pp_handler){
		$pp_handler->SetProgress ();
	}
    return 1;
}




#----------------------------------------------------------------------------

sub initEventHandler{
    my ($self) = @_;
    my $parser = $self->getConfigParser ();
    if (!defined $parser){
        $self->AddError ("No configParser defined");
        return undef;
    }
    my $handlerClass;
    my $customModules = $parser->getCustomModules ();
    if (defined $customModules){
        $handlerClass = $customModules->{EventHandler};
    }
    if (defined $handlerClass){
        local @INC = @INC;
        if ($self->isa ('SDB::Install::Kit')){
            push @INC, $self->getArchiveDir();
            push @INC, $self->getCustomModulesDir();
        }
        eval ("require $handlerClass;");
        if ($@){
            $self->AddError ("Cannot initialize custom event handler: $@");
            return undef;
        }
        $self->{eventHandler} = new {$handlerClass} ();
    }
    else{
        $self->{eventHandler} = new SDB::Install::InstallationEventHandler ();
    }
    return 1;
}


#----------------------------------------------------------------------------

sub setConfigParser{
    $_[0]->{configParser} = $_[1];
}

#----------------------------------------------------------------------------

sub getConfigParser{
    return $_[0]->{configParser};
}

#----------------------------------------------------------------------------

sub getEventHandler{
    return $_[0]->{eventHandler};
}

#----------------------------------------------------------------------------

sub getProductName{
    my ($self) = @_;
    if (!defined $self->{_product}){
        my $configparser = $self->getConfigParser ();
        if (! defined $configparser){
            return $self->SUPER::getProductName ();
        }
        my $product = $configparser->getProduct ();
        if (!defined $product){
            return $self->SUPER::getProductName ();
        }
        $self->{_product} = $product;
    }
    return $self->{_product}->{ProductName};
}

#----------------------------------------------------------------------------

sub getShortProductName{
    my ($self) = @_;
    if (!defined $self->{_product}){
        my $configparser = $self->getConfigParser ();
        if (!defined $configparser){
            return $self->SUPER::getShortProductName ();
        }
        my $product = $configparser->getProduct ();
        if (!defined $product){
            return $self->SUPER::getShortProductName ();
        }
        $self->{_product} = $product;
    }
    return $self->{_product}->{ShortProductName};
}

#----------------------------------------------------------------------------

sub getProductKey{
    my ($self) = @_;
    if (!defined $self->{_product}){
        my $configparser = $self->getConfigParser ();
        if (! defined $configparser){
            return undef;
        }
        my $product = $configparser->getProduct ();
        if (!defined $product){
            return undef;
        }
        $self->{_product} = $product;
    }
    return $self->{_product}->{ProductKey};
}

#----------------------------------------------------------------------------

sub getLicenseFile{
    my ($self) = @_;
    if (!defined $self->{_product}){
        my $configparser = $self->getConfigParser ();
        if (! defined $configparser){
            return undef;
        }
        my $product = $configparser->getProduct ();
        if (!defined $product){
            return undef;
        }
        $self->{_product} = $product;
    }
    if ($self->{_product}->{License}){
        return $self->{archive_dir} . $path_separator . $self->{_product}->{License};
    }
    return '';
}

#----------------------------------------------------------------------------

sub skipInstallationSteps {

	my ($self) = @_;
	my $parser = $self->getConfigParser();
	if (!$parser) {
		return $self->SUPER::skipInstallationSteps();
	}
	my $p2call = $parser->getP2Call();
    if (!defined $p2call || !%$p2call) {
		return $self->SUPER::skipInstallationSteps();
    }
	return 0;
}

#----------------------------------------------------------------------------

sub getCustomModulesDir{
    return $_[0]->getArchiveDir().$path_separator.'CustomModules';
}

#----------------------------------------------------------------------------

sub DESTROY{
	my ($self) = @_;
	$self->SUPER::DESTROY ();
	if (defined $self->{runProgramAfterInstallation}){
		my $cmd = $self->{runProgramAfterInstallation};
		print "Executing external program: $cmd->{ExecCommand}";
		foreach my $arg (@{$cmd->{ExecArg}}) {
			print ' ' . $arg;
		}
		print "\n";
		my $cfg = {};
		my $rc;
		my $msg;
		# open new block within which the environment may be changed
		{
			local %ENV = %ENV;
			if (defined $cmd->{ExecEnv} && %{$cmd->{ExecEnv}}) {
				foreach my $var (keys %{$cmd->{ExecEnv}}) {
					$ENV{$var} = $cmd->{ExecEnv}->{$var};
				}
			}
			if ($isWin) {
				$rc = ShellExecute ($cmd->{ExecCommand},$cmd->{ExecArg}, $cmd->{ExecDir});
			} else {
				$msg = $self->getMsgLst()->addMessage ("Starting external program");
				$msg->getSubMsgLst()->injectIntoConfigHash ($cfg);
				$rc = exec_program ($cmd->{ExecCommand},$cmd->{ExecArg},$cfg);
			}
		}
		if ((!defined $rc || $rc != 0) && !$cmd->{IgnoreError}){

			if (defined $msg) {
				print "External program failed: \n" . ${$msg->getSubMsgLst()->getMsgLstString ()} . "\n";
			}
			return undef;
		}
	}
}

1;
