package SDB::Install::Kit::GenericServerPlugin;

use SDB::Install::Kit::Generic;
use SDB::Install::System qw (makedir isSameFile);
use SAPDB::Install::System::Unix qw (lchown);
use SDB::Install::SysVars qw ($path_separator $isWin);
use SDB::Install::Globals qw ($gProductNameEngine);
use SDB::Install::Configuration qw ($bool_false_pattern $bool_true_pattern);

use strict;

our @ISA = qw (SDB::Install::Kit::Generic);

our $SQL_PREPARE_INITIALIZE = 'CALL SYS.AFLPM_ONLINE_REGISTRATION_INITIALIZE (?)';
our $SQL_PREPARE_FINALIZE   = 'CALL SYS.AFLPM_ONLINE_REGISTRATION_FINALIZE (?)';
our $SQL_CLEANUP            = 'CALL SYS.AFLPM_ONLINE_REGISTRATION_CLEANUP ()';
our $AUTO_DU_DIR_BASENAME = 'auto_content';

sub canInstall{
	my ($self, $system, $instconfig) = @_;
	my $kitVersion = $self->GetVersion();
	my $kitManifest = $self->getManifest();
	my $systemManifest = $system->getManifest();
	my $instance = $system->getNewDBInstances()->[0];
	my $plugin = $instance->getPluginInstallation ($self->getProductKey ());
	if (defined $plugin) {
		# upgrade
		if ($kitManifest->getVersion() < $plugin->GetVersion()) {
			$self->AddError ("Cannot update " . $self->getProductName() . ": newer version " . $plugin->GetVersion() .
			" already installed.");
			if ($instconfig->getIgnore('check_version')) {
				$self->AddMessage("Ignore error due to ignore option 'check_version'");
			} else {
				return undef;
			}
		} else {
			return 1;
		}
	} else {
		# new installation
		if (!$kitManifest->checkComponentDependency($systemManifest)) {
			$self->AddMessage ("Plugin " . $self->getProductName() . " " . $kitManifest->getVersion() .
				" is not compatible with $gProductNameEngine " . $systemManifest->getVersion() .
				" and will not be activated");
		}
		return 1;
	}
}

sub skipPluginInstallationSteps{
    my $self = shift;
    my $baseSkip = $self->skipInstallationSteps (@_);
    if (!$baseSkip){
        # package was installed
        # links have to be verified
        return 0;
    }
    my $instconfig = $self->getInstallation()->getConfiguration();
    my $system = $instconfig->getSystem();
    my $instance = $system->getNewDBInstances()->[0];
    my $plugin = $instance->getPluginInstallation ($self->getProductKey ());
    if ($self->checkComponentDependency($system)) {
        # check whether plugin is already activated
        if (defined $plugin){
            # check active plugin
            if (!isSameFile($plugin->GetInstallationPath(),$instconfig->get_InstallationPath())){
                # there is another version activated
                # => activate own version
                return 0;
            }
        }
        else{
            # there is no active plugin
            # => activation required
            return 0;
        }
    }
    else{
        # check whether plugin has already a '*_new' link
        # => activation required
        my $new_link = $instconfig->getPluginPath() . $path_separator . $self->getProductKey() . '_new';
        if (!-d $new_link){
            # there is no *_new link
            # => plugin preparation required
            return 0;
        }
        if (!isSameFile ($new_link,$instconfig->get_InstallationPath())){
            # *_new link belongs to another version
            # => plugin preparation required
            return 0;
        }
    }
    # all links are up-to-date
    return 1;
}


sub Install{
	my $self = $_[0];
	my $rc = shift->SUPER::Install (@_);
	if (!defined $rc) {
		return undef;
	}

    if ($self->skipPluginInstallationSteps()) {
        $self->AddProgressMessage ("Installation is up-to-date", undef);
        return $rc;
    }
    my $instconfig = $self->getInstallation()->getConfiguration();
	my $system = $instconfig->getSystem();
	if (!$isWin) {
		# Linux

		require SDB::Install::ServerVersionLinker;
		# create symbolic links
		my $linker = new SDB::Install::ServerVersionLinker($instconfig->getPluginPath(),
														undef,
														$self->getProductKey (),
														$self->getProductKey () . '_' . $self->getVersionIdString (),
														$system->getUID(),
														$system->getGID());

		if (!defined $linker->prepareNewVersion()) {
			$self->AddError(undef, $linker);
			return undef;
		}
		$self->AddMessage(undef, $linker);
	}

	# restart system if plugin is compatible with server
	if ($self->checkComponentDependency($system)) {
		if (!$self->activatePlugin ()){
			return undef;
	    }
    	## plugin installation/update : DU import
    	my $parser = $instconfig->{configparser};
    	my $doTheImport = $parser->getImportContent();
        if($doTheImport =~ /$bool_true_pattern/i) {
            if (!$instconfig->getValue ('NoStart')) {
            	my $system = $instconfig->getSystem();
        	    my $instance = $system->getNewDBInstances()->[0];
                my $auto_content_dir;
                if ($isWin) {
                    $auto_content_dir = $instance->get_pluginsDir() . $path_separator . $self->getProductKey() . $path_separator . $AUTO_DU_DIR_BASENAME;
                } else {
                    $auto_content_dir = $instconfig->get_InstallationPath() . $path_separator . $AUTO_DU_DIR_BASENAME;
                }
                if (!-d $auto_content_dir) {
                    $self->AddError ("Import of delivery units failed: auto content directory $auto_content_dir not found", $instance);
                    $self->AddMessage(undef, $instance);
                    return undef;
                }
                if($instance->isSecondarySystem()){
                    $self->AddMessage("Skipping import of delivery units because this is a secondary system");
                } elsif (!$instance->importDeliveryUnits($instconfig, $auto_content_dir, 1)) {
                    $self->AddError  ('Import of delivery units failed', $instance);
                    $self->AddMessage(undef, $instance);
                    return undef;
                }
                $self->AddMessage(undef, $instance);
            }
        }
	}
	else {
		$self->AddProgressMessage("Server version not compatible with plugin, skipping activation of plugin");
	}

	return 1;
}


sub callDBProcWithParam{
    my ($self, $sql, $stmnt, $param) = @_;
    my $sqlOk = 1;
    if (!defined $sql->prepare ($stmnt)){
        $self->setErrorMessage ("Prepare '$stmnt' failed", $sql->getErrMsgLst ());
        $sqlOk = 0;
    }
    else{
        $sql->bindParam (1, $param);
    }
    if ($sqlOk){
        my $sql_execute = $stmnt;
        $sql_execute =~ s/\?/'$param'/;
        if (!defined $sql->execute ()){
            $self->setErrorMessage ("Execution of '$sql_execute' failed", $sql->getErrMsgLst ());
            $sqlOk = 0;
        }
    }
    return $sqlOk;
}


sub activatePlugin{
	my ($self) = @_;
	$self->getInstallation()->CloseRegistry(); # prevent permission denied error on Windows when renaming plugin dir
	my $instconfig = $self->getInstallation()->getConfiguration();

    my $msg;
    my $system = $instconfig->getSystem();
    my ($instance, $user, $password);
    my $keyName = $self->getProductKey ();
    my $onlineActivationSql = $instconfig->getOnlineActivationSQLConnection ();
    my $onlineActivatioOK = 1;
    my ($ssoCert, $useHttps);
    if (defined $onlineActivationSql){
        if ($instconfig->isUpdate ()){
            my $msg = $self->getMsgLst ()->addProgressMessage ("Initializing ONLINE REGISTRATION");
            $onlineActivationSql->setMsgLstContext ([$msg->getSubMsgLst ()]);
            $onlineActivatioOK = $self->callDBProcWithParam ($onlineActivationSql, $SQL_PREPARE_INITIALIZE, $keyName);
        }
    }
    elsif (!$instconfig->getValue('SystemIsOffline')){
        # stop system
        $instance = $system->getNewDBInstances()->[0];
        $user = $system->getUser()->getSidAdm()->getname ();
        $password = $instconfig->getValue ('Password');
        $msg = $self->getMsgLst()->addProgressMessage ("Stopping system...");
        $instance->setMsgLstContext ([$msg->getSubMsgLst (), $self->getErrMsgLst ()]);
        $useHttps = $instconfig->isUseHttps ();
        $ssoCert = $instconfig->getValue ('SSOCertificate');
        if (!defined $instance->stopAllHosts($user,
                                             $password,
                                             0, # ignore not running nodes
                                             $instconfig->getTimeout ('stop_instance'),
                                             $useHttps,
                                             $ssoCert)) {
            $msg->endMessage (0, 'Stop system');
            return undef;
        }
        $msg->endMessage (0, 'Stop system');
    }

	# activate plugins
	$msg = $self->AddProgressMessage ("Activating plugin...");
	$self->SetFormatInfo ($msg, 'h1', 'Activate plugin');
    $system->setMsgLstContext ([$msg->getSubMsgLst (), $self->getErrMsgLst ()]);
	if (!defined $system->activateServerPlugins($instconfig->getValue('CustomPluginsPath'))) {
		return undef;
    }

    if (defined $onlineActivationSql){

        if ($onlineActivatioOK ){
            my $msg = $self->getMsgLst ()->addProgressMessage ("Finalizing ONLINE REGISTRATION");
            $onlineActivationSql->setMsgLstContext ([$msg->getSubMsgLst ()]);
            $onlineActivatioOK = $self->callDBProcWithParam ($onlineActivationSql, $SQL_PREPARE_FINALIZE, $keyName);
        }

        if (!$onlineActivatioOK){
            my $msg = $self->getMsgLst ()->addProgressMessage ("Cleaning up failed ONLINE REGISTRATION");
            $onlineActivationSql->setMsgLstContext ([$msg->getSubMsgLst ()]);
            if (!defined $onlineActivationSql->execute ($SQL_CLEANUP)){
                 $self->appendErrorMessage ("Execution of '$SQL_CLEANUP' failed", $onlineActivationSql->getErrMsgLst ());
            }
            $self->appendErrorMessage ("Plugin activation in online mode failed.");
            $self->appendErrorMessage ("Please restart HANA Database manually to activate the plugin!");
            return undef;
        }
    }
    else{
        # start system
        if (!$instconfig->getValue ('NoStart')) {
            $msg = $self->getMsgLst()->addProgressMessage ("Starting system...");
            $instance->setMsgLstContext ([$msg->getSubMsgLst (), $self->getErrMsgLst ()]);
            if (!defined $instance->startAllHosts($user,
                                                  $password,
                                                  $instconfig->getTimeout ('start_instance'),
                                                  $useHttps,
                                                  $ssoCert)) {
                $msg->endMessage (0, 'Start system');
                return undef;
            }
            $msg->endMessage (0, 'Start system');
        }
    }
    return 1;
}


sub createLink{
	my ($self, $lnksrc, $lnktgt) = @_;
    my $instconfig = $self->getInstallation()->getConfiguration();
	my $system = $instconfig->getSystem();

	if (-l $lnksrc){
		unlink ($lnksrc);
	}
	if (!symlink ($lnktgt, $lnksrc)){
		$self->AddError ("Cannot create symbolic link $lnksrc -> $lnktgt: $!");
		return undef;
	}
	if (!lchown ($system->getUID, $system->getGID, $lnksrc)) {
		$self->AddError ("Cannot set owner and group of symbolic link $lnksrc: $!");
		return undef;
	}
	return 1;
}

sub createDir{
	my ($self, $dir) = @_;
    my $instconfig = $self->getInstallation()->getConfiguration();
	my $system = $instconfig->getSystem();
	my $cfg = {};

	if (! -d $dir && !defined makedir ($dir, $cfg)){
		$self->AddError ("Cannot create directory '$dir'", $cfg);
		return undef;
	}
	if (!chown ($system->getUID, $system->getGID, $dir)) {
		$self->AddError ("Cannot set owner and group of directory $dir: $!");
		return undef;
	}
	return 1;
}

1;
