package SDB::Install::LayeredConfig::IniFile;

use SDB::Install::BaseLegacy;
use SDB::Install::SysVars qw ($path_separator $isWin);
use SDB::Install::IniFile;
use SDB::Install::Tools qw (trim ltrim rtrim);
use strict;

use Exporter;

our @ISA = qw (SDB::Install::BaseLegacy Exporter);


our (
	$CFG_LAYER_DEFAULT,
	$CFG_LAYER_SYSTEM,
	$CFG_LAYER_TENANT,
	$CFG_LAYER_HOST,
	$CFG_IS_LAYERED) = (1,2,4,8,16);


our @CFG_LAYER_NAMES = qw(
	Default
	System
	Tenant
	Host
);

our @EXPORT = qw (
	$CFG_LAYER_DEFAULT
	$CFG_LAYER_SYSTEM
	$CFG_LAYER_TENANT
	$CFG_LAYER_HOST
	$CFG_IS_LAYERED);

sub new {
	my $self = shift->SUPER::new ();
	($self->{name}, my $data,$self->{layer},$self->{uid},$self->{gid}) = @_;
	$self->{mask} = $data->[0];
	$self->{metadata} = $data->[1];
	return $self;
}


sub getIndexByFlag{
	my ($flag) = @_;
	my $val = 1;
	my $index = 0;
	while ($val <= $flag){
		if ($flag == $val){
			return $index;
		}
		$val <<= 1;
		$index++;
	}
	return -1;
}


sub isLayered{
	return ($_[0]->{mask} & $CFG_IS_LAYERED);
}

sub readValues{
	my ($self, $reload) = @_;
	
	if (defined $self->{data} && !$reload){
		return $self->{data};
	}
	$self->{file_buffer} = [];
	
	my $mask = $self->{mask};
	my ($lineNr, $line, $section, $key, $value);
	my %data;
	$self->{data} = \%data;
	if ($self->{mask} & $CFG_IS_LAYERED){
		my $flag = 1;
		my $file;
		foreach my $i (0..$#CFG_LAYER_NAMES){
			my @file_buffer;
			if ($mask & $flag && defined $self->{layer}->[$i]){
				$file = $self->{layer}->[$i] . $path_separator . $self->{name};
				if (-f $file){
					$self->AddMessage ("Layer '$CFG_LAYER_NAMES[$i]' found");
					if (!open (FD, $file)){
						$self->AddError ("Cannot open file '$file': $!");
						return undef;
					}
					@file_buffer = <FD>;
					close (FD);
					$self->{file_buffer}->[$i] = \@file_buffer;
					
					foreach $lineNr (0..$#file_buffer){
						$line = $file_buffer[$lineNr];
						chomp ($line);
						if ($line =~/^\s*#/){
							next;
						}
						trim (\$line);
						if ($line =~/^\[.*\]$/){
							($section) = ($line =~/^\[(.*)\]$/);
							if (!defined $data{$section}){
								$data{$section} = {};
							}
							next;
						}
						if ($line =~ /\S.*=/){
							($key,$value) = ($line =~ /^([^=]+)=(.*)/);
							rtrim (\$key);
							ltrim (\$value);
							if ($key){
								if (!defined $data{$section}->{$key}){
									$data{$section}->{$key} = [];
								}
								$data{$section}->{$key}->[$i] = $value;
								
							}
						}
					}
				
				}
				else{
					$self->AddMessage ("Layer '$CFG_LAYER_NAMES[$i]' not found");
				}
			}
			$flag <<= 1;
		}
	}
    return $self->{data};
}

sub getValue{
	my ($self,$section, $key, $rlayer) = @_;
    my $layer;
    if (ref ($rlayer) eq 'SCALAR'){
        $layer = $$rlayer;
    }
    else{
        $layer = $rlayer;
    }

	if (!defined $self->{data}->{$section}){
		$self->AddError ("Section '$section' is not known in '$self->{name}'");
		return undef;
	}
	if (!defined $self->{data}->{$section}->{$key}){
		$self->AddError ("Value '$key' is not known in section '$section' in '$self->{name}'");
		return undef;
	}
	if (!($self->{mask} & $CFG_IS_LAYERED)){
		if (defined $layer){
			$self->AddWarning ("Inifile '$self->{name}' is not layered => ignoring layer parameter");
		}
		return $self->{data}->{$section}->{$key};
	}
	
	if (defined $layer){
		if (!($layer & $self->{mask})){
			$self->AddError ('Invalid layer');
			return undef;
		}
		my $index = getIndexByFlag ($layer);
		return $self->{data}->{$section}->{$key}->[$index];
	}
	my $flag = 8;
	foreach my $index (0..$#CFG_LAYER_NAMES){
		$index = $#CFG_LAYER_NAMES - $index;
		if ($flag & $self->{mask}){
			if (defined $self->{data}->{$section}->{$key}->[$index]){
				$self->AddMessage ("Found value '$section/$key' = '$self->{data}->{$section}->{$key}->[$index]' in layer '$CFG_LAYER_NAMES[$index]'");
                if (defined $rlayer && ref ($rlayer) eq 'SCALAR'){
                    $$rlayer = 1 << $index;
                }
				return $self->{data}->{$section}->{$key}->[$index];
			}
		}
		$flag >>= 1;
	}
	return undef;
}

sub existsValue{
    my ($self, $section, $key) = @_;
    if (defined $self->{data} && defined $self->{data}->{$section} &&
        defined $self->{data}->{$section}->{$key}){
        return 1;
    }
    return 0;
}

sub setValue{
	my ($self,$layer, $section, $key, $value, $comment) = @_;
	
	my $index = getIndexByFlag ($layer);
	
	if (!($layer & $self->{mask})){
		$self->AddError ("Ini file '$self->{name}' doesn't support layer '$CFG_LAYER_NAMES[$index]'");
		return undef;
	}
	
	if ($layer == $CFG_LAYER_DEFAULT){
		$self->AddError ("Cannot set value ($section/$key) in layer '$CFG_LAYER_NAMES[$index]' : layer is read only");
		return undef;
	}
	my $file = $self->{layer}->[$index] . $path_separator . $self->{name};
	if (!defined $self->{inis}){
		$self->{inis} = [];
	}
	
	if (!defined $self->{inis}->[$index]){
		if (!-f $file){
			$self->AddMessage ("Creating ini file '$file'");
			if (!open (FD, '>'.$file)){
				$self->AddError ("Cannot create inifile '$file': $!");
				return undef;
			}
			close (FD);
		}
		
		if (!$isWin){
			if ($> == 0 && defined $self->{uid} && defined $self->{gid}){
				chown ($self->{uid}, $self->{gid},$file);
			}
			chmod (0640,$file);
		}
		
		$self->{inis}->[$index] = new SDB::Install::IniFile ($file);
		if ($self->{inis}->[$index]->ErrorState ()){
			$self->AddError ("Cannot read inifile", $self->{inis}->[$index]);
			return undef;
		}
	}
	$self->{inis}->[$index]->setMsgLstContext ([$self->getMsgLst()]);
	if (!defined $self->{inis}->[$index]->setValue ($section, $key, $value, $comment)){
		$self->AddError ("Cannot set value '$section/$key'", $self->{inis}->[$index]);
		return undef;
	}
	$self->{data}->{$section}->{$key}->[$index] = $value;
	return 1;
}

sub removeKey{
    my ($self,$layer, $section, $key) = @_;

    my $index = getIndexByFlag ($layer);

    if (!($layer & $self->{mask})){
        $self->AddError ("Ini file '$self->{name}' doesn't support layer '$CFG_LAYER_NAMES[$index]'");
        return undef;
    }

    if ($layer == $CFG_LAYER_DEFAULT){
        $self->AddError ("Cannot remove key ($section/$key) in layer '$CFG_LAYER_NAMES[$index]' : layer is read only");
        return undef;
    }
    my $file = $self->{layer}->[$index] . $path_separator . $self->{name};
    if (!defined $self->{inis}){
        $self->{inis} = [];
    }

    if (!defined $self->{inis}->[$index]){
        if (!-f $file){
            return 1;
        }
        $self->{inis}->[$index] = new SDB::Install::IniFile ($file);
        if ($self->{inis}->[$index]->ErrorState ()){
            $self->AddError ("Cannot read inifile", $self->{inis}->[$index]);
            return undef;
        }
    }
    $self->{inis}->[$index]->setMsgLstContext ([$self->getMsgLst()]);
    if (!defined $self->{inis}->[$index]->getValue($section,$key)){
        return 1;
    }
    if (!defined $self->{inis}->[$index]->removeKey ($section, $key)){
        $self->AddError ("Cannot remove key '$section/$key'", $self->{inis}->[$index]);
        return undef;
    }
    return 1;
}


sub importIniFile{
    my ($self, $layer, $iniFile) = @_;
    if (!defined $iniFile || !$iniFile->isa ('SDB::Install::IniFile')){
        $self->AddError ("No valid inifile obj");
        return undef;
    }
    my ($key,$value);
    foreach my $section (sort @{$iniFile->getSections()}){
        foreach $key (sort @{$iniFile->getKeys($section)}){
            $self->setValue ($layer, $section, $key, $iniFile->getValue($section, $key))
        }
    }
    return 1;
}

sub getSectionKeys{
    my ($self,$section) = @_;
    if (defined $self->{data} && defined $self->{data}->{$section}){
        return [keys %{$self->{data}->{$section}}];
    }
    return undef;
}

sub getSections{
    if (defined $_[0]->{data}){
        return [keys %{$_[0]->{data}}];
    }
    return undef;
}

sub setOwner{
    my $self = shift;
    ($self->{uid}, $self->{gid}) = @_;
    return 1;
}

sub write{
	my ($self) = @_;
	my $rc;
	my $isUpToDate = 1;
	my $ini;
	foreach my $index (1..$#CFG_LAYER_NAMES){
		$ini = $self->{inis}->[$index];
		if (defined $ini){
			$ini->setMsgLstContext ([$self->getMsgLst ()]);
			$rc = $ini->write ();
			if (!defined $rc){
				$self->setErrorMessage ("Cannot flush $self->{name}", $ini->getErrMsgLst ());
				return undef;
			}
			if ($rc != 2){
				$isUpToDate = 0;
			}
		}
	}
	if (!$isUpToDate){
		$self->readValues (1);
	}
	return 1;
}

1;
