#!/usr/bin/perl
#

package SDB::Install::Log;
use SAPDB::Install::Hostname qw(hostname);
use SDB::Install::System;
use SDB::Install::SysVars qw($path_separator $isWin);
use SDB::Install::DebugUtilities qw(dumpThings);
use SDB::Install::Globals qw ($gLogTime $gSapsysGroupName $gTmpDir  );
use SDB::Install::Tools qw (getFilenameTimeStamp getFilenameTimeStampPattern);
use Exporter;
use strict;

our @ISA = qw (Exporter);


our @EXPORT = qw (
    LOG_FORMAT_PLAIN
    LOG_FORMAT_MSGLIST
    LOG_HISTSIZE_UNLIMITED
    LOG_INDICATE_LOCATION_ON_STDOUT
    LOG_DONT_INDICATE_LOCATION_ON_STDOUT
    LOG_DESTINATION_TRY_ONLY
    addLogFileLocation
);

#----------------------------------------------------------------------------

sub new{
    my (
       $type,
       $path
    ) = @_;
    my $self = bless ({}, $type); 
    $self->{suppress} = 0;
    $self->{'data'} = [];
    $self->{'indicateLocationOnStdout'} = 1;
    return $self;
}

#----------------------------------------------------------------------------

our $LOG_FORMAT_PLAIN     = 2;
our $LOG_FORMAT_MSGLIST   = 3;

our $LOG_HISTSIZE_UNLIMITED = 0;

our $LOG_DONT_INDICATE_LOCATION_ON_STDOUT = 0;
our $LOG_INDICATE_LOCATION_ON_STDOUT      = 1;

our $LOG_DESTINATION_TRY_ONLY      = 1;


sub LOG_FORMAT_PLAIN {return $LOG_FORMAT_PLAIN};
sub LOG_FORMAT_MSGLIST {return $LOG_FORMAT_MSGLIST};

sub LOG_HISTSIZE_UNLIMITED {return $LOG_HISTSIZE_UNLIMITED};

sub LOG_DONT_INDICATE_LOCATION_ON_STDOUT {return $LOG_DONT_INDICATE_LOCATION_ON_STDOUT};
sub LOG_INDICATE_LOCATION_ON_STDOUT {return $LOG_INDICATE_LOCATION_ON_STDOUT};

sub LOG_DESTINATION_TRY_ONLY {return $LOG_DESTINATION_TRY_ONLY};

#----------------------------------------------------------------------------

=example
$self->{'data'} =
[
    {
        'format'   => LOG_FORMAT_PLAIN,
        'path'     => '/some/path',
        'histsize' => 1,
        'name'     => 'install.log',
        'indicateLocationOnStdout' => LOG_DONT_INDICATE_LOCATION_ON_STDOUT
    },
    {
        'format'   => LOG_FORMAT_MSGLIST,
        'path'     => '/some/path',
        'histsize' => 1,
        'name'     => 'install.msg',
        'indicateLocationOnStdout' => LOG_DONT_INDICATE_LOCATION_ON_STDOUT
    },
    {
        # trace directory, for access via NewDB Studio
        'format'   => LOG_FORMAT_PLAIN,
        'path'     => '/some/trc/path',
        'histsize' => 3,
        'name'     => 'hdbinst_LOGDATETIME.log',
        'indicateLocationOnStdout' => LOG_DONT_INDICATE_LOCATION_ON_STDOUT
    },
    {
        'format'   => LOG_FORMAT_PLAIN,
        'path'     => '/some/trc/path',
        'histsize' => 3,
        'name'     => 'hdbremovehost_LOGDATETIME.log',
        'indicateLocationOnStdout' => LOG_DONT_INDICATE_LOCATION_ON_STDOUT,
        'tryOnly' => LOG_DESTINATION_TRY_ONLY
    }
];
=cut
sub addLogFileLocation {
    my (
        $self,
        $format,
        $path,
        $histsize,        # up to 'histsize' many logfiles are kept
                          # i.e. the top 'histsize', according to descending lexicographic 
                          # order. such an order can be established by use
                          # of macros (currently only 'LOGDATETIME') in the filenames.
        $name,
        $indicateLocationOnStdout,
        $uid,             # optional
        $gid,             # optional
        $tryOnly,         # no error when logfile destination does not exist, optional
        $slaveCopy,       # hash map: how to copy slave log files
        $copyDir,         # if specified, this log directory is copied to $copyDir
    ) = @_;

    if (!$isWin && !defined $gid) {
        $gid = getgrnam ($gSapsysGroupName);
    }

    my $location = {
        'format'                   => $format,
        'path'                     => $path,
        'histsize'                 => $histsize,
        'name'                     => $name,
        'indicateLocationOnStdout' => $indicateLocationOnStdout,
    };

    $location->{'uid'}       = $uid       if (defined $uid);
    $location->{'gid'}       = $gid       if (defined $gid);
    $location->{'tryOnly'}   = $tryOnly   if (defined $tryOnly);
    $location->{'slaveCopy'} = $slaveCopy if (defined $slaveCopy);
    $location->{'copyDir'}   = $copyDir   if (defined $copyDir);
    push(@{$self->{'data'}}, $location);
}


#-------------------------------------------------------------------------------
# Copies all files except '*.msg' from $location->{path} to $location->{copyDir}

sub copyAllLogFiles {

    my ($self, $location) = @_;

    my $sourceDir = $location->{path};
    my $destinDir = $location->{copyDir};

    if (!defined $sourceDir || !-d $sourceDir || !defined $destinDir) {
        return 1;
    }

    my $rc = 1;
    my $original_umask;

    if (!$isWin){
        # user: 0 (rwx), group: 3 (r), other: 7 (none)  ==>  rwxr-----
        $original_umask = umask (037);
        if ($original_umask == 037){
            undef $original_umask;
        }
    }

    if (!-d $destinDir) {

        my $config     = {};
        $config->{uid} = $location->{uid} if (defined $location->{uid});
        $config->{gid} = $location->{gid} if (defined $location->{gid});
        $rc = makedir($destinDir, $config);
        if (!$rc) {
            $self->getMsgLst()->addMessage
                              ("Cannot create directory '$destinDir'", $config);
        }
    }

    if ($rc) {
        $rc = copySelectedFiles($sourceDir,
                               $destinDir,
                               $self->getMsgLst(),
                               undef,            # newPostfix
                               undef,            # oldFilenames
                               $location->{uid},
                               $location->{gid},
                               undef,            # hostname
                               undef,            # pattern
                               '.msg$');         # skipPattern
    }

    if ($rc
        && ($location->{'indicateLocationOnStdout'} == $LOG_INDICATE_LOCATION_ON_STDOUT)
        && (    $self->{'indicateLocationOnStdout'} == $LOG_INDICATE_LOCATION_ON_STDOUT)) {
        print "Log files copied to '$destinDir' on host '" . hostname() . "'.\n";
    }

    if (defined $original_umask){
        umask ($original_umask);
    }

    return $rc;
}


#----------------------------------------------------------------------------

sub copyLogFilesFromSlaveHosts {

    my ($self, $location) = @_;

    my $instanceDir = $location->{slaveCopy}->{instanceDir};
    my $destinDir   = $location->{slaveCopy}->{destinDir};

    if (!defined $destinDir   || !-d $destinDir   ||
        !defined $instanceDir || !-d $instanceDir) {
        return 1;
    }

    my $rc = 1;
    my $slaveHosts = $location->{slaveCopy}->{hostNames};
    my $uid        = $location->{slaveCopy}->{uid};
    my $gid        = $location->{slaveCopy}->{gid};
    my $actionID   = $location->{slaveCopy}->{actionID};
    my $pattern    = (defined $actionID) ? '_' . $actionID : undef;
    my $original_umask;

    if (!$isWin){
        # user: 0 (rwx), group: 3 (r), other: 7 (none)  ==>  rwxr-----
        $original_umask = umask (037);
        if ($original_umask == 037){
            undef $original_umask;
        }
    }

    foreach my $currHost (@$slaveHosts) {

        my $currDir   = join($path_separator, $instanceDir, $currHost, 'trace');
        my $domainIdx = index ($currHost, '.', 0);
        my $hostPrefix= ($domainIdx > 0) ? substr($currHost, 0, $domainIdx)
                                         : $currHost;

        my $rcCopy = copySelectedFiles($currDir,
                                       $destinDir,
                                       $self->getMsgLst(),
                                       undef,  # newPostfix
                                       undef,  # oldFilenames
                                       $uid,
                                       $gid,
                                       $hostPrefix,
                                       $pattern);
        $rc = $rcCopy if (!$rcCopy);
    }

    if (defined $original_umask){
        umask ($original_umask);
    }

    return $rc;
}

#----------------------------------------------------------------------------

sub removeLogFileLocation {
	my ($self, $path) = @_;
	return if(!defined($self->{data}));
	my @newDataArray = grep {$_->{path} ne $path} @{$self->{data}};
	$self->{data} = \@newDataArray;
}

#----------------------------------------------------------------------------

sub setIndicateLocationOnStdout {
    my (
        $self,
        $doIt
    ) = @_;
    $self->{'indicateLocationOnStdout'} = $doIt;
}

#----------------------------------------------------------------------------

sub Suppress{
    shift->{suppress} = defined $_[1] ? $_[1] : 1;
}

#----------------------------------------------------------------------------

sub setMsgLst {
    $_[0]->{msglst} = $_[1];
}

sub getMsgLst {
    return $_[0]->{msglst};
}

#----------------------------------------------------------------------------

sub writeLogs{
    my ($self) = @_;
    my $retval = undef;
    if (!defined $self->{msglst}){
        return 1;
    }
    if ($self->{'already_written'} || $self->{suppress}){
        $retval = 1;
    }
    else {
        $self->{'with_timestamp'} = 1;
        my $logWritten  = 0;
        my $plainLogBuf = $self->{msglst}->getMsgLstString (undef, 1, 1);
        my $xmlLogBuf = $self->{msglst}->getMsgLstXML(1);
        foreach my $location (@{$self->{'data'}}) {
            my $rc = $self->writeLog($location, $plainLogBuf, $xmlLogBuf);
            if($location->{'indicateLocationOnStdout'} == $LOG_INDICATE_LOCATION_ON_STDOUT && 
               $self->{'indicateLocationOnStdout'} == $LOG_INDICATE_LOCATION_ON_STDOUT &&
               $rc) {
                my $stdoutMsg = "Log file written to '$location->{'path'}$path_separator$location->{'expandedName'}'"
                              . " on host '" . hostname() . "'.";
                print $stdoutMsg."\n";
            }
            $logWritten = 1 if ($rc);
            if (defined $location->{'slaveCopy'}) {
                $self->copyLogFilesFromSlaveHosts($location);
            }
            if (defined $location->{'copyDir'}) {
                $self->copyAllLogFiles($location);
            }
        }
        LCM::DevelopmentTrace::RemoveTempDevelopmentTrace() if ($logWritten);
        $self->{'already_written'} = 1;
        $retval = 1;
    }
    return $retval;
}

#----------------------------------------------------------------------------

sub writeLog{
    my (
        $self,
        $location,
        $plainLogBuf,
        $xmlLogBuf
    ) = @_;
    if(defined $location->{'tryOnly'} && $location->{'tryOnly'} == $LOG_DESTINATION_TRY_ONLY && not -d $location->{'path'}) {
        return 1;
    }
    $location->{'expandedName'} = $location->{'name'};
    $location->{'namePattern'} = $location->{'name'};
    # currently, LOGDATETIME is the only possible macro in the
    # location name. if there are more to come, we have to do a
    # systematic iteration here:
    if($location->{'name'} =~ /LOGDATETIME/) {
        my $datetime = getFilenameTimeStamp($gLogTime);
        my $pattern = getFilenameTimeStampPattern();
        $location->{'expandedName'} =~ s/LOGDATETIME/$datetime/;
        $location->{'namePattern'} =~ s/LOGDATETIME/$pattern/;
    }
    my $filename = $location->{'path'}.$path_separator.$location->{'expandedName'};
    my $uid = $>;
    if (!$isWin && $uid && -f $filename && ! -w _ ){
        my $name      = $location->{'expandedName'};
        my $extension = '';
        if ($name eq $location->{'name'}) {
            my $idx = rindex($name, '.');
            if ($idx > 0) {
                $extension = substr($name, $idx);
                $name      = substr($name, 0, $idx);
            }
        }
        $filename = $location->{'path'} . $path_separator
                  . $name . '_' .  $uid . $extension;
    }

    my $original_umask;

    if (!$isWin){
        # user: 0 (rwx), group: 3 (r), other: 7 (none)  ==>  rwxr-----
        $original_umask = umask (037);
        if ($original_umask == 037){
            undef $original_umask;
        }
    }

    my $rc = open (LOG,'>'.$filename);

    if (defined $original_umask){
        umask ($original_umask);
    }

    if (!$rc){
        print STDERR "Could not create logfile '${filename}': $!\n";
        return undef;
    }
    
    if (!$isWin && ($> == 0)                               # $> - effective UID
        && (defined $location->{uid} || defined $location->{gid})) {

        my $newUID = (defined $location->{uid}) ?  $location->{uid} : -1;
        my $newGID = (defined $location->{gid}) ?  $location->{gid} : -1;
        if($location->{'path'} ne $gTmpDir){
            chown ($newUID, $newGID, $location->{'path'});
        }
        chown ($newUID, $newGID, $filename);
    }
    my $printResult;
    if($location->{'format'} == $LOG_FORMAT_PLAIN) {
        $printResult = print LOG $$plainLogBuf;
    }
    elsif($location->{'format'} == $LOG_FORMAT_MSGLIST) {
        $printResult = print LOG $$xmlLogBuf;
    }

    if (!$printResult){
        print STDERR "Cannot write to file '$filename': $!\n";
    }

    close LOG;
    if($location->{'histsize'} > 1) {
        # delete oldest logfiles:
        if(!opendir(DH, $location->{'path'})) {
            print STDERR "Could not open directory '$location->{'path'}': $!\n";
            return undef;
        }
        my @content = readdir(DH);
        closedir(DH);
        my @installerLogfiles;
        foreach my $entry (@content) {
            if($entry =~ /$location->{'namePattern'}/) {
                push @installerLogfiles, $entry;
            }
        }
        @installerLogfiles = reverse(sort(@installerLogfiles));
        for(my $i = $location->{'histsize'}; $i < $#installerLogfiles+1; $i++) {
            unlink($location->{'path'}.$path_separator.$installerLogfiles[$i]);
        }
    }
    return 1;
}

#----------------------------------------------------------------------------

sub DESTROY{
    my ($self) = @_;

    return if ${^GLOBAL_PHASE} eq 'DESTRUCT';
    unless ($self->{suppress}){
        if (not defined $self->writeLogs ()){
            print STDERR ($self->GetErrorString ()) . "\n";
        }
    }
}

#----------------------------------------------------------------------------

1;
