package SDB::Install::Manifest;

use base SDB::Install::BaseLegacy;
use strict;
use SDB::Install::ComponentVersion;
use SDB::Install::SysVars qw($currentPlatformName);
use SDB::Install::Version;
use SDB::Install::Tools qw (trim);
use SDB::Install::Globals qw ($gFlavourPlatform $gFlavourExpress $gFlavourCockpit);
use SDB::Install::Globals qw ($gKeynameEngine
                              $gKeynameClient
                              $gKeynameClientNoCrypto
                              $gKeynameStudio
                              $gKeynameAFL
                              $gKeynameLCA
                              $gKeynameHLM
                              $gKeynameSDA
                              $gKeynameStreaming
                              $gKeynameEs
                              $gKeynameAccelerator
                              $gKeynameRDSync
                              $gKeynameStreaming
                              $gKeynameInstaller
                              $gKeynameLMStructure
                              $gKeynameCockpitStack
                              $gKeynameXS
                              $gKeynameXS2
                              $gKeynameOfficeClient
                              $gKeynameLSS);

# Multi-value manifest keys might be contained more than once in a single file.
# Although the key is same, different <key,value> pairs should be unique
my %MULTI_VALUE_MANIFEST_KEYS = (
	'upgrade-restriction' => 1,
	'required-components' => 1,
	'restrict-supported-components' => 1,
);

sub new{
	my $self = shift->SUPER::new ();
	($self->{file}) = @_;
	if (defined $self->{file} && -f $self->{file}){
		$self->read ();
	}
	return $self;
}

sub read{
	my ($self, $file, $no_cache) = @_;
	if ((!defined $no_cache || !$no_cache) && defined $self->{data}){
		return $self->{data};	
	}	

	if (!defined $file){
		$file = $self->{file};
		if (!defined $file){
			$self->AddError ("manifest file is not defined");
			return undef;
		}
	}
	
	$self->{file} = $file;
	
	if (!open (FD, $file)){
		$self->AddError ("Cannot open file '$file': $!");
		return undef;
	}
	my ($key, $value);
	my %data;
	while (my $line = <FD>){
		if ($line =~ /^\s*#/){
			next;
		}
		($key, $value) = ($line =~ /^\s*([^:]+?)\s*:\s*(.*?)\s*$/);

		next if (! $key);

		if ($key eq 'compilebranch' && defined $ENV{HDB_COMPILEBRANCH}){
			$value = $ENV{HDB_COMPILEBRANCH};
		}
		if($MULTI_VALUE_MANIFEST_KEYS{$key}){
# Using autovivification here. if(! exists($data{$key})) -> perl will self execute $data{$key} = [];
			push(@{$data{$key}}, $value);
			next;
		}
		$data{$key} = $value;
	}
	close (FD);
	$self->{data} = \%data;

    if (defined $data{keyname} && $data{keyname} eq $gKeynameCockpitStack){
        require SDB::Install::Manifest::OneOrTwoDigitReleaseManifest;
        bless ($self, 'SDB::Install::Manifest::OneOrTwoDigitReleaseManifest');
    }
	return $self->{data};
}

sub _isComponent {
	my ($self, $componentKeyname) = @_;
	return undef if (!defined $self->{data});
	my $keyname = $self->getKeyname();
	return defined $keyname && $keyname eq $componentKeyname;
}

sub isStudio{
	my ($self) = @_;
	return $self->_isComponent($gKeynameStudio);
}

sub isStudioWithEmbededJvm{
    if (!$_[0]->isStudio()){
        return undef
    }
    return substr( $_[0]->getVersion(), 0, 1) >= 2 ;
}

sub isServer{
	my ($self) = @_;
	return $self->_isComponent($gKeynameEngine);
}

sub getHANAFlavour {
    my ($self) = @_;

    return undef unless $self->isServer();

    return $gFlavourExpress if defined $self->{data}->{'subkeyname'} && $self->{data}->{'subkeyname'} eq $gFlavourExpress ||
        defined $self->{data}->{'keycaption'} && $self->{data}->{'keycaption'} eq 'SAP HANA DATABASE EXPRESS';
    return $gFlavourCockpit if defined $self->{data}->{'subkeyname'} && $self->{data}->{'subkeyname'} eq $gFlavourCockpit;

    return $gFlavourPlatform;
}

sub isHanaExpressOrExpressForCockpit {
    my ($self) = @_;
    if($self->isServer()) {
        my $subkeyname = $self->{'data'}->{'subkeyname'};
        if(defined $subkeyname) {
            if($subkeyname eq $gFlavourExpress || $subkeyname eq $gFlavourCockpit) {
                return 1;
            }
        }
        my $keycaption = $self->{'data'}->{'keycaption'};
        if(defined $keycaption) {
            if($keycaption eq 'SAP HANA DATABASE EXPRESS') {
                return 1;
            }
        }
    }
    return 0;
}

sub isAFL{
	my ($self) = @_;
	return $self->_isComponent($gKeynameAFL);
}

sub isLCAPPS{
	my ($self) = @_;
	return $self->_isComponent($gKeynameLCA);
}

sub isHLM{
	my ($self) = @_;
	return $self->_isComponent($gKeynameHLM);
}

sub isCockpitStack{
	my ($self) = @_;
	return $self->_isComponent($gKeynameCockpitStack);
}

sub isReferenceData {
	my $referenceData = $_[0]->{data}->{'reference-data'};
	return ( defined $referenceData && $referenceData eq '1' ) ? 1 : 0;
}

sub isES {
	my ($self) = @_;
	return $self->_isComponent($gKeynameEs);
}

sub isSDA {
	my ($self) = @_;
	return $self->_isComponent($gKeynameSDA);
}

sub isInstaller {
	my ($self) = @_;
	return $self->_isComponent($gKeynameInstaller);
}

sub isLMStructure {
	my ($self) = @_;
	return $self->_isComponent($gKeynameLMStructure);
}

sub isStreaming {
	my ($self) = @_;
	return $self->_isComponent($gKeynameStreaming);
}

sub isAccelerator {
	my ($self) = @_;
	return $self->_isComponent($gKeynameAccelerator);
}

sub isRDSync {
	my ($self) = @_;
	return $self->_isComponent($gKeynameRDSync);
}

sub isXS2 {
	my ($self) = @_;
	return $self->_isComponent($gKeynameXS2);
}

sub isLSS{
    my ($self) = @_;
    return $self->_isComponent($gKeynameLSS);
}

sub isXS2Application { return 0; }

sub isClient{
	my ($self) = @_;
	return undef if (!defined $self->{data});
	my $keyname = $self->getKeyname();
	return defined $keyname && ($keyname eq $gKeynameClient || $keyname eq $gKeynameClientNoCrypto);
}

sub isKnownComponent {
    my ($self) = @_;
    return ($self->isServer() || $self->isClient() || $self->isServer() || $self->isStudio() || $self->isLCAPPS() || $self->isAFL()
            || $self->isHLM() || $self->isSDA() || $self->isInstaller() || $self->isLMStructure() || $self->isCockpitStack()
            || $self->isStreaming() || $self->isES() || $self->isAccelerator() || $self->isRDSync() || $self->isXS2()
            || $self->isXS2Application() || $self->isServerPlugin() || $self->isReferenceData() || $self->isLSS());
}

sub getCompId{
    return $_[0]->{data}->{'compversion-id'};
}

sub getKeyname{
    return $_[0]->{data}->{'keyname'};
}

#_TODO_: remove after official switch of XS2 ID to the enw one
sub setCompId{
    $_[0]->{data}->{'compversion-id'} = $_[1];
}

sub getComponentKey{
    return $_[0]->{data}->{'component-key'};
}

sub getBuildstring{
    return $_[0]->getValue('fullversion');
}

sub isInternalBuild {
	my ($self) = @_;
	my $release = $self->getValue('release');
	return ($release =~ /^\d+\.50/) ? 1 : 0;
}

sub isServerPlugin{
    my $flag = $_[0]->{data}->{'server-plugin'};
    if (defined $flag && ($flag eq '1' || lc($flag) eq 'true' || lc($flag) eq 'yes' || lc($flag) eq 'on')) {
        return 1;
    } else {
        return 0;
    }
}

sub isServerPluginOnlineUpdatable{
    if (!$_[0]->isServerPlugin ()){
        return 0;
    }
    my $flag = $_[0]->{data}->{'online-upgrade-plugin'};
    if (defined $flag && ($flag eq '1' || lc($flag) eq 'true' || lc($flag) eq 'yes' || lc($flag) eq 'on')) {
        return 1;
    }
    return 0;
}

sub isCustomerReleaseBranch{
    my ($self) = @_;
    if (!defined $self->getValue ('git-mergeepoch')){
        # old style manifest, 'compilebranch' marks customer delivery
        return  $self->getValue('compilebranch') eq 'NewDB100_REL';
    }
    return  $self->getValue ('compiletype') eq 'rel';
}

sub isOfficeClient{
    if (!defined $_[0]->{data} || !defined $_[0]->{data}->{'keyname'}){
        return undef
    }
    
    return ($_[0]->{data}->{'keyname'} eq $gKeynameOfficeClient);
}


sub is64bit {
	return ($_[0]->getValue ('platform') !~ /intel/);
}


sub getVersion{
	my ($self) = @_;

	if ( ! $self->{version} ) {
		my $version = $self->getVersionObject ();
		$self->{version} = $version->getVersion();
	}

    return $self->{version};
}

sub getValue{
	my ($self, $key) = @_;

	my $data = $self->read ();
	if (!defined $data){
		return undef;
	}
	if (!exists $data->{$key}){
		return undef;
	}
	if($key eq 'platform'){
		my $supportedPlatforms = $self->getSupportedPlatforms();
		return ($self->isPlatformSupported($currentPlatformName))?$currentPlatformName:$supportedPlatforms->[0];
	}
	return $data->{$key};
}

sub getFileName{
    return $_[0]->{file};
}

sub exists{
    if (defined $_[0]->{file} && -f $_[0]->{file}){
        return 1;
    }
    return 0;
}

sub checkComponentDependency{
    my ($self, $manifest, $ownProductName) = @_;

    my $parser = $self->getDependencyParser();
    if(!defined $parser || $parser == 1){
        return $parser;
    }
    return $self->checkComponentDependencyForParser($manifest, $ownProductName, $parser);
}

sub checkComponentDependencyForParser{
	my ($self, $manifest, $ownProductName, $parser) = @_;

	my $keyVendor = $manifest->getValue('keyvendor');
	my $keyName   = $manifest->getValue('keyname');

	my $minDependency = $parser->getMinVersion($keyVendor, $keyName);
	my $maxDependency = $parser->getMaxVersion($keyVendor, $keyName);

	my $release = $manifest->getValue('release');
	my $sp = $manifest->getValue('rev-number');
	my $pl = $manifest->getValue('rev-patchlevel');
	my $cl = $manifest->getValue('rev-changelist');

	# check minimum and maximum requirements
	if ((defined $minDependency) && (!$minDependency->isCompatible($release, $sp, $pl, $cl)) ||
		(defined $maxDependency) && (!$maxDependency->isCompatible($release, $sp, $pl, $cl))) {

		my $dependencyStr = $parser->getDependencyString ();
		$self->setErrorMessage ('Unresolved dependency: ' .
			$parser->getDependencyString ($keyVendor, $keyName, $ownProductName));
		return undef;
	}
	return 1;
}


sub getVersionObject{
    require SDB::Install::ComponentVersion;
    require SDB::Install::EsComponentVersion;

    my ($self) = @_;
    my $release = $self->getValue('release') // '0';
    my $revision = $self->getValue('rev-number');
    my $patch = $self->getValue('rev-patchlevel') // '00';
    my $change = $self->getValue('rev-changelist');

    if($self->isES()){
        return new SDB::Install::EsComponentVersion(split('\.', $release), $revision, $patch, $change);
    }
    return new SDB::Install::ComponentVersion(split('\.', $release), $revision, $patch, $change);
}

sub getVersionObjectWithoutRevChangeList {
	my ($self) = @_;
	require SDB::Install::ComponentVersion;

    my $version = new SDB::Install::ComponentVersion (split ('\.', $self->getValue ('release')), $self->getValue ('rev-number'),
        $self->getValue ('rev-patchlevel'));
    return $version;
}

sub getThirdPartyLibs {
	my ($self) = @_;
	if (!defined $self->{'thirdPartyLibs'}) {
		foreach my $key (keys (%{$self->{data}})) {
			if ($key =~ /^3rd-party/) {
				my ($name, $vendor, $version, $filenames);
				foreach my $thirdPartyValuePart (split(';', $self->getValue ($key))) {
					trim (\$thirdPartyValuePart);
					my ($left, $right) = split('=', $thirdPartyValuePart);
					trim (\$left);
					trim (\$right);
					$right =~ s/^"(.*)"$/$1/;
					if ($left eq 'name') {$name = $right;}
					elsif ($left eq 'vendor') {$vendor = $right;}
					elsif ($left eq 'version') {$version = $right;}
					elsif ($left eq 'filenames') {$filenames = $right;}
				}
				if (defined $name && defined $version && defined $filenames) {
					$self->{'thirdPartyLibs'}->{$name}->{'vendor'} = $vendor;
					$self->{'thirdPartyLibs'}->{$name}->{'version'} = $version;
					$self->{'thirdPartyLibs'}->{$name}->{'filenames'} = $filenames;
				}
			}
		}
	}
	return $self->{'thirdPartyLibs'};
}

#
# check whether there is a upgrade limitation in the target manifest
# due to sps change
#

our $max_sps_revnumber_pattern = 'max_sps(\d+)_rev-number';
our $max_sps_revnumber_pattern_with_rel = 'max_rel([\d\.]+)_sps(\d+)_rev-number';

sub checkMaxSPSRevision{
    my ($self, $srcManifest) = @_;
    my $srcRel = $srcManifest->getValue('release');
    my $srcRev = $srcManifest->getValue('rev-number');
    my $srcRelVersion = new SDB::Install::Version (split (/\./, $srcRel));
    my $maxPatch;
    my $format_without_rel;
    my ($num,$rel);
    foreach my $key (keys (%{$self->{data}})){
        ($num) = ($key =~ /$max_sps_revnumber_pattern/);
        if (defined $num){
            $rel = "1.00";
            $format_without_rel = 1;
        }
        else{
            $format_without_rel = 0;
            ($rel,$num) = ($key =~ /$max_sps_revnumber_pattern_with_rel/);
        }
        if (!defined $rel){
            next;
        }
        if (!$srcRelVersion->isEqual(new SDB::Install::Version (split (/\./, $rel)))){
            next;
        }

        if ($srcRev == $self->{data}->{$key}){
            if ($format_without_rel){
                $maxPatch = $self->{data}->{"max_sps${num}_patchlevel"};
            }
            else{
                $maxPatch = $self->{data}->{"max_rel${rel}_sps${num}_patchlevel"};
            }
            if (defined $maxPatch){
                last;
            }
        }
    }
    if (!defined $maxPatch){
        $self->getMsgLst()->addMessage ("No max sps patch defined for revision $srcRev");
        return 1;
    }
    my $srcPatch = $srcManifest->getValue('rev-patchlevel');
    if ($srcPatch > $maxPatch){
        $self->setErrorMessage ("'$srcPatch' > maximal sps patch '$maxPatch' for revision '$srcRev'");
        return 0;
    }
    $self->getMsgLst()->addMessage ("max sps patch '$maxPatch' check passed for revision  $srcRev (patch '$srcPatch')");
    return 1;
}


sub _getSPSNum{
    my ($self) = @_;
    my $rev = $self->getValue ('rev-number');
    return (int ($rev/10));
}

sub checkSPSUpgradeLimitation{
    my ($self, $srcManifest) = @_;

    #
    # checking for SPS upgrade
    #

    return 0 if (!$self->_checkUpgradeRestrictions($srcManifest));

    my $release = $self->getValue ('release');
    my $releaseVersion = new SDB::Install::Version (split (/\./, $release));
    my $srcRelease = $srcManifest->getValue ('release');
    my $srcReleaseVersion = new SDB::Install::Version (split (/\./, $srcRelease));
    my $msglst = $self->getMsgLst ();
    my $srcPatchLevel =  $srcManifest->getValue ('rev-patchlevel');


    my $version_1_00 = new SDB::Install::Version (1,0);
    my $isSrcRelease_1 =  $srcReleaseVersion->isEqual ($version_1_00);

    if (!$self->checkMaxSPSRevision ($srcManifest)){
        my $msg = $msglst->addMessage ("Looking for upgrade limitation exceptions");
        $srcManifest->setMsgLstContext ([$msg->getSubMsgLst]);
        my $isRelease_1_00 = $releaseVersion->isEqual ($version_1_00);
        my $rc = $srcManifest->checkSPSUpgradeLimitationException ($self, $isRelease_1_00 ? undef : $release, $self->_getSPSNum());
        if (!$rc){
            $self->appendErrorMessage ("Upgrade from version " .
                $srcManifest->getVersion() .
                ' to ' .
                $self->getVersion() .
                " is not allowed. Please see SAP note 1948334!" );
        }
        return $rc;
    }
    $msglst->addMessage ("Upgrade limitations check successfully passed.");
    return 1;
}

sub _findLimitationExecptionKey{
    my ($self, $targetRelNum, $targetSpNum) = @_;
    my $min_sps_rev_key_pattern;
    my $min_sps_rev_key;
    my ($targetRelVersion, $targetSpVersion);
    my $spVersionIndex = 0;
    if (defined $targetRelNum){
        $spVersionIndex = 1;
        $min_sps_rev_key_pattern = '^min_rel([\.\d]+)_sps(\d+)_rev-number$';
        $targetRelVersion = new SDB::Install::Version (split('\.', $targetRelNum));
    }
    else{
        $min_sps_rev_key_pattern = '^min_sps(\d+)_rev-number$';
    }
    $targetSpVersion = new SDB::Install::Version ($targetSpNum);
    my @version;
    my $data = $self->{data};
    foreach my $key (keys (%$data)){
        @version = ($key =~ /$min_sps_rev_key_pattern/);
        if (!@version){
            next;
        }
        if ($spVersionIndex){
            if (!$targetRelVersion->isEqual(new SDB::Install::Version (split ('\.', $version[0])))){
                next;
            }
        }
        if ($targetSpVersion->isEqual(new SDB::Install::Version ($version[$spVersionIndex]))){
           $min_sps_rev_key = $key;
           last;
        }
    }
    return $min_sps_rev_key;
}

#
# check whether there is an exception in the source manifest
# to allow the upgrade in spite of the sps upgrade limitation
# in the target manifest
#
# if there is an exception, the return value is 1, otherwise 0
#

sub checkSPSUpgradeLimitationException{
    my ($self, $targetManifest, $targetRelNum, $targetSpNum) = @_;
    my $min_sps_rev_key = $self->_findLimitationExecptionKey ($targetRelNum, $targetSpNum);
    my $min_sps_rev_number;
    if (defined $min_sps_rev_key){
        $min_sps_rev_number = $self->getValue ($min_sps_rev_key);
    }
    my $msglst = $self->getMsgLst ();
    if (!defined $min_sps_rev_number || $min_sps_rev_number eq ''){
        $msglst->addMessage ("No upgrade limitation exception in source manifest found ($min_sps_rev_key).");
        return 0;
    }

    if ($targetManifest->getValue ('rev-number') < $min_sps_rev_number){
        $msglst->addMessage ("Upgrade limitation exception rev-number < $min_sps_rev_number ($min_sps_rev_key)");
        return 0;
    }
    my $min_sps_patchlevel_key = "min_sps${targetSpNum}_patchlevel";
    my $min_sps_patchlevel = $self->getValue ($min_sps_patchlevel_key);
    if (defined $min_sps_patchlevel){
        if ($targetManifest->getValue ('rev-patchlevel') < $min_sps_patchlevel){
            $msglst->addMessage ("Upgrade limitation exception rev-patchleval < $min_sps_patchlevel ($min_sps_patchlevel_key)");
            return 0;
        }
    }
    $msglst->addMessage ("Upgrade limitation exception from source version fits ($min_sps_rev_key = $min_sps_rev_number, $min_sps_patchlevel_key = $min_sps_patchlevel)!");
    return 1;
}

sub isSystemRestartRequired {
	my ($self) = @_;
    my $servicesToRestart = $self->getServicesToRestart();
    return 0 if (@$servicesToRestart);

	my $requiredRestartString = $self->getValue('requires-restart');
	return $requiredRestartString =~ /^\s*system\s*,*/;
}

sub isInstanceRestartRequired {
	my ($self, $a) = @_;
    my $servicesToRestart = $self->getServicesToRestart();
    return 0 if (@$servicesToRestart);

    my $requiredRestartString = $self->getValue('requires-restart');
    my $rc = $requiredRestartString =~ /^\s*instance\s*,*/;
    return $rc;
}

sub getServicesToRestart {
    my ($self) = @_;
    my $requiredRestartString = $self->getValue('requires-restart');
    if ($requiredRestartString =~ /^.*services=\"(.*?)\"$/) {
        my $csvServicesToRestart = $1;
        my @aServicesToRestart = split(',', $csvServicesToRestart);
        return \@aServicesToRestart;
    }
    return [];
}

sub getSupportedPhases {
	my ($self) = @_;
	my $suportedPhasesString = $self->getValue('supported-phases');

	if(defined($suportedPhasesString) && length($suportedPhasesString) > 0){
		$suportedPhasesString =~ s/\s//ig;
		return [ split(',', $suportedPhasesString) ];
	}
	return [];
}

sub _checkUpgradeRestrictions {
	my ($self, $srcManifest) = @_;
	my $upgradeRestrictions = $self->getValue('upgrade-restriction');

	return 1 if(! defined($upgradeRestrictions));

	for my $upgradeRestriction (@{$upgradeRestrictions}) {
		if(!$self->_checkSingleUpgradeRestriction($srcManifest, $upgradeRestriction)){
			$self->setErrorMessage($self->_getUpgradeRestrictionMessage($upgradeRestriction));
			return 0;
		}
	}
	return 1;
}

sub _getUpgradeRestrictionInterval {
	my ($self, $upgradeRestriction) = @_;
	return ($upgradeRestriction =~ /sourceVersion="(.*?)";/) ? $1 : undef;
}

sub _getUpgradeRestrictionMessage {
	my ($self, $upgradeRestriction) = @_;
	return ($upgradeRestriction =~ /message="(.*)"$/) ? $1 : undef;
}

sub _checkSingleUpgradeRestriction {
	my ($self, $srcManifest, $upgradeRestriction) = @_;
	my $restrictionInterval = $self->_getUpgradeRestrictionInterval($upgradeRestriction);

# Upgrade restriction interval does not exist
	return 1 if(! defined($restrictionInterval));

	my $includesLeftEndpoint = ($restrictionInterval =~ /^\[/) ? 1 : 0;
	my $includesRightEndpoint = ($restrictionInterval =~ /\]$/) ? 1 : 0;

# Remove any brackets around the two intervals
	$restrictionInterval =~ s/[\[\]\(\)]//g;

	my ($leftEndpoint, $rightEndpoint) = split(/\s*,\s*/, $restrictionInterval);
	my $currentVersionString = join('.', ($srcManifest->getValue('release'),
										  $srcManifest->getValue('rev-number'),
										  $srcManifest->getValue('rev-patchlevel')));
	my $leftEndpointVersion = new SDB::Install::Version (split(/\./, $leftEndpoint));
	my $rightEndpointVersion = new SDB::Install::Version (split(/\./, $rightEndpoint));
	my $currentVersion = new SDB::Install::Version (split(/\./, $currentVersionString));

	return 0 if($currentVersion->isEqual($leftEndpointVersion) && $includesLeftEndpoint);
	return 0 if($currentVersion->isEqual($rightEndpointVersion) && $includesRightEndpoint);
	return 0 if($currentVersion->isNewerThan($leftEndpointVersion) && $rightEndpointVersion->isNewerThan($currentVersion));

	return 1;
}

sub isPlatformSupported{
	my ($self, $currentPlatformName ) = @_;
	my $currentPlatformNameTolower=lc($currentPlatformName);
	return scalar grep{lc($_) eq $currentPlatformNameTolower} @{$self->getSupportedPlatforms()};
}
sub getSupportedPlatforms{
    my ($self ) = @_;
    my $data = $self->read ();
    if (!defined $data){
        return undef;
    }
    if (!exists $data->{'platform'}){
        return undef;
    }
    my $componentPlatformsString = $data->{'platform'};
    if(defined($componentPlatformsString) && length($componentPlatformsString) > 0){
        $componentPlatformsString =~ s/\s//ig;
        return  [split(',', $componentPlatformsString)];
    }
    return [];    
}

sub _createDependencyParser {
    my($self, $dependencyEntries) = @_;

    if (!defined $dependencyEntries) {
        return 1;
    }
    require SDB::Install::DependencyParser;
    my $parser = new SDB::Install::DependencyParser();
    my $dependencyTree = $parser->parse($dependencyEntries);
    if (!defined $dependencyTree) {
        $self->AddError(undef, $parser);
        return undef;
    }
    return $parser;
}

sub getDependencyParser{
	my($self)=@_;
	my $dependencyEntries = $self->getValue ('required-components');
	return $self->_createDependencyParser($dependencyEntries);
}

sub getSupportedComponentsParser{
    my($self)=@_;
    my $dependencyEntries = $self->getValue ('restrict-supported-components');
    return $self->_createDependencyParser($dependencyEntries);
}

sub isAutoInitializeServiceSupported {
	my ($self) = @_;
    my $supportedDbFeatures = $self->getValue('supported-db-features');
    my $rc = $supportedDbFeatures =~ /.*auto_initialize_services.*/;
    return $rc;
}
1;
