#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: base class of MaxDB installation package


package SDB::Install::Package;

use SDB::Install::BaseLegacy;

our @ISA = qw (SDB::Install::BaseLegacy);

use SDB::Install::Tools;
use SDB::Install::PackageManager;
use SDB::Install::System qw (canLockProcess
                             getFileSystemInfo
                             getMountPoint
                             isAbsolute
                             isSameFile2
                             normalizeRelativePath
                             removeEmptyDirs);
use SAPDB::Install::MD5Sum;
use SAPDB::Install::ProcState qw (SDB_PROCSTATE_FLAG_MODULES);
use SDB::Install::SysVars qw($isWin $path_separator);
use LCM::ProcessExecutor;
use SDB::Install::Version;

use strict;



our %fake_features = ();


our %fake_package_features = ();

#----------------------------------------------------------------------------

sub new{
	my $self = shift->SUPER::new ();	
	$self->{interfaceversion} = undef;
	$self->{data} = {};
	$self->{data}->{name} = undef;
	$self->{id} = undef;
	$self->{data}->{version} = undef;
	$self->{data}->{buildstring} = undef;
	$self->{data}->{files} = undef;
	$self->{data}->{type} = undef;
	$self->{data}->{checksum} = undef;
	$self->{data}->{size} = undef;
	$self->{data}->{desc} = undef;
	$self->{installation} = undef;
    $self->{global} = undef;
    $self->{isClientPackage} = undef;
	return $self;	
}

#----------------------------------------------------------------------------

sub LogInformations{
	my ($self) = @_;
	$self->AddMessage ("Name = $self->{data}->{name}");
	$self->AddMessage ("Id = $self->{id}");
	$self->AddMessage ("Version = $self->{data}->{buildstring}");
	$self->AddMessage ("GitHash = $self->{data}->{git_hash}");
	$self->AddMessage ("Number of Files = " . $self->GetNumberOfFiles);
	$self->AddMessage (sprintf ("Size = %.3f mb", $self->GetSize / 0x100000));
	$self->AddMessage ("Description = $self->{data}->{desc}");
	$self->AddMessage ("Checksum = $self->{data}->{checksum}");
	if (defined $self->{data}->{requires}){
		foreach my $req (@{$self->{data}->{requires}}){
			$self->AddMessage ("Requires = $req->{str}");
		}
	}
	if (defined $self->{data}->{require_conditional}){
		foreach my $req (@{$self->{data}->{require_conditional}}){
			$self->AddMessage ("InstallAfter = $req->{str}");
		}
	}
	if (defined $self->{debugPackage}){
		$self->AddMessage ("Debug Archive = $self->{data}->{debugArchive}");
	}
}

#----------------------------------------------------------------------------

sub CheckDependency{
	my ($self,$dependency,$package_list, $dont_keep) = @_;
	
	if (!$dont_keep){	
		unless (defined $self->{dependencies}->{$dependency->{id}}){
			$self->{dependencies}->{$dependency->{id}} = {};
		}
		$self->{dependencies}->{$dependency->{id}}->{str} = $dependency->{str};
	}
			
	if (exists $package_list->{$dependency->{id}}){
		if (exists $dependency->{operand}){
			my $required_version = new SDB::Install::Version (split ('\.', $dependency->{version}));
			my $version =  new SDB::Install::Version (split ('\.', $package_list->{$dependency->{id}}->{data}->{version}));
			my $operand = $dependency->{operand};
			
			if ($operand eq '=='){
				if ($version->isEqual($required_version)){
					return 1;
				}
			}
			elsif ($operand eq '>'){
				if ($version->isNewerThan ($required_version)){
					return 1;
				}
			}
			elsif ($operand eq '<'){
				if ($required_version->isNewerThan ($version)){
					return 1;
				}
			}
			elsif ($operand eq '>='){
				if (!$required_version->isNewerThan ($version)){
					return 1;
				}
			}
			elsif ($operand eq '<='){
				if (!$version->isNewerThan ($required_version)){
					return 1;
				}
			}
		}
		else{
			return 1;
		}
	}
	return 0;
		
}

#----------------------------------------------------------------------------

#
# checks the dependencies on already installed packages
#

sub CheckDependenciesByInstallation{
	my ($self,$installation) = @_;
	my $rc = 1;

	my $requires = $self->{data}->{requires};
	
	foreach my $dependency (@$requires){
		if ($self->CheckDependency ($dependency,$installation->GetPackageList)){
			if (exists $self->{dependencies}->{$dependency->{id}}){
				$self->{dependencies}->{$dependency->{id}}->{resolved_by_installation} = 1;
			}
		}
		else{
			if (exists $self->{dependencies}->{$dependency->{id}}){
				$self->{dependencies}->{$dependency->{id}}->{resolved_by_installation} = 0;
				$rc = 0;
			}
		}
	}
	return $rc;
}

#----------------------------------------------------------------------------

sub getExternalFileListName{
    return undef;
}

#----------------------------------------------------------------------------

sub isSelected ($){
	$_[0]->{selected};
}


#----------------------------------------------------------------------------

sub isGlobal {
	my $retval = 0;
    if(defined $_[0]->{'data'}->{'global'}) {
    	$retval = $_[0]->{'data'}->{'global'};
    }
    return $retval;
}

#----------------------------------------------------------------------------

sub isClientPackage{
    my $retval = 0;
    if(defined $_[0]->{'data'}->{'isClientPackage'}) {
        $retval = $_[0]->{'data'}->{'isClientPackage'};
    }
    return $retval;

}

#----------------------------------------------------------------------------

sub isServerPackage{
	return !isDBStudioPackage(@_) && !isClientPackage (@_);
}

#----------------------------------------------------------------------------

sub isDBStudioPackage{
	$_[0]->{id} eq 'newdbstudioDirector';
}

#----------------------------------------------------------------------------

sub getPath{
	if ($_[0]->isGlobal ()){
		return $_[0]->{installation}->get_globalSidDir;
	}
	return $_[0]->{installation}->getProgramPath ();
}

sub getOriginFileBackupPath{
	my ($self) = @_;
	return $self->{installation}->getRegistryPath () . $path_separator . 'subst';
}

#----------------------------------------------------------------------------

sub Register (){
	return 1;
}

#----------------------------------------------------------------------------

sub evaluateScript{
	my ($self) = @_;
	return 1 if $self->{evaluated};
	my ($parentpackage_name) = ("$self" =~ /(.*)=.*/);
	
	my $checksum = MD5Str ($self->{data}->{script});

	my $package_name = $self->{id} . $checksum;
	
	$package_name =~ s/\s/_/g;
		
	$package_name = $parentpackage_name . '::' . $package_name;
    my $symbolTableName = '%' . $package_name . '::';
    my $symbolTable;
    eval ('$symbolTable = \\'.$symbolTableName . ';');
	if (not %$symbolTable){
		my $script = 	'package '. $package_name . ";\n".
			'our @ISA = qw (' . $parentpackage_name . ");\n".
			'# line 1 "installation script of package '.$self->{data}->{name}."\"\n".
			$self->{data}->{script};
		no strict "vars"; # work around buggy installer package scripts
		eval ($script);
		use strict;
		if ($@){
			$self->AddError ('Cannot evaluate package script: '.$@);
			return undef;
		}
	}
	bless ($self,$package_name);
	$self->{evaluated} = 1;
	$self->initPackage ();
	return 1;
}

#----------------------------------------------------------------------------

sub initPackage (){
	return 1;
}

#----------------------------------------------------------------------------

sub Provides ($){
	if (exists $fake_package_features{$_[0]->{id}}){
		my %result;
		foreach my $feature (@{$fake_package_features{$_[0]->{id}}}){
			$result{$feature} = $fake_features{$feature};
		}
		return \%result;
	}
	{};	
}

#----------------------------------------------------------------------------

sub GetSize ($) {
	$_[0]->{data}->{size};
}


#----------------------------------------------------------------------------

sub GetFiles ($) {
	$_[0]->{data}->{files};
}

#----------------------------------------------------------------------------

sub GetNumberOfFiles ($) {
	my $files = $_[0]->GetFiles;
	return  (defined $files ? (scalar keys %$files) : 0); 
}

#----------------------------------------------------------------------------

sub GetName ($) {
	$_[0]->{data}->{name};
}

#----------------------------------------------------------------------------

sub GetVersion ($) {
	$_[0]->{data}->{version};
}

#----------------------------------------------------------------------------

sub GetId ($) {
	$_[0]->{id};
}

#----------------------------------------------------------------------------

sub GetGitHash{
	$_[0]->{data}->{git_hash};
}

sub GetMakeId;
*GetMakeId = \&GetGitHash;

#----------------------------------------------------------------------------

sub GetDescription ($) {
	$_[0]->{data}->{desc};
}

#----------------------------------------------------------------------------

sub GetTestFileMagic ($) {
	$_[0]->{data}->{test_file_magic};
}

#----------------------------------------------------------------------------

sub GetObjectMode ($) {
	my $magic = $_[0]->{data}->{test_file_magic};
	if (defined $magic){
		return $magic =~ /64/ ? '64' : '32';
	}
	return '';
}

#----------------------------------------------------------------------------

sub GetInstallation ($){
	$_[0]->{installation};
}

#----------------------------------------------------------------------------

sub GetPath;

*GetPath = \&getPath;

#----------------------------------------------------------------------------

sub IsUpdate{
	return 0;
}

#----------------------------------------------------------------------------


=functionComment

Creates a data structure containing all file systems (identified by mount point)
affected by this package, and per file system, the number of bytes used by 
this package in this file system.
The return value is a reference to a hash with the mount points as keys and
filesystem utilization in bytes as values.
 
=cut
sub getFootprintsByFilesystem {
    my ($self)                        = @_;
    my $retval                        = {};
    my $success                       = 1;
    my $errmsg                        = undef;
    my $files                         = $self->GetFiles();
    my $mountPoint                    = undef;
    my $deviceId                      = undef;
    my $fileSystemTotalSizeBlocks     = undef;
    my $filesystemAvailableSizeBlocks = undef;
    my $filesystemBlockSizeBytes      = undef;
    my $footprints                    = undef;
    my $instRoot                      = $self->getPath();
    # if > 1, the scaling factor modifies the unit of fields
    # 'fileSystemTotalSizeBytes' and 'filesystemAvailableSizeBytes'.
    # e.g. if scalingFactor is 1024 (1024*1024), the unit is kilobytes
    # (megabytes). 
    my $scalingFactor                 = undef;
    # in the unix case, we want to cache mount points which we have already found:
    my $footprintIncrement            = undef;
    foreach my $file (keys %$files) {
        ($mountPoint, $success, $errmsg) = 
            getMountPoint($instRoot, $file);
        if(!$success) {
            return (undef, $success, $errmsg);
        }
        $footprints = $retval->{$mountPoint};
        if(!defined $footprints) {
            $retval->{$mountPoint} = $footprints = {};
            ($deviceId,
             $fileSystemTotalSizeBlocks,
             undef,
             $filesystemAvailableSizeBlocks,
             $filesystemBlockSizeBytes,
             $scalingFactor,
             $success,
             $errmsg) = 
                getFileSystemInfo($mountPoint);
            if(!$success) {
                return (undef, $success, $errmsg);
            }
            $footprints->{'deviceId'}                                    = $deviceId;
            $footprints->{'filesystemBlockSizeBytes'}                    = $filesystemBlockSizeBytes;
            $footprints->{'fileSystemTotalSizeBlocks'}                   = $fileSystemTotalSizeBlocks;
            $footprints->{'filesystemAvailableSizeBlocks'}               = $filesystemAvailableSizeBlocks;
            $footprints->{'scalingFactor'}                               = $scalingFactor;
            $footprints->{'estimatedNewInstallationFootprintBlocks'}     = 0;
            $footprints->{'estimatedUpdatedInstallationFootprintBlocks'} = 0;
            $footprints->{'actualInstallationFootprintBlocks'}           = 0;
        }
        $footprintIncrement = divideIntegrally($files->{$file}->{"size"}, $filesystemBlockSizeBytes) + 1;
        $footprints->{'estimatedNewInstallationFootprintBlocks'}     += $footprintIncrement;
        $footprints->{'estimatedUpdatedInstallationFootprintBlocks'} += 1; ## TODO: implement this, the value may be negative !!!
        $footprints->{'actualInstallationFootprintBlocks'}           += 1; ## TODO: implement this using 'stat' !!!
    }
    return ($retval, $success, $errmsg);
}

#----------------------------------------------------------------------------

=functionComment

Creates a data structure containing all files of the package which are currently
in use. The return value is a reference to a hash where the keys are the PIDs of
the locking processes and the values are lists of (unique) files locked by the process.
Example:
        {
            '30000'=>
                {
                    'lockedFiles'=>
                        [
                            'bbb.jpg',
                            '/aaa.dll',
                        ],
                    'commandLine'=>'/usr/bin/program01 -someopt',
                },
            '100000'=>
                {
                    'lockedFiles'=>
                        [
                            'bbb.pbx',
                            '/aaa/zzz.htm',
                        ],
                    'commandLine'=>'/usr/bin/program02 -someotheropt',
                }
        }
 
=cut
sub getFilesInUseByPID {
    my ($self,$procStateObj)         = @_;
    my $retval         = {};
    my $success        = 1;
    my $errmsg         = undef;
    my $files          = $self->GetFiles();
    my $instRoot       = $self->getPath;
	if (!defined $procStateObj){
		$procStateObj   = new SAPDB::Install::ProcState (SDB_PROCSTATE_FLAG_MODULES);
	}
    my $absFile        = undef; 
    my $file           = undef; 
    my $pidsPerFile    = undef;
    my $pid            = undef;
    my $pidInfo        = undef;
    my $normalizedFile = undef;
    my $commandLine    = undef;
    
    
    if(!defined $procStateObj) {
        $errmsg = "SDB::Install::Package::getFilesInUseByPID: Cannot create ProcState object.";
        $success = 0;
        return (undef, $success, $errmsg);
    }
    foreach my $file (keys %$files) {
        if (defined $files->{$file}->{isSymlink} && $files->{$file}->{isSymlink}){
            next;
        }
        if(canLockProcess($file)) {
		    $absFile = $instRoot.'/'.$file;
            $pidsPerFile = $procStateObj->WhoUsesModule($absFile);
            if(defined $pidsPerFile) {
				($normalizedFile, $success, $errmsg) = normalizeRelativePath($instRoot, $file);
                if(!$success) {
                    return (undef, $success, $errmsg);
                }
                foreach $pid (@$pidsPerFile) {
                    $pidInfo = $retval->{$pid};
                    if(!defined $pidInfo) {
                        $retval->{$pid} = $pidInfo = {};
                        $commandLine = $procStateObj->GetArgs($pid);
                        if(!defined $commandLine) {
                            $commandLine = $procStateObj->GetCmd($pid);
                            if(!defined $commandLine) {
                                delete $retval->{$pid};
                                next;
                            }
                            $commandLine = $commandLine." [command line parameters not known]";
                        }
                        $pidInfo->{'commandLine'} = $commandLine;
                    }
                    push @{$pidInfo->{'lockedFiles'}}, ($normalizedFile);
                    
                    if (!exists $files->{$file}->{forceExtract} || !$files->{$file}->{forceExtract}){
						$pidInfo->{lock_check_failed} = 1;
                    }
                    
                }
            }
        }
    }
    return ($retval, $success, $errmsg);
}

#----------------------------------------------------------------------------

=functionComment

Checks whether this package does not have
the 'skip' attribute set. Returns 1 if this is the case.

=cut
sub isNonSkipPackage {
    my ($self) = @_;
    my $retval = 0;
    if(!defined $self->{'skip'} || !$self->{'skip'}) {
        $retval = 1;
    }
    return $retval;
}

#----------------------------------------------------------------------------

=functionComment

This function is overridden by 'getShortcutDefinitions' from 'script.pm' of
the respective package, if it is defined there.

=cut
sub getShortcutDefinitions {
    return undef;
}

#----------------------------------------------------------------------------

sub createPackageShortcuts {
    my ($self) = @_;
	if (!$isWin){
		return 1;
	}

	require SDB::Install::System::Shortcut;

	my $inst = $self->{'installation'};
	my $defs = $self->getShortcutDefinitions($inst->{'kit'});
	if (defined $defs && @$defs){
		my $startMenuPath = $inst->getStartMenuPath ();
		if (!defined $startMenuPath){
			$self->setErrorMessage ("Cannot create shortcut", $inst->getErrMsgLst ());
			return undef;
		}
		my $sc;
		foreach my $def (@$defs){
			my $lnk_path = $startMenuPath . $path_separator . $def->{folder}  . $path_separator . $def->{name};
			my $path_template = $lnk_path . ' (%s).lnk';
			my $name_template = $def->{name} . ' (%s)';

			$lnk_path .= '.lnk';
			my $name = $def->{name};

			if (-e $lnk_path){
				my $i;
				for ($i = 2;;$i++){
					$lnk_path = sprintf ($path_template, $i);
					$name = sprintf ($name_template, $i);
					if (!-e $lnk_path){
						last;
					}
				}
			}

			$sc = new SDB::Install::System::Shortcut ($startMenuPath . $path_separator . $def->{folder}, $def->{target},$name,1);
			if (!defined $sc->create()){
				$self->AddError ("Cannot create shortcut", $sc);
				return undef;
			}
		}
	}
	return 1;
}

#----------------------------------------------------------------------------

sub removePackageShortcuts {

    my ($self) = @_;

	if (!$isWin){
		return 1;
	}

	require SDB::Install::System::Shortcut;

	my $inst = $self->{'installation'};
	my $defs = $self->getShortcutDefinitions($inst);

	if (defined $defs && @$defs){
		my $sc;
		my $scFolder;
		my @dirEntries;
		my $pattern;
		foreach my $def (@$defs){
			$scFolder = $inst->getStartMenuPath () . $path_separator . $def->{folder};
			$pattern = '^' . quotemeta ($def->{name}) . '\.lnk$';
			$pattern .= '|^'. quotemeta ($def->{name}) . '\s\(\d+\)\.lnk$';
			if (!opendir (DH, $scFolder)){
				next;
			}
			@dirEntries = grep {/$pattern/i} readdir (DH);
			closedir (DH);
			my @buf = SAPDB::Install::System::Win32::API::GetFileId ($def->{target});

			foreach my $scName (@dirEntries){
				$scName =~ s/\.lnk$//i;
				$sc = new SDB::Install::System::Shortcut ($scFolder, undef, $scName);
				if (defined $sc->{target} && (lc ($def->{target}) eq lc ($sc->{target})) || (isSameFile2($sc->{target}, undef, \@buf)) ){
					$sc->delete();
				}
			}
		}

		# remove shortcut directory if empty
		if (opendir (DH, $scFolder)){
			@dirEntries = readdir (DH);
			closedir (DH);
			if (scalar @dirEntries == 2) {
				removeEmptyDirs($scFolder);
			}
		} else {
			closedir (DH);
		}
	}
	return 1;
}

# -----------------------------------------------------------------

# LCMsdk consumers should use this function (instead of using ProgramExecutor directly),
# to stay away from future compatibility problems:
sub _runProgram() {
    my (
        $self,
        $cmd,       # either absolute, or relative to installation path, mandatory
        $args,      # ref to array of strings, mandatory
        $logMessage # log file message, mandatory
    ) = @_;
    if(!isAbsolute($cmd)) {
        my $instpath = $self->getPath();
        $cmd = $instpath.$path_separator.$cmd;
    }
    my $msg = $self->AddMessage($logMessage);
    my $exer = new LCM::ProcessExecutor($cmd, $args);
    my $exitCode = $exer->executeProgram();
    if (!defined $exitCode || $exitCode){
        $self->AddSubMsgLst ($msg, $exer->getMsgLst());
        $self->AddError($exer->getErrorString());
        return undef;
    }
    $self->AddSubMsgLst ($msg, $exer->getMsgLst());
    return 1;
} 

# -----------------------------------------------------------------

# LCMsdk consumers should use this function
# (instead of using the logging framework directly),
# to stay away from future compatibility problems:
sub _logInfo() {
    my (
        $self,
        $line # without line terminator
    ) = @_;
    my $msg = $self->AddMessage($line);
    return 1;
} 

# -----------------------------------------------------------------

# LCMsdk consumers should use this function
# (instead of using the logging framework directly),
# to stay away from future compatibility problems:
sub _logError() {
    my (
        $self,
        $line # without line terminator
    ) = @_;
    my $msg = $self->AddError($line);
    return 1;
} 

# -----------------------------------------------------------------

# LCMsdk consumers should use this function
# (instead of using configuration objects directly),
# to stay away from future compatibility problems:
sub _getParamValue() {
    my (
        $self,
        $name
    ) = @_;
    my $inst  = $self->{'installation'};
    my $conf  = $inst->getConfiguration();
    my $value = undef;
    if( (exists $conf->{'params'}) &&
        (exists $conf->{'params'}->{$name}) &&
        (exists $conf->{'params'}->{$name}->{'value'}) ) {
            $value = $conf->{'params'}->{$name}->{'value'};
    }
    return $value;
} 

#----------------------------------------------------------------------------

sub DESTROY{
	my ($self) = @_;
	$self->SUPER::DESTROY();
	return 1 unless $self->{evaluated};
}

#----------------------------------------------------------------------------

1;
