#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: manage MaxDB package dependencies 


package SDB::Install::PackageManager;

use SDB::Install::BaseLegacy;
use SDB::Install::Tools qw (arrayToHash);
use SDB::Install::System qw (getFileSystemInfo);
use SDB::Install::SysVars qw ($path_separator);
use SDB::Install::Manifest;
use SDB::Install::DebugUtilities; ##########
use SDB::Install::Globals;
use SAPDB::Install::ProcState qw(SDB_PROCSTATE_FLAG_MODULES);
use SDB::Install::Version;
use strict;

our @ISA = qw (SDB::Install::BaseLegacy);

our @base_ids = qw ();
our @server_ids = qw ();
our @client_ids = qw ();

our @runtime_ids = qw ();


#
# dependency tree
#
#   $self-->{tree}+->{$package_id}+-->
#                |               |
#                .               +-->{$package_id}+
#                .               |                | 
#                .               .                +-->{$package_id}
#                                .                 
#                                .                 


#
# tree of conditional dependencies
#
#  $self-->{cond_tree}+->{$package_id}+-->
#                		|               |
#                		                +-->{$package_id}+
#                		.               |                | 
#                      .                +-->{$package_id}
#                		                .
#                               	    .




#
# package list
#
#  $self-->{packages}+-->{$package_id}+-->{package}->{tree_node}
#								   ->{cond_tree_node}	
#                   |                
#                   .                
#                   .
#                   .
#

#----------------------------------------------------------------------------

sub new {
	my $self = $_[0]->SUPER::new ();
	$self->{'packages'} = undef;
	$self->{'tree'} = undef;
    $self->{'applicationOptions'}         = undef;  # the options of the application, choosen by the user either on the
                                                    # commandline or in the gui
	return $self;
}

#----------------------------------------------------------------------------

=functionComment

The 'applicationOptions' field contains the commandline options and their values,
as set application wide.

=cut
sub setApplicationOptions{
    my (
        $self,
        $options
    ) = @_;
    $self->{'applicationOptions'} = $options;
}

#----------------------------------------------------------------------------

sub GenerateTree{
	my ($self,$inst, $msglst) = @_;
	
	if (!defined $msglst){
		$msglst = $self;
	}
	
	if (!defined $self->{'packages'}){
		if (!defined $self->GenPackageList ($msglst == $self ? undef :
				$msglst)){
			
			$msglst->AddError('Cannot get package list', $msglst);
			return undef;
		}
		if (!defined $self->{'packages'}){
			return undef;
		}
		
	}

	my $rc = 1;
	
	my @err_list;
	foreach my $package (values (%{$self->{'packages'}})){
		
		if (!defined $package->{'cond_tree_node'}){
			$self->AddPackageConditional($package);
		}		
		
		if (defined  $package->{'tree_node'}){
			next;
		}
		unless (defined $self->AddPackage ($package, $inst)){
			my $msg = $self->GenMsg ('ERR','cannot add package ' .
					$package->{'data'}->{'name'});
			$msg->{'submsg'} = $self->{'last_error'};
			undef $self->{'last_error'}; 
			push @err_list, $msg; 
			$rc = undef;
		}
	}
	
	unless (defined $rc){
		$msglst->{'last_error'} = \@err_list;
		$msglst->AddError ('Cannot create dependency tree', $msglst);
	}
	return $rc;
}

#----------------------------------------------------------------------------

sub PrintTree{
	my ($self,$node,$prefix) = @_;
	foreach my $child (keys (%$node)){
		print $prefix.$child."\n";
		$self->PrintTree ($node->{$child},$prefix . "\t");
	}
}

#----------------------------------------------------------------------------

sub AddPackageConditional{
	my ($self,$package,$force_add) = @_;
	
	if (defined $package->{'cond_tree_node'}){
		return $package->{'cond_tree_node'};
	}
	
	unless (defined $package->{data}->{'require_conditional'}){
		if ($force_add){
			unless (defined $self->{'cond_tree'}){
				$self->{'cond_tree'} = {};
			}
			$self->{'cond_tree'}->{$package->{'id'}} = {};
			$package->{'cond_tree_node'} = $self->{'cond_tree'}->{$package->{'id'}};
			return $self->{'cond_tree'}->{$package->{'id'}};
		}
		else{	
			return {};
		}
	}
	my $node = {};
	foreach my $dependency (@{$package->{data}->{require_conditional}}){
		if ($package->CheckDependency ($dependency, $self->{'packages'},1)){
			
			my $parentnode = $self->AddPackageConditional ($self->{'packages'}->{$dependency->{id}},1);
			$parentnode->{$package->{'id'}} = $node;
		}
	}
	$package->{'cond_tree_node'} = $node;
	return $node;
}


#----------------------------------------------------------------------------

sub GetChildren{
	my ($self,$node,$visited) =  @_;
	unless (defined $visited){
		$visited = {};
	}
	my @packages;
	foreach my $child (sort {$self->{'packages'}->{$a}->{'data'}->{'name'} cmp 
		 	$self->{'packages'}->{$b}->{'data'}->{'name'}} keys (%$node)){
		
		next if $visited->{$child};
		if (%{$node->{$child}}){
			unshift @packages, $self->GetChildren ($node->{$child},$visited);
		}
		$visited->{$child} = 1;
		if (!$self->{'packages'}->{$child}->{'skip'} && $self->{'packages'}->{$child}->{'selected'}){
			unshift @packages, $self->{'packages'}->{$child};
		}
	}
	return @packages;
}

#----------------------------------------------------------------------------

sub GetPackageById ($){
	
	if (!defined $_[0]->{packages}){
		return undef;
	}

	$_[0]->{packages}->{$_[1]};
}

#----------------------------------------------------------------------------

sub GetPackageByName ($$) {

	foreach my $package (values %{$_[0]->{packages}}){
		if ($package->GetName() eq $_[1]){
			return $package;
		}
	}
	return undef;
}

#----------------------------------------------------------------------------

sub GetPackageList ($){
	return $_[0]->{'packages'};
}

#----------------------------------------------------------------------------

sub GetDependentPackages{
	my ($self,$package,$recursive) = @_;
	
	my $tree_node;
	
	if (!defined $package){
		$tree_node = $self->{'tree'}; 
	}
	elsif(!defined $self->{packages} || !exists $self->{'packages'}->{$package->{'id'}}){
		$self->AddError ("Package \"$package->{data}->{name}\" not found");
		return undef;
	}
	elsif (!exists $package->{'tree_node'} || !%{$package->{tree_node}}){
		return [];	
	}
	else{
		$tree_node =  $package->{'tree_node'};		
	}
	
	my @packages;
	
	if ($recursive){
		my $child_packages;
		foreach my $child_id (keys %$tree_node){
			push @packages, $self->{'packages'}->{$child_id};
			$child_packages = $self->GetDependentPackages ($self->{'packages'}->{$child_id},1);
			if (defined $child_packages && @$child_packages){
				push @packages, @$child_packages;
			}
		}
	}
	else{
		foreach my $child_id (keys %$tree_node){
			push @packages, $self->{'packages'}->{$child_id};
		}
	}

	return \@packages;
}

#----------------------------------------------------------------------------

sub GetRequiredPackages{
	my ($self,$package) = @_;
	
	if (!defined $package){
		return undef;
	}
	
	my $id = $package->{'id'};
	
	if (!defined $self->{packages} || !exists $self->{'packages'}->{$id}){
		$self->AddError ("Package \"$package->{data}->{name}\" not found");
		return undef;
	}
	
	my @required_packages;	

	foreach my $pack (values %{$self->{packages}}){
		next if ($pack->{id} eq $id);
		next if !%{$pack->{tree_node}};
		if (exists $pack->{'tree_node'}->{$id}){
			push @required_packages, $pack;
		}
	}
	return \@required_packages;
}

#----------------------------------------------------------------------------

sub SelectAllPackages ($){
		
	if (!defined $_[0]->{packages}){
		return undef;
	}
	
	foreach my $package (values %{$_[0]->{packages}}){
		$package->{'selected'} = 1;
	}

	return 1;
}


#----------------------------------------------------------------------------

sub AreAllPackagesSelected ($){
      
    if (!defined $_[0]->{packages}){
        return undef;
    }
    
    foreach my $package (values %{$_[0]->{packages}}){
        if (!$package->{'selected'}) { return 0; }
    }

    return 1;
}

#----------------------------------------------------------------------------

sub HasSelectedPackages ($){
      
    if (!defined $_[0]->{packages}){
        return undef;
    }
    
    foreach my $package (values %{$_[0]->{packages}}){
        if ($package->{'selected'}) { return 1; }
    }

    return 0;
}


#----------------------------------------------------------------------------


sub HasPackages ($){
	defined $_[0]->{packages};
}



#----------------------------------------------------------------------------


sub HasDBKernel ($){
	defined $_[0]->{packages} && exists $_[0]->{packages}->{binaries};
}

#----------------------------------------------------------------------------


sub IsDBKernelSelected ($){
	defined $_[0]->{packages} && exists $_[0]->{packages}->{binaries} &&
	$_[0]->{packages}->{binaries}->{selected};
}


#----------------------------------------------------------------------------

sub getCompId ($){
    my $mf = $_[0]->getManifest ();
    if (defined $mf){
        return  $mf->getCompId();
    }
    return undef;
}


#----------------------------------------------------------------------------

sub isComponent ($){
    my $mf = $_[0]->getManifest ();
    if (defined $mf){
        return  $mf->getCompId() eq $_[1];
    }
    return undef;
}

#----------------------------------------------------------------------------

sub checkComponentDependency ($){
    my ($self, $pm) = @_;
    my $mf = $self->getManifest ();
    if (!defined $mf){
        return undef;
    }
    my $rc = $mf->checkComponentDependency ($pm->getManifest (), $pm->getProductName ());
    if (!$rc){
        $self->AddError (undef, $mf);
    }
    return $rc;
}

#----------------------------------------------------------------------------

sub isStudio ($){
	my $mf = $_[0]->getManifest ();
	if (defined $mf){
		return  $mf->isStudio ();
	}
	defined $_[0]->{packages} && exists $_[0]->{packages}->{newdbstudioDirector};
}

sub IsDBStudioDistribution ($);
*IsDBStudioDistribution = \&isStudio;


#----------------------------------------------------------------------------

sub isServer ($){
	if (defined $_[0]->{mf}){
		return  $_[0]->{mf}->isServer ();
	}
	return $_[0]->HasDBKernel ();
}

#----------------------------------------------------------------------------

sub isClient ($){
	my $mf = $_[0]->getManifest ();
	if (defined $mf){
		return  $mf->isClient ();
	}
	defined $_[0]->{packages} && exists $_[0]->{packages}->{sqldbc};
}

#----------------------------------------------------------------------------

sub isOfficeClient ($){
	my $mf = $_[0]->getManifest ();
	if (defined $mf){
		return  $mf->isOfficeClient ();
	}
	defined $_[0]->{packages} && exists $_[0]->{packages}->{officedbc};
}

#----------------------------------------------------------------------------

sub isServerPlugin ($){
	my $mf = $_[0]->getManifest ();
	if (defined $mf){
		return  ($mf->isAFL() || $mf->isLCAPPS() || $mf->isServerPlugin());
	}
	return 0;
}


#----------------------------------------------------------------------------

sub getProductName() {
	my ($self) = @_;
	if ($self->isServer()) {
		return $gProductNameEngine;
	} elsif ($self->isClient()) {
		return $gProductNameClient;
	} elsif ($self->isStudio()) {
		return $gProductNameStudio;
	} elsif ($self->isOfficeClient()) {
		return $gProductNameOfficeClient;
	} else {
		return 'unknown';
	}
}

#----------------------------------------------------------------------------

sub getShortProductName() {
	my ($self) = @_;
	if ($self->isServer()) {
		return $gShortProductNameEngine;
	} elsif ($self->isClient()) {
		return $gShortProductNameClient;
	} elsif ($self->isStudio()) {
		return $gShortProductNameStudio;
	} else {
		return 'unknown';
	}
}

#----------------------------------------------------------------------------

sub getProductKey() {
    return undef;
}

#----------------------------------------------------------------------------

sub getPlatform ($){
	if (defined $_[0]->{mf}){
		return  $_[0]->{mf}->getValue ('platform');
	}
	return undef;
}

#----------------------------------------------------------------------------

sub isDebugBuild ($){
    my $manifest = $_[0]->getManifest ();
    if (defined $manifest){
        return  $manifest->getValue ('compiletype') eq 'dbg';
    }
    return undef
}

#----------------------------------------------------------------------------

sub is64bit() {
	my $manifest = $_[0]->getManifest ();
    if (defined $manifest){
        return  $manifest->is64bit();
    }
    return undef
}


#----------------------------------------------------------------------------
#
# Returns a regex, which programs should be ignored during busy files check
# or undef if there is no exception.
# It's currently overwritten in SDB::Install::SAPSystem class.
#

sub getIgnoreBusyProgramsPattern{
    return undef;
}

#----------------------------------------------------------------------------

sub getManifest ($$){
    my ($self,$nocache) = @_;
    if (defined $self->{mf} && !$nocache){
        return  $self->{mf};
    }
    my $file = $self->getManifestDir () . $path_separator . 'manifest';
    $! = 0;
    if (-f $file){
        $self->{mf} = new SDB::Install::Manifest ($file);
        return $self->{mf};
    }
    $self->AddError ("Cannot access manifest file '$file': ". ($! ? $! : 'Not a file'));
    return undef
}

#----------------------------------------------------------------------------

sub getManifestDir ($){
    return undef;
}

#----------------------------------------------------------------------------

sub SelectServerPackages{
	my ($self) = @_;

	if (!defined $self->{packages}){
		return undef;
	}

	foreach my $id (@base_ids, @server_ids){
		if (exists $self->{'packages'}->{$id}){
			$self->SelectPackage ($self->{'packages'}->{$id});
		}
	}
	return 1;
}

#----------------------------------------------------------------------------

sub SelectBasePackages{
	my ($self) = @_;

	if (!defined $self->{packages}){
		return undef;
	}

	foreach my $id (@base_ids){
		if (exists $self->{'packages'}->{$id}){
			$self->SelectPackage ($self->{'packages'}->{$id});
		}
	}
	return 1;
}



#----------------------------------------------------------------------------

sub SelectClientPackages{
	my ($self) = @_;
    
    if (!defined $self->{packages}){
		return undef;
	}
    
    foreach my $id (@base_ids, @client_ids){
		if (exists $self->{'packages'}->{$id}){
			$self->SelectPackage ($self->{'packages'}->{$id});
		}
	}
	return 1;
}

#----------------------------------------------------------------------------

sub UnselectAllPackages{
	my ($self) = @_;
	
	if (!defined $self->{packages}){
		return undef;
	}
	
	foreach my $package (values %{$self->{packages}}){
		$package->{'selected'} = 0;
	}
	return 1;
}

sub MergeCondTree{
	my ($self,$node) = @_;
			
	if (!defined $node){
		$node = $self->{tree};
		if (!defined $self->{cond_tree}){
			return $node;
		}
	}

	my %return_value;
	my $dirty = 0;

	foreach my $id (keys (%$node)){
		my $cond_node = $self->{packages}->{$id}->{cond_tree_node};
		$return_value{$id} = $node->{$id};
		my $newnode = $self->MergeCondTree ($node->{$id});
		
		if ($newnode !=  $node->{$id}){
			$return_value{$id} = $newnode;
			$dirty = 1;
		}
				
		if (defined $cond_node && %$cond_node){
			my %new_node = %{$return_value{$id}};
			foreach my $cond_id (keys (%$cond_node)){
				$new_node{$cond_id} = $self->MergeCondTree ($self->{packages}->{$cond_id}->{tree_node});			
			}
			$return_value{$id} = \%new_node;
			$dirty = 1;
		}
	}
	
	if ($dirty){
		return \%return_value;
	}
	
	return $node;	
}


#----------------------------------------------------------------------------

=functionComment

note that this function has dependency tree generation as important side effect.

=cut
sub GetPackages{
    my ($self, $inst, $ignore_dependency_errors, $msglst) = @_;
    
    #
    # resolve dependecies && generate dependency tree
    #
    unless (defined $self->{'tree'}){

		my $rc = $self->GenerateTree ($inst, $msglst);

        if (!defined $rc && !defined $ignore_dependency_errors){
            return undef;
        }
    }


    #foreach my $key (keys (%{$self->{tree}})){
    #   print ">>> KEY $key\n";
    #}

	#$self->PrintTree ($self->{tree},'TREE: ');


	#$self->PrintTree ($self->{cond_tree},'COND TREE: ');

    my $tree = $self->MergeCondTree();
    
    
    
	#$self->PrintTree ($tree,'MERGED TREE: ');
    
    
    #
    # serialize dependency tree 
    #
    
    my @list = $self->GetChildren ($tree);
	
	return \@list;
}

#----------------------------------------------------------------------------

=functionComment

an "abstract" method, i.e. a placeholder for the respective implementations in
the "subclasses" 'Kit' and 'Installation'.

=cut
sub GenPackageList ($){
    $_[0]->AddError('SDB::Install::PackageManager::GenPackageList: meaningful implementation only in sub classes');
    return undef;
}

#----------------------------------------------------------------------------

sub GetSelectedPackageList {
    my ($self) = @_;
    my $retval = undef;    
    my $packages = $self->{'packages'};
    if (!defined $packages){
        my $packages = $self->GenPackageList(); # we assume that we are a subclass here.
    }
    if(defined $packages) {
        $retval = {};
        foreach my $packageKey (keys %{$packages}){
            my $package = $packages->{$packageKey};
            if($package->{'selected'}) {
                $retval->{$packageKey} = $package;
            }
        }
    }
    return $retval;
}

#----------------------------------------------------------------------------

sub RemoveFromTree{
	my ($self,$package_id,$tree) = @_;
	unless (defined $tree){
		$tree = $self->{'tree'};
		unless (defined ($tree)){
			return 1;
		}
	}
	foreach my $id (keys %$tree){
		if ($id eq $package_id){
			delete $tree->{$id};
		}
		else{
			$self->RemoveFromTree ($package_id, $tree->{$id});
		}
	}
	return 1;
}

#----------------------------------------------------------------------------

sub RemovePackage{
	my ($self,$package, $is_update) = @_;
	if (defined $package->{'tree_node'} && %{$package->{tree_node}}){
			
		#
		# there is at least one dependent package
		# => invalidate dependency tree 
		#
		
		if (!$is_update){
			my $msg = $self->AddWarning ('There are dependencies to package "' . $package->{'data'}->{'name'} . '": invalidate dependency tree');
			foreach my $id (keys (%{$package->{tree_node}})){
				$self->AddSubMessage ($msg,'Package "'.$self->{'packages'}->{$id}->{'data'}->{'name'}. '" is dependent');
			}
		}
	
		undef $self->{'tree'};
	}
	

	if ( defined $package->{cond_tree_node} and %{$package->{cond_tree_node}}){
			
		#
		# there is at least one dependent package
		# => invalidate conditional dependency tree 
		#
		if (!$is_update){
			my $msg = $self->AddWarning ('There are conditional dependencies to package "' . $package->{'data'}->{'name'} . '": invalidate conditional dependency tree');
			foreach my $id (keys (%{$package->{cond_tree_node}})){
				$self->AddSubMessage ($msg,'Package "'.$self->{'packages'}->{$id}->{'data'}->{'name'}. '" is dependent');
			}
		}	
		undef $self->{'cond_tree'};
	}

	
	if (defined $self->{'tree'}){
		$self->RemoveFromTree ($package->{'id'});
		
	}
	
	if (defined $self->{'cond_tree'}){
		$self->RemoveFromTree ($package->{'id'}, $self->{'cond_tree'});
		
	}
	
	if (defined $self->{'version'} && $self->{'version'} eq $package->{'data'}->{'version'}){
		#
		# invalidate version
		#
		undef $self->{'version'};
	}

	delete $self->{'packages'}->{$package->{'id'}};
}

#----------------------------------------------------------------------------

sub SetPackageProgressHandler{
	my ($self,$handler) =  @_;
	foreach my $package (values %{$self->{packages}}){
		$package->SetProgressHandler ($handler);	
	}
}

#----------------------------------------------------------------------------

sub GetPackageProgressHandler{
	my ($self) =  @_;
	my $handler;
	foreach my $package (values %{$self->{packages}}){
		$handler = $package->GetProgressHandler ();
		if (defined $handler){
			return $handler;
		}	
	}
	return undef;
}


#----------------------------------------------------------------------------

sub FreePackages{
	my ($self) =  @_;
	undef $self->{'tree'};
	undef $self->{'version'};
	undef $self->{'packages'};
    return 1;
}

#----------------------------------------------------------------------------

#
# returns kernel version,
# if there is no kernel it returns the most recent version
# of all other packages
#

sub GetVersion{
	my ($self) = @_;
	if (defined $self->{'version'}){
		return $self->{'version'};	
	}
	if (defined $self->{mf}){
		$self->{'version'} = $self->{mf}->getVersion ();
		if (defined $self->{'version'}){
			return $self->{'version'};
		}
	}
	my $max_version = 0;
	foreach my $package (values (%{$self->{packages}})){
		my $version = new SDB::Install::Version (split ('\.',$package->{'data'}->{'version'}));
		if (!defined $max_version){
			$max_version = $version;
			next;
		}
		if ($version->isNewerThan ($max_version)){
			$self->{'version'} = $package->{'data'}->{'version'};
			$max_version = $version;
		}
	}
	return $self->{'version'};
}

#----------------------------------------------------------------------------

sub GetSize{
	my ($self) = @_;
	
	if (!defined $self->{packages}){
		return undef;
	}
	
	my $size = 0;
	foreach my $package (values %{$self->{packages}}){
		$size += $package->GetSize;
	}
	return $size;
}

#----------------------------------------------------------------------------

sub GetSelectedSize{
	my ($self) = @_;
	
	if (!defined $self->{packages}){
		return undef;
	}
	
	my $size = 0;
	my $package;
	
	foreach my $package (values %{$self->{packages}}){
		if ($package->{'skip'} || !$package->{'selected'}){
			next;
		}
		$size += $package->GetSize;
	}
	return $size;
}

#----------------------------------------------------------------------------

sub GetNumberOfPackages{
	my ($self) = @_;
	
	if (!defined $self->{packages}){
		return undef;
	}
	
	return scalar keys %{$self->{packages}};
}

#----------------------------------------------------------------------------


#
# returns a version string to use it e.g. for directory or file names
#

sub getVersionIdString{
    my ($self) = @_;
    my $manifest = $self->getManifest();
    my $versionId = $self->GetVersion ();
    if (defined $manifest){
        my $git_hash = $manifest->getValue('git-hash');
        if (defined $git_hash and $git_hash =~/\S/){
            $versionId .= '_' . $git_hash;
        }
        else{
            my $make_date = $manifest->getValue('date');
            $make_date =~ s/\s/_/g;
            $make_date =~ s/://g;
            $versionId .= '_' . $make_date;
        }
    }
    return $versionId
}

#----------------------------------------------------------------------------

=functionComment

Creates a data structure containing all file systems (identified by mount point)
affected by this installation, and per file system, the number of bytes used by 
this installation in this file system.
The return value is a reference to a hash with the mount points as keys and
refs to hashes containing filesystem utilization information as values.
NOTE: if > 1, 'scalingFactor' modifies the unit of fields
'*Blocks'.
e.g. if scalingFactor is 1024 (or 1024*1024), the unit is kiloblocks
(resp. megablocks). 

Example:
        {
            '/'=>
                {
                    'fileSystemTotalSizeBlocks'=>'7224600',
                    'filesystemBlockSizeBytes'=>'1024',
                    'actualInstallationFootprintBlocks'=>'288',
                    'estimatedUpdatedInstallationFootprintBlocks'=>'288',
                    'deviceId'=>'2050',
                    'filesystemAvailableSizeBlocks'=>'1890908',
                    'estimatedNewInstallationFootprintBlocks'=>'137964728',
                    'scalingFactor'=>'1',
                },
            '/home'=>
                {
                    'fileSystemTotalSizeBlocks'=>'27095116',
                    'filesystemBlockSizeBytes'=>'1024',
                    'actualInstallationFootprintBlocks'=>'130',
                    'estimatedUpdatedInstallationFootprintBlocks'=>'130',
                    'deviceId'=>'2052',
                    'filesystemAvailableSizeBlocks'=>'13063428',
                    'estimatedNewInstallationFootprintBlocks'=>'56547314',
                    'scalingFactor'=>'1024',
                },
            '/tmp/mptest/fs02'=>
                {
                    'fileSystemTotalSizeBlocks'=>'19362',
                    'filesystemBlockSizeBytes'=>'1024',
                    'actualInstallationFootprintBlocks'=>'170',
                    'estimatedUpdatedInstallationFootprintBlocks'=>'170',
                    'deviceId'=>'1794',
                    'filesystemAvailableSizeBlocks'=>'17157',
                    'estimatedNewInstallationFootprintBlocks'=>'81203400',
                    'scalingFactor'=>'1024',
                },
        }
 
=cut
sub getFootprintsByFilesystem {
    my ($self) = @_;
    # keys: mount points
    # values: refs to hashes containing filesystem footprint information.
    my $installationFootprints     = undef;
    my $packageFootprints          = undef;
    my $fileSystemRoot             = undef;
    my $size                       = undef;
    my $packagesHash               = $self->GetSelectedPackageList();
    my @packagesArray              = values(%$packagesHash);
    my $packages                   = \@packagesArray;
    my $installationFileSystemInfo = undef; 
    my $packageFileSystemInfo      = undef;
    my $success                    = 1;
    my $errmsg                     = undef;
    if(!defined $packages) {
        $errmsg = "SDB::Install::PackageManager::getFootprintsByFilesystem: Fatal error: No packages to process.\n";
        $success = 0;
        return (undef, $success, $errmsg);
    }
    $installationFootprints = {};
    foreach my $package (@$packages) {
        if($package->{'skip'}) {
            next;
        }
        ($packageFootprints, $success, $errmsg) =
            $package->getFootprintsByFilesystem();
        if(!$success) {
            return (undef, $success, $errmsg);
        }
        foreach $fileSystemRoot (keys %$packageFootprints) {
            $installationFileSystemInfo = $installationFootprints->{$fileSystemRoot};
            $packageFileSystemInfo = $packageFootprints->{$fileSystemRoot};
            if(!defined $installationFileSystemInfo) {
                # we have not seen this file system before:
                $installationFootprints->{$fileSystemRoot} = $packageFileSystemInfo;
            }
            else {
                # file system info for this file system already exists, merge:
                $installationFileSystemInfo->{'estimatedNewInstallationFootprintBlocks'} +=
                    $packageFileSystemInfo->{'estimatedNewInstallationFootprintBlocks'};
                $installationFileSystemInfo->{'estimatedUpdatedInstallationFootprintBlocks'} +=
                    $packageFileSystemInfo->{'estimatedUpdatedInstallationFootprintBlocks'};
                $installationFileSystemInfo->{'actualInstallationFootprintBlocks'} +=
                    $packageFileSystemInfo->{'actualInstallationFootprintBlocks'};
            }
        }
    }
    # we want to have the dynamic part of the file system info from the OS
    # (i.e. available space) as up to date as possible:
    foreach $fileSystemRoot (keys %$installationFootprints) {
        my (undef, undef, undef, $filesystemAvailableSizeBlocks, undef, undef, $success, $errmsg) = 
            getFileSystemInfo($fileSystemRoot);
        if(!$success) {
            return (undef, $success, $errmsg);
        }
        $installationFileSystemInfo = $installationFootprints->{$fileSystemRoot};
        $installationFileSystemInfo->{'filesystemAvailableSizeBlocks'} = $filesystemAvailableSizeBlocks;
    }
    return (
        $installationFootprints,
        $success,
        $errmsg
    );
}

#----------------------------------------------------------------------------

=functionComment

Checks whether this installation has any packages which don't have
the 'skip' attribute set. Returns 1 if this is the case.

=cut
sub hasNonSkipPackages {
    my ($self)       = @_;
    my $packagesHash               = $self->GetSelectedPackageList();
    my @packagesArray              = values(%$packagesHash);
    my $packages                   = \@packagesArray;
    my $retval       = 0;
    if(defined $packages) {
        foreach my $package (@$packages) {
            if($retval = $package->isNonSkipPackage()) {
                last;
            }
        }
    }
    return $retval;
}

#----------------------------------------------------------------------------

=functionComment

Creates a data structure containing all files of this installation which are currently
in use and belong to a package which is to be installed (i.e. does not have the
'skip' attribute set; this can only happen in the update case.),
together with the commandline of the locking process.
The return value is a reference to a hash where the keys are the PIDs of
the locking processes and the values are arrays of hashes,
one and only one for each locked file.
The keys in these subhashes point to the paths of the locked files and to
the respective commandline of the locking process.
Example:
        {
            '30000'=>
                {
                    'lockedFiles'=>
                        [
                            'bbb.jpg',
                            '/aaa.dll',
                        ],
                    'commandLine'=>'/usr/bin/program01 -someopt',
                },
            '100000'=>
                {
                    'lockedFiles'=>
                        [
                            'bbb.pbx',
                            '/aaa/zzz.htm',
                        ],
                    'commandLine'=>'/usr/bin/program02 -someotheropt',
                }
        }
=cut
sub getFilesInUseByPID {
    my (
        $self,
        $procState
    ) = @_;
    my $success           = 1;
    my $errmsg            = undef;
    my $packagesHash               = $self->GetSelectedPackageList();
    my @packagesArray              = values(%$packagesHash);
    my $packages                   = \@packagesArray;
    if(!defined $packages) {
        $errmsg = "SDB::Install::PackageManager::getFilesInUseByPID: Fatal error: No packages to process.\n";
        $success = 0;
        return (undef, $success, $errmsg);
    }
    my $packageFilesInUse = undef;
    if(!defined $procState) {
        $procState = new SAPDB::Install::ProcState (SDB_PROCSTATE_FLAG_MODULES);
    }
    my %retval;
    my $progress_handler = $self->GetProgressHandler();

    foreach my $package (@$packages) {
        if($package->isNonSkipPackage()) {
            ($packageFilesInUse, $success, $errmsg) =
                $package->getFilesInUseByPID($procState);
            if(!$success) {
                return (undef, $success, $errmsg);
            }
            foreach my $processId (keys %$packageFilesInUse) {
                my $lockedFiles = $packageFilesInUse->{$processId}->{'lockedFiles'};
                my $commandLine = $packageFilesInUse->{$processId}->{'commandLine'};
                if(exists $retval{$processId}) {
                    foreach my $entry (@$lockedFiles) {
                        if(!exists $retval{$processId}->{'lockedFiles'}->{$entry}) {
                            $retval{$processId}->{'lockedFiles'}->{$entry} = $entry;
                        }
                    }
                }
                else {
                    $retval{$processId}->{'lockedFiles'} = arrayToHash($lockedFiles);
                    $retval{$processId}->{'commandLine'} = $commandLine;
                }
                
                if ($packageFilesInUse->{$processId}->{lock_check_failed}){
					$retval{$processId}->{lock_check_failed} = 1;
                }
            }
        }
    }
    # converting from hashes to arrays again:
    foreach my $processId (keys %retval) {
        my $lockedFilesAsHash = $retval{$processId}->{'lockedFiles'};
        my @fkeys = keys %$lockedFilesAsHash;
        $retval{$processId}->{'lockedFiles'} = \@fkeys;
    }
    return (
        \%retval,
        $success,
        $errmsg
    );
}

#----------------------------------------------------------------------------

1;
