package SDB::Install::RemoteHostctrlHosts;

use strict;
use SDB::Install::DebugUtilities;
use SDB::Install::RemoteHosts;
use SDB::Install::SAPHostControl;
use SDB::Install::HdbInstallerOutputParser qw (parseHdbInstallerErrorMessages);

our @ISA = qw (SDB::Install::RemoteHosts);

our $NOT_SUPPORTED = 'Not supported for remote execution via SAPHostControl';


#-------------------------------------------------------------------------------
# Constructor

sub new {
    my $self = shift->SUPER::new (@_);
    #$self->setDump();
    return $self;
}


#-------------------------------------------------------------------------------
sub destroy { }


sub setCredentials {
	my ($self, $username, $password) = @_;
	$self->{username} = $username;
	$self->{password} = $password;
}

#-------------------------------------------------------------------------------
# Returns a hash containing arguments for all hosts.

sub _buildCommonArguments {

    my ($self,
        $instconfig,
        $paramIDs,     # string array containing parameter IDs for all hosts
        $passwordKeys, # string array containing parameter IDs of password parameters
        $optIgnore,    # SDB::Install::Configuration::OptionIgnore
        $optTimeout,   # SDB::Install::Configuration::OptionTimeout
        $optionMap     # map of hostctrl options with their values
       ) = @_;

    my $arguments;

    if (defined $paramIDs) {

        foreach my $id (@$paramIDs) {

            my $param = $instconfig->{params}->{$id};
            my $arg   = (defined $param) ? $param->{hostctrl_opt} : undef;

            if (defined $arg) {
                my $value = $instconfig->getValue($id);
                if ($param->{type} =~ /bool/) {
                    $arguments->{$arg} = ($value) ? 'on' : 'off';
                }
                elsif (defined $value) {
                    $arguments->{$arg} = $value;
                }
            }
        }
    }

    if (defined $passwordKeys) {
        my $xml = $instconfig->getXmlPasswordStream($passwordKeys);
        # getXmlPasswordStream() returns undef, or a ref to an array containing a single entry
        $arguments->{XMLINPUT} = $xml->[0] if (defined $xml);
    }

    my $handlerIgnore = $instconfig->getOptionIgnore();

    if (defined $optIgnore && defined $handlerIgnore) {
        my $ignoreArgs = $handlerIgnore->getSetKeysValidFor($optIgnore);
        my $ignoreValue = $handlerIgnore->getStringFromArgList($ignoreArgs);
        $arguments->{IGNORE} = $ignoreValue if (defined $ignoreValue);
    }

	my $currentOptionTimeout = $instconfig->getOptionTimeout();

    if (defined $optTimeout && defined $currentOptionTimeout) {
        my $timeoutArgs = $currentOptionTimeout->getSetKeysValidFor($optTimeout);
        my $timeoutValue = $currentOptionTimeout->getStringFromArgList($timeoutArgs);
        $arguments->{TIMEOUTS}= $timeoutValue if (defined $timeoutValue);
    }

    if (defined $optionMap) {
        my ($arg, $value);
        while (($arg, $value) = each %$optionMap) {
            $arguments->{$arg} = $value;
        }
    }

    $self->_dumpStructure($arguments, 'Common arguments:');
    return $arguments;
}


#-------------------------------------------------------------------------------
# Returns a hash containing the existing args
# and the arguments for a specififc host.

sub _buildHostArguments {

    my ($self,
        $existingArgs,
        $wantedHost,
        $hostMapOptionMap, # host map containing a map of hostctrl options with their values
       ) = @_;

    if (!defined $hostMapOptionMap || !defined $hostMapOptionMap->{$wantedHost}) {
        return $existingArgs;
    }

    my $hostArgs;

    # copy existing arghostArgsuments to hostArgs
    my ($arg, $value);
    while (($arg, $value) = each %$existingArgs) {
        $hostArgs->{$arg} = $value;
    }

    while (($arg, $value) = each %{$hostMapOptionMap->{$wantedHost}}) {
        $hostArgs->{$arg} = $value;
    }

    $self->_dumpStructure($hostArgs, "Arguments of host '$wantedHost':");
    return $hostArgs,
}


#-------------------------------------------------------------------------------
# Dumps the specified structure if '_isDump' is set.

sub _dumpStructure {

    my ($self, $struct, $headline) = @_;

    if ($self->{_isDump}) {

        my $delimiterLine = "\n" . ('=' x length($headline)) . "\n";

        print($delimiterLine);
        print("$headline\n"); dumpThings($struct, 8, 2);
        print("$delimiterLine\n");
    }
}


#-------------------------------------------------------------------------------
# Executes the specified command parallel on each remote host or on selected
# hosts (parameter $executeOnlyOnHosts).
# Returns 0 if successfully executed (does not return a Perl retcode)
#
# These parameters are set implicitly:
#    --batch
#    --read_password_from_stdin=xml
#    XMLINPUT (hostctrl parameter containing the xml password stream)
#
# Additional hostctrl options can be specified in $paramIDs or in $optionMap.
# If host specific options have to be passed, used $hostMapParamMap.
#
# Parameters specified in $paramIDs, are accessed via $instconfig.
# These parameters should provide the entry 'hostctrl_opt':
#    $instconfig->{params}->{KeepUser}->{hostctrl_opt} = 'KEEP_USER';
#
# Example of $optionMap that defines additional hostctrl options:
#    my $optionMap-> = {'FORCE' => (($force) ? 'on' : 'off')};
#
#
# Example of $hostMapOptionMap that defines additional hostctrl options
#                                                            for specific hosts:
#    my $hostMapOptionMap =
#       # <hostname>     <hostctrl_option>      <value>
#        {'lu000111' => {'HOSTNAME'          => 'lu000111',
#                        'ROLES'             => 'worker,extended_storage_worker',
#                        'GROUP'             => 'default',
#                        'STORAGE_PARTITION' => 2,
#                       },
#         'lu000222' => {'HOSTNAME'          => 'lu000222',
#                        'ROLES'             => 'standby'},
#        };

our $operationOptionAsynchron = {'mTimeout' =>  0};
our $operationOptionSynchron  = {'mTimeout' => -1};

sub executeHostctrlParallel {

    my ($self,
        $operation,          # name of the '.conf' file under 'operations.d' (e.g. 'AddHost')
        $instconfig,         # e.g. SDB::Install::Configuration::AddHost
        $paramIDs,           # string array containing parameter IDs (e.g. ['CheckOnly', 'NoStart'])
        $passwordKeys,       # string array containing parameter IDs of password parameters (e.g. ['Password'])
        $optIgnore,          # SDB::Install::Configuration::OptionIgnore  valid for remote tool
        $optTimeout,         # SDB::Install::Configuration::OptionTimeout valid for remote tool
        $progress_message,   # e.g. "Adding host '%s'..."
        $done_message,       # e.g. "Host '%s' added."
        $error_message,      # e.g. "Adding host '%s' failed!"
        $optionMap,          # map of additional hostctrl options with their values (see example above)
        $hostMapOptionMap,   # host map containing a map of hostctrl options with their values (see example above)
        $executeOnlyOnHosts, # array of hostnames, on which to execute the command
        $doNotFailOnError,   # does not abort the subroutine in case of send/receive error
        $suppressConsoleMsg, # suppresses console output
        $isCollectHostInfo   # suppress logging of the remote output
       ) = @_;
    my $hosts = $self->_filterRemoteHosts($executeOnlyOnHosts);
    my @outputbuffers;
    $self->{output_buffer} = \@outputbuffers;
    if (!defined $hosts) {
        return 1;
    }

    my $commonArgs = $self->_buildCommonArguments($instconfig,
                                                  $paramIDs,
                                                  $passwordKeys,
                                                  $optIgnore,
                                                  $optTimeout,
                                                  $optionMap);
    my $useHttp  = $instconfig->getValue('UseHttp');
    my $sso_cert = $instconfig->getValue('SSOCertificate');

    # As a default setting use the 'sapadm' credentials for remote authentication
    my $username = $instconfig->getValue('HostagentUserName') || 'sapadm';
    my $password = $instconfig->getValue('HostagentPassword');

    # Use <sid>adm credentials if explicitly told to or as a fallback option when no 'sapadm' credentials are present
    if (($self->isUseSidadm() && $instconfig->hasValue('Password')) || !defined($password)) {
        $username = $instconfig->getSAPSystem()->getUser()->getSidAdmName();
        $password = $instconfig->getValue('Password');
    }

	if(defined $self->{username} && defined $self->{password}){
		$username = $self->{username};
		$password = $self->{password};
	}
	if(!$self->_deploySHAConfigurationsIfNeeded($hosts, $username, $password, $useHttp, $sso_cert)){
		$self->getMsgLst()->AddWarning("Deployment of SAP Host Agent configurations on remote hosts failed");
	}
	
    my $rc = 0;
    my @mainmsgs;
    my @hostctrls;
    my @operationIds;
    my ($currHost, $hostArgs, $info, $result, $errorInfo, $lines, $hostctrl);

    #
    # start operation asynchronously for each host
    #

    foreach my $i (0..(@$hosts - 1)) {
        $currHost = $hosts->[$i];
        $hostctrl = $hostctrls[$i] = new SDB::Install::SAPHostControl($currHost,
                                                        $username,
                                                        $password,
                                                        $useHttp,
                                                        $sso_cert);

        $hostArgs = $self->_buildHostArguments($commonArgs,
                                                  $currHost,
                                                  $hostMapOptionMap);

        $info = (defined $progress_message)
                   ? sprintf($progress_message, $currHost)
                   : "Executing hostctrl operation '$operation' on host '$currHost'...";
        $mainmsgs[$i] = ($suppressConsoleMsg) ? $self->AddMessage        ($info)
                                         : $self->AddProgressMessage($info);
        $hostctrl->setMsgLstContext ([$mainmsgs[$i]->getSubMsgLst()]);
        $result = $hostctrl->ExecuteOperation($operation,
                                              $hostArgs,
                                              $operationOptionAsynchron);

        $self->_dumpStructure($result, "ExecuteOperation result of host '$currHost':");


        $errorInfo = (defined $error_message)
                        ? sprintf($error_message, $currHost)
                        : "Executing hostctrl operation '$operation'"
                        . " failed on host '$currHost'!";

        if (!defined $result) {
            $self->appendErrorMessage($errorInfo,
                $hostctrl->errorState() ?
                    $hostctrl->getErrMsgLst () :
                    $hostctrl->getMsgLst ());
            $rc |= 1;
        }
        else  {
            my $faultString = $hostctrl->getLastResult_faultstring();
            $operationIds[$i] = $hostctrl->getLastResult_mOperationID();
            if (!defined $operationIds[$i]){
                $self->appendErrorMessage ($errorInfo . ' ' . $faultString);
                if ($faultString =~ '^.*Invalid\sCredentials.*') {
                    $rc |= 2;
                } else {
                    $rc |= 1;
                }
            }
        }
        $mainmsgs[$i]->endMessage();
    }

    # fetching results
    my ($exitCode, $faultcode);
    my @fetchedLineNumbers;
    my @newlines;
    my $outputHandler;
    my $prevOutputHost = undef;
    $self->{exit_codes} = [];
    while (1){
        if (!(grep {defined $_} @operationIds)){
            # no more pending operations
            last;
        }
        foreach my $i (0..(@$hosts - 1)) {
            $currHost = $hosts->[$i];
            if (!defined $operationIds[$i]){
                next;
            }
            $hostctrl = $hostctrls[$i];
            $result = $hostctrl->GetOperationResults (
                $operationIds[$i],
                $operationOptionAsynchron);

            $self->_dumpStructure($result, "GetOperationResults result of host '$currHost':");

            if (!defined $fetchedLineNumbers[$i]) {
                $fetchedLineNumbers[$i] = 0;
            }
            $lines = $hostctrl->getLastResultOutputlines();
            my $ouputMsgLst = $mainmsgs[$i]->getSubMsgLst ();
            if (defined $lines && @$lines > $fetchedLineNumbers[$i]){
                @newlines = @$lines;
                splice(@newlines,0, $fetchedLineNumbers[$i]);
                $fetchedLineNumbers[$i] = @$lines;
                $outputHandler = $self->getOutputHandler ($currHost);
                my $isContinueOutput = (defined $prevOutputHost &&
                                        ($prevOutputHost eq $currHost)) ? 1 : 0;
                if (!$isContinueOutput && !$isCollectHostInfo) {
                    $self->addRemoteOutputToMsglst($ouputMsgLst,
                                                   $currHost,
                                                   \@newlines);
                    $prevOutputHost = $currHost;
                }

                if (defined $outputHandler || $isContinueOutput) {
                    foreach my $newLine (@newlines){
                        if (defined $outputHandler) {
                            $outputHandler->addLine($newLine);
                        }
                        if ($isContinueOutput && !$isCollectHostInfo) {
                            $ouputMsgLst->addMessage('|  ' . $newLine);
                        }
                    }
                }
            }
            $faultcode = $hostctrl->getLastResult_faultcode();
            $errorInfo = (defined $error_message)
                ? sprintf($error_message, $currHost)
                : "Executing hostctrl operation '$operation'"
                . " failed on host '$currHost'!";
            if (defined $faultcode){
                if ($faultcode == 255){
                    # 255 => error timeout => means EAGAIN
                    next;
                }
                elsif (!defined $hostctrl->getLastResult_exitcode ()){
                    my $faultstring = $hostctrl->getLastResult_faultstring ();
                    $mainmsgs[$i]->getSubMsgLst->addError
                        ("Operation '$operation' finished on host '$currHost' with error: $faultcode, $faultstring");
                    $mainmsgs[$i]->endMessage();
                    if (defined $lines) {
                        $outputbuffers[$i] = join ("\n", @$lines);
                    }
                    $operationIds[$i] = undef;
                    $rc = 1;
                    my $errmsglst = new SDB::Install::MsgLst ();
                    $errmsglst->addError ($faultstring . " (code = $faultcode)");
                    $self->appendErrorMessage($errorInfo, $errmsglst);
                }
            }

            $exitCode = $hostctrl->getLastResult_exitcode ();
            if (defined $exitCode){
                $mainmsgs[$i]->getSubMsgLst->addMessage('');
                $mainmsgs[$i]->getSubMsgLst->addMessage
                    ("Operation '$operation' finished on host '$currHost' with exit code $exitCode");
                $mainmsgs[$i]->endMessage();
                $self->{exit_codes}->[$i] = $exitCode;
                if ($exitCode != 0){
                    my $errlst = parseHdbInstallerErrorMessages ($lines);
                    $self->appendErrorMessage($errorInfo,
                        !$errlst->isEmpty () ? $errlst :
                        ($hostctrl->errorState() ?
                            $hostctrl->getErrMsgLst () :
                            $hostctrl->getMsgLst ()));
                    $rc = 1;
                }
                else{
                    $info = (defined $done_message)
                        ? sprintf($done_message, $currHost)
                        : "Hostctrl operation '$operation' done on host '$currHost'";
                    if ($suppressConsoleMsg) {
                        $self->AddMessage($info);
                    }
                    else {
                        $self->AddProgressMessage($info);
                    }
                }
                # remove finished operation id
                if (defined $lines) {
                    $outputbuffers[$i] = join ("\n", @$lines);
                }
                $operationIds[$i] = undef;
            }
        }
    }

    return $rc;
}

sub _deploySHAConfigurationsIfNeeded {
	my ($self, $hosts, $username, $password, $useHttp, $sso_cert) = @_;
	require LCM::SHA::OperationsDeployer;
	require LCM::SHA::DeploymentStrategy::DeployConfiguration;
	my $deployer = new LCM::SHA::OperationsDeployer();
	my $strategy = new LCM::SHA::DeploymentStrategy::DeployConfiguration($username, $password, $useHttp, $sso_cert);
	my $returnCode = $deployer->deployConfigurations($hosts, $strategy, 1, 1);
	$self->getMsgLst()->addMessage(undef, $deployer->getMsgLst());
	return (defined($returnCode) && $returnCode == 0);
}

#-------------------------------------------------------------------------------
# Executes the specified command serial on each remote host (see executeParallel)
#
# Returns 0 if successfully executed (does not return a Perl retcode)

sub executeHostctrlSerial {

    my ($self,
        $operation,          # name of the '.conf' file under 'operations.d' (e.g. 'AddHost')
        $instconfig,         # e.g. SDB::Install::Configuration::AddHost
        $paramIDs,           # string array containing parameter IDs (e.g. ['CheckOnly', 'NoStart'])
        $passwordKeys,       # string array containing parameter IDs of password parameters (e.g. ['Password'])
        $optIgnore,          # SDB::Install::Configuration::OptionIgnore  valid for remote tool
        $optTimeout,         # SDB::Install::Configuration::OptionTimeout valid for remote tool
        $progress_message,   # e.g. "Adding host '%s'..."
        $done_message,       # e.g. "Host '%s' added."
        $error_message,      # e.g. "Adding host '%s' failed!"
        $optionMap,          # map of additional hostctrl options with their values (see example above)
        $hostMapOptionMap,   # host map containing a map of hostctrl options with their values (see example above)
        $executeOnlyOnHosts, # array of hostnames, on which to execute the command
        $doNotFailOnError,   # does not abort the subroutine in case of send/receive error
        $suppressConsoleMsg  # suppresses console output
       ) = @_;

    my $hosts = $self->_filterRemoteHosts($executeOnlyOnHosts);

    if (!defined $hosts) {
        return 1;
    }

    my $commonArgs = $self->_buildCommonArguments($instconfig,
                                                  $paramIDs,
                                                  $passwordKeys,
                                                  $optIgnore,
                                                  $optTimeout,
                                                  $optionMap);
    my $useHttp  = $instconfig->getValue('UseHttp');
    my $sso_cert = $instconfig->getValue('SSOCertificate');

    # As a default setting use the 'sapadm' credentials for remote authentication
    my $username = $instconfig->getValue('HostagentUserName') || 'sapadm';
    my $password = $instconfig->getValue('HostagentPassword');

    # Use <sid>adm credentials if explicitly told to or as a fallback option when no 'sapadm' credentials are present
    if (($self->isUseSidadm() && $instconfig->hasValue('Password')) || !defined($password)) {
        $username = $instconfig->getSAPSystem()->getUser()->getSidAdmName();
        $password = $instconfig->getValue('Password');
    }

	if(defined $self->{username} && defined $self->{password}){
		$username = $self->{username};
		$password = $self->{password};
	}
	
	if(!$self->_deploySHAConfigurationsIfNeeded($hosts, $username, $password, $useHttp, $sso_cert)){
		$self->getMsgLst()->AddWarning("Deployment of SAP Host Agent configurations on remote hosts failed");
	}
	
    my $rc = 0;
    $self->{exit_codes} = [];
    foreach my $currHost (@$hosts) {

        my $hostctrl = new SDB::Install::SAPHostControl($currHost,
                                                        $username,
                                                        $password,
                                                        $useHttp,
                                                        $sso_cert);

        my $hostArgs = $self->_buildHostArguments($commonArgs,
                                                  $currHost,
                                                  $hostMapOptionMap);

        my $mainmsg;
        my $info = (defined $progress_message)
                   ? sprintf($progress_message, $currHost)
                   : "Executing hostctrl operation '$operation' on host '$currHost'...";
        $mainmsg = ($suppressConsoleMsg) ? $self->AddMessage        ($info)
                                         : $self->AddProgressMessage($info);

        my $result = $hostctrl->ExecuteOperation($operation,
                                                 $hostArgs,
                                                 $operationOptionSynchron);

        $self->_dumpStructure($result, "Execute operation result of host '$currHost':");

        $mainmsg->getSubMsgLst->appendMsgLst($hostctrl->getMsgLst());

        my $errorInfo = (defined $error_message)
                        ? sprintf($error_message, $currHost)
                        : "Executing hostctrl operation '$operation'"
                        . " failed on host '$currHost'!";

        if (!defined $result) {
            $self->setErrorMessage($errorInfo, ($hostctrl->errorState())
                                               ? $hostctrl->getErrMsgLst()
                                               : $hostctrl->getMsgLst());
            $rc |= 1;
            if (!$doNotFailOnError) {
                return $rc;
            }
        }
        else  {
            my $lines = $hostctrl->getLastResultOutputlines();

            if (defined $lines && @$lines) {
                my $outputmsglst = new SDB::Install::MsgLst();
                $self->addRemoteOutputToMsglst($outputmsglst,
                                               $currHost,
                                               $lines);
                $outputmsglst->addMessage('');
                $self->AddSubMsgLst($mainmsg, $outputmsglst);
            }
            my $faultcode = $hostctrl->getLastResult_faultcode();
            my $exitcode  = $hostctrl->getLastResult_exitcode();
            push @{$self->{exit_codes}}, $exitcode;
            if ((defined $exitcode && ($exitcode != 0)) || defined $faultcode) {
                $self->setErrorMessage($errorInfo . ': '
                                      . $hostctrl->getLastResult_faultstring());
                $rc |= 1;
                if (!$doNotFailOnError) {
                    return $rc;
                }
            }
            else {
                $info = (defined $done_message)
                        ? sprintf($done_message, $currHost)
                        : "Hostctrl operation '$operation' done on host '$currHost'";
                if ($suppressConsoleMsg) {
                    $self->AddMessage($info);
                }
                else {
                    $self->AddProgressMessage($info);
                }
            }
        }
        $mainmsg->endMessage();
    }

    return $rc;
}


#-------------------------------------------------------------------------------
# Returns a subset of remote hosts according to $executeOnlyHosts.
#
# Return undef if none of $executeOnlyHosts matches the remote hosts.

sub _filterRemoteHosts {

     my ($self,
         $executeOnlyOnHosts  # Array of hostnames, on which to execute the command
        ) = @_;

    if (!defined $executeOnlyOnHosts) {
        return $self->getHostNames();
    }

    my $hosts = undef;

    if (defined $executeOnlyOnHosts) {

        foreach my $currFromAllHosts (@{$self->{_hosts}}) {

            foreach my $currWantedHost (@{$executeOnlyOnHosts}) {
                if($currFromAllHosts eq $currWantedHost) {
                    push (@$hosts, $currWantedHost);
                }
            }
        }
    }

    if (!defined $hosts) {
        $self->PushError ("Host for remote execution does not exist");
    }

    return $hosts;
}

#-------------------------------------------------------------------------------

sub isHostctrl {
    return 1;
}


#-------------------------------------------------------------------------------
# removes hosts to the internal host array.

sub removeRemoteHosts {

    my ($self,
        $removeHosts # reference to an array containing hosts to be removed
        ) = @_;

    my @newHosts  = ();
    my %removeMap = ();

    foreach my $curr (@$removeHosts) {
        $removeMap{$curr} = 1;
    }

    my $i = 0;
    foreach my $currHost (@{$self->{_hosts}}) {

        if (!exists $removeMap{$currHost}) {
            push @newHosts, $currHost;
        }
        $i++;
    }
    $self->{_hosts} = \@newHosts;
}


#-------------------------------------------------------------------------------
# Enables or disables the dump of arguments and parameters

sub setDump {
    $_[0]->{_isDump} = 1;
}


#-------------------------------------------------------------------------------
# Resets hostctrl user to the default:
# Uses the parameter 'HostagentUserName' (or 'sapadm') and the password
# parameter 'HostagentPassword' for hostctrl.


sub useHostagentUser {
    delete $_[0]->{_useSidadm};
}


#-------------------------------------------------------------------------------
# Uses the sidadm user and the password parameter 'Password' for hostctrl.

sub useSidadm {
    $_[0]->{_useSidadm} = 1;
}

sub isUseSidadm {
    return (exists $_[0]->{_useSidadm}) ? $_[0]->{_useSidadm} : 0;
}


#===============================================================================
# Not supported subroutines

sub authKey {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

sub authOk {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

sub authPassword {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

sub connect {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

sub copyConnections {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

sub copyTree {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

sub deltree {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

sub executeParallel {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

sub executeSerial {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

sub existsFile {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

sub getPubKey {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

sub getUserName {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

sub isAuthenticationOk {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

sub mkdir {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

sub setPassPhrase {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

sub setUserName {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

sub setUserPass  {
    $_[0]->PushError($NOT_SUPPORTED);
    return undef;
}

1;
