package SDB::Install::RemoteHosts;

use strict;
use base "SDB::Install::BaseLegacy";
use SDB::Install::Globals qw($gProductNameSystem);
use SDB::Install::System qw (exec_program loadSSLRequiringPackage);
use SDB::Install::SysVars qw ($path_separator $isWin);
use Fcntl qw(:DEFAULT :mode);
use Time::HiRes qw (usleep);
use SDB::Install::HdbInstallerOutputParser qw (parseHdbInstallerErrorMessages);

our $can_exit_signal;

sub new{
    my $self = shift->SUPER::new();
    $self->{_hosts} = \@_;
    return $self;
}

sub setUserName{
    my ($self, $name, $installSSHKey) = @_;
    if (defined $self->{_user} && ($self->{_user} eq $name)) {
        return 1;
    }
    $self->{_user} = $name;
    my @ent = getpwnam ($name);
    if (@ent){
        my $home = $ent[7];
        if (-f "$home/.ssh/id_rsa"){
            $self->{_privkey} = "$home/.ssh/id_rsa";
        }
        else{
            $self->{_privkey} = undef;
        }
        if (-f "$home/.ssh/id_rsa.pub"){
            $self->{_pubkey} = "$home/.ssh/id_rsa.pub";
        }
        else{
            $self->{_pubkey} = undef;
        }

        if (defined $self->{_pubkey} && defined $self->{_privkey}){
            $self->AddMessage ("rsa keys found");
        }
        else {
            $self->AddMessage ("no rsa keys found");
            if ($installSSHKey) {
                if (! -x '/usr/bin/ssh-keygen') {
                    $self->AddWarning ("'/usr/bin/ssh-keygen' cannot be executed");
                }
                else {
                    my $msglst = new SDB::Install::MsgLst ();
                    $self->AddMessage ("Generating rsa keys", $msglst);
                    if ($self->_createSSHKey($home, $msglst)) {
                        $self->AddMessage("Keys generated successfully");
                    } else {
                        $self->AddWarning("Error occured while geneating SSH keys. Remote hosts authentication will be performed with the root credentials");
                    }
                }
            }
        }

    }
    return 1;
}

sub _createSSHKey {
    my ($self, $homedir, $msglst) = @_;
    if (defined $self->{_privkey}) {
        unlink $self->{_privkey};
    }
    my $rc = SDB::Install::System::exec_program ('/usr/bin/ssh-keygen',
            ['-t', 'rsa', '-b', '4096', '-N', '', '-f', "$homedir/.ssh/id_rsa"],
            $msglst);

    if (defined $rc && $rc == 0){
        if (-f "$homedir/.ssh/id_rsa" && -f "$homedir/.ssh/id_rsa.pub"){
            $self->{_privkey} = "$homedir/.ssh/id_rsa";
            $self->{_pubkey} = "$homedir/.ssh/id_rsa.pub";
            return 1;
        }
    }
    return 0;
}

sub _check_can_exit_signal{
    require SDB::Install::Version;
    my $versionString = Net::SSH2->version ();
    $versionString =~ s/[^\d\.].*$//;
    my $version = new SDB::Install::Version (split ('\.', $versionString));
    my $minVersion = new SDB::Install::Version (1,2,8);
    if ($minVersion->isNewerThan ($version)){
        #
        # used SDK doesn't support it
        #
        $can_exit_signal = 0;
    }
    else{
        #
        # try to check libssh2 runtime
        #
        $can_exit_signal = 0; #default is false
        require SDB::Install::RPM;
        my $rpm = new SDB::Install::RPM ();
        my $libssh2Package = $rpm->getPackage ('libssh2-1');
        if (!defined $libssh2Package){
            $libssh2Package = $rpm->getPackage ('libssh2');
        }
        if (!defined $libssh2Package){
            return;
        }
        $versionString = $libssh2Package->getVersion ();
        $versionString =~ s/[^\d\.].*$//;
        if ($versionString){
            $version = new SDB::Install::Version (split ('\.', $versionString));
            if (!$minVersion->isNewerThan ($version)){
                $can_exit_signal = 1;
            }
        }
    }
}

sub connect{
    my ($self) = @_;
    $self->{_connections} = [];
    my $socket;
    my $rc = 0;
    my $errmsglst = new SDB::Install::MsgLst ();
    if (!loadSSLRequiringPackage ('Net::SSH2',$errmsglst, $errmsglst)){
        $self->setErrorMessage ("Loading libssh2 failed", $errmsglst);
        return 2;
    }

    if (!defined $can_exit_signal){
        _check_can_exit_signal ();
    }

    foreach my $i (0 .. (scalar @{$self->{_hosts}} -1)){
        $socket = Net::SSH2->new ();
        #$socket->debug(1);
        if (!$socket->connect ($self->{_hosts}->[$i])){
            $self->PushError (sprintf ("Connect to host '$self->{_hosts}->[$i]' failed: rc=" . join(', ', grep { $_ ne '' } $socket->error())));
            $rc = 1;
            next;
        }
        $self->{_connections}->[$i] = $socket;
    }
    return $rc;
}

#
# key passphrase
#

sub setPassPhrase{
    $_[0]->{_passphrase} =  $_[1];
}

#
# user password
#

sub setUserPass{
    $_[0]->{_password} = $_[1];
}

sub addHost{
    push @{$_[0]->{_hosts}}, $_[1];
}

sub getNumberOfHosts{
    return scalar @{$_[0]->{_hosts}};
}

sub authKey{
    my ($self, $ignorePubKeyMissing) = @_;
    my $rc = 0;
    if (!$self->{_user}){
        $self->AddError ('User name unknown');
        return 1;
    }
    if (!$self->{_pubkey}){
        my $msg = 'No public rsa key';
        if ($ignorePubKeyMissing) {
            $self->AddMessage($msg);
        }
        else {
            $self->AddError ($msg);
        }
        return 2;
    }
    if (!$self->{_privkey}){
        $self->AddError ('No private rsa key');
        return 4;
    }

    foreach my $i (0 .. (scalar @{$self->{_hosts}} -1)){
        if (!defined $self->{_connections}->[$i]){
            my $msg = "authKey failed: host '$self->{_hosts}->[$i]' is not connected";
            if ($ignorePubKeyMissing) {
                $self->AddMessage($msg);
            }
            else {
                $self->AddError($msg);
            }
            $rc |= 8;
            next;
        }
        if ($self->{_connections}->[$i]->auth_ok()){
            next;
        }
        $self->{_connections}->[$i]->auth_publickey ($self->{_user}, $self->{_pubkey}, $self->{_privkey},$self->{_passphrase});
        if (!$self->{_connections}->[$i]->auth_ok()){
            my $msg = sprintf ("authKey for host '$self->{_hosts}->[$i]' failed: rc=" . join(', ', grep { $_ ne '' } $self->{_connections}->[$i]->error));
            if ($ignorePubKeyMissing) {
                $self->AddMessage($msg);
            }
            else {
                $self->AddError($msg);
            }
            $rc |= 16;
        }
    }
    return $rc;
}


sub getPubKey{
    my ($self) = @_;
    if (!$self->{_pubkey}){
        $self->AddError ("No public rsa key");
        return undef;
    }
    if (!open (FH, $self->{_pubkey})){
        $self->AddError ("Cannot open public rsa key '$self->{_pubkey}': $!");
        return undef;
    }
    my $keystr = <FH>;
    close (FH);
    return $keystr;
}


sub authOk{
    my ($self) = @_;
    my $rc = 0;
    foreach my $i (0 .. (scalar @{$self->{_hosts}} -1)){
        if (!defined $self->{_connections}->[$i]){
            $self->PushError ("Host '$self->{_hosts}->[$i]' not accessible");
            $rc |= 1;
            next;
        }
        if (!$self->{_connections}->[$i]->auth_ok()){
            $self->PushError ("Connection to host '$self->{_hosts}->[$i]' is not authorized");
            $rc |= 2;
        }
    }
    return $rc;
}

sub isAuthenticationOk{
    my ($self) = @_;
    my $connections = $self->{_connections};
    foreach my $i (0 .. (scalar @{$self->{_hosts}} - 1)){
        if (!defined($connections->[$i]) || !$connections->[$i]->auth_ok()){
			return 0;
        }
    }
    return 1;
}

sub _canAuthPassword{
    my ($self, $i) = @_;
    if (!defined $self->{_canAuthPassword}){
        $self->{_canAuthPassword} = [];
    }
    if (!defined $self->{_canAuthPassword}->[$i]){
        my $tmpSocket = Net::SSH2->new ();
        if (!$tmpSocket->connect ($self->{_hosts}->[$i])){
            print sprintf ("Connect to host '$self->{_hosts}->[$i]' failed: rc=" . join(', ', grep { $_ ne '' } $tmpSocket->error()) . "\n");
            return undef;
        }
        $self->{_canAuthPassword}->[$i] =
            (grep {$_ eq 'password'} $tmpSocket->auth_list()) ?
            1 : 0;
    }
    return $self->{_canAuthPassword}->[$i];
}

sub authPassword{
    my ($self, $installSSHKey) = @_;
    my $rc = 0;
    if (!$self->{_user}){
        $self->AddError ('User name not specified');
        return 1;
    }
    if (!$self->{_password}){
        $self->AddError ('Password not specified');
        return 2;
    }
    my $authError;
    foreach my $i (0 .. (scalar @{$self->{_hosts}} -1)){
        if (!defined $self->{_connections}->[$i]){
            $self->PushError ("Host '$self->{_hosts}->[$i]' not accessible");
            $rc |= 8;
            next;
        }
        if ($self->{_connections}->[$i]->auth_ok()){
            next;
        }
        $authError = 0;
        if ($self->_canAuthPassword ($i)){
            if(!$self->{_connections}->[$i]->auth_password ($self->{_user}, $self->{_password})){
                $authError = 1;
            }
        }
        else{
            if (!$self->{_connections}->[$i]->auth_keyboard ($self->{_user}, $self->{_password})){
                $authError = 1;
            }
        }
        if ($authError){
            my $msg = $self->PushError ("Password authorization on host '$self->{_hosts}->[$i]' failed: " .
                join (', ', $self->{_connections}->[$i]->error()));
            $rc |= 16;
            if ($self->{_userCouldBeChanged}){
                # support several attempts with different users
                # => reconnect
                $self->{_connections}->[$i] = Net::SSH2->new ();
                $self->{_connections}->[$i]->connect ($self->{_hosts}->[$i]);
            }
        }
        else {
            $self->AddMessage ("Connection to host '$self->{_hosts}->[$i]' established");
            my $keystr = ($installSSHKey) ? $self->getPubKey() : undef;
	        if ($keystr){
	            $self->{_connections}->[$i]->blocking (1);
	            my $sftp = $self->{_connections}->[$i]->sftp();            
	            if ($sftp){
	                $self->AddMessage ("Distributing public rsa key to host '$self->{_hosts}->[$i]'");
	                if (!defined $sftp->stat ('.ssh')){
	                    $sftp->mkdir ('.ssh', 0700);
	                }
	                my $file = $sftp->open ('.ssh/authorized_keys', O_CREAT|O_WRONLY|O_APPEND, 0600);
	                if ($file){
	                    $file->seek($file->stat ()->{'size'});
	                    if (!defined $file->write ($keystr)){
	                        $self->AddError ("Cannot distribute public rsa key: write to '.ssh/authorized_keys' failed (host '$self->{_hosts}->[$i]')");
	                    }
	                }
	                else{
	                    $self->AddError ("Cannot distribute public rsa key: cannot open'.ssh/authorized_keys' (host '$self->{_hosts}->[$i]')");
	                }
	            } 
	        }
        }
    }
    return $rc;
}


#-------------------------------------------------------------------------------
# Executes the specified command parallel on each remote host or on selected
# hosts (parameter $executeOnlyOnHosts).
#
# Calling a specific main program
# -------------------------------
# If a '-main' parameter is specified, the main method of that Perl class
# is executed instead of the specified program.
# Example:
#    '-main SDB::Install::App::Console::DBUpgradeHost::main'
# Note:
#    The C-program (helpers.c) does not detect a main parameter followed by
#    an equal sign:
#         '-main=SDB...' is wrong, but '-main SDB...' is correct
#
# Passwords are passed via input stream to the host.
# --------------------------------------------------
# If a password is required, add the parameter '--read_password_from_stdin[=raw]'
# to the command and specify a password array for the parameter $stdin_input.
# Example: ['SysAdm123456', 'DBsys12345678']
#
# If xml password stream is wanted, add the parameter
# '--read_password_from_stdin=xml' to the command.
# The function '$instconfig->getXmlPasswordStream' returns the required array
# for the parameter $stdin_input.
#
#
# Template values
# ---------------
# If $cmd contains template symbols '%s', these placeholders are replaced by
# the values contained in an array for each host.
# Example: $cmd        = '... --hostname=%s --roles=%s%s'
#          $cmdTmplRpl = [['lu000111', 'worker', ' --group=default'],
#                         ['lu000222', 'standby', '']]
#
#          '... --hostname=lu000111 --roles=worker --group=default' is sent to $self->{_hosts}->[0]
#          '... --hostname=lu000222 --roles=standby'                is sent to $self->{_hosts}->[1]

sub executeParallel{
    my $rc = 0;
    my ($self,
        $cmd,                # e.g. 'hana/shared/<SID>/global/hdb/install/bin/hdbreg <parameters>'
        $progress_message,   # e.g. "Upgrading host '%s'..."
        $done_message,       # e.g. "Upgrade host '%s' done"
        $error_message,      # e.g. "Upgrade host '%s' failed"
        $stdin_input,        # Password input stream (see above)
        $cmdTmplRpl,         # Template values (see above)
        $executeOnlyOnHosts, # Array of hostnames, on which to execute the command
                             # (use when you don't want to execute it on all hosts)
        $doNotFailOnError,
        $isNotVerbose,       # suppress console output
        $isCollectHostInfo   # suppress logging of the remote output
       ) = @_;
    
    if (!defined $self->{channels}){
        $self->{channels} = [];
    }
    
    my $channels = $self->{channels};
    my $connections = $self->{_connections};
    my $hosts;
    if($executeOnlyOnHosts) { 
        my @ownChannels;
        my @ownConnections;
        my $i = 0;          
        $hosts = $executeOnlyOnHosts;
        foreach my $host (@{$self->{_hosts}}) {
            foreach my $ownHost (@{$executeOnlyOnHosts}) {
                if($host eq $ownHost) {
                    push (@ownChannels, $channels->[$i]);
                    push (@ownConnections, $connections->[$i]);
                }
            }
            $i++;
        }
        $channels = \@ownChannels;
        $connections = \@ownConnections;
    } else {
        $hosts = $self->{_hosts};
    }
        
    my ($buffer,@rcs,@rest,@lines, $got,@output_buffer,$len, $written);
    my ($msglst, $errlst);

    my @stdin_buffers;

    #
    # copy stdin buffer for each host
    #

    if (defined $stdin_input && @$stdin_input){
       foreach my $i (0 .. (scalar @$hosts -1)){
            push @stdin_buffers, [@$stdin_input];
       }
    }

    my $cmdTemplate;
    if (defined $cmdTmplRpl){
        $cmdTemplate = $cmd;
    }

    #
    # spawn remote processes
    #
    my @mainmsg;
    foreach my $i (0 .. (scalar @{$hosts} -1)){
        $output_buffer[$i] = '';
    	if (defined $executeOnlyOnHosts) {
    		my $currentHost = $hosts->[$i];
            if (not (grep { /^$currentHost/i } @{$self->{_hosts}})) {
    			next;
    		}
    	}    	
        if (!defined $connections->[$i]){
            $self->PushError ("executeParallel failed: host '$hosts->[$i]' is not connected");
            $rc |= 1;
            next;
        }
        if (!$connections->[$i]->auth_ok ()){
            $self->PushError ("connection to host '$hosts->[$i]' is not authorized");
            $rc |= 2;
            next;
        }

        if (defined $cmdTemplate){
            $cmd = sprintf ($cmdTemplate,@{$cmdTmplRpl->[$i]});
        }
        $connections->[$i]->blocking (1);
        $channels->[$i] = $connections->[$i]->channel ();
        if (!defined $channels->[$i]) {
            $self->PushError("Could not open a channel over the connection to host '$hosts->[$i]'");
            $rc |= 16;
            next;
        }
        $channels->[$i]->blocking (0);
        $channels->[$i]->ext_data ('merge');
        
        if ($progress_message){
          if ($isNotVerbose){
            $mainmsg[$i] = $self->AddMessage (sprintf ($progress_message ,$hosts->[$i])); 
          } else {
            $mainmsg[$i] = $self->AddProgressMessage (sprintf ($progress_message ,$hosts->[$i]));
          }
        }
        else{
            $mainmsg[$i] = $self->AddMessage ("Performing '$cmd' on host '$hosts->[$i]'");            
        }
        $channels->[$i]->exec ($cmd);
    }

    #
    # i/o loop
    #

    my ($outputmsglst, $stdin_string, $outputHandler, $endNewLines);
    while (1){
        foreach my $i (0 .. (scalar @{$hosts} -1)){
            next if defined $rcs[$i]; # already finished
            next if !defined $channels->[$i]; # pervious error, e.g. no connection

            if (defined $stdin_buffers[$i] && @{$stdin_buffers[$i]}){
                #
                # write to stdin
                #

                # get the current input string
                $stdin_string =  $stdin_buffers[$i]->[0];
                $stdin_string .= "\n";
                $len = length ($stdin_string);
                $written = $channels->[$i]->write ($stdin_string);
                if ($written == $len){
                    # remove the input string from the host buffer
                    shift @{$stdin_buffers[$i]};
                    $channels->[$i]->send_eof();
                }
            }

            #
            # read stdout/stderr
            #

            $got = $channels->[$i]->read ($buffer, 1024);
            if (!defined $got){
                    if ($channels->[$i]->error == -37){
                        # EAGAIN
                        # read failed => no data present
                        usleep (300000); # 0.3s
                        next;
                    }
            }
            if (defined $got && $got==0 && $channels->[$i]->eof){

                #
                # output channel closed
                # => process terminated
                #

                if ($rest[$i]){
                    $outputHandler = $self->getOutputHandler ($hosts->[$i]);
                    if (defined $outputHandler){
                        $outputHandler->addLine($rest[$i]);
                    }
                }
                if($progress_message && !$isNotVerbose) {
                    $mainmsg[$i]->endMessage ();
                } else {
                    if ($rest[$i]){
                        $self->AddMessage ("$hosts->[$i]: $rest[$i]");
                    }
                }
                if (defined $cmdTemplate){
                    $cmd = sprintf ($cmdTemplate,@{$cmdTmplRpl->[$i]});
                }
                $channels->[$i]->close ();
                $channels->[$i]->wait_closed ();
                my $signal;
                if ($can_exit_signal){
                    $signal = $channels->[$i]->exit_signal();
                }
                if ($signal){
                    $rcs[$i] = 130;
                }
                else{
                    $rcs[$i] = $channels->[$i]->exit_status();
                }

                $channels->[$i] = undef;
                $msglst = new SDB::Install::MsgLst ();
                $outputmsglst = new SDB::Install::MsgLst ();
                $outputmsglst->addMessage("Executing command on host '$hosts->[$i]':");
                my $isFirst = 1;
                foreach my $cmdItem (split(/[\s*]--/, $cmd)) {
                    if ($isFirst) {
                        $outputmsglst->addMessage("    $cmdItem");
                        $isFirst = 0;
                    }
                    else {
                        $outputmsglst->addMessage("        --$cmdItem");
                    }
                }
                if (!$isCollectHostInfo) {
                    $self->addRemoteOutputToMsglst($outputmsglst,
                                               $hosts->[$i],
                                               [split("\n", $output_buffer[$i])]);
                }
                $outputmsglst->addMessage('');
                $outputmsglst->addMessage("Command finished on host '$hosts->[$i]' with exit code $rcs[$i]");
                $self->AddSubMsgLst ($mainmsg[$i], $outputmsglst);
                if ($rcs[$i] != 0 && !$doNotFailOnError){
                    $msglst = parseHdbInstallerErrorMessages ([split("\n", $output_buffer[$i])]);
                    if ($error_message){
                        $self->PushError (sprintf ($error_message, $hosts->[$i]), $msglst);
                    }
                    else{
                        $self->PushError ("Command '$cmd' finished on host '$hosts->[$i]' with error (rc = $rcs[$i]". ($signal ? " signal = $signal" : '') .")", $msglst);
                    }
                    $rc |= 8;
                }
                else{
                    if ($done_message){
                     if ($isNotVerbose){
                        $self->AddMessage (sprintf ($done_message ,$hosts->[$i]), $msglst);
                     } else {
                        $self->AddProgressMessage (sprintf ($done_message ,$hosts->[$i]), $msglst);
                     }
                    }
                    else{
                        $self->AddMessage ("Cmd '$cmd' done on host '$hosts->[$i]'", $msglst);
                    }
                }
                next;
            }
            $output_buffer[$i] .= $buffer;
            my $currLines = (($i <= $#rest) && defined $rest[$i])
                            ? $rest[$i] . $buffer : $buffer;
            @lines = split ("\n", $currLines);
            ($endNewLines) = ($currLines =~ /(\n+)$/);
            if ($endNewLines){
                foreach my $n (1 .. length ($endNewLines)){
                    push @lines, '';
                }
            }
            if (@lines == 1){
                    $rest[$i] = $currLines;
                    next;
            }
            $rest[$i] = pop @lines;
            $outputHandler = $self->getOutputHandler ($hosts->[$i]);
            if (defined $outputHandler){
                foreach my $newLine (@lines){
                    $outputHandler->addLine($newLine);
                }
            }
        }
        my $finished = 1;
        foreach my $i (0 .. (scalar @{$hosts} -1)){
            next if !defined $channels->[$i];
            if (!defined $rcs[$i]){
                $finished = 0;
                last;
            }
        }
        if ($finished){
            last;
        }
    }
    $self->{exit_codes} = \@rcs;
    $self->{output_buffer} = \@output_buffer;
    return $rc;
}

sub getOutputBuffer{
    return $_[0]->{output_buffer};
}

sub freeOutputBuffer{
    delete $_[0]->{output_buffer};
}


#-------------------------------------------------------------------------------
# Executes the specified command serial on each remote host (see executeParallel)

sub executeSerial{
    my $rc = 0;
    my ($self,
        $cmd,                # e.g. 'hana/shared/<SID>/global/hdb/install/bin/hdbreg <parameters>'
        $force,              # do not fail on error
        $progress_message,   # e.g. "Upgrading host '%s'..."
        $done_message,       # e.g. "Upgrade host '%s' done"
        $error_message,      # e.g. "Upgrade host '%s' failed"
        $stdin_input,        # Password input stream (see executeParallel)
        $cmdTmplRpl,         # Template values (see executeParallel)
       ) = @_;

    if (!defined $self->{channels}){
        $self->{channels} = [];
    }
    my $channels = $self->{channels};
    my $connections = $self->{_connections};
    my $hosts = $self->{_hosts};
   #my ($output_buffer, $buffer,$rest,$ec, @lines, $got, $msglst, $errlst,
    my ($output_buffer, $buffer, $ec, $got, $msglst, $errlst,
        $outputmsglst, $len, $written, $stdin_string);

    my $cmdTemplate;
    if ($cmdTmplRpl){
        $cmdTemplate = $cmd;
    }

    my @stdin_buffers;

    #
    # copy stdin buffer for each host
    #

    if (defined $stdin_input && @$stdin_input){
       foreach my $i (0 .. (scalar @{$self->{_hosts}} -1)){
            push @stdin_buffers, [@$stdin_input];
       }
    }
    my $mainmsg;
    foreach my $i (0 .. (scalar @{$self->{_hosts}} -1)){
        $output_buffer = '';
        if (!defined $connections->[$i]){
            $self->PushError ("executeSerial failed: host '$hosts->[$i]' is not connected");
            $rc |= 1;
            next;
        }
        if (!$connections->[$i]->auth_ok ()){
            $self->PushError ("connection to host '$hosts->[$i]' is not authorized");
            $rc |= 2;
            next;
        }
        if (defined $cmdTemplate){
            $cmd = sprintf ($cmdTemplate,@{$cmdTmplRpl->[$i]});
        }
        $connections->[$i]->blocking (1);
        $channels->[$i] = $connections->[$i]->channel ();
        if (!defined $channels->[$i]) {
            $self->PushError("Could not open a channel over the connection to host '$hosts->[$i]'");
            $rc |= 16;
            next;
        }
        $channels->[$i]->blocking (0);
        $channels->[$i]->ext_data ('merge');
        if ($progress_message){
            $mainmsg = $self->AddProgressMessage (sprintf ($progress_message, $self->{_hosts}->[$i]));
        }
        else{
            $mainmsg = $self->AddMessage ("Performing '$cmd' on host '$self->{_hosts}->[$i]'");
        }
        $channels->[$i]->exec ($cmd);
        while (1){
            if (defined $stdin_buffers[$i] && @{$stdin_buffers[$i]}){

                #
                # write to stdin
                #

                ($stdin_string) =  $stdin_buffers[$i]->[0];
                $stdin_string .= "\n";
                $len = length ($stdin_string);
                $written = $channels->[$i]->write ($stdin_string);
                if ($written == $len){
                    shift @{$stdin_buffers[$i]};
                    $channels->[$i]->send_eof();
                }
            }

            $got = $channels->[$i]->read ($buffer, 1024);
            if (!defined $got){
                    if ($channels->[$i]->error == -37){
                        # EAGAIN
                        # read failed => no data present
                        usleep (300000); # 0.3s
                        next;
                    }
            }

            if (defined $got && $got==0 && $channels->[$i]->eof){
                #if ($rest){
                #   $self->AddProgressMessage ("$hosts->[$i]: $rest");
                #}
                $mainmsg->endMessage ();

                $channels->[$i]->close ();
                $channels->[$i]->wait_closed ();

                my $signal;
                if ($can_exit_signal){
                    $signal = $channels->[$i]->exit_signal();
                }

                if ($signal){
                    $ec = 130;
                }
                else{
                    $ec = $channels->[$i]->exit_status();
                }

                $channels->[$i] = undef;

                $msglst = new SDB::Install::MsgLst ();
                $outputmsglst = new SDB::Install::MsgLst ();
                $outputmsglst->addMessage("Running command '$cmd'");
                $self->addRemoteOutputToMsglst($outputmsglst,
                                               $hosts->[$i],
                                               [split("\n", $output_buffer)]);
                $outputmsglst->addMessage('');
                $outputmsglst->addMessage("Command finished on host '$hosts->[$i]' with exit code $ec");
                $self->AddSubMsgLst ($mainmsg, $outputmsglst);
                if ($ec != 0){
                    if ($error_message){
                        $self->PushError (sprintf ($error_message, $hosts->[$i]), $msglst);
                    }
                    else{
                        $self->PushError ("Command '$cmd' finished on host '$hosts->[$i]' with error (rc = $ec". ($signal ? " signal = $signal" : '') .")",$msglst);
                    }
                    $rc |= 8;
                    if (!$force){
                        return $rc;
                    }
                }
                else{
                    if ($done_message){
                        $self->AddProgressMessage (sprintf ($done_message, $hosts->[$i]), $msglst);
                    }
                    else{
                        $self->AddMessage ("Cmd '$cmd' done on host '$hosts->[$i]'", $msglst);
                    }
                }
                last;
            }
            $output_buffer .= $buffer;
            #@lines = split ("\n",$rest . $buffer);
            #if (@lines == 1){
            #        $rest .= $buffer;
            #        next;
            #}
            #$rest = pop @lines;
            #foreach my $line (@lines){
            #    $self->AddProgressMessage ("$hosts->[$i]: $line");
            #}
        }
    }
    return $rc;
}


#-------------------------------------------------------------------------------
# Writes a header and each output line to the specified message list.

sub addRemoteOutputToMsglst {
    my ($self,
        $outputMsgLst,
        $host,
        $outputLines, # reference to an array of output line
       ) = @_;

    $outputMsgLst->addMessage('');
    $outputMsgLst->addMessage(" ----- Output from host '$host' -----");
    $outputMsgLst->addMessage('|');
    my $isFirst = 1;

    foreach my $currLine (@$outputLines) {
        if (!$isFirst || ((length($currLine) > 0) && ($currLine !~ /^\s*$/))) {
            $outputMsgLst->addMessage('|  ' . $currLine);
            $isFirst = 0;
        }
    }
}


sub copyTree{
    my ($self,$localSrcDir, $remoteDestDir, $items) = @_;
    my ($lPath, $rPath);
    my $rc = 0;

    my $connections = $self->{_connections};
    my ($rFile, $offset, $written, $len, $sftp, $buf,$code, $str, $dsc);

    foreach my $item (@$items){
        $lPath = $localSrcDir . $path_separator . $item;
        $rPath = $remoteDestDir . $path_separator . $item;
        my @statbuf = stat ($lPath);
        my $bufSize= $statbuf [11] || 32768;
        if (! -e _){
            $self->PushError ("local path '$lPath' doesn't exist");
            $rc |= 1;
            next;
        }
        if (-f _){
            if (!open (SRC, $lPath)){
                $self->PushError ("Cannot open local file '$lPath': $!");
                $rc |= 2;
                next;
            }
            foreach my $i (0 .. (scalar @{$self->{_hosts}} -1)){
                $self->{_connections}->[$i]->blocking (1);
                $sftp = $self->{_connections}->[$i]->sftp();
                $rFile = $sftp->open ($rPath, O_CREAT|O_WRONLY, $statbuf[2]);
                if (!$rFile){
                    ($code, $str, $dsc) = $sftp->error();
                    $self->PushError ("Cannot create remote file '$rPath': $str - $dsc");
                    $rc |= 4;
                    next;
                }
                $offset = 0;
                sysseek (SRC,0,0);
                while($len = sysread(SRC,$buf,$bufSize)){
                    if (!defined $len){
                        next if $! =~ /^Interrupted/;
                        $self->PushError ($lPath . ' read failure: ' . $!);
                        $rc |= 8;
                        last;
                    }
                   $rFile->seek ($offset);
                   $written = $rFile->write ($buf);
                   if (!defined $written){
                        $self->PushError ('Write failure: ' . $!);
                        return undef;
                    }
                    $len-=$written;
                    $offset+=$written;
                }
            }
        }
        elsif (-d _){
            if (!opendir (DH, $lPath)){
                $self->PushError ("Cannot open local directory '$lPath': $!");
                $rc |= 16;
                next;
            }
            my @subitems = grep {!/^\.{1,2}$/} readdir (DH);
            closedir (DH);
            foreach my $i (0 .. (scalar @{$self->{_hosts}} -1)){
                $self->{_connections}->[$i]->blocking (1);
                $sftp = $self->{_connections}->[$i]->sftp();
                if (!defined $sftp->stat ($rPath)){
                    if (!defined $sftp->mkdir ($rPath, $statbuf[2])){
                        ($code, $str, $dsc) = $sftp->error ();
                        if ($code != 0){
                            $self->PushError ("Cannot create remote directotry '$rPath': $str - $dsc");
                            $rc |= 32;
                            next;
                        }
                    }
                }
            }
            $rc |= $self->copyTree ($lPath, $remoteDestDir . $path_separator . $item, \@subitems);
        }
    }
    return $rc;
}

sub deltree{
    my ($self,$remotePath, $sftp, $errlst) = @_;
    my $rc = 0;
    if (!defined $errlst){
        $errlst = $self;
    }
    my ($code, $str, $dsc);
    if (defined $sftp){
        my $stat = $sftp->stat ($remotePath);
        if (!defined $stat){
           # already gone
           return 0;
        }
        if (S_ISDIR ($stat->{mode})){
            my $dir = $sftp->opendir ($remotePath);
            if (!defined $dir){
                ($code, $str, $dsc) = $sftp->error ();
                $errlst->PushError ("Cannot open directory '$remotePath': $str - $dsc");
                $rc |= 1;
                return $rc;
            }
            my $entry;
            while ($entry = $dir->read()){
                next if $entry->{name} eq '.';
                next if $entry->{name} eq '..';
                $rc |= $self->deltree ($remotePath . $path_separator . $entry->{name}, $sftp, $errlst);
            }
            if ($rc == 0){
                $sftp->rmdir ($remotePath);
                ($code, $str, $dsc) = $sftp->error ();
                if ($code != 0){
                    $errlst->PushError ("Cannot delete directory '$remotePath': $str - $dsc");
                    $rc |= 2;
                }
            }
        }
        else{
            $sftp->unlink ($remotePath);
            ($code, $str, $dsc) = $sftp->error ();
            if ($code != 0){
                $errlst->PushError ("Cannot delete file '$remotePath': $str - $dsc");
                $rc |= 4;
            }
        }
        return $rc;
    }

    my $connections = $self->{_connections};
    my $hosterrlst;
    my $lrc;
    foreach my $i (0 .. (scalar @{$self->{_hosts}} -1)){
        $connections->[$i]->blocking (1);
        $sftp = $self->{_connections}->[$i]->sftp();
        $hosterrlst = new SDB::Install::MsgLst ();
        $lrc = $self->deltree ($remotePath, $sftp, $hosterrlst);
        if ($lrc != 0){
            $errlst->PushError ("Cannot remove $remotePath on host '$self->{_hosts}->[$i]'", $hosterrlst);
            $rc |= $lrc;
        }
    }
    return $rc;
}

sub mkdir{
    my ($self,$rPath, $mode) = @_;
    my $rc = 0;
    my $connections = $self->{_connections};
    my ($sftp,$code, $str, $dsc);
    foreach my $i (0 .. (scalar @{$self->{_hosts}} -1)){
        $self->{_connections}->[$i]->blocking (1);
        $sftp = $self->{_connections}->[$i]->sftp();
        $sftp->mkdir ($rPath, $mode);
        ($code, $str, $dsc) = $sftp->error();
        if ($code != 0){
            $self->PushError ("Cannot create remote directotry '$rPath': $str - $dsc");
            $rc |= 1;
            next;
        }
    }
    return $rc;
}

sub existsFile{
    my ($self,$rPath, $isDirectory, $outStatError, $hosts) = @_;
    my $rc = 0;
    my $connections = $self->{_connections};
    my ($sftp,$code, $str, $dsc);
    $self->ResetError();
    if (!defined $hosts){
        $hosts = $self->{_hosts};
    }

    my $i;
    foreach my $host (@$hosts){
        $i = $self->getIndexByHostName ($host);
        if (!defined $i){
            $self->appendErrorMessage ("host '$host' is not handled in this RemoteHosts instance");
            $rc |= 16;
            next;
        }
        $self->{_connections}->[$i]->blocking (1);
        $sftp = $self->{_connections}->[$i]->sftp();
        if (!defined $sftp){
            ($code,$str, $dsc) = $self->{_connections}->[$i]->error ();
            $self->appendErrorMessage ("Cannot get sftp channel for host '$self->{_hosts}->[$i]': $code, $str - $dsc");
            $rc |= 8;
            next;
        }
        
        my $stat = $sftp->stat ($rPath);
        if(!defined $stat) {
        	$stat = $sftp->stat ($rPath);
        }
        
        if (!defined $stat){
            my @statError = $sftp->error ();
            if ($statError[0] == 2){
                $self->PushError ("File '$rPath' does not exist on host '$self->{_hosts}->[$i]': $statError[1]");
            }
            elsif ($statError[0] == 3){
                $self->PushError ("File '$rPath' cannot be accessed on host '$self->{_hosts}->[$i]': $statError[1]");
            }
            else{
                $self->PushError ("stat ('$rPath') on host '$self->{_hosts}->[$i]' failed: " .
                    join (', ', @statError));
            }
            if (defined $outStatError){
                push @$outStatError, [@statError, $self->{_hosts}->[$i]];
            }
            $rc |= 1;
            next;
        }
        
        if($isDirectory){
            if (!S_ISDIR ($stat->{mode})){
	         $self->PushError ("'$rPath' is not a directory on host '$self->{_hosts}->[$i]'");
		 $rc |= 4;
            }
        }
        else{
            if (!S_ISREG ($stat->{mode})){
                $self->PushError ("File '$rPath' is no regular file on host '$self->{_hosts}->[$i]'");
		$rc |= 2;
            }
        }
    }
    return $rc;
}

sub getHostNameByIndex{
    return $_[0]->{_hosts}->[$_[1]];
}

sub getIndexByHostName{
    my $index;
    my $hosts = $_[0]->{_hosts};
    foreach my $i (0 .. (scalar @{$hosts} -1)){
        if ($hosts->[$i] eq $_[1]){
            $index = $i;
            last;
        }
    }
    return $index;
}

sub getExitCode{
    my ($self, $index) = @_;
    if (!defined $self->{exit_codes}){
        return undef;
    }
    return $self->{exit_codes}->[$index];
}


sub getHostNames{
    return $_[0]->{_hosts};
}

sub getUserName{
    return $_[0]->{_user};
}


#-------------------------------------------------------------------------------
# Copys connections from the class $allHosts to this class.
#
# Returns 0 in case of host not found in $allHosts.

sub copyConnections {

    my ($self,
        $allHosts,  # SDB::Install::RemoteHosts
        $msglst     # for writing error message
        ) = @_;

    if (!defined $self->{_hosts} || ($self->getNumberOfHosts() == 0)) {
        return 1;
    }

    my %idxMapAllHosts = ();

    my $i = 0;
    foreach my $curr (@{$allHosts->{_hosts}}) {
        $idxMapAllHosts{$curr} = $i;
        $i++;
    }

    my $rc          = 1;
    my @connections = ();

    foreach my $currHost (@{$self->{_hosts}}) {

        my $allHostIdx = $idxMapAllHosts{$currHost};

        if (defined $allHostIdx) {
            push @connections, $allHosts->{_connections}->[$allHostIdx];
        }
        else {
            push @connections, undef;
            $msglst->PushError
                    ("Host '$currHost' is not part of the $gProductNameSystem.");
            $rc = 0;
        }
    }

    $self->{_connections} = \@connections;
    delete $self->{_canAuthPassword};

    return $rc;
}


#-------------------------------------------------------------------------------
# removes hosts to the internal host array.

sub removeRemoteHosts {

    my ($self,
        $removeHosts # reference to an array containing hosts to be removed
        ) = @_;

    my @newHosts       = ();
    my @newConnections = ();
    my %removeMap      = ();

    foreach my $curr (@$removeHosts) {
        $removeMap{$curr} = 1;
    }

    my $i = 0;
    foreach my $currHost (@{$self->{_hosts}}) {

        if (!exists $removeMap{$currHost}) {
            push @newHosts,       $currHost;
            push @newConnections, $self->{_connections}->[$i];
        }
        $i++;
    }
    $self->{_hosts}       = \@newHosts;
    $self->{_connections} = \@newConnections;
    delete $self->{_canAuthPassword};
}


#-------------------------------------------------------------------------------

sub getOutputHandler {
    my ($self, $host) = @_;
    if (!defined $self->{_outputHandler}){
        return undef;
    }
    return $self->{_outputHandler}->{$host};
}

#-------------------------------------------------------------------------------
sub setOutputHandler {
    my ($self, $host, $outputHandler) = @_;
    if (!defined $self->{_outputHandler}){
        $self->{_outputHandler} = {};
    }
    $self->{_outputHandler}->{$host} = $outputHandler;
    return 1;
}


#-------------------------------------------------------------------------------
sub destroy{
	my $self = shift;
	if(defined $self->{_connections}){
		for(@{$self->{_connections}}){
			if(defined $_){
			    $_->disconnect();
			}
		}
		$self->{_connections} = undef;
		delete $self->{_canAuthPassword};
	}
}


#-------------------------------------------------------------------------------

sub isHostctrl {
    return 0;
}

1;
