package SDB::Install::SAPControl;

use base SDB::Install::BaseLegacy;
use strict;
use Net::HTTP;
use SDB::Install::XMLParser;
use SAPDB::Install::Hostname;
use SDB::Install::SysVars qw($isWin);
use SDB::Install::System qw (loadSSLRequiringPackage);
use SDB::Common::Utils qw(toBase64);
use SAPDB::Install::ProcState;

our $param_tmpl = '<%s>%s</%s>';

our $body_tmpl = << '__BODY__END__';
<?xml version="1.0" encoding="UTF-8"?>
<SOAP-ENV:Envelope
    xmlns:SOAP-ENV="http://schemas.xmlsoap.org/soap/envelope/"
    xmlns:SOAP-ENC="http://schemas.xmlsoap.org/soap/encoding/"
    xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance"
    xmlns:xsd="http://www.w3.org/2001/XMLSchema"
    xmlns:SAPControl="urn:SAPControl"
    xmlns:SAPCCMS="urn:SAPCCMS"
    xmlns:SAPHostControl="urn:SAPHostControl">
    <SOAP-ENV:Body>
        <SAPControl:%s>
            %s
        </SAPControl:%s>
    </SOAP-ENV:Body>
</SOAP-ENV:Envelope>
__BODY__END__

our %header_tmpl = (
        'Content-Type'      => 'text/xml; charset=utf-8',
        'Transfer-Encoding' => 'chunked');

our $socket_timeout = 120;
our $keep_alive     = 0;

sub new{
    my $self = shift->SUPER::new (@_);
    ( $self->{_host},
      $self->{_nr},
      $self->{_user},
      $self->{_passwd},
      $self->{_https},
      $self->{_cert}) = @_;
    if (!defined $self->{_https}){
        $self->{_https} = $isWin ? 0 : 1;
    }
    if (!defined $self->{_host}){
           $self->{_https} = 0;
    }
    $self->init ();
    return $self;
}



sub createTcpConnection{
    my ($self, $noAuth) = @_;
    my $conn;
    if ($self->{_https}){
        if (!defined $noAuth){
            $noAuth = 0;
        }
        $conn = Net::HTTPS->new (
            'Host'            => sprintf('%s:5%02d14', $self->{_host}, $_[0]->{_nr}),
            'SSL_verify_mode' => 0x00, # SSL_VERIFY_NONE,
            #'SSL_verify_mode' => IO::Socket::SSL::SSL_VERIFY_PEER(),
            #'SSL_fingerprint' => '',
            !$noAuth && defined $self->{_cert} ? ('SSL_cert_file'   => $self->{_cert}) : (),
            'Timeout'         => $socket_timeout,
            'KeepAlive'       => $keep_alive);
    }
    else{
        $conn = new Net::HTTP (
            'Host'      => sprintf('%s:5%02d13', $_[0]->{_host}, $_[0]->{_nr}),
            'Timeout'   => $socket_timeout,
            'KeepAlive' => $keep_alive);
    }
    return $conn;
}


sub isServiceRunning{
    my ($self, $processCheckPattern) = @_;

    if (!$isWin && defined $processCheckPattern){
        my $ps = new SAPDB::Install::ProcState();
        my ($cmd, $pid);
        my @matches;
        foreach my $proc (@{$ps->{procs}}){
            $pid = $proc->{pid};
            $cmd = $ps->GetArgs($pid);
            if ($cmd =~ /$processCheckPattern/){
                push @matches, "$cmd (pid=$pid)";
            }
        }
        my $msglst = $self->getMsgLst();
        if (!@matches){
            $msglst->addMessage ("No sapstartsrv process found");
            return undef;
        }
        my $submsglst = $msglst->addMessage('process is still running')->getSubMsgLst();
        foreach my $sapstartsrv (@matches){
            $submsglst->addMessage($sapstartsrv);
        }
    }

    if (!defined $self->{_host}){
        if (defined $self->connectUnixDomainSocket ()){
            return 1;
        }
        return 0;
    }
    if (defined $self->createTcpConnection (1)){
        return 1;
    }
    return 0;
}

sub _handleTcpError{
    my ($self) = @_;
    my $error = $@;
    my $prot = $self->{_https} ? 'HTTPS' : 'HTTP';
    if (!$@ && $self->{_https} && $IO::Socket::SSL::SSL_ERROR){
        $error = $IO::Socket::SSL::SSL_ERROR;
    }
    $self->setErrorMessage ("Cannot establish $prot connection to '$self->{_host}': $error");
    return 1;
}


sub accessCheck{
    my ($self, $user, $passwd ) = @_;
    my $credentials;

    if (!defined $user){
        $credentials = $self->{cred};
    }
    else{
        $credentials = toBase64(join (':', $user, $passwd));
    }

    my $conn = $self->createTcpConnection ();

    if (!defined $conn){
        $self->_handleTcpError ();
        return 1;
    }

    my %headers = %header_tmpl;
    if (defined $credentials && !defined $self->{_cert}){
        $headers{'Authorization'} = 'Basic ' . $credentials;
    }

    if (! $conn->write_request ('POST','/', %headers)){
        $self->AddError ("Cannot send http request: $!");
        return 2;
    }

    my $method = 'AccessCheck';
    my $func = '<function>Stop</function>';


    my $body= sprintf ($body_tmpl, $method, $func, $method);


    if (! $conn->write_chunk ($body)){
        $self->AddError ("Cannot send body: $!");
        return undef;
    }

    if (! $conn->write_chunk_eof ()){
        $self->AddError ("Cannot send eof: $!");
        return undef;
    }

    my ($code, $mess);

    eval {
        ($code, $mess, %headers) = $conn->read_response_headers ();
    };

    if ($@){
        $self->AddError ('Caught Net::HTTP exception (read_response_headers): ' . $@);
        return 3;
    }

    $self->{body} = '';

    my ($n, $buf);

    while (1){
        eval {
            $n = $conn->read_entity_body($buf, 1024);
        };
        if ($@){
            $self->AddError ('Caught Net::HTTP exception (read_entity_body): ' . $@);
            return 4;
        }
        if (!$n){
            last;
        }
        $self->{body} .= $buf;
    }

    if (!defined $n){
        $self->AddError ("Error reading body: $!");
        return 5;
    }

    if (int $code != 200){
        my ($detail) = ($self->{body} =~ /<faultstring>(.*)<\/faultstring>/m);
        $self->AddError ("$method request failed: $detail");
        return 6;

    }
    return 0;
}



sub init{
    my ($self) = @_;
    my $userName = $self->{_user};
    my $password = $self->{_passwd};
    my $credentialsString = defined($userName) && defined($password) ? toBase64("${userName}:${password}") : '';

    $self->{cred} = $credentialsString;

    if ($self->{_https}){
        my $messageList = new SDB::Install::MsgLst();
        my $isLoaded = loadSSLRequiringPackage ('Net::HTTPS', $messageList, $messageList);

        $self->getMsgLst()->appendMsgLst($messageList);
        if (!$isLoaded){
            die (${$messageList->getMsgLstString()} . "\n");
        }
    }
    return 1;
}

sub connect{
    my ($self) = @_;
    if ($self->{conn}){
        return 1;
    }

    if (!defined $self->{_host}){
        return $self->connectUnixDomainSocket ();
    }

    $self->{conn} = $self->createTcpConnection ();

    if (!defined $self->{conn}){
        $self->_handleTcpError ();
        return undef;
    }
    $self->{uds} = 0;
    return 1;
}

sub connectUnixDomainSocket{
    my ($self) = @_;
    if ($self->{conn}){
        return 1;
    }
    require IO::Socket::UNIX;
    import IO::Socket::UNIX qw (SOCK_STREAM);

    my $socketPath = sprintf ('/tmp/.sapstream5%02d13',$self->{_nr});

    $self->{conn} = new IO::Socket::UNIX (
        'Type' => SOCK_STREAM,
        'Peer' => $socketPath,
        'Timeout' => $socket_timeout
        );
    if (!defined $self->{conn}){
        $self->AddError ("Cannot establish http connection to unix domain socket '$socketPath' ($!)");
        return undef;
    }
    bless ($self->{conn},'Net::HTTP');
    if ($keep_alive){
        $self->{conn}->keep_alive(1);
    }
    $self->{conn}->http_version('1.1');
    $self->{conn}->peer_http_version('1.0');
    $self->{conn}->max_line_length(8*1024);
    $self->{conn}->max_header_lines(128);
    ${*{$self->{conn}}}{'http_buf'} = '';
    $self->{uds} = 1;
    return 1;
}

sub disconnect{
    $_[0]->{conn} = undef;
}

sub request{
    my ($self,$method, $args, $retries) = @_;

    if (!$keep_alive){
        $self->disconnect();
    }

    my $body= sprintf ($body_tmpl, $method,defined $args ? $args : '',$method);

    if (!defined $args){
        $args = '';
    }

    my $credentials = $self->{cred};

    if (!defined $self->connect ()){
        return undef;
    }

    my %headers = %header_tmpl;
    if (defined $credentials && !$self->{uds}  && !defined $self->{_cert}){
        $headers{'Authorization'} = 'Basic ' . $credentials;
    }

    if (! $self->{conn}->write_request ('POST','/', %headers)){
        $self->AddError ("Cannot send http request: $!");
        return undef;
    }

    if (! $self->{conn}->write_chunk ($body)){
        $self->AddError ("Cannot send body: $!");
        return undef;
    }
    if (!$self->{conn}->write_chunk_eof ()){
        $self->AddError ("Cannot send eof: $!");
        return undef;
    }

    my ($code, $mess);

    eval {
        ($code, $mess, %headers) = $self->{conn}->read_response_headers ();
    };

    if ($@){
        $self->AddError ('Caught Net::HTTP exception (read_response_headers): ' . $@);
        if ($retries){
            return $self->request ($method, $args, --$retries);
        }
        return undef;
    }

    $self->{body} = '';

    my ($n, $buf);

    while (1){
        eval {
            $n = $self->{conn}->read_entity_body($buf, 1024);
        };
        if ($@){
            $self->AddError ('Caught Net::HTTP exception (read_entity_body): ' . $@);
            if ($retries){
                return $self->request ($method, $args, --$retries);
            }
            return undef;
        }
        if (!$n){
            last;
        }
        $self->{body} .= $buf;
    }

    if (!defined $n){
        $self->AddError ("Error reading body: $!");
        return undef;
    }

    if (int $code != 200){
        my ($detail) = ($self->{body} =~ /<faultstring>(.*)<\/faultstring>/m);
        if ($method eq 'Start' && ($detail =~ /already started/)){
            $self->getMsgLst()->addMessage ("$method request failed: $detail");
        }
        else{
            $self->AddError ("$method request failed: $detail");
            return undef;
        }

    }
    return $self->{body};
}

sub callFunction{
    $_[0]->request ($_[1]->[0]);
}

sub set_callback{
    $_[0]->{_f_callback} = $_[1];
}

sub _status{
    my ($self,$service,$msglst) = @_;
    my $ret;
    if (!defined $service){
        $service = 'hdbdaemon';
    }
    $ret = $self->getProcessList ();
    if (!defined $ret){
        return undef;
    }
    my $rc = 'unknown';
    if (defined $msglst){
        $msglst =  $msglst->addMessage("SAPControl::GetProcessList result")->getSubMsgLst();
    }
    foreach my $p (@$ret){
        if (exists $p->{name} && exists $p->{textstatus}){
            if ($p->{name} =~ /^$service/){
                $rc = $p->{textstatus};
            }
        }
        if (!defined $msglst){
            next;
        }
        $msglst->addMessage ("$p->{name} $p->{textstatus}");
    }
    return $rc;
}


sub isRunning{
    my $status = $_[0]->_status($_[1], $_[2]);
    if (!defined $status){
        return undef;
    }
    return ('Running' eq $status);
}

sub getProcessList{
    my ($self) = @_;

    my $body = $self->request('GetProcessList', undef, 3);

    if (!defined $body){
        return undef;
    }

    my $parser = new SDB::Install::XMLParser ();

    eval {
        $parser->parsestring ($body);
    };

    if ($@){
        $self->AddError ("Cannot parse reply: $@");
        return undef;
    }

    my $node = $parser->getElementByTagName ('SAPControl:GetProcessListResponse');

    if (!defined $node){
        my $msglst = new SDB::Install::MsgLst ();
        $msglst->addMessage (">>> SAPControl:GetProcessListResponse not found in respose");
        $msglst->addMessage (">>> respose = '$body'");
    }

    my @ll;
    my %i;
    my $i = 0;
    my $item;

    foreach my $n (qw (name description dispstatus
                        textstatus starttime elapsedtime pid)){
        $i{$n} = $i++;
    }

    if (defined $node){
        $item = $parser->getElementByTagName ('item',$node);
        if (!defined $item){
            my $msglst = new SDB::Install::MsgLst ();
            $msglst->addMessage (">>> 'item' not found in respose");
            $msglst->addMessage (">>> respose = '$body'");
        }
        while (defined $item){
            my %l;
            $node = $item->{child};
            while (defined $node){
                if (exists $i{$node->{type}}){
                    $l{$node->{type}} = $node->{content};
                    if ($node->{type} eq 'dispstatus'){
                        $l{$node->{type}} =~ s/^SAPControl-//;
                    }
                }
                $node = $node->{neighbor};
            }
            if (%l){
                push @ll,\%l;
            }
            $item = $item->{neighbor};
        }
    }
    if (!@ll){
        my $msglst = new SDB::Install::MsgLst ();
        $msglst->addMessage (">>> populated empty list");
        $msglst->addMessage (">>> respose = '$body'");
    }
    return \@ll;
}


sub WaitforStarted{
    my ($self, $maxtries, $sleep, $hostInfo, $tolerance) = @_;
    if (!defined $tolerance) {
        $tolerance = 3;
    }
    if (!defined $sleep){
        $sleep = 2;
    }
    if (!defined $maxtries){
        $maxtries = _setMaxtries($maxtries,$sleep);
    }
    my $i = 0;
    my @pending_processes;
    my @running_processes;
    my $daemon_stopped_retries = 20;
    my $oldPending = 0;
    my $host = (defined $self->{_host}) ? $self->{_host} : hostname();
    $hostInfo= "'$host'" if (!defined $hostInfo);
    my $toStart;
    my ($name, $tenant);
    while (1){
        my $pl = $self->getProcessList ();
        if (!defined $pl){
            return undef;
        }
        if (@$pl){
            @pending_processes = ();
            @running_processes = ();
            foreach my $p (@$pl){
                if (exists $p->{name} && exists $p->{textstatus}){
                    if ($p->{textstatus} eq 'Stopped'){
                        if ($p->{name} =~ /hdbdaemon/){
                            if ($daemon_stopped_retries--){
                                push @pending_processes, $p->{name};
                                last;
                            }
                        }
                        $self->AddError ("FAIL: process $p->{name} $p->{description} not running");
                        return undef;
                    }
                    $name = $p->{name};
                    if ($name eq 'hdbindexserver'){
                        ($tenant) = ($p->{description} =~ /^\S+-(\S+)$/);
                        if ($tenant){
                            $name .= " ($tenant)";
                        }
                    }
                    if ($p->{textstatus} eq 'Running'){
                        push @running_processes, $name;
                    }
                    else{
                        push @pending_processes, $name;
                    }
                }
            }

            my $pending = @pending_processes;
            if($pending != 0) {
                my $header = "    Starting on $hostInfo: ";
                if($oldPending == 0 || $pending > $oldPending) {
                    $toStart = $pending;
                    my $plural = ($toStart != 1) ? "es" : "";
                    my $msgtext = "Starting $toStart process$plural on host $hostInfo:";
                    $self->getMsgLst()->addProgressMessage ($msgtext);
                    $msgtext = $header;
                    my $first = 1;
                    foreach my $p (sort @pending_processes) {
                        if($first) {
                            $first = 0;
                        }
                        else {
                            $msgtext = $msgtext.", ";
                        }
                        $msgtext = $msgtext.$p;
                    }
                    $self->getMsgLst()->addProgressMessage ($msgtext);
                }
                if($oldPending != 0 && $pending < $oldPending) {
                    my $progressMsg = $header.join (', ',sort @pending_processes);
                    $self->getMsgLst()->addProgressMessage ($progressMsg);
                }
            }
            $oldPending = $pending;

            if (@pending_processes){
                # there is at least one pending process
                # $self->AddMessage ("Waiting for " . join (',',@pending_processes));
            }
            elsif (@running_processes > 1){
                # no pending process
                # at least 2 processes are in state 'Running'
                # $self->AddMessage ('Started. Running Processes: ' . join (',',@running_processes));
                if ($i >= $tolerance) {
                    $self->getMsgLst()->addProgressMessage ("  All server processes started on host $hostInfo.");
                    return 1;
                }
            }
            else{
                # no pending process, but less than 2 processes are running
                # $self->AddMessage ('Waiting. Running Process: ' . join (',',@running_processes));
            }
        }
        $i++;
        if ($i >= $maxtries && $maxtries != 0){
            $self->AddError ("Timeout reached");
            return undef;
        }
        $self->sleep($sleep);
        if (defined $self->{_f_callback}){
            &{$self->{_f_callback}} ();
        }
    }
    return 1;
}

sub WaitforStopped{
    my ($self, $maxtries, $sleep, $services, $hostInfo) = @_;
    if (!defined $sleep){
        $sleep = 2;
    }
    if (!defined $maxtries){
        $maxtries = 400;
    }
    my $i = 0;
    my @pending_processes;
    my @stopped_processes;
    my $servicePattern;
    if (defined $services && @$services){
        $servicePattern = join ('|', @$services);
    }
    my $oldPending = 0;
    my $host = (defined $self->{_host}) ? $self->{_host} : hostname();
    $hostInfo= "'$host'" if (!defined $hostInfo);
    my $toStop;
    my ($name, $tenant);
    my $stoppedMsg = (defined $servicePattern ? 'S' : 'All s') . "erver processes stopped on host $hostInfo.";
    
    while (1){
        my $pl = $self->getProcessList ();
        if (!defined $pl){
            return undef;
        }
        if (@$pl){
            @pending_processes = ();
            @stopped_processes = ();
            foreach my $p (@$pl){
                if (exists $p->{name} && exists $p->{textstatus}){
                    if (defined $servicePattern && ($p->{name} !~ /$servicePattern/)){
                        next;
                    }
                    $name = $p->{name};
                    if ($name eq 'hdbindexserver'){
                        ($tenant) = ($p->{description} =~ /^\S+-(\S+)$/);
                        if ($tenant){
                            $name .= " ($tenant)";
                        }
                    }
                    if ($p->{textstatus} eq 'Stopped'){
                        push @stopped_processes, $name;
                    }
                    else{
                        push @pending_processes, $name;
                    }
                }
            }
            my $pending = @pending_processes;
            if($pending != 0) {
                my $header = "    Stopping on $hostInfo: ";
                if($oldPending == 0 || $pending > $oldPending) {
                    $toStop = $pending;
                    my $plural = ($toStop != 1) ? "es" : "";
                    my $msgtext = "  Stopping $toStop process$plural on host $hostInfo:";
                    $self->getMsgLst()->addProgressMessage ($msgtext);
                    $msgtext = $header;
                    my $first = 1;
                    foreach my $p (sort @pending_processes) {
                        if($first) {
                            $first = 0;
                        }
                        else {
                            $msgtext = $msgtext.", ";
                        }
                        $msgtext = $msgtext.$p;
                    }
                    $self->getMsgLst()->addProgressMessage ($msgtext);
                }
                if($oldPending != 0 && $pending < $oldPending) {
                    my $progressMsg = $header.join (', ',sort @pending_processes);
                    $self->getMsgLst()->addProgressMessage ($progressMsg);
                }
            }
            else {
                $self->getMsgLst()->addProgressMessage ("  $stoppedMsg");
                return 1;
            }
            $oldPending = $pending;

        }
        else {
            # empty process list, i.e. nothing to stop
            $self->getMsgLst()->addProgressMessage ("  $stoppedMsg");
            return 1;
        }
        $i++;
        if ($i >= $maxtries && $maxtries != 0){
            $self->AddError ("Timeout reached");
            return undef;
        }
        $self->sleep($sleep);
        if (defined $self->{_f_callback}){
            &{$self->{_f_callback}} ();
        }
    }
    return 1;

}

sub _setMaxtries{
    my ($maxtries,$sleep) = @_;
    if (defined $ENV{HDB_INSTALLER_TIMEOUT} &&
        $ENV{HDB_INSTALLER_TIMEOUT} =~ /^\d+$/){
        $maxtries = int ($ENV{HDB_INSTALLER_TIMEOUT});
        if ($maxtries > 0 && $sleep != 0){
            $maxtries = int ($maxtries / $sleep);
            if ($maxtries == 0){
                $maxtries = 1;
            }
        }
    }
    else{
        $maxtries = 0;
    }
    return $maxtries;
}

sub StartWait{
    my ($self, $maxtries, $sleep, $hostInfo) = @_;
    if ($_[0]->isRunning ()){
        $_[0]->AddMessage ('Instance is already running');
        return 1;
    }
    $_[0]->AddMessage ('Sending HTTP request, function = Start, maximum number of retries = 3');
    if (!defined $_[0]->request ('Start', undef, 3)){
        return undef;
    }
    if (!defined $self->WaitforStarted ($maxtries, $sleep, $hostInfo)){
        return undef;
    }
    return 1;
}


our $softtimoutParamName  = 'softtimeout';

sub Stop{
    my ($self, $softtimeout, $maxtries) = @_;
    my $arg;
    my $timeout_str = '';
    if (defined $softtimeout){
        $arg = sprintf ($param_tmpl, $softtimoutParamName, $softtimeout, $softtimoutParamName);
        $timeout_str = "(softtimeout = $softtimeout)";
    }
    $self->getMsgLst ()->addMessage ("Sending HTTP request, function = Stop$timeout_str, maximum number of retries = $maxtries");
    return $self->request ('Stop', $arg, $maxtries);
}


sub Start{
    if ($_[0]->isRunning ()){
        $_[0]->AddMessage ('Instance is already running');
        return 1;
    }
    return $_[0]->callFunction (['Start']);
}

sub StopWait{
    my ($self, $wait_retries, $delay, $hostInfo) =  @_;
    my $softtimeout;
    if (!$delay){
        $delay = 1;
    }
    if ($wait_retries){
        $softtimeout = $wait_retries * $delay;
        my $waitTimeout = $softtimeout + 60;
        $wait_retries = int ($waitTimeout / $delay);
    }

    if (!defined $self->Stop ($softtimeout, 3)){
        return undef;
    }

    if (!defined $self->WaitforStopped ($wait_retries, $delay, undef, $hostInfo)){
        return undef;
    }
    return 1;
}

sub restartService{
    my ($self) = @_;

    #
    # save connection for waitForService
    #

    if (!defined $self->connect ()){
        return undef;
    }

    $self->{restart_check_conn} =  $self->{conn};

    $self->disconnect ();

    if (!defined $_[0]->request ('RestartService')){
        return undef;
    }
    return 1;
}

sub sleep{
    CORE::sleep($_[1]);
}

sub stopService{
    my ($self) = @_;

    #
    # save connection for waitForService
    #

    if (!defined $self->connect ()){
        return undef;
    }

    $self->{restart_check_conn} =  $self->{conn};

    $self->disconnect ();

    if (!defined $_[0]->request ('StopService')){
        return undef;
    }
    return 1;
}

sub stopServiceWait{
    my ($self,$maxtries) = @_;

    if (!defined $self->stopService()){
        return undef;
    }

    if (!defined $self->waitForServiceStop ($maxtries)){
        return undef;
    }
    return 1;
}

sub waitForServiceStop{
    my ($self, $maxtries, $msglst, $waitSecondsAfterStopDetection) = @_;
    if (!$self->{restart_check_conn}){
        $self->sleep(5);
        return 1;
    }

    if (!defined $maxtries){
        $maxtries = 60;
    }

    if (!defined $msglst){
        $msglst = $self;
    }

    my $conn = $self->{restart_check_conn};

    require IO::Select;

    my $read_set = new IO::Select ();

    $read_set->add ($conn);

    local %SIG = %SIG;
    $SIG{PIPE} = sub {$conn = undef;};

    my $msg = $msglst->AddMessage ("Waiting until service is stopped...");

    my $i = 0;
    my $rc;
    while ($conn && $maxtries--){
        if (!$read_set->can_read (1)){
            $self->AddSubMessage ($msg,'Service is stopping ('. $i++ . ')');
            if (defined $self->{_f_callback}){
                &{$self->{_f_callback}} ();
            }
            next;
        }
        eval {
            $rc = $conn->my_readline ();
        };

        if ($@ || !defined $rc){
            $self->AddSubMessage ($msg,"Service stopped");
            last;
        }
    }

    if (!$maxtries){
        $self->AddError ("Cannot restart service (sapstartsrv): Stop timeout reached");
        return undef;
    }

    delete $self->{restart_check_conn};
    $self->disconnect();
    if ($waitSecondsAfterStopDetection){
        $self->sleep($waitSecondsAfterStopDetection);
    }
    return 1;
}


sub waitForService{
    my ($self, $maxtries, $msglst,$waitSecondsAfterStopDetection, $checkProcessPattern) = @_;

    if ($self->{restart_check_conn}){
        return $self->waitForServiceRestart ($maxtries, $msglst, $waitSecondsAfterStopDetection);
    }

    if (!defined $msglst){
        $msglst = $self;
    }

    if (!defined $maxtries){
        $maxtries = 120;
    }
    my $timeout = $maxtries;
    my $leave_with_error;
    while ($maxtries--){
        my $rc = $self->isServiceRunning ($checkProcessPattern);
        if ($rc){
            $msglst->AddMessage ("sapstartsrv is running");
            return 1;
        }
        if (!defined $rc){
            $msglst->AddError ("sapstartsrv process is gone.");
            return undef;
        }
        $leave_with_error = !$isWin && !$!{ECONNREFUSED} && !$!{ENOENT};
        $msglst->AddMessage ("sapstartsrv is not ready: $!");
        if ($leave_with_error){
           $msglst->AddError ("Waiting for sapstartsrv failed: $!");
            return 0;
        }
        if (defined $self->{_f_callback}){
            &{$self->{_f_callback}} ();
        }
        $self->sleep(1);
    }
    $msglst->AddError ("Waiting for sapstartsrv failed: timeout reached ($timeout)");
    return 0;
}



sub waitForServiceRestart{
    my ($self, $maxtries, $msglst, $waitSecondsAfterStopDetection) = @_;

    if (!defined $self->waitForServiceStop ($maxtries, $msglst, $waitSecondsAfterStopDetection)){
        return undef;
    }

    if (!defined $maxtries){
        $maxtries = 120;
    }
    if (!defined $msglst){
        $msglst = $self;
    }

    my $msg = $msglst->AddMessage ("Waiting until service is restarted...");
    my $i = 0;
    while ($maxtries-- && !$self->isServiceRunning){
        $self->AddSubMessage ($msg,'Service is starting ('. $i++ . ')');
        if (defined $self->{_f_callback}){
            &{$self->{_f_callback}} ();
        }
        $self->sleep(1);
    }

    if ($maxtries){
        $self->AddSubMessage ($msg,'Service started');
    }
    else{
        $self->AddError ("Cannot restart service (sapstartsrv): Start timeout ($i) reached");
        return undef;
    }
    return 1;
}

sub updateSystemPKI{
    my ($self) = @_;

    if (!defined $self->connect ()){
        return undef;
    }

    if (!defined $_[0]->request ('UpdateSystemPKI')){
        return undef;
    }
    return 1;
}

1;




