package SDB::Install::SAPHostControl;

use base SDB::Install::Base;
use SDB::Install::SOAP::Client;
use SDB::Install::SOAP::HostAgent::WSDL;
use SDB::Install::SysVars;

use strict;

#---------------------------------------------------------------------------------------------

sub new {
    my $self = shift->SUPER::new (); 
    my (
        $host,
        $user,
        $passwd,
        $forceUseHttp,  # uses http instead of https
        $sso_cert,
        $log_hostname
    ) = @_;
    $self->_init($host, $user, $passwd, $forceUseHttp, $sso_cert, $log_hostname);
    return $self;
}

#---------------------------------------------------------------------------------------------

sub _init{
    my (
        $self,
        $host,
        $user,
        $passwd,
        $forceUseHttp,
        $sso_cert,
        $log_hostname
    ) = @_;
    my $hostAgentPort = getservbyname('saphostctrl','tcp');
    if(!defined $hostAgentPort) {
        $hostAgentPort = 1128;
    }
    my $hostAgentSecPort = getservbyname('saphostctrls','tcp');
    if(!defined $hostAgentSecPort) {
        $hostAgentSecPort = 1129;
    }
    my $baseurl = sprintf ('http:/tmp/.sapstream%d', $hostAgentPort);
    my $socketMode = 'uds';
    my $hostAgentRealm = undef;
    my $useHttps = 0;
    if(defined $host) {
        $socketMode = 'tcp';
        $hostAgentRealm = 'gSOAP Web Service';
        if($isWin || $forceUseHttp) {
            $baseurl = sprintf ('http://%s:%d', $host, $hostAgentPort);
        }
        else {
            $baseurl = sprintf ('https://%s:%d', $host, $hostAgentSecPort);
            $useHttps = 1;
        }
    }
    $self->{'soapclientParams'} = {
        'WsdlBuf'      => $HOST_AGENT_WSDL, 
        'Host'         => $host, 
        'User'         => $user,
        'Passwd'       => $passwd,
        'Port'         => $useHttps ? $hostAgentSecPort : $hostAgentPort,
        'Realm'        => $hostAgentRealm,
        'BaseUrl'      => $baseurl,
        'SocketMode'   => $socketMode,
        'Https'        => $useHttps,
        'sso_cert'     => $sso_cert
    };
    $self->{'lastResult'}            = undef;
    $self->{'lastResultProcessed'}   = undef;
    $self->{'logHostName'} = $log_hostname;
}

#---------------------------------------------------------------------------------------------

#OperationOptions
#  <element name="ListInstances">
#   <complexType>
#    <sequence>
#     <element name="aSelector" type="SAPHostControl:InstanceSelector" minOccurs="1" maxOccurs="1"/>
#    </sequence>
#   </complexType>
#  </element>
#
sub ListInstances {
    my ($self) = @_;
    my ($query, $format, $style) = ('uniprot:wap_rat', 'fasta', 'raw');
    my $soapclient = $self->_createSOAPClient();
    if(not defined $soapclient) {
        return undef;
    }
    $self->_logWebServiceInformation('ListInstances', {});
    $self->getErrMsgLst()->setMsgLst($soapclient->getErrMsgLst());
    $self->{'lastResult'}->{'raw'} = $soapclient->{'service'}->ListInstances($query, $format, $style);
    return $self->{'lastResult'}->{'raw'};
}

#---------------------------------------------------------------------------------------------

sub DeployConfiguration {
    my ($self, $configurations) = @_;
    my $soapclient = $self->_createSOAPClient();
    if(not defined $soapclient) {
        return undef;
    }
    $self->_logWebServiceInformation('DeployConfiguration', {});
    $self->getErrMsgLst()->setMsgLst($soapclient->getErrMsgLst());
    $self->{'lastResult'}->{'raw'} = $soapclient->{'service'}->DeployConfiguration(_createSOAPData_DeployConfiguration($configurations));
    return $self->{'lastResult'}->{'raw'};
}

#---------------------------------------------------------------------------------------------

sub DetectManagedObjects {
    my ($self, $manifestPath) = @_;
    my $soapclient = $self->_createSOAPClient();
    if(not defined $soapclient) {
        return undef;
    }
    my $detectManagedObjectsArguments = $manifestPath ? {'manifest' => $manifestPath} : {};
    $self->_logWebServiceInformation('DetectManagedObjects', $detectManagedObjectsArguments);
    $self->getErrMsgLst()->setMsgLst($soapclient->getErrMsgLst());
    $self->{'lastResult'}->{'raw'} = $soapclient->{'service'}->DetectManagedObjects($manifestPath ? $manifestPath : ());
    return $self->{'lastResult'}->{'raw'};
}

#---------------------------------------------------------------------------------------------

sub _callWebMethod{
    my (
        $self,
        $method,
        $aProperties,
        $operationOptions
    ) = @_;

    my $timeout = undef;
    if(defined $operationOptions) {
        $timeout = $operationOptions->{'mTimeout'};
    }
    my $soapclient = $self->_createSOAPClient ($timeout);
    if(not defined $soapclient) {
        return undef;
    }
    my $service = $soapclient->{'service'};
    my $fMethod = $service->can($method);
    my $methodName = 'SAPHostControl::' . $method;
    if (!defined $fMethod){
        $self->setErrorMessage ("Unknown web sevice method '$methodName'");
        return undef;
    }
    my $msglst = $self->getMsgLst ();
    my $infoTimeout = (defined $timeout) ? ", timeout: $timeout" : '';
    my $msg = $msglst->addMessage("Calling web service method '$methodName' on local host (url: '"
                    . $self->{soapclientParams}->{BaseUrl} . "'$infoTimeout)");
    my $submsglst = $msg->getSubMsgLst ();
    my $currVal;
    my $argMsgLst = $submsglst->addMessage ('Arguments')->getSubMsgLst ();
    foreach my $currArg (sort keys %$aProperties) {
        $currVal = ($currArg =~ /Password$/) ? '***' : $aProperties->{$currArg};
        $argMsgLst->addMessage("$currArg: '$currVal'");
    }
    $soapclient->setMsgLstContext ([$submsglst, $self->getErrMsgLst ()]);
    $self->{'lastResult'}->{'raw'} = $fMethod->($service,
        _createSOAPData_ArrayOfProperty($aProperties, 'aArguments'),
        _createSOAPOperationOptions($operationOptions, 'aOptions')
    );
    my $loglines = $self->getLastResultLoglines ();

    if (defined $loglines){
        my $resultMsgLst = $submsglst->addMessage ('Web service log')->getSubMsgLst ();
        foreach my $line (@$loglines){
            $resultMsgLst->addMessage ($line);
        }
    }

    my $faultcode = $self->getLastResult_faultcode ();

    if (!$self->getErrMsgLst ()->isEmpty ()){
        $msglst->addWarning("$methodName() failed", $self->getErrMsgLst ());
    }
    elsif (!defined $faultcode){
        my $resultHash = $self->getLastResultHash();
        if (defined $resultHash){
            my $resultMsglst = $submsglst->addMessage ('Web service result')->getSubMsgLst ();
            foreach my $key (sort keys %$resultHash){
                $resultMsglst->addMessage ("'$key' = '$resultHash->{$key}'");
            }
        }
        my $successMsg = $msglst->addMessage ("$methodName() successfully finished");
    }

    if (defined $faultcode){
        $msglst->addWarning("$methodName() failed: " . $self->getLastResult_faultstring ());
    }

    return $self->{'lastResult'}->{'raw'};
}

#---------------------------------------------------------------------------------------------

sub GetDatabaseProperties{
    my $self = shift;
    return $self->_callWebMethod('GetDatabaseProperties', @_);
}

#---------------------------------------------------------------------------------------------

#  <element name="UnregisterInstanceService">
#   <complexType>
#    <sequence>
#     <element name="aInstance" type="SAPHostControl:SAPControlInstance" minOccurs="1" maxOccurs="1"/>
#     <element name="aOptions" type="SAPHostControl:OperationOptions" minOccurs="1" maxOccurs="1"/>
#    </sequence>
#   </complexType>
#  </element>
#
sub UnregisterInstanceService {
    my (
        $self,
        $sid,
        $hostname,
        $systemNumber,
        $operationOptions
    ) = @_;
    my $timeout = undef;
    if(defined $operationOptions) {
        $timeout = $operationOptions->{'mTimeout'};
    }
    my $soapclient = $self->_createSOAPClient($timeout);
    if(not defined $soapclient) {
        return undef;
    }
    $self->getErrMsgLst()->setMsgLst($soapclient->getErrMsgLst());
    $self->{'lastResult'}->{'raw'} = $soapclient->{'service'}->UnregisterInstanceService(
        _createSOAPData_SAPControlInstance({'mSid' => $sid, 'mHostname' => $hostname, 'mSystemNumber' => $systemNumber}, 'aInstance'),
        _createSOAPOperationOptions($operationOptions, 'aOptions')
    );
    return $self->{'lastResult'}->{'raw'};
}

#---------------------------------------------------------------------------------------------

sub RegisterInstanceService {
    my (
        $self,
        $sid,
        $hostname,
        $systemNumber,
        $operationOptions
    ) = @_;
    my $timeout = undef;
    if(defined $operationOptions) {
        $timeout = $operationOptions->{'mTimeout'};
    }
    my $soapclient = $self->_createSOAPClient($timeout);
    if(not defined $soapclient) {
        return undef;
    }
    $self->getErrMsgLst()->setMsgLst($soapclient->getErrMsgLst());
    $self->{'lastResult'}->{'raw'} = $soapclient->{'service'}->RegisterInstanceService(
        _createSOAPData_SAPControlInstance({'mSid' => $sid, 'mHostname' => $hostname, 'mSystemNumber' => $systemNumber}, 'aInstance'),
        _createSOAPOperationOptions($operationOptions, 'aOptions')
    );
    return $self->{'lastResult'}->{'raw'};
}

#---------------------------------------------------------------------------------------------

sub SetDatabaseProperty{
    my $self = shift;
    return $self->_callWebMethod('SetDatabaseProperty', @_);
}

#---------------------------------------------------------------------------------------------

#  <element name="ExecuteOperation">
#   <complexType>
#    <sequence>
#     <element name="aOperation" type="xsd:string" minOccurs="0" maxOccurs="1" nillable="true"/>
#     <element name="aArguments" type="SAPHostControl:ArrayOfProperty" minOccurs="1" maxOccurs="1" nillable="false"/>
#     <element name="aOptions" type="SAPHostControl:OperationOptions" minOccurs="1" maxOccurs="1"/>
#    </sequence>
#   </complexType>
#  </element>
#
sub ExecuteOperation {
    my (
        $self,
        $operation,             # name of Operation, string, mandatory
        $arguments,             # params of Operation, hash ref , optional (default: {}), e.g.
                                # {
                                #    'SID'      => 'XYZ',
                                #    'instNumb' => 37
                                # }
                                #
        $options                # optional (default: {}), see 'sub _createSOAPOperationOptions'
    ) = @_;

    my $timeout = undef;
    if(defined $options) {
        $timeout = $options->{'mTimeout'};
    }
    my $infoTimeout = (defined $timeout) ? ", timeout: $timeout" : '';
    my $msgLst      = $self->getMsgLst();
    $msgLst->addMessage("Executing operation '$operation' on host '"
                    . $self->{soapclientParams}->{Host}    . "' (user: '"
                    . $self->{soapclientParams}->{User}    . "', url: '"
                    . $self->{soapclientParams}->{BaseUrl} . "'$infoTimeout)");

    foreach my $currArg (sort keys %$arguments) {
        my $currVal = ($currArg eq 'XMLINPUT') ? '***' : $arguments->{$currArg};
        $msgLst->addMessage("    $currArg: '$currVal'");
    }

    my $soapclient = $self->_createSOAPClient($timeout);
    if(not defined $soapclient) {
        return undef;
    }
    $self->getErrMsgLst()->setMsgLst($soapclient->getErrMsgLst());
    $self->{'lastResult'}->{'raw'} =  $soapclient->{'service'}->ExecuteOperation(
        $operation,
        _createSOAPData_ArrayOfProperty($arguments, 'aArguments'),
        _createSOAPOperationOptions($options, 'aOptions')
    );
    
    $self->_handleError();
    return $self->{'lastResult'}->{'raw'};
}

#---------------------------------------------------------------------------------------------

sub _handleError {
	my ( $self, ) = @_;
	my $errMsgList = ${$self->getErrMsgLst()->getMsgLstString()};
	my $nocertificatesErrPattern = '^500.*1129.*$';
	
	if ( $errMsgList =~ $nocertificatesErrPattern ) {
		my $sapNoteMessage = new SDB::Install::MsgLst();
		
		my $shaName = "SAP Host Agent";
		$sapNoteMessage->addError( "Connecting to $shaName via HTTPS failed!" );
		$sapNoteMessage->addError( "See SAP Note 2059960 for more information." );
		
		$self->getErrMsgLst()->addError( undef, $sapNoteMessage );
	}
}

#---------------------------------------------------------------------------------------------

#  <element name="GetOperationResults">
#   <complexType>
#    <sequence>
#     <element name="aOperationID" type="xsd:string" minOccurs="0" maxOccurs="1" nillable="true"/>
#     <element name="aOptions" type="SAPHostControl:OperationOptions" minOccurs="1" maxOccurs="1"/>
#    </sequence>
#   </complexType>
#  </element>
#
sub GetOperationResults {
    my (
        $self,
        $operationID,           # string, the id of an Operation, as returned by (e.g.) 'ExecuteOperation()'
        $options                # optional (default: {}), see 'sub _createSOAPOperationOptions'
    ) = @_;
    my $timeout = undef;
    if(defined $options) {
        $timeout = $options->{'mTimeout'};
    }
    my $soapclient = $self->_createSOAPClient($timeout);
    if(not defined $soapclient) {
        return undef;
    }
    $self->getErrMsgLst()->setMsgLst($soapclient->getErrMsgLst());
    $self->{'lastResult'}->{'raw'} = $soapclient->{'service'}->GetOperationResults(
        $operationID,
        _createSOAPOperationOptions($options, 'aOptions')
    );
    return $self->{'lastResult'}->{'raw'};
}

#---------------------------------------------------------------------------------------------

sub getLastResult {
        my (
            $self
        ) = @_;
        return $self->{'lastResult'}->{'raw'};
}

#---------------------------------------------------------------------------------------------

sub getLastResultHash {
        my (
            $self
        ) = @_;
        if(not $self->{'lastResultProcessed'}) {
            $self->_processLastResult();
        }
        return $self->{'lastResult'}->{'resulthash'};
}

#---------------------------------------------------------------------------------------------

sub getLastResultLoglines {
        my (
            $self
        ) = @_;
        if(not $self->{'lastResultProcessed'}) {
            $self->_processLastResult();
        }
        return $self->{'lastResult'}->{'loglines'};
}

#---------------------------------------------------------------------------------------------

sub getLastResultOutputlines {
        my (
            $self
        ) = @_;
        if(not $self->{'lastResultProcessed'}) {
            $self->_processLastResult();
        }
        return $self->{'lastResult'}->{'outputlines'};
}

#---------------------------------------------------------------------------------------------

sub getLastResult_mOperationID {
        my (
            $self
        ) = @_;
        return $self->{'lastResult'}->{'raw'}->{'mOperationID'};
}

#---------------------------------------------------------------------------------------------

sub getLastResult_exitcode {
        my (
            $self
        ) = @_;
        my $exitCode = $self->_getLastResultMsgByKey('exitcode');
        if (!defined $exitCode) {
            $exitCode = $self->_getLastResultMsgByKey('faultcode');
        }
        return $exitCode;
}

#---------------------------------------------------------------------------------------------

sub getLastResult_description {
        my (
            $self
        ) = @_;
        return $self->_getLastResultMsgByKey('description');
}

#---------------------------------------------------------------------------------------------

sub getLastResult_faultstring {
        my (
            $self
        ) = @_;
        return $self->_getLastResultMsgByKey('faultstring');
}

#---------------------------------------------------------------------------------------------

sub getLastResult_mFaultCode {
        my (
            $self
        ) = @_;
        return $self->_getLastResultMsgByKey('mFaultCode');
}

#---------------------------------------------------------------------------------------------

sub getLastResult_faultcode {
        my (
            $self
        ) = @_;
        return $self->_getLastResultMsgByKey('faultcode');
}

##############################################################################################

#  <complexType name="Property">
#   <sequence>
#     <element name="mKey" type="xsd:string" minOccurs="0" maxOccurs="1" nillable="true"/>
#     <element name="mValue" type="xsd:string" minOccurs="0" maxOccurs="1" nillable="true"/>
#   </sequence>
#  </complexType>
#  <complexType name="ArrayOfProperty">
#   <sequence>
#    <element name="item" type="SAPHostControl:Property" minOccurs="0" maxOccurs="unbounded" nillable="true"/>
#   </sequence>
#  </complexType>
sub _createSOAPData_ArrayOfProperty {
        my (
            $hashParams,
            $name
        ) = @_;
        if(not defined $hashParams) {
            $hashParams = {};
        }
        my @list;
        my ($paramKey, $paramValue);
        foreach my $key (keys (%{$hashParams})) {
            $paramKey   = SOAP::Data->new('name' => 'mKey',   'value' => $key,                'type' => 'string');
            $paramValue = SOAP::Data->new('name' => 'mValue', 'value' => $hashParams->{$key}, 'type' => 'string');
            push @list, SOAP::Data->new('value' => [$paramKey,$paramValue], 'type' => 'tns:Property');
        }
        return SOAP::Data->new('name' => $name, 'value' => \@list, 'type' => 'tns:ArrayOfProperty');
}

#---------------------------------------------------------------------------------------------

#  <complexType name="OperationOptions">
#   <sequence>
#     <element name="mTimeout" type="xsd:long" minOccurs="1" maxOccurs="1"/>
#     <element name="mOptions" type="SAPHostControl:ArrayOfInstanceOptionsFlags" minOccurs="1" maxOccurs="1" nillable="false"/>
#     <element name="mSoftTimeout" type="xsd:long" minOccurs="1" maxOccurs="1" default="0"/>
#     <element name="mProperties" type="SAPHostControl:ArrayOfProperty" minOccurs="0" maxOccurs="1" nillable="true"/>
#   </sequence>
#  </complexType>
#
#  <complexType name="ArrayOfInstanceOptionsFlags">
#   <sequence>
#    <element name="item" type="SAPHostControl:InstanceOptionsFlags" minOccurs="0" maxOccurs="unbounded" nillable="true"/>
#   </sequence>
#  </complexType>
#
#  <simpleType name="InstanceOptionsFlags">
#   <restriction base="xsd:string">
#    <enumeration value="O-INSTANCE"/>
#    <enumeration value="O-SERVICE"/>
#    <enumeration value="O-PREHOOK"/>
#    <enumeration value="O-POSTHOOK"/>
#    <enumeration value="O-CLEANUP"/>
#    <enumeration value="O-FORCE"/>
#    <enumeration value="O-ERRORHOOK"/>
#    <enumeration value="O-LASTOPTION"/>
#   </restriction>
#  </simpleType>
sub _createSOAPOperationOptions{
        my (
            $plainOperationOptions, # hostagent related options for operation, hash ref, optional (default: {}), e.g.
                                    # {
                                    #    'mTimeout'     => 100,                                     ## number
                                    #    'mOptions'     => ['O-INSTANCE', 'O-SERVICE'],             ## array of strings
                                    #    'mSoftTimeout' => 50,                                      ## number
                                    #    'mProperties'  => {'key1' => 'value1', 'key2' => 'value2'} ## hash
                                    # }
                                    #
            $name
        ) = @_;
        if(not defined $plainOperationOptions) {
            $plainOperationOptions = {};
        }
        my @list;
        if(defined $plainOperationOptions->{'mTimeout'}) {
            push @list, SOAP::Data->new('name' => 'mTimeout', 'value' => $plainOperationOptions->{'mTimeout'}, 'type' => 'long');
        }
        if(defined $plainOperationOptions->{'mOptions'}) {
            push @list, SOAP::Data->new('name' => 'mOptions', 'value' => $plainOperationOptions->{'mOptions'}, 'type' => 'tns:ArrayOfInstanceOptionsFlags');
        }
        if(defined $plainOperationOptions->{'mSoftTimeout'}) {
            push @list, SOAP::Data->new('name' => 'mSoftTimeout', 'value' => $plainOperationOptions->{'mSoftTimeout'}, 'type' => 'long');
        }
        if(defined $plainOperationOptions->{'mProperties'}) {
            push @list, _createSOAPData_ArrayOfProperty($plainOperationOptions->{'mProperties'}, 'mProperties');
        }
        return SOAP::Data->new('name' => $name, 'value' => \@list, 'type' => 'tns:OperationOptions');
}

#---------------------------------------------------------------------------------------------

#  <complexType name="SAPControlInstance">
#   <complexContent>
#    <extension base="SAPHostControl:SAPInstance">
#     <sequence>
#     <element name="mStartProfilePath" type="xsd:string" minOccurs="0" maxOccurs="1" nillable="true"/>
#     <element name="mUsername" type="xsd:string" minOccurs="0" maxOccurs="1" nillable="true"/>
#     <element name="mPassword" type="xsd:string" minOccurs="0" maxOccurs="1" nillable="true"/>
#     </sequence>
#    </extension>
#   </complexContent>
#  </complexType>
#  <complexType name="SAPInstance">
#   <sequence>
#     <element name="mSid" type="xsd:string" minOccurs="0" maxOccurs="1" nillable="true"/>
#     <element name="mHostname" type="xsd:string" minOccurs="0" maxOccurs="1" nillable="true"/>
#     <element name="mSystemNumber" type="xsd:string" minOccurs="0" maxOccurs="1" nillable="true"/>
#     <element name="mSapVersionInfo" type="xsd:string" minOccurs="0" maxOccurs="1" nillable="true"/>
#   </sequence>
#  </complexType>
sub _createSOAPData_SAPControlInstance{
        my (
            $plainControlInstance,
            $name
        ) = @_;
        if(not defined $plainControlInstance) {
            $plainControlInstance = {};
        }
        my @list;
        if(defined $plainControlInstance->{'mSid'}) {
            push @list, SOAP::Data->new('name' => 'mSid', 'value' => $plainControlInstance->{'mSid'}, 'type' => 'string');
        }
        if(defined $plainControlInstance->{'mHostname'}) {
            push @list, SOAP::Data->new('name' => 'mHostname', 'value' => $plainControlInstance->{'mHostname'}, 'type' => 'string');
        }
        if(defined $plainControlInstance->{'mSystemNumber'}) {
            push @list, SOAP::Data->new('name' => 'mSystemNumber', 'value' => $plainControlInstance->{'mSystemNumber'}, 'type' => 'string');
        }
        if(defined $plainControlInstance->{'mSapVersionInfo'}) {
            push @list, SOAP::Data->new('name' => 'mSapVersionInfo', 'value' => $plainControlInstance->{'mSapVersionInfo'}, 'type' => 'string');
        }
        if(defined $plainControlInstance->{'mStartProfilePath'}) {
            push @list, SOAP::Data->new('name' => 'mStartProfilePath', 'value' => $plainControlInstance->{'mStartProfilePath'}, 'type' => 'string');
        }
        if(defined $plainControlInstance->{'mUsername'}) {
            push @list, SOAP::Data->new('name' => 'mUsername', 'value' => $plainControlInstance->{'mUsername'}, 'type' => 'string');
        }
        if(defined $plainControlInstance->{'mPassword'}) {
            push @list, SOAP::Data->new('name' => 'mPassword', 'value' => $plainControlInstance->{'mPassword'}, 'type' => 'string');
        }
        return SOAP::Data->new('name' => $name, 'value' => \@list, 'type' => 'tns:OperationOptions');
}

#---------------------------------------------------------------------------------------------
# $configurations is an array ref to an array of configuration objects
sub _createSOAPData_DeployConfiguration{
    my ($configurations) = @_;
    my @hostAgentConfigurationSArray = ();

    for my $configuration (@{$configurations}){
        my $targetElement = SOAP::Data->new('name' => 'target', 'value' => $configuration->getTargetType(), 'type' => 'string');
        my $nameElement = SOAP::Data->new('name' => 'name', 'value' => $configuration->getName(), 'type' => 'string');
        my @contentElements = map({SOAP::Data->new('value' => "$_", 'type' => 'string')} @{$configuration->getContent()});
        my $contentArrayElement = SOAP::Data->new('name' => 'configuration', 'value' => \@contentElements, 'type' => 'tns:ArrayOfString');
        my $configurationElement = SOAP::Data->new('name' => 'item', 'value' => [$targetElement, $nameElement, $contentArrayElement], 'type' => 'tns:HostAgentConfiguration');

        push(@hostAgentConfigurationSArray, $configurationElement);
    }

    return SOAP::Data->new('name' => 'configurations', 'value' => \@hostAgentConfigurationSArray, 'type' => 'tns:ArrayOfHostAgentConfiguration');
}

#---------------------------------------------------------------------------------------------

sub _createSOAPClient{
    my (
        $self,
        $timeout
    ) = @_;
    $self->_clearLastResult();
    my $soapclient;
    my %params = %{$self->{'soapclientParams'}};
    $params{'Keepalive'} = 1;
    # we set the soap client read timeout:
    # if the saphostctrl timeout is undef or 0 (async operation), we wait 30 seconds for saphostctrl's response
    # if the saphostctrl timeout is negative (sync operation), we wait 1h.
    # else, we add 30 seconds to any specific saphostctrl timeout.
    if(defined $timeout) {
        if($timeout < 0) {
            $params{'Timeout'} = 3600;
        }
        else {
            $params{'Timeout'} = $timeout + 30;
        }
    }
    else {
        $params{'Timeout'} = 30;
    }
    eval{
        $soapclient = SDB::Install::SOAP::Client->new(%params);
    };
    if ($@){
        $self->getErrMsgLst()->addError($@);
        return undef;
    }
    #---------------------------------------------------------------------------------
    sub SOAP::Serializer::as_ArrayOfProperty {
        my $self = shift;
        my $value = shift;
        my $name = shift;
        my $type = shift;
        my $attr = shift;
        return [$name, { 'xsi:type' => 'tns:ArrayOfProperty', %$attr }, $value];
    }
    #---------------------------------------------------------------------------------
    sub SOAP::Serializer::as_ArrayOfString {
        my $self = shift;
        my $value = shift;
        my $name = shift;
        my $type = shift;
        my $attr = shift;
        return [$name, { 'xsi:type' => 'tns:ArrayOfString', %$attr }, $value];
    }
    #---------------------------------------------------------------------------------
    sub SOAP::Serializer::as_ArrayOfHostAgentConfiguration {
        my $self = shift;
        my $value = shift;
        my $name = shift;
        my $type = shift;
        my $attr = shift;
        return [$name, { 'xsi:type' => 'tns:ArrayOfHostAgentConfiguration', %$attr }, $value];
    }
    #---------------------------------------------------------------------------------
    sub SOAP::Serializer::as_SAPControlInstance {
        my $self = shift;
        my $value = shift;
        my $name = shift;
        my $type = shift;
        my $attr = shift;
        return [$name, { 'xsi:type' => 'tns:SAPControlInstance', %$attr }, $value];
    }
    #---------------------------------------------------------------------------------
    sub SOAP::Serializer::as_OperationOptions {
        my $self = shift;
        my $value = shift;
        my $name = shift;
        my $type = shift;
        my $attr = shift;
        return [$name, { 'xsi:type' => 'tns:OperationOptions', %$attr }, $value];
    }
    #---------------------------------------------------------------------------------
    return $soapclient;
}

#---------------------------------------------------------------------------------------------

sub _getLastResultMsgByKey {
        my (
            $self,
            $key
        ) = @_;
        if(not $self->{'lastResultProcessed'}) {
            $self->_processLastResult();
        }
        return $self->{'lastResult'}->{'keys'}->{$key};
}

#---------------------------------------------------------------------------------------------

sub _processLastResult {
    my (
        $self
    ) = @_;
    $self->{'lastResultProcessed'} = 1;
    if ((not defined $self->{'lastResult'}) ||
        (not defined $self->{'lastResult'}->{'raw'})) {
        return 1;
    }
    my @outputLines; # have no line terminators
    my @logLines;
    my $lastresult  = $self->{'lastResult'}->{'raw'};
    my $results     = $lastresult->{'mOperationResults'};

    if (defined $lastresult->{'faultcode'}) {
        my $faultcode   = undef;
        my $faultstring = undef;
        my $opException = undef;
        if(defined $lastresult->{'detail'} && ref($lastresult->{'detail'}) =~ /HASH/) {
            $opException = $lastresult->{'detail'}->{'OperationException'};
        }
        if (defined $opException) {
            $faultcode   = $opException->{'mFaultCode'};
            $faultstring = $opException->{'mMessage'};
            $results     = $opException->{'mResponseMessages'};
        }

        $faultcode   = $lastresult->{'faultcode'  } if (!defined $faultcode);
        $faultstring = $lastresult->{'faultstring'} if (!defined $faultstring);

        $self->{'lastResult'}->{'keys'}->{'faultstring'} = $faultstring;
        $self->{'lastResult'}->{'keys'}->{'faultcode'}   = $faultcode;
    }
    elsif (!defined $results){
        $results =  $lastresult;
    }
    my %resulthash;
    if (defined $results) {
        my $item       = undef;
        my $resultsRef = ref($results);
        if ($resultsRef =~ /SCALAR/) {
            push @outputLines, $results;
        }
        elsif ($resultsRef =~ /HASH/) {
            $item = $results->{'item'};
        }

        if (defined $item && (ref($item) =~ /ARRAY/)) {
            foreach my $pair (@{$item}) {
                $self->_processMessagePair($pair, \@outputLines, \@logLines);
                $self->_processResultPair ($pair, \%resulthash);
            }
        }
        elsif (defined $item) {
            $self->_processMessagePair($item, \@outputLines, \@logLines);
            $self->_processResultPair ($item, \%resulthash);
        }
    }

    if (@outputLines) {
        $self->{'lastResult'}->{'outputlines'} = \@outputLines;
    }
    if (@logLines) {
        $self->{'lastResult'}->{'loglines'} = \@logLines;
    }
    if (%resulthash){
            $self->{'lastResult'}->{'resulthash'} = \%resulthash;
    }
    return 1;
}

#---------------------------------------------------------------------------------------------

sub _processMessagePair {
    my (
        $self,
        $messagePair,
        $outputLines, # reference to array of strings
        $logLines
    ) = @_;

    if (ref($messagePair) =~ /HASH/) {
        my $key = $messagePair->{'mMessageKey'};
        my $val = $messagePair->{'mMessageValue'};
        if (defined $key && ($key eq '')) {
            # i.e. stdout
            $val = '' if (!defined $val);
            push @$outputLines, $val;
        }
        elsif (defined $key) {
            # these 'keys' can occur more than once (like '').
            # but currently, we handle this only for '' (by creating an array).
                $self->{'lastResult'}->{'keys'}->{$key} = $val;
            if (defined $logLines && $key eq 'LogMsg/Text'){
                push @$logLines, $val;
            }
        }
    }
    return 1;
}

#---------------------------------------------------------------------------------------------

sub _processResultPair{
    my (
        $self,
        $resultPair,
        $resulthash # reference to a hash
    ) = @_;
    if (ref($resultPair) =~ /HASH/) {
        my $key = $resultPair->{'mKey'};
        my $val = $resultPair->{'mValue'};
        if (defined $key) {
            $resulthash->{$key} = $val;
        }
    }
    return 1;
}

#---------------------------------------------------------------------------------------------

sub _clearLastResult {
    my (
        $self
    ) = @_;
    $self->{'lastResultProcessed'}   = 0;
    $self->{'lastResult'}            = undef;
}

#---------------------------------------------------------------------------------------------

sub _logWebServiceInformation {
	my ($self, $webService, $arguments) = @_;
	my $host = $self->{soapclientParams}->{Host};
    if (!defined $host){
        $host = $self->{'logHostName'};
        if (!defined $host){
            $host = 'localhost';
        }
    }
    my $user = $self->{soapclientParams}->{User};
    my $url = $self->{soapclientParams}->{BaseUrl};
	my @argumentPairs = map {"'$_' - '" . $arguments->{$_} . "'"} keys(%{$arguments});
	my $argumentsString = ", arguments: " . join(', ', @argumentPairs);
    $self->getMsgLst()->addMessage("Calling SAP Host Control web service operation '$webService' (host: '$host', user: '$user', url: '$url'$argumentsString)");
}

#---------------------------------------------------------------------------------------------

1;
