package SDB::Install::SAPInstance::HanaTraceDirAnalyzer;

use strict;
use base 'SDB::Install::Base';
use SDB::Install::MonitoredTraceDirectory;
use SDB::Install::Globals qw ($gLogDir);
use SDB::Install::SysVars qw ($path_separator);
use File::stat;

sub createMonitoredTraceDirectory{
    my ($self,$hdbInstance, $host) = @_;

    if (!defined $host){
        $host = $hdbInstance->get_host();
        $self->{_defaultHost} = $host;
    }

    my $traceDir = $hdbInstance->get_hostNameTraceDir($host);

    if (!defined $traceDir){
        $self->setErrorMessage("Cannot get HANA trace directory", $hdbInstance->getErrMsgLst());
        return undef;
    }

    foreach my $dir ($traceDir, $gLogDir){
        my $stat = File::stat::stat($dir);
        if (!defined $stat){
            $self->setErrorMessage("Cannot access directory '$dir': $!");
            return undef;
        }
        if (!-d $stat){
            $self->setErrorMessage("'$dir' is not a directory");
            return undef;
        }
    }

    my %traceDirs = ('SYSTEMDB' => $traceDir);
    my $save_context = $hdbInstance->setMsgLstContext([$self->getMsgLst()]);
    my $tenantDatabases = $hdbInstance->getTenantDatabases();
    $hdbInstance->setMsgLstContext($save_context);
    if (defined $tenantDatabases){
        my $tenantName;
        foreach my $db_ent (@$tenantDatabases){
            $tenantName = $db_ent->[0];
            $traceDirs{$tenantName} =  $traceDir . $path_separator . $tenantName;
        }
    }

    foreach my $dbName (keys (%traceDirs)){
        $self->{_traceDirs}->{$host}->{$dbName} = new SDB::Install::MonitoredTraceDirectory($traceDirs{$dbName}, $gLogDir . $path_separator . $dbName . '_' . $host . '_tracediff.tgz');
    }
    return 1;
}

sub _getTraceDirsPerHost{
    my ($self, $host, $no_error) = @_;
    if (!defined $host){
        $host = $self->{_defaultHost};
        if (!defined $host){
            return undef;
        }
    }
    my $traceDirs = $self->{_traceDirs}->{$host};
    if (!defined $traceDirs){
        if (!$no_error){
            $self->setErrorMessage("host '$host' is unknown");
        }
        return undef;
    }
    return $traceDirs;
}

sub existsMonitoredTraceDirectory{
    my ($self, $host) = @_;
    return defined $self->_getTraceDirsPerHost($host,1) ? 1 : 0;
}

sub _getSortedDBNames{
    my ($traceDirs) = @_;
    return ('SYSTEMDB', sort grep {$_ ne 'SYSTEMDB'} keys %$traceDirs);
}

sub takeSnapshot{
    my ($self, $host) = @_;
    my $traceDirs = $self->_getTraceDirsPerHost($host);
    if (!defined $traceDirs){
        return undef;
    }
    my $msglst = $self->getMsgLst();
    foreach my $traceDir (values %$traceDirs){
        $msglst->addMessage ("Taking trace directory snapshot of '$traceDir->{fullDirName}'");
        $traceDir->takeSnapshot();
    }
    return 1;
}

sub analyzeDirectoryDeltas{
    my ($self, $host, $msglst, $createArchive) = @_;
    my $traceDirs = $self->_getTraceDirsPerHost($host);
    if (!defined $traceDirs){
        return undef;
    }
    my ($traceDir, $msg);
    foreach my $dbName (_getSortedDBNames($traceDirs)){
        $traceDir = $traceDirs->{$dbName};
        $msg = $msglst->addMessage ("Messages from trace files of database '$dbName'");
        $traceDir->analyzeDirectoryDeltas($msg->getSubMsgLst(), $createArchive);
    }
    return 1;
}

1;
