package SDB::Install::IniParser::SAPProfile;
use SDB::Install::SysVars;
use SDB::Install::SysInfo;
use Exporter;

use strict;

our $pattValue = '^\s*([a-zA-Z0-9_/]+)\s*=\s*([^\#\n\r]*)\s*(#.*)?$';
our $pattSection = '^\[([a-zA-Z0-9_!\"#$%&\'`()*+,-./:;<=>?@\\{\|}]+)\]\s*(#.*)?$';

sub getBackupName{
	my ($filePath, $newPath) = @_;
	my $backupName = '';

	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
	$mon++;
	$year += 1900;

	my $time_stamp = "~_${year}-${mon}-${mday}_${hour}.${min}.${sec}";

	if (defined $newPath){
		my ($basename) = ($filePath =~ /[\\\/]([^\\\/]+)$/);
		return $filePath . $path_separator . $basename . $time_stamp;
	}
	else{
		return $filePath . $time_stamp;
	}
}

use SDB::Install::BaseLegacy;

our @ISA = qw (SDB::Install::BaseLegacy Exporter);

our @EXPORT = qw (getBackupName);

sub new{
	my ($class, $fullname) = @_;
	my $self = $class->SUPER::new ();
	$self->{fullname} = $fullname;
	$self->{lines} = [];
	$self->{new} = 0;
	return $self;
}

sub read{
	my ($self, $msglst) = @_;
	
	if (@{$self->{lines}}){
		return 1;
	}
	
	
	if (!defined $msglst){
		$msglst = $self;
	}
	
	if (-f $self->{fullname}){
		if (!open (FD,$self->{fullname})){
			$msglst->AddError ("Cannot open file $self->{fullname}: $!");
			return undef;
		}
		while (my $line = <FD>){
			chomp ($line);
			push @{$self->{lines}}, $line;
		}
		close (FD);
	}
	elsif ( -e _ ){
		$msglst->AddError ("$self->{fullname} is no regular file");
			return undef;
	}
	else{
		$self->{new} = 1;
	}
	return 1;
}

sub setValue{
	my ($self,
	    $valuename,
	    $value,
	    $active,             # active (i.e. not a comment)
	    $blockComment,       # if specified, new value is inserted into this block
	    $withSpaces,         # with spaces (i.e. ' = ')
	    $insertBehindValName # if specified, new value is inserted behind this entry
	                         # (cannot be combined with $blockComment)
	   ) = @_;

	if (!defined $active) {$active = 1;}
	if (!defined $withSpaces) {$withSpaces = 0;}
	my $equal = '=';
	if ($withSpaces){
		$equal = ' = ';
	}
	my $valueEntry = ($active) ?       $valuename . $equal . $value
	                           : '#' . $valuename . $equal . $value;
	if ($blockComment){
		$blockComment =~ s/^\s*//;
		$blockComment =~ s/\s*$//;
		if ($blockComment !~ /^#/){ 
			$blockComment = '#'  . $blockComment;
		}
	}
	my $index = $self->getIndexOfValuename($valuename, $blockComment);
	if ($index > -1){  #change
		$self->{lines}->[$index] = $valueEntry;
	}
	else{           #new
		if ($blockComment){
			if ($self->getIndexOfValuename ($valuename) > -1){
				die ("$valuename exist in another block and not in $blockComment");
			}
			my $index = $self->getBlockAppendIndex ($blockComment);
			if ($index > -1){
				splice (@{$self->{lines}}, $index, 0, $valueEntry);
				return 1;
			}
		}
		elsif (defined $insertBehindValName) {
			my $indexBefore = $self->getIndexOfValuename($insertBehindValName);
			if ($indexBefore > -1) {
				splice (@{$self->{lines}}, $indexBefore+1, 0, $valueEntry);
				return 1;
			}
		}

		my $maxIndex = scalar (@{$self->{lines}}) - 1;

		if ($blockComment && $self->getBlockCommentIndex ($blockComment) == -1){
			push @{$self->{lines}}, $blockComment;
		}
		push @{$self->{lines}}, $valueEntry;
	}
}

sub getLines{
	return $_[0]->{lines};
}

sub hasLine{
	return ($_[0]->getLineIndex ($_[1]) != -1)
}

sub hasValue{
	return ($_[0]->getIndexOfValuename ($_[1]) > -1);
}

sub getValue{
	my ($self, $valuename, $withComment) = @_;
	if (!defined $withComment) {$withComment = 1;}
	my $index = $self->getIndexOfValuename ($valuename);
	if ($index < 0){
		return undef;
	}
	my ($value) = ($self->{lines}->[$index] =~ /=(.*)/);
	if (!$withComment){
		$value = (split('#',$value))[0];
	}
	$value =~ s/^\s*//;
	$value =~ s/\s*$//;
	return $value
}

sub delIndex($){
    my ($self,$index) = @_;
    splice (@{$self->{lines}}, $index, 1);
    return 1;
}


sub delValue{
	my ($self, $valuename, $blockComment) = @_;
	my $index = $self->getIndexOfValuename ($valuename, $blockComment);
	if ($index < 0){
		return undef;
	}
	$self->delIndex($index);
	return 1;
}

sub delLine{
	my ($self, $line) = @_;
	my $index = $self->getLineIndex($line);
	if ($index < 0){
		return undef;
	}
	$self->delIndex($index);
	return 1;
}

sub delLineRegex{
	my ($self, $regex) = @_;
	my $index = $self->getRegexIndex($regex);
	if ($index < 0){
		return undef;
	}
	my $line = $self->{lines}->[$index];
	$self->delIndex($index);
	return $line;
}

sub getNextFreeIndexInBlock{
	my ($self, $blockComment) = @_;
	my $found = scalar (@{$self->{lines}}) + 1;
	my $i=0;
	if ($blockComment){
		$blockComment =~ s/^\s*//;
		$blockComment =~ s/\s*$//;
		if ($blockComment !~ /^#/){ 
			$blockComment = '#'  . $blockComment;
		}
		my $blockCommentPattern = $blockComment;
		$blockCommentPattern =~ s/^\s*#*//;
		$blockCommentPattern = '^\s*#*'. quotemeta ($blockCommentPattern) . '\s*$';

		foreach my $l (@{$self->{lines}}){
			if ($l =~ /^\s*#/){
				if ($l =~ /$blockCommentPattern/){
					$found = $i;
					last;
				}
			}
			$i++;
		}
	}
	my $len = @{$self->{lines}};
	if ($found < $len){
		foreach $i ($found + 1 .. $len){
			if ($self->{lines}->[$i] !~ /\S/){
				$found = $i;
				last;
			}
		}
	}
	return $found;
}

sub getIndexOfValuename{
		my ($self, $valuename, $blockComment) = @_;
		my $start = -1;
		my $end = @{$self->{lines}};
		$valuename =~ s/^\s*//;
		$valuename =~ s/\s*$//; 
		if (defined $blockComment){
			$start = $self->getBlockCommentIndex ($blockComment);
			$end = $self->getNextBlockIndex($start + 1);
		}
		my $valuename_pattern = '^\s*'. quotemeta ($valuename) . '\s*=';
		foreach my $i ($start + 1 .. $end){
			if (defined $self->{lines}->[$i] &&
               ($self->{lines}->[$i] =~ /$valuename_pattern/)){
				return $i;
			}
		}
		#for i in range(start+1, end):
		#    if self.lines[i].find(valuename.strip()) > -1:
		#        if self.lines[i].split('=', 1)[0].strip().lstrip('#').strip() == valuename:
		#            return i
		return -1;
}

sub getBlockCommentIndex{
	my ($self, $blockComment) = @_;
	my $index = -1;
	if (defined $blockComment){
		$blockComment =~ s/^\s*//; 
		$blockComment =~ s/\s*$//;
		if ($blockComment !~ /^#/){ 
			$blockComment = '#'  . $blockComment;
		}
		my $blockCommentPattern = $blockComment;
		$blockCommentPattern =~ s/^\s*#*//;
		$blockCommentPattern = '^\s*#\s*'. quotemeta ($blockCommentPattern) . '\s*$';

		my $i = 0;
		foreach my $l (@{$self->{lines}}){
			if ($l =~ /$blockCommentPattern/){
				$index = $i;
				last;
			}
			$i++;
		}
	}
	return $index;
}

sub getLineIndex{
	my ($self, $line) = @_;
	my $linePattern = '^\s*' . quotemeta ($line) . '\s*$';
	return $self->getRegexIndex($linePattern);
}

sub getRegexIndex{
	my ($self, $regex) = @_;
	my $index = -1;
	my $i = 0;
	foreach my $l (@{$self->{lines}}){
		if ($l =~ /$regex/){
			$index = $i;
			last;
		}
		$i++;
	}
	return $index;
}

sub isBlockHeader{
	my ($self, $line) = @_;
	if ($line !~ /$pattValue/){
		my $pattern = '^#[^=]*$';
		if ($line =~ /$pattern/){
			return 1;
		}
	}
	return 0;
}

sub getNextBlockIndex{
	my ($self, $startIndex) = @_;
	my $len = @{$self->{lines}};
	if ($startIndex >= $len){
		return $len;
	}
	my $i;
	foreach $i ($startIndex .. $len - 1){
		if ($self->isBlockHeader ($self->{lines}->[$i])){
			return $i;
		}
	}
	return $len - 1;
}

sub getBlockAppendIndex{
	my ($self, $blockComment) = @_;
	my $insertIndex = -1;
	my $startIndex = $self->getBlockCommentIndex ($blockComment);
	if ($startIndex > -1){
		foreach my $i  ($startIndex + 1 .. scalar (@{$self->{lines}})){
			if (!defined $self->{lines}->[$i] || ($self->{lines}->[$i] !~ /\S/)){
				$insertIndex = $i;
				last;
			}
			if ($self->isBlockHeader ($self->{lines}->[$i])){
				$insertIndex = $i;
				last;
			}
		}
	}
	return $insertIndex;
}

sub write{
	my ($self, $backupPath, $fullname, $mode, $uid, $gid) = @_;
	my $new_name = 1;
	if (!defined $fullname){
		$fullname = $self->{fullname};
		$new_name = 0;
	}
	my $newname = $fullname . '.new'; 
	if (!open (FD, '>' . $newname)){
		$self->AddError ("Cannot create file '$newname': " . $!);
		return undef;
	}
	foreach my $line (@{$self->{lines}}){
		print FD $line . "\n";
	}

	close (FD);

	if (!$isWin) {
		my @statbuf = stat ($newname);
		my $oldmode = $statbuf[2] & 07777;
		if (defined $mode && ($oldmode != $mode)) {
			if (!chmod($mode, $newname)) {
				$self->AddWarning("Cannot chmod '$mode' of file '$newname': $!");
			}
		}
		if ((defined $uid && ($statbuf[4] != $uid)) ||
		    (defined $gid && ($statbuf[5] != $gid))) {
			$uid = -1 if (!defined $uid);
			$gid = -1 if (!defined $gid);
			if (!chown($uid, $gid, $newname)) {
				$self->AddWarning
				    ("Cannot chown (uid=$uid, gid=$gid) of file '$newname': $!");
			}
		}
	}

	$self->_backup ($backupPath, $new_name ? $fullname : undef);
	rename ($newname, $fullname);
	return 1;
}

sub _backup{
	my ($self, $backupPath, $fullname) = @_;
	if (!defined $fullname){
		$fullname = $self->{fullname};
	}
	if (-f $fullname){
		my $backup = getBackupName ($fullname, $backupPath);
		if (-e $backup){
			unlink ($backup);
		}
		#
		# TODO implement move across filesystem boundaries 
		#
		rename ($fullname, $backup);
	}
}	

1;

package SDB::Install::SAPProfiles;
use SDB::Install::SysVars;
use SDB::Install::NewDBUser;
import SDB::Install::IniParser::SAPProfile;
use strict;

use SDB::Install::BaseLegacy;

our @ISA = qw (SDB::Install::BaseLegacy);


sub tell_platform{
	$isWin ? 'ntamd64' : 'linuxx86_64';
}

sub new{ 
	my ($class, $profileDir, $sid, $globalHostName) = @_;
	my $self = $class->SUPER::new ();
	$self->{_profileDir} = $profileDir;
    $self->{_sid} = $sid;
    $self->{_globalHostName} = $globalHostName;
    
    if (defined $profileDir){
		$self->{_defaultProfile} = $profileDir . $path_separator . 'DEFAULT.PFL';
    }
    
    return $self;
}

sub get_DefaultProfile{
	if (defined $_[0]->{_defaultProfile}){
		return $_[0]->{_defaultProfile};
	}
	$_[0]->{_defaultProfile} = $_[0]->{_profileDir} . $path_separator . 'DEFAULT.PFL';
}

sub createDefaultProfile{
	my ($self, $msglst) = @_;
	
	if (!defined $msglst){
		$msglst = $self;
	}

	#create default profile:
	if (! -f $self->{_defaultProfile}){
           $msglst->AddMessage ('Creating ' .  $self->{_defaultProfile});
            my $pf = new SDB::Install::IniParser::SAPProfile ($self->{_defaultProfile});
            if (!defined $pf->read ()){
				$msglst->AddError ('Error reading default profile', $pf);
				return undef ;
            }
            $pf->setValue('SAPSYSTEMNAME', $self->{_sid}, undef, undef, 1);
            $pf->setValue('SAPGLOBALHOST', lc ($self->{_globalHostName}), undef, undef, 1);
            if (!$isWin){
				$pf->setValue ('HDB_LINKED_BINARIES', 'yes', undef, undef, 1);
				$pf->setValue ('HDB_SHARED_BINARIES', 'yes', undef, undef, 1);
                $pf->setValue ('hdbConfigType', 'DEFAULT', undef, undef, 1);
			}
            if (!defined $pf->write ()){
				$msglst->AddError ('Error writing default profile', $pf);
				return undef ;
            }
	}
	else{
		 $msglst->AddMessage ($self->{_defaultProfile} . ' already exists');
	}
	return 1;
}

sub getInstanceProfile{
	my ($self, $instanceNr, $hostName) = @_;
	return $self->{_profileDir} . $path_separator .  $self->{_sid} . '_HDB' . $instanceNr . '_' . $hostName;
}


sub createInstanceProfile{
	my ($self, $instanceNr, $hostName, $autostart, $mode_installation, $manifest) = @_;
	#create host specific profile
	if (!defined $autostart){
		$autostart = 1;
	}

	my $instanceProfile = $self->getInstanceProfile ($instanceNr, $hostName);
	if (-e $instanceProfile){
		#
		# TODO: move across filesystem boundaries
		#
		rename ($instanceProfile, getBackupName($instanceProfile));
	}

	$self->AddMessage ("Creating $instanceProfile");
	if (!open (FD, '>' . $instanceProfile)){
		$self->AddError ("Cannot create file $instanceProfile: $!");
		return undef;
	}

    my $sysinfo = new SDB::Install::SysInfo ();

	my $lines = [
		'SAP_RETRIEVAL_PATH = $(DIR_INSTANCE)' . $path_separator .
			'$(SAPLOCALHOST)',
		'DIR_HOME = $(SAP_RETRIEVAL_PATH)' . $path_separator . 'trace',
		'DIR_PERF = $(SAP_RETRIEVAL_PATH)' . $path_separator . 'trace',
		'DIR_LOGGING = $(SAP_RETRIEVAL_PATH)' . $path_separator . 'log',
		'SECUDIR = $(SAP_RETRIEVAL_PATH)' . $path_separator . 'sec',
		'Autostart = '. ($autostart ? '1' : '0'),
		'SETENV_00 = PATH=$(DIR_INSTANCE)' . $path_separator . 'exe' .
			$env_path_separator . '%(PATH)',
		'SETENV_01 = SAP_RETRIEVAL_PATH=$(SAP_RETRIEVAL_PATH)',
		'SETENV_02 = PYTHONPATH=$(SAP_RETRIEVAL_PATH)' .
			$env_path_separator . '$(DIR_EXECUTABLE)' .
			$env_path_separator . join ($path_separator,
				'$(DIR_INSTALL)','global','hdb','custom','python_support') .
			$env_path_separator . '$(DIR_EXECUTABLE)' .$path_separator .
				'python_support',
		'SAPSYSTEMNAME = ' . $self->{_sid},
		'SAPLOCALHOST = ' . $hostName,
		($isWin ?
			'SAPGLOBALHOST = ' . lc ($self->{_globalHostName})
		:
			()
		),
		'SAPSYSTEM = ' . $instanceNr,
		'INSTANCE_NAME = HDB' . $instanceNr,
		(!$isWin ?
			'DIR_CT_RUN = $(DIR_EXE_ROOT)' . $path_separator .
			tell_platform() . $path_separator . 'hdb'
		:
			'DIR_CT_RUN = $(DIR_EXE_ROOT)' . $path_separator .
			tell_platform()),
		'DIR_EXECUTABLE = $(DIR_INSTANCE)' . $path_separator . 'exe',
		'DIR_PROFILE = $(DIR_INSTALL)' . $path_separator . 'profile',
		'ccms/enable_agent = 0',
		'service/status_procs = hdbdaemon',
		'service/protectedwebmethods = SDEFAULT',
		($mode_installation ?
			('service/init_system_pki = ON')
		:
			()),
		'_PF = $(DIR_PROFILE)' . $path_separator .
			"$self->{_sid}_HDB${instanceNr}_$hostName",
		(!$isWin ?
			('SETENV_03 = DIR_LIBRARY=$(DIR_LIBRARY)',
			'SETENV_04 = LD_LIBRARY_PATH=$(DIR_LIBRARY):%(LD_LIBRARY_PATH)',
			'SETENV_05 = SHLIB_PATH=$(DIR_LIBRARY):%(SHLIB_PATH)',
			'SETENV_06 = LIBPATH=$(DIR_LIBRARY):%(LIBPATH)')
		:
			()),
        '#-----------------------------------------------------------------------',
        '# Run hdbupdconf',
        '#-----------------------------------------------------------------------',
        'HDBUPDCONF = $(DIR_INSTALL)'.$path_separator.'global'.$path_separator.'hdb'.$path_separator.
            'install'.$path_separator.'bin'.$path_separator.'hdbupdconf$(FT_EXE)',
        ($isWin ?
            ('Start_Program_00 = immediate $(HDBUPDCONF)')
        :
            ('Execute_00 = immediate $(HDBUPDCONF)')
        ),
        '#-----------------------------------------------------------------------',
        '# Copy SAP Executables',
        '#-----------------------------------------------------------------------',
		($isWin ?
			('_CPARG0 = list:$(DIR_CT_RUN)' . $path_separator . 'hdbbin.lst',
			'Start_Program_01 = immediate $(DIR_EXECUTABLE)' . $path_separator .
				'sapcpe$(FT_EXE) pf=$(_PF) $(_CPARG0)')
		:
			()),
		'#-----------------------------------------------------------------------',
		'# Update HDB Configuration',
		'#-----------------------------------------------------------------------',
		(!$isWin ?
			('_PY = $(DIR_HOME)/py.sap$(SAPSYSTEMNAME)_$(INSTANCE_NAME)',
			'Execute_01 = immediate rm -f $(_PY)',
			'Execute_02 = immediate ln -s -f $(DIR_EXECUTABLE)/Python/bin/python$(FT_EXE) $(_PY)')
		:
			'_PY = $(DIR_EXECUTABLE)\Python\python$(FT_EXE)'),
		'#-----------------------------------------------------------------------',
		'# Start SAP HDB',
		'#-----------------------------------------------------------------------',
		(!$isWin ?
			('_HDB = $(DIR_HOME)/hdb.sap$(SAPSYSTEMNAME)_$(INSTANCE_NAME)',
			'Execute_03 = local rm -f $(_HDB)',
			'Execute_04 = local ln -s -f $(DIR_EXECUTABLE)/hdbdaemon$(FT_EXE) $(_HDB)')
		:
			'_HDB = $(DIR_EXECUTABLE)' . $path_separator . 'hdbdaemon$(FT_EXE)'),
		'Start_Program_02 = local $(_HDB) -d -nw -f $(SAP_RETRIEVAL_PATH)' . $path_separator . 'daemon.ini pf=$(_PF)'
	];

	foreach my $line (@$lines){
		print FD $line . "\n";
	}
	close (FD);
	
	if (!$isWin){
		my $user = new SDB::Install::NewDBUser ($self->{_sid});
		$user->possess ($instanceProfile,0,0644);
	}
	
}

1;
