package SDB::Install::SAPSystemRegister;

use strict;
use SDB::Install::FsManager;
use SDB::Install::Globals qw($gLogDir
                             $gOperationRegisterHost
                             $gOperationUnregisterHost
                             $gHostRoleXS2Standby
                             $gHostRoleXS2Worker
                             $gProductNameEngine);
use SDB::Install::Installation;
use SDB::Install::SAPInstance;
use SDB::Install::SAPSystem;
use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::SAPInstance::TrexInstance;
use SDB::Install::Saphostagent qw(unregisterInstanceService);
use SDB::Install::System qw (changeOwn);

# use SDB::Install::DebugUtilities qw(dumpThings);

our @ISA = qw (SDB::Install::SAPSystemUtilities);


#-------------------------------------------------------------------------------
# Constructor

sub new {
    my $self = shift->SUPER::new (@_);
    return $self;
}


#-------------------------------------------------------------------------------
# Starts or stops the local instance and sends the command to other hosts.
#
# The method is called from SDB::Install::App::Console::HdbReg
#
# Returns int retCode

sub startStopOnly {

    my ($self,
        $app,              # SDB::Install::App::Console::HdbReg
        $sapSys,           # SDB::Install::SAPSystem
        $startStopParamID, # 'StartOnly' or 'StopOnly'
        $passwordParamId,  # 'Password', 'srcPasswd', etc.; default 'Password'
        $setStopServiceErr # does not ignore error when stopping sapstartsrv
       ) = @_;

    my $instconfig = $app->getInstconfig();

    if (!defined $passwordParamId){
        $passwordParamId = 'Password';
    }

    my $sid               =  $instconfig->getValue('SID');
    my $sidadmPassword    =  $instconfig->getValue($passwordParamId);
    my $scope             =  $instconfig->getValue($startStopParamID);
    my $isNoStart         = ($instconfig->getValue('NoStart')) ? 1 : 0;
    my $isStartWanted     = ($startStopParamID eq 'StartOnly') ? 1 : 0;
    my $isScopeSystemOnly = ($scope eq 'system_only')          ? 1 : 0;
    my $instance          = (defined $sapSys) ? $instconfig->getOwnInstance()
                                              : undef;
    if (!defined $sid) {
        $sapSys->AddError("SAP System ID not defined");
        return undef;
    }

    if (!defined $instance) {
        $sapSys->AddError("$gProductNameEngine '$sid' not found\n");
        return undef;
    }

    if ($instconfig->{isSlave} || !defined $scope) {
        $scope = 'instance';
    }

    $instance->SetProgressHandler($sapSys->getMsgLst()->getProgressHandler());

    if ($isStartWanted && !$isScopeSystemOnly) {

        if (!$self->startSapstartsrv($sapSys, $instance, $instconfig)) {
            return undef;
        }

        if (!$isNoStart && ($scope ne 'system')) {
            if (!$self->startLocalInstance($sapSys,
                                           $instance,
                                           $sid,
                                           $sidadmPassword, # used for Windows
                                           $instconfig)) {
                return undef;
            }
        }
    }
    elsif (!$isStartWanted) {

        if (!$self->stopSystem($sapSys,
                               $instance,
                               $sidadmPassword,
                               $instconfig->getTimeout ('stop_service'),
                               $instconfig->getTimeout ('stop_instance'),
                               $instconfig->isUseHttps(),
                               $instconfig->getValue ('SSOCertificate'),
                               !$isScopeSystemOnly, # stop local instance only
                               $setStopServiceErr,
                               $isScopeSystemOnly   # do no stop sapstartsrv
                              )) {
            return undef;
        }
        if (!$isScopeSystemOnly) {
            $self->tryKillProcessesRestartHandover($sapSys, $instance);
        }
    }

    if (!$instconfig->{isSlave} && $instconfig->isRemoteScope($scope)) {

        if ($isStartWanted && ($scope eq 'system')) {
            # start sapstartsrv on remote hosts only
            $instconfig->{params}->{NoStart}->{value} = 1;
        }

        my $rc = $self->startStopRemoteHosts($instconfig,
                                             $sapSys,
                                             $startStopParamID,
                                             $app->getActionProgressive(),
                                             $app->getActionDone());

        $instconfig->{params}->{NoStart}->{value} = $isNoStart;
        if (!$rc) {
            return undef;
        }
    }

    if ($isStartWanted && !$isNoStart &&
                          (($scope eq 'system') || $isScopeSystemOnly)) {

        if (!$self->startSystem($sapSys,
                                $instconfig,
                                $sidadmPassword,
                                $app->getAction())) {
            return undef;
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Register a system
# This method is called from SDB::Install::App::Console::HdbReg
#
# This method does not change or create any object in case of check only.
# The method cleanupSAPServicesEntry (calling TrexInstance::removeSapservicesEntry)
# is not called any more because TrexInstance::cleanupLocalLeftovers is called
# when starting sapstartsrv.
#
# Parameters: $app    SDB::Install::App::Console::HdbReg
#             $sapSys SDB::Install::SAPSystem
#
# Returns int retCode

sub registerSystem {
    
    my ($self, $app, $sapSys) = @_;

    my $sid         = $sapSys->get_sid();
    my $instconfig  = $app->getInstconfig();
    my $hostHashMap = $instconfig->getValue('HostMap');
    my $password    = $instconfig->getValue('Password');
    my $instance    = $instconfig->getOwnInstance();

    $instance->setMsgLstContext ([$sapSys->getMsgLst]);
    $instance->SetProgressHandler ($app->GetProgressHandler());

    if (!$instconfig->{isSlave}) {

        $instconfig->addMsgHostMap($sapSys);
    }

    if ($app->isStopSystem() && !$instconfig->isCheckOnly()
                                              && $sapSys->getUser()->exists()) {

        if (!$self->stopSystem(
                            $sapSys,
                            $instance,
                            $password,
                            $instconfig->getTimeout ('stop_service'),
                            $instconfig->getTimeout ('stop_instance'),
                            $instconfig->isUseHttps(),
                            $instconfig->getValue ('SSOCertificate'),
                            $instconfig->isAnyHostChanged())) { # local host only
            $sapSys->AddError ('Cannot stop system', $sapSys);
            return undef;
        }
    }

    my $oldUid;
    my $oldGid;
    my $msg;

    if (!$isWin){
        my $errlst = new SDB::Install::MsgLst();
        my $cfg = $sapSys->getUserConfig ($errlst);
        if (!defined $cfg){
            $sapSys->AddError ('Cannot get user configuration', $errlst);
            return undef;
        }
        $oldUid = $cfg->{sidadm_id};
        $oldGid = $cfg->{sapsys_groupid};
    }

    my $remoteParamIDs = ['InstallHostagent', 'NoStart'];
    
    if ($instconfig->isCheckOnly()) {

        if (!$instconfig->shouldSkipHostagentCalls()) {
            my $rc = $self->tryRegisterRemoteHosts($instconfig, $app->getActionProgressive(),
                                                   $app->getActionDone(), $gOperationRegisterHost,
                                                   $remoteParamIDs);
            if (!$rc) {
                $sapSys->AddError (undef, $instconfig);
                return undef;
            }
        }
        return 1;
    }

    my $usrSapSid    = $sapSys->getUsrSapSid;
    my $globalSidDir = $sapSys->get_globalSidDir();
    my $newUid       = $instconfig->getValue('UID');
    my $newGid       = $instconfig->getValue('GID');
    my $user_changed = 0;
    if (!$isWin && ($oldUid != $newUid || $oldGid != $newGid)){
        $user_changed = 1;
    }
    
    my $msgRoot    = $sapSys->AddMessage ("Creating root...");

    my $rc = $sapSys->createRoot ($msgRoot->getSubMsgLst (), $newUid, $newGid);

    $msgRoot->endMessage(1, 'Create root');
    
    if (!defined $rc) {
        $sapSys->AddError ('Cannot create root', $sapSys);
        return undef;
    }

    if (defined $instconfig->{existingSidAdm}
                            && $instconfig->{existingSidAdm}->{changeUserCfg}) {
        $sapSys->{_user} = new SDB::Install::NewDBUser($sid);
    }

    my $configureUser     = 0;
    ($rc, $configureUser) = $self->tryCreateSysAdminUser($instconfig,
                                                         $sapSys,
                                                         $password);
    if (!$rc) {
        return undef;
    }
    if (($configureUser == 1) && !$instconfig->{isSlave}) {
        # new sys admin user is created on master host
        $user_changed = 1;
    }

    $instconfig->tryRemoveUnregisterPersFile();

    my $nr = $instance->get_nr();

    my $fsManager = new SDB::Install::FsManager ($usrSapSid,
                                                 $globalSidDir,
                                                 $newUid,
                                                 $newGid);

    $fsManager->set_instanceNr ($nr);
    $sapSys->setFileSystemManager($fsManager);

    if (!$isWin){
        
        if (($configureUser > 0) || $instconfig->getValue('InitUserHomeDir')) {

            $sapSys->getUser()->configureHome
                                    ($sapSys->get_globalTrexInstallSupportDir(),
                                     $instance->get_instanceDir(),
                                     $instconfig->{localHostName});
                                 
            $sapSys->AddMessage(undef, $sapSys->getUser());
            $sapSys->getUser()->resetMsgLstContext();
        }
        
        $fsManager->setUserSettings($newUid, $newGid, $sapSys->getShmGID);
                                    
        if (!$fsManager->createLocalIpcDirectory ()) {
            
            $sapSys->AddError ("Cannot create ipc directory", $fsManager);
            $sapSys->AddMessage (undef, $fsManager);
            return undef;
        }
        $sapSys->AddMessage (undef, $fsManager);
    }

    if (!defined $fsManager->createUsrSapSidLinks()){
        
        $sapSys->AddError ("Cannot create links under '/usr/sap/$sid'", $fsManager);
        return undef;
    }

    if (!defined $fsManager->createInstanceLink()){
        
        $sapSys->AddError ('Cannot create links to instance', $fsManager);
        return undef;
    }

    if (!$isWin) {
        
        my $msg    = $sapSys->AddProgressMessage ('Changing owner'
                                . " (uid: $newUid, gid: $newGid) of files...");
        my $errlst = new SDB::Install::MsgLst;
        
        if (!$instconfig->{isSlave}) {

            my $globSidDir = $sapSys->get_globalSidDir();

            # change all UIDs and GIDs of <sapmnt>/<SID>
            # - except root IDs
            # - except IDs contained in SKIP_CHOWN_DIR

            if (!changeOwn ($oldUid,
                            $newUid,
                            $oldGid,
                            $newGid,
                            $globSidDir,
                            $errlst,
                            $msg->getSubMsgLst(), # info list
                            SKIP_CHOWN_DIR(),     # skip path map
                            0,                    # change root IDs
                            1,                    # never change root IDs
                           )) {

                $sapSys->AddError ("Cannot change owner of files under '$globSidDir'", $errlst);
                return undef;
            }
        }
        
        my $currOldUid = $oldUid;
        my $currOldGid = $oldGid;

        if ($instconfig->{isSlave}) {

            if (-l $usrSapSid){
                $currOldUid = (lstat (_))[4];
                $currOldGid = (lstat (_))[5];
            }
            elsif (-e $usrSapSid){
                $currOldUid = (stat (_))[4];
                $currOldGid = (stat (_))[5];
            }
        }

        # change matching UIDs and GIDs of /usr/sap/<SID>
        # - including root IDs
        # - except IDs contained in SKIP_CHOWN_DIR

        if (!changeOwn ($currOldUid,
                        $newUid,
                        $currOldGid,
                        $newGid,
                        $usrSapSid,
                        $errlst,
                        $msg->getSubMsgLst(), # info list
                        SKIP_CHOWN_DIR(),     # skip path map
                        1                     # change root IDs
                       )) {

            $sapSys->AddError ("Cannot change owner of files under '$usrSapSid'", $errlst);
            return undef;
        }
    }
        
    if (!$isWin && $user_changed && !$instconfig->{isSlave}) {

        if (!defined $sapSys->changePackageOwnerInRegistry($oldUid, $newUid,
                                                           $oldGid, $newGid)) {
            $sapSys->AddError ('Cannot change package owner in registry', $sapSys);
            return undef;
        }
    }

    $instance->{_user} = $sapSys->getUser();

    if ($instconfig->{isSlave}) {
        $instance->createHWInfoFile ($app->GetSysInfo());
        if (!$instconfig->isSlaveUpgrade()
            &&
            ($instance->isUsingInternalInterface()
             || ($instance->isUsingGlobalInterface() && $instance->getInternalNetworkPrefix()))) {
            $instance->setMsgLstContext([$sapSys->getMsgLst ()]);
            if (!defined $instance->reconfigureTrexNet ($instconfig)){
                $sapSys->setErrorMessage (undef, $instance->getErrMsgLst());
                return undef;
            }
        }
    }
    else {
        my $oldHostRoles = $instance->getActiveHostRoles();
        if (!defined $oldHostRoles) {
            $sapSys->setErrorMessage (undef, $instance->getErrMsgLst());
            return undef;
        }

        my $configMsg = $sapSys->AddProgressMessage
                                     ("Updating system configuration files...");

        my $sapprofileinitempate = new SDB::Install::IniFile (
            $sapSys->get_globalTrexInstallTemplateDir () .
            $path_separator . 'sapprofile.ini');

        my $defaultprofile = new SDB::Install::IniFile (
            $sapSys->get_ProfileDir () .
            $path_separator .'DEFAULT.PFL');

        my $inis = [$sapprofileinitempate, $defaultprofile];

        my $rc = 1;
        my ($val, $hostkey,$i);

        foreach my $ini (@$inis){
            if ($ini->ErrorState()){
                $sapSys->PushError ('Cannot access ini-file', $ini);
                $rc = 0;
                next;
            }
            foreach $hostkey ('SAPLOCALHOST','SAPGLOBALHOST'){
                $val = $ini->getValue (undef, $hostkey);
                if (defined $val) {
                    my $newHostName = $hostHashMap->{$val};
                    if (defined $newHostName) {
                        $ini->setValue (undef, $hostkey, $newHostName);
                    }
                }

                #foreach $i (0..$#pairsOriginNewHosts){
                #    if ($val eq $pairsOriginNewHosts[$i]->[0]){
                #        $ini->setValue (undef, $hostkey, $pairsOriginNewHosts[$i]->[1]);
                #        last;
                #    }
                #}
            }
        }

        if ($user_changed){
            $sapSys->saveConfig ();
        }

        if (!$rc){
            return undef;
        }

        foreach my $ini (@$inis){
            if (!defined $ini->write ()){
                $sapSys->PushError ('Cannot write ini-file', $ini);
                $rc = 0;
            }
            $sapSys->AddSubMessage($configMsg,
                                   "Updating '$ini->{fullname}'",
                                   $ini);
        }
        if (!$rc){
            return undef;
        }

        $configMsg = $sapSys->AddProgressMessage ("Adjusting instance...");
        $instance->setMsgLstContext([$configMsg->getSubMsgLst ()]);
        if (!defined $instance->renameInstance
                 ($sid, # old sid
                  $sid, # new sid
                  $nr,  #old nr
                  $nr,  #new nr
                  $hostHashMap,
                  $instconfig,
                  (!$isWin && $user_changed) ? [$oldUid, $newUid, $oldGid, $newGid]
                                             : undef,
                  $app->GetSysInfo(),
                  undef,
                  $oldHostRoles)) {

            $sapSys->AddError ("Adjust instance failed", $instance);
            return undef;
        }
    }
    
    if (!$self->adaptOperatingSystemConfig($sapSys, $instconfig)) {
        $sapSys->AddError ('Cannot adapt operating system configuration', $sapSys);
        return undef;
    }

    if (!$instconfig->{isSlave}){
        $sapSys->_linkRedHatCompatCPlusPlusLib ();
    }

    $msg = $sapSys->AddProgressMessage ("Creating sapservices entry...");
    $instance->setMsgLstContext([$msg->getSubMsgLst ()]);
    if (!defined $instance->createService ($password)) {

        $sapSys->AddError ("Cannot create sapservices entry", $instance);
        return undef;
    }

    if ($instconfig->{isSlave}) {

        my $newHostName = $instance->get_host();
        my $oldHostName = $instconfig->{oldHost};
        $oldHostName    = $newHostName if (!defined $oldHostName);

        if ($instconfig->{isRenameAccelerator}) {
            my $chownList = ($isWin) ? undef
                                     : [$instconfig->{oldUidAccelerator}, $newUid,
                                        $instconfig->{oldGidAccelerator}, $newGid];

            if (!$instance->renameAcceleratorInstance($instconfig,
                                                     $sid,         $sid,
                                                     $nr,          $nr,
                                                     $oldHostName, $newHostName,
                                                     $chownList)) {
                return undef;
            }
        }

        if ($instconfig->{isRenameExtendedStorage}) {
            my $chownList = ($isWin) ? undef
                                     : [$instconfig->{oldUidEs}, $newUid,
                                        $instconfig->{oldGidEs}, $newGid];

            if (!$instance->renameExtendedStorageInstance($instconfig,
                                                  $sid,         $sid,
                                                  $nr,          $nr,
                                                  $oldHostName, $newHostName,
                                                  $chownList)) {
                return undef;
            }
        }

        if ($instconfig->{isRenameRDSync}) {
            my $chownList   = ($isWin) ? undef
                                       : [$instconfig->{oldUidRDSync}, $newUid,
                                          $instconfig->{oldGidRDSync}, $newGid];

            if (!$instance->renameRDSyncInstance($instconfig,
                                                 $sid,         $sid,
                                                 $nr,          $nr,
                                                 $oldHostName, $newHostName,
                                                 $chownList)) {
                return undef;
            }
        }
        if ($instconfig->{isRenameXS2}){
            return undef if(!$instance->recreateXSSpaceIsolationUsers($instconfig, $sid, $sid)); 
            return undef if(!$instance->recreateXsEaDataPath($instconfig, $oldHostName));
        }
    }

    if (!$instconfig->trySetAutoStartLocal($instance, $sapSys)) {
        return undef;
    }

    if ($instconfig->{isSlave}) {
        $instconfig->pers_remove();
    }

    if (!$self->startSapstartsrv($sapSys, $instance, $instconfig)) {
        return undef;
    }

    my $startSystemWanted = 0;
    my $instanceStarted   = 0;
    $self->setRenameMasterStartedHandler ($instconfig);
    if ($app->isStartInstance() && !$instconfig->getValue('NoStart')) {
        if($instconfig->noHostStart()) {
            $startSystemWanted = 1;
        }
        else {
            if (!$self->startLocalInstance($sapSys,
                                          $instance,
                                          $sid,
                                          $password, # used for Windows only
                                          $instconfig)) {
                return undef;
            }
            $instanceStarted = 1;
        }
    }
    $instconfig->setMsgLstContext ([$sapSys->getMsgLst ()]);

    my $shouldSkipSHACalls = $instconfig->shouldSkipHostagentCalls();
    if (!$shouldSkipSHACalls && !$instconfig->{isSlave} && defined $instconfig->getRemoteHosts()) {
        my $isRemoteReconfigWanted =
                ($instance->isUsingInternalInterface()
                 ||
                 ($instance->isUsingGlobalInterface() &&
                  defined $instance->getInternalNetworkPrefix())) ? 1 : 0;

        if (!$self->tryRegisterRemoteHosts ($instconfig,
                                            $app->getActionProgressive(),
                                            $app->getActionDone(),
                                            $gOperationRegisterHost,
                                            $remoteParamIDs,
                                            undef, # $cmdLineParams
                                            undef, # $theseHostsOnly
                                            $isRemoteReconfigWanted, # is serial
                                           )) {
            $sapSys->AddError ('Cannot register remote hosts', $instconfig);
            return undef;
        }
    }

    if ($startSystemWanted && !$instconfig->{isSlave}) {

        my $systemInfo = "$gProductNameEngine system $sid (HDB$nr)";

        if (!$self->startSystem($sapSys,
                                $instconfig,
                                $password,
                                $app->getAction())) {
            return undef;
        }
        $instanceStarted = 1;
    }

    if (!$instconfig->{isSlave} && $instanceStarted
        && $instconfig->getValue('ChangeSystemPasswd')) {

        if (!$self->changeSystemUserPassword($sapSys, $instance, $instconfig)) {
            return undef;
        }
    }

    return 1;
}


#-------------------------------------------------------------------------------
# Unregisters a local instance and deletes the host specific directory
# (trace files) from the installation path.
#
# This method does not stop the system and does not change any object
# in case of check only.
#
# If the system or the installation path is not accessible, this method
# deletes the system administrator user and the directory '/usr/sap/<sid>'.
#
# The method is called from SDB::Install::App::Console::HdbReg
#
# Parameters: $app    SDB::Install::App::Console::HdbReg
#             $sapSys SDB::Install::SAPSystem  (may be undef)
#
# Returns int retCode

sub unregisterInstance {

    my ($self, $app, $sapSys) = @_;

    my $instconfig   = $app->getInstconfig();
    my $sid          = $instconfig->getValue('SID');
    my $unregister   = $instconfig->getValue('Unregister');
    my $ipcDir       = join($path_separator, '', 'var', 'lib', 'hdb', $sid);
    my $msgList      = $app->getMsgLst ();

    $instconfig->setMsgLstContext ([$msgList]);

    if (!defined $sid) {
        $app->setErrorMessage ("SAP System ID not defined", $instconfig->getErrMsg ());
        return undef;
    }

    if (!defined $sapSys) {
        $msgList->addMessage ("$gProductNameEngine '$sid' not found\n");
    }
    else {
        my $instance = $instconfig->getOwnInstance();

        if (defined $instance) {

            my @checkPathList = ($sapSys->get_globalSidDir(),
                            $instance->getDataPath(1),
                            $instance->getLogPath(1));

            my @pathNames    = ('installation', 'data', 'log');
            my $usrSapSidDir = join($path_separator, '', 'usr', 'sap', $sid);
            my $pattern      = quotemeta("$usrSapSidDir".$path_separator);
            my $i            = 0;

            foreach my $currPath (@checkPathList) {
                if (($currPath eq $usrSapSidDir) || ($currPath =~ "^$pattern")){
                    $app->setErrorMessage ("Cannot unregister instance $sid: "
                        . $pathNames[$i]
                        . " path '$currPath' is contained in '$usrSapSidDir'");
                    return undef;
                }
                $i++;
            }

            if (!$instconfig->isCheckOnly()) {

                if (defined $instconfig->pers_filename()) {
                    $instconfig->pers_remove();
                }

                if (($unregister eq 'instance')
                    && $instance->exists_remote_host()) {

                    my $persFilename = $instance->get_hostNameDir()
                                     . $path_separator
                                     . $instconfig->getPersistentFile();

                    $instconfig->pers_store ($persFilename,
                        {'oldHost'        => $instance->get_host(),
                         'unregisterHost' => 1});    # ignore any error
                }

                if (!$self->stopSystem(
                                $sapSys,
                                $instance,
                                $instconfig->getValue   ('Password'),
                                $instconfig->getTimeout ('stop_service'),
                                $instconfig->getTimeout ('stop_instance'),
                                $instconfig->isUseHttps(),
                                $instconfig->getValue ('SSOCertificate'))) {
                    return undef;
                }
                if(!$instconfig->getValue('SkipModifySudoers')){
                    $sapSys->tryRemoveSudoersEntries();
                }

                $instance->removeHWInfoFile(); # ignore errors
                $self->tryKillProcessesRestartHandover($sapSys, $instance);

                if (!$instconfig->shouldSkipHostagentCalls()) {
                    unregisterInstanceService($sapSys,
                                              0,   # $isProgressMsg
                                              $gLogDir,
                                              $sid,
                                              $instance->get_nr(),
                                              $instance->get_host(),
                                              -1); # $timeoutSeconds (-1 = synchronous)
                    # ignore saphostagent errors
                }

                $self->cleanupSAPServicesEntry($sapSys, $instance); # ignore errors
                $self->_tryDeleteXSSpaceIsolationUsers($sapSys, $instance, $instconfig); # ignore errors
                $self->_tryDeleteXsEaDataPath($sapSys, $instance, $instconfig); # ignore errors
            }
        }
    }

    if (!$instconfig->isCheckOnly()) {
        my $keepUserHomeDir = $instconfig->getValue('KeepUserHomeDir');
        my $homeDir;
        if ($keepUserHomeDir) {
            my $user = new SDB::Install::NewDBUser($sid);
            $homeDir = $user->getSidAdmHome();
        }
        if (!$instconfig->getValue('KeepUser')) {

            $self->deleteSysAdminUser($msgList,
                                      $sid,
                                      $keepUserHomeDir,
                                      $instconfig->getIgnore('scan_password_database'));
            # ignore errors
        }

        $self->cleanupUsrSapSidDirectory($msgList, undef, $sid, $keepUserHomeDir, $homeDir);    # ignore errors
        $self->deleteOrUnlinkDirectory  ($msgList, undef, $ipcDir) ;# ignore errors
    }

    my $shouldSkipSHACalls = $instconfig->shouldSkipHostagentCalls();
    if (!$shouldSkipSHACalls && !$instconfig->{isSlave} && $instconfig->isRemoteScope($unregister)) {
        my $cmdLineOption = ' ' . $instconfig->getOpt('Unregister');
        $instconfig->setMsgLstContext ([$app->getMsgLst()]);
        if (!$self->tryRegisterRemoteHosts ($instconfig,
                                            $app->getActionProgressive(),
                                            $app->getActionDone(),
                                            $gOperationUnregisterHost,
                                            ['KeepUser', 'KeepUserHomeDir', 'KeepXsUsers', 'SkipModifySudoers'], # Remove KeepXsUsers in case of downport to SPS 11!
                                            $cmdLineOption)) {
            $app->setErrorMessage (undef, $instconfig->getErrMsgLst ());
            return undef;
        }
    }

    return 1;
}

sub _tryDeleteXSSpaceIsolationUsers {
	my ($self, $sapSys, $instance, $instconfig) = @_;
	my $hostRoles = $instance->getHostRolesByIniFile();
	my $hasXsRole = defined($hostRoles) && grep {$_ eq $gHostRoleXS2Worker || $_ eq $gHostRoleXS2Standby} @{$hostRoles};
	my $saveContext = $instance->setMsgLstContext($sapSys->getMsgLstContext());
	my $returnValue = $hasXsRole ? $instance->deleteXSSpaceIsolationUsers($instconfig, $instconfig->getSID()) : 1;

	$instance->setMsgLstContext($saveContext);
	return $returnValue;
}

sub _tryDeleteXsEaDataPath {
	my ($self, $sapSys, $instance, $instconfig) = @_;
	my $hostRoles = $instance->getHostRolesByIniFile();
	my $hasXsRole = defined($hostRoles) && grep {$_ eq $gHostRoleXS2Worker || $_ eq $gHostRoleXS2Standby} @{$hostRoles};
	my $saveContext = $instance->setMsgLstContext($sapSys->getMsgLstContext());
	my $returnValue = $hasXsRole ? $instance->deleteXsEaDataPath($instconfig, $instance->get_host()) : 1;

	$instance->setMsgLstContext($saveContext);
	return $returnValue;
}

sub getInstanceRenameArgs{
    my ($self, $instconfig) = @_;
    my $sid         = $instconfig->getValue ('SID');
    my $hostHashMap = $instconfig->getValue ('HostMap');
    my $instance    = $instconfig->getOwnInstance ();
    my $nr          = $instance->get_nr ();
    return [$sid, $sid, $nr, $nr, $hostHashMap];
}

1;
