package SDB::Install::SOAP::Client;

use base 'SDB::Install::Base';
use SDB::Install::MsgLst;
use SDB::Install::DebugUtilities;
use SDB::Install::Tools qw(trim);
use SAPDB::Install::MD5Sum qw(MD5Str);
use SDB::Install::System qw (loadSSLRequiringPackage);
use Cwd;


use strict;


our $soapLiteServicePrototypes = {};


# ctor parameters:
#        'WsdlBuf'      buffer containing the wsdl of the service
#        'Host'         host
#        'User'         user
#        'Passwd'       password
#        'Port'         port
#        'Realm'        realm
#        'BaseUrl'      base url
#        'SocketMode'   either uds (unix domain socket), or tcp
#        'Keepalive'    1 for keepalive, 0 or undef for not
#        'Timeout'      soap client read timeout (seconds)
#        'Https'        1 for https, 0 or undef for not (only effective for SocketMode=='tcp')
#
# ctor may die (i.e. "throws exception"), use eval to catch:
sub new {
    my $self = shift->SUPER::new ();
    my %ctorParams = @_;
    $self->{'ctorParams'} = \%ctorParams;
	$self->_create();
	return $self;
}


sub _create{
    my (
        $self
    ) = @_;
    eval{
        require SOAP::Lite;
        require SDB::Install::SOAP::Schema;
    };
    if ($@){
        die("Could not create SDB::Install::SOAP::Client object: $@");
    }
    my $credentials;
    if($self->{'ctorParams'}->{'SocketMode'} =~ /tcp/i) {
        $credentials = [
                $self->{'ctorParams'}->{'Host'}.':'.$self->{'ctorParams'}->{'Port'},
                $self->{'ctorParams'}->{'Realm'},
                $self->{'ctorParams'}->{'User'},
                $self->{'ctorParams'}->{'Passwd'}
            ];
    }
    elsif($self->{'ctorParams'}->{'SocketMode'} =~ /uds/i) {
        $credentials = undef;
    }
    $self->_setLWPProtocolImplementor();
    my $wsdlHash = MD5Str($self->{'ctorParams'}->{'WsdlBuf'});
    if(not defined $soapLiteServicePrototypes->{$wsdlHash}) {
        my $schema = new SDB::Install::SOAP::Schema();
        $schema->setWsdlBuf($self->{'ctorParams'}->{'WsdlBuf'});
        my $soapLiteObj = new SOAP::Lite();
        $soapLiteObj->schema($schema);
        local $SIG{__WARN__} = sub {};
        $soapLiteServicePrototypes->{$wsdlHash} = $soapLiteObj->service($self->{'ctorParams'}->{'BaseUrl'});
    }

    my $ssl_opts = {};
    my $verify_host = 0;

    if( ($self->{'ctorParams'}->{'SocketMode'} =~ /tcp/i) && $self->{'ctorParams'}->{'Https'} ) {
        $ssl_opts->{'verify_hostname'} = $verify_host;
        if (defined $self->{'ctorParams'}->{'sso_cert'}){
            $ssl_opts->{'SSL_cert_file'} = $self->{'ctorParams'}->{'sso_cert'};
        }
    }
    my $service = $soapLiteServicePrototypes->{$wsdlHash}->proxy(
        $self->{'ctorParams'}->{'BaseUrl'},
        'keep_alive' => $self->{'ctorParams'}->{'Keepalive'},
        'timeout'=> $self->{'ctorParams'}->{'Timeout'},
        'credentials' => $credentials,
        'ssl_opts' => [%$ssl_opts]
    );

    $service->on_fault (
        ## this is called on network errors from SOAP::Lite :
        sub {
            my $soap = shift;
            my $res = shift;
            if(!ref($res)) {
                my $error = $soap->transport ()->status ();
                if (defined $IO::Socket::SSL::SSL_ERROR){
                    $error .= ":  $IO::Socket::SSL::SSL_ERROR";
                }
                $self->getErrMsgLst ()->addError ($error);
                return;
            }
            # we currently don't process the error code and string attached to the SOAP::SOM
            # parameter $res into an error msg, since this handler gets called with high freq, but the
            # "faults" are mostly informational seen from the pov of the installer.
            #
            ## This is only a workaround, since in the 1st place message filtering
            ## should be done at the places in the installer code, where large amounts of
            ## false faults are to be expected during "normal" operation, e.g. when connections
            ## are probed.
            ##
            ## We _should_ enable error msgs for the SOAP::SOM faults again then, since 
            ## from what we saw, they looked potentially very helpful in analysing hostagent/connection
            ## problems.
            return $res->fault();
        }

    );
    $self->{'service'} = $service;
	return $self;
}

=functionComment

The so-called LWP Protocol Implementor can only be set modally
for soaplite.

=cut

sub _setLWPProtocolImplementor {
    my (
        $self
    ) = @_;
    if($self->{'ctorParams'}->{'SocketMode'} =~ /tcp/i) {
        require LWP::Debug;
        if ($self->{'ctorParams'}->{'Https'}) {
            if (!loadSSLRequiringPackage ('LWP::Protocol::https', $self->getMsgLst (), $self->getErrMsgLst ())){
                die ($self->getErrorString() . "\n");
            }
            LWP::Protocol::implementor( 'https' => 'LWP::Protocol::https');
        }
        else {
            require LWP::Protocol::http;
            LWP::Protocol::implementor( 'http' => 'LWP::Protocol::http');
        }
    }
    elsif($self->{'ctorParams'}->{'SocketMode'} =~ /uds/i) {
        require LWP::Debug;
        require LWP::Protocol::http::SocketUnixAlt;
        LWP::Protocol::implementor( 'http' => 'LWP::Protocol::http::SocketUnixAlt' );
    }
}


1;
