#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: 


package SDB::Install::Saphostagent;
use strict;

require Exporter;
our @ISA = qw (Exporter);

our @EXPORT = qw (
    installSaphostagent
    getSaphostagentPpmsId
    sapadmUserExists
    unregisterInstanceService
    getActiveSaphostexecDir
    getSHAVersion
    getSaphostexecPath
    setDatabaseProperties
);

use File::Spec;
use LCM::ProcessExecutor;

use SAPDB::Install::Hostname;
use SDB::Install::MsgLst;
use SDB::Install::Globals qw ($gOperationCheckSHAVersion $gProductNameSHA);
use SDB::Install::RemoteHostctrlHosts;
use SDB::Install::SysVars;
use SDB::Install::DebugUtilities;
use SDB::Install::BaseLegacy;
use SDB::Install::System qw (
    exec_program
    getSysProgPath
    isAdmin
    isSidadmin
);
use SDB::Install::Tools qw (trim generatePassword);
use SDB::Install::NewDBUser;
use SDB::Install::SAPHostControl;
use File::stat;

#----------------------------------------------------------------------------

sub installSaphostagent {
        my (
            $path,          # installation path. Mandatory.
            $sapadmPasswd,  # only needed for first installation of hostagent on this machine. Optional.
            $msgHandler,    # a 'SDB::Install::BaseLegacy' object, used for info, warning and error messages. Mandatory
            $sapadmUserName,# optional, alternative user name (instead of 'sapadm') to run hostagent.
                            # must be fully qualified for windows (i.e. <domain>\<username>).
                            # only needed for first installation.
            $isProgressMsg, # AddProgressMsg is used to display start of installation/update. Optional.
            $restoreDir     # we want to avoid Cwd::getcwd, but have to chdir to the 
                            # hostagent installation kit temporarily and then restore the cwd. Mandatory.
        ) = @_;
        my $exeExt               = $isWin ? '.exe' : '';
        my $activeSaphostexecDir = getActiveSaphostexecDir();
        my $activeSaphostexec    = $activeSaphostexecDir.$path_separator.'saphostexec'.$exeExt;
        my $upgradeMode          = 0;
        if(-x $activeSaphostexec) {
            $upgradeMode = 1;
        }
        my $hostagentKit = $path.$path_separator.'global'.$path_separator.'hdb'.$path_separator.'saphostagent_setup';
        my $saphostexec = $hostagentKit.$path_separator.'saphostexec'.$exeExt;
        if(-x $saphostexec) {
            my $sapadmLinuxUserExists = ($isWin) ? 0 : sapadmUserExists($sapadmUserName);
            if( (not $upgradeMode) && (not $sapadmLinuxUserExists) && (not defined $sapadmPasswd)) {
                $msgHandler->AddMessage("Not installing SAP Host Agent (no sapadm passwd given).");
                $msgHandler->AddMessage("Run '$saphostexec -install -passwd' manually.");
                return 1;
            }
            my $exec_cfg = {};
            my $rc;
            chdir($hostagentKit);
            my $initialMsg = $msgHandler->AddMessage("Running '$saphostexec' to install/upgrade SAP Host Agent.");
            my $saphostexecVersion = getSaphostagentVersionString($hostagentKit);
            my @exec_args;
            my $msg;
            my $msgText;
            if($upgradeMode) {
                @exec_args = ('-upgrade');
                my $activeSaphostexecVersion = getSaphostagentVersionString($activeSaphostexecDir);
                my $installedSHAVersionMsg = "Detected installed $gProductNameSHA version '$activeSaphostexecVersion'";
                my $detectedSHAVersionMsg = "Detected $gProductNameSHA version '$saphostexecVersion' in installation kit";
                my $callingInstallerMsg = "Running $gProductNameSHA installer...";
                if ($isProgressMsg) {
                    $msg = $msgHandler->AddProgressMessage($installedSHAVersionMsg);
                    $msg = $msgHandler->AddProgressMessage($detectedSHAVersionMsg);
                    $msg = $msgHandler->AddProgressMessage($callingInstallerMsg);
                }
                else {
                    $msg = $msgHandler->AddMessage($installedSHAVersionMsg);
                    $msg = $msgHandler->AddMessage($detectedSHAVersionMsg);
                    $msg = $msgHandler->AddMessage($callingInstallerMsg);
                }
            }
            else {
                $msgText = "Installing SAP Host Agent version $saphostexecVersion...";
                if ($isProgressMsg) {
                    $msg = $msgHandler->AddProgressMessage($msgText);
                }
                else {
                    $msg = $msgHandler->AddMessage($msgText);
                }
                if ($sapadmLinuxUserExists) {
                    @exec_args = ('-install');
                }
                else {
                    if (!$isWin){
                        my $defaultCallback;
                        if (%{Wx::} && exists ${Wx::}{wxTheApp}) {
                            $defaultCallback = sub {Wx::Yield()};
                        }
                        $exec_cfg->{PTY_IO} = 1;
                        $exec_cfg->{callback} = sub {
                            my ($sess) = @_;
                            if (defined $defaultCallback){
                                $defaultCallback->();
                            }
                            my $read_buffer = SAPDB::Install::PipeExec::GetBufferXS ($sess);
                            if (!defined $read_buffer){
                                return;
                            }
                            if ($read_buffer =~ /enter.+password/im){
                                SAPDB::Install::PipeExec::WriteXS ($sess, $sapadmPasswd ."\n");
                                SAPDB::Install::PipeExec::CloseStdinXS ($sess);
                                SAPDB::Install::PipeExec::SetCallbackXS ($sess,$defaultCallback);
                            }
                        };
                    }
                    else{
                        $exec_cfg->{stdin_buf} = [$sapadmPasswd];
                    }
                    @exec_args = ('-install', '-passwd');
                }
                if(defined $sapadmUserName && ($sapadmUserName ne 'sapadm')) {
                    push @exec_args, ('-user', $sapadmUserName);
                }
            }
            $msg->getSubMsgLst ()->injectIntoConfigHash ($exec_cfg);
            if(!$isWin) {
                local %ENV = %ENV;
                $ENV{PATH} = '/usr/sbin:' . $ENV{PATH};
                $rc = exec_program($saphostexec, \@exec_args, $exec_cfg);
            }
            else {
                $rc = exec_program($saphostexec, \@exec_args, $exec_cfg);
            }
            if(!defined $rc || $rc != 0) {
                $msgHandler->AddError("Installation/Upgrade of SAP Host Agent FAILED", $msg->getSubMsgLst ());
                chdir($restoreDir);
                return undef;
            }
            chdir($restoreDir);
        }
        else {
            $msgHandler->AddError("Could not find Program '$saphostexec'.");
            return undef;
        }
        return 1;
}

#----------------------------------------------------------------------------

sub getActiveSaphostexecDirUx {
     return  join ($path_separator, '', qw (usr sap hostctrl exe));
}

our $programsPath;

sub getActiveSaphostexecDirWin {
    if (!defined $programsPath){
        $programsPath = getSysProgPath ();
    }
    return join ($path_separator, $programsPath, qw (SAP hostctrl exe));
}

sub getActiveSaphostexecDir;
*getActiveSaphostexecDir = $isWin ?
    \&getActiveSaphostexecDirWin :
    \&getActiveSaphostexecDirUx;

sub getSaphostexecPath {
    my $activeSaphostexecDir = getActiveSaphostexecDir();
    return undef if (! -d File::stat::stat($activeSaphostexecDir));
    my $saphostexec = File::Spec->catfile($activeSaphostexecDir, 'saphostexec');
    return (-f File::stat::stat($saphostexec)) ? $saphostexec : undef;
}

sub getSaphostagentPpmsId {
    my $saphostexecDir = getActiveSaphostexecDir();
    my $manifestFile = join($path_separator, ($saphostexecDir, "hostagent.mf"));

    return undef if(!-f $manifestFile || !open (FH, $manifestFile));

    my $ppmsId = undef;
    while(my $line = <FH>) {
        if($line =~ /^\s*compversion-id:\s*(\d+)/) {
            $ppmsId = $1;
        }
    }
    close(FH);
    return $ppmsId;
}

#----------------------------------------------------------------------------

sub getSaphostagentVersionString {
        my (
            $saphostexecDir,# directory of the saphostexec executable in question
        ) = @_;
    	my $outbuffer;
    	my $manifestFile = $saphostexecDir.$path_separator."hostagent.mf";
        my ($krel, $pnum) = (undef, undef);
        if(-f $manifestFile) {
            if(!open (FH, $manifestFile)) {
                return "UNKNOWN.UNKNOWN";
            }
            while(<FH>) {
                chomp;
                my $line = $_;
                $line =~ s/^\s(.*)\s$/$1/; #trim whitespace
                if($line !~ /^#/ and $line !~ /^$/) {
                    # not a comment and not empty:
                    my ($_krel) = ($line =~ /^hostagent release:(.*)/i);
                    my ($_pnum) = ($line =~ /^hostagent patch number:(.*)/i);
                    if(defined $_krel) {
                        trim(\$_krel);
                        $krel = $_krel;
                    }
                    if(defined $_pnum) {
                        trim(\$_pnum);
                        $pnum = $_pnum;
                    }
                }
            }
            close(FH);
        }
        else {
            return "UNKNOWN.UNKNOWN";
        }
	    if((not defined $krel) || length($krel) == 0) { # WTF Larry, why these stupid games with operator precedence ?
	        $krel = 'UNKNOWN';
	    }
	    if((not defined $pnum) || length($pnum) == 0) {
	        $pnum = 'UNKNOWN';
	    }
        return "$krel.$pnum";
}

#----------------------------------------------------------------------------
# Returns the version of the installed saphostagent
# or undef if saphostagent is not installed.
#

sub _getSHAVersionFromManifest {
    my $activeSaphostexecDir = getActiveSaphostexecDir();

    if (-d $activeSaphostexecDir) {
        return getSaphostagentVersionString($activeSaphostexecDir);
    }

    return undef;
}

#-----------------------------------------------------------------------------
# Gets the SHA version on the local host
#
# Returns 'undef' if no SHA is installed _OR_ 
# its version could not be retrieved properly _OR_
# you do not provide a $config argument in non-root case
#
sub getSHAVersion {
    my ($config) = @_;
    if (isAdmin()) {
        return _getSHAVersionAsRoot($config);
    } else {
        return _getSHAVersionAsSidadm($config);
    }
}

#-----------------------------------------------------------------------------
# Gets the SHA version from the output of 'saphostexec -version' through the
# 'CheckSHAVersion' SHA configuration
#
# Needs a Configuration object to make the 'saphostctrl' call
#
sub _getSHAVersionAsSidadm {
    my ($config) = @_;
    my $saphostexec = getSaphostexecPath();
    return _getSHAVersionFromManifest() if (!defined $config);
    my $sid = $config->getValue('SID');
    if (!defined $sid || !defined $saphostexec || !isSidadmin($sid)) {
        return _getSHAVersionFromManifest();
    }

    my $localHost = new SDB::Install::RemoteHostctrlHosts(hostname());
    $localHost->setMsgLstContext($config->getMsgLstContext());

    my $passwordKeys = ['Password'];
    my $optionsMap = {
        SID => uc($config->getSID()),
    };

    my $rc = $localHost->executeHostctrlParallel($gOperationCheckSHAVersion, $config, undef, $passwordKeys, undef, undef, undef, undef, undef, $optionsMap, undef, undef, undef, 1);
    if ($rc) {
        $config->getMsgLst()->addMessage("Could not retrieve SAP Host Agent version through 'saphostexec'. Falling back to version in the manifest file.");
        return _getSHAVersionFromManifest();
    }
    my $outputBuffer = $localHost->getOutputBuffer();
    my $hostOutput = $outputBuffer->[0];
    if (!defined $hostOutput || length($hostOutput) == 0) {
        $config->getMsgLst()->addMessage("Could not retrieve SAP Host Agent version through 'saphostexec'. Falling back to version in the manifest file.");
        return _getSHAVersionFromManifest();
    }
    my $output = [ split('\n', $hostOutput) ];
    return _parseSaphostexecVersionOutput($output);
}

#-----------------------------------------------------------------------------
# Gets the SHA version from the output of 'saphostexec -version'
#
sub _getSHAVersionAsRoot {
    my ($config) = @_;
    my $saphostexec = getSaphostexecPath();
    return undef if (! isAdmin() || !defined $saphostexec);

    my $executor = new LCM::ProcessExecutor($saphostexec, ['-version']);
    $executor->setMsgLstContext($config->getMsgLstContext()) if (defined $config);
    my $rc = $executor->executeProgram();
    if(!defined($rc) || $rc != 0) {
        if (defined $config) {
            $config->getMsgLst()->addMessage("Could not retrieve SAP Host Agent version through 'saphostexec'. Falling back to version in the manifest file.");
        }
        return _getSHAVersionFromManifest();
    }
    my $output = $executor->getOutputLines();
    return _parseSaphostexecVersionOutput($output);
}

#-----------------------------------------------------------------------------
# Takes the 'kernel release' and 'patch number' output from the
# 'saphostexec -version' call and adds missing dot in for the 'kernel release'
#
sub _parseSaphostexecVersionOutput {
    my ($outputLines) = @_;
    my $kernelRelease = (grep {$_ =~ s/^kernel release\s+//} @$outputLines)[0];
    my $patchNumber   = (grep {$_ =~ s/^patch number\s+//} @$outputLines)[0];
    substr($kernelRelease, 1, 0) = '.';
    return "$kernelRelease.$patchNumber";
}

#----------------------------------------------------------------------------
# Returns 1 if the 'sapadm' user exists, 0 if not
#

sub sapadmUserExists {
    my (
        $alternativeName  # optional, the name to use when it is not 'sapadm'.
    ) = @_;
    my $preexistingSapadmUserName = (defined $alternativeName) ? $alternativeName : 'sapadm';
    my $preexistingSapadmUser = new SDB::Install::User($preexistingSapadmUserName);
    return $preexistingSapadmUser->exists();
}


sub isSapHostAgentRunning{
    my ($msglst, $skipText, $restoreDir) = @_;

    my $exeExt               = $isWin ? '.exe' : '';
    my $activeSaphostexecDir = getActiveSaphostexecDir();
    my $activeSaphostexec    = $activeSaphostexecDir.$path_separator.'saphostexec'.$exeExt;

    if(not -x $activeSaphostexec) {
        $msglst->addWarning("No SAP Host Agent installed at '$activeSaphostexecDir': $skipText");
        return 0;
    }
    my $exec_cfg = {};
    $msglst->injectIntoConfigHash ($exec_cfg);
    my @exec_args = ('-status');

    if (!defined $restoreDir){
        require Cwd;
        $restoreDir = Cwd::getcwd();
    }

    chdir($activeSaphostexecDir);

    my $rc = exec_program($activeSaphostexec, \@exec_args, $exec_cfg);

    if (defined $restoreDir){
        chdir($restoreDir);
    }

    if(!defined $rc || $rc != 0) {
        $msglst->addWarning("SAP Host Agent not running (or status unknown): $skipText");
        return 0;
    }
    return 1;

}



#----------------------------------------------------------------------------
# Registers a system with sap hostagent. 
# 
# should work in principle, but is not tested.
# test before usage and add to the export list.
#
# note that it has the same soft error handling as unregisterInstanceService().
#

sub registerInstanceService {
    my (
            $msgHandler,    # a 'SDB::Install::BaseLegacy' object, used for info, warning and error messages. Mandatory
            $isProgressMsg, # AddProgressMsg is used to display start of installation/update. Optional.
            $restoreDir,    # we want to avoid Cwd::getcwd, but have to chdir to the 
                            # hostagent installation kit temporarily and then restore the cwd. Mandatory.
            $sid,           # the SID
            $instNumb,      # the instance number
            $saplocalhost,  # the host we (want to) run on
            $timeoutSeconds # saphostcontrol timeout, cf HostAgent documentation on asynchronous operations
                            # -1 is asynchronous (i.e. no wait), 0 is wait until response from host agent complete,
                            # > 0 is time to wait in secs, optional, defaults to -1.
    ) = @_;
    if($isWin) {
        return 1;
    }
    my $msg;
    my $msgText = "Unregistering SAP Host Agent instance service for sid '$sid', instance '$instNumb', host '$saplocalhost'";
    if ($isProgressMsg) {
        $msg = $msgHandler->AddProgressMessage($msgText);
    }
    else {
        $msg = $msgHandler->AddMessage($msgText);
    }
    my $skipText = "Skipping instance service unregistration.";
    if (!isSapHostAgentRunning ($msg->getSubMsgLst (), $skipText, $restoreDir)){
        return 1;
    }
    $msgHandler->AddMessage("Registering via SOAP over Unix Domain Socket...");
    my $hostcontrol = SDB::Install::SAPHostControl->new(); # ctor w/o params: HTTP over unix domain socket.
    my $result = $hostcontrol->RegisterInstanceService($sid, $saplocalhost, $instNumb);
    if (!$hostcontrol->getErrMsgLst()->isEmpty){
        my $msgstr = ${$hostcontrol->getErrMsgLst()->getMsgLstString()};
        $msgHandler->AddWarning("Saphostagent::registerInstanceService() failed: $msgstr");
        chdir($restoreDir);
        return 1;
    }
    my $successText = "Successfully registered SAP Host Agent instance service for sid '$sid', instance '$instNumb', host '$saplocalhost'.";
    $msgHandler->AddMessage($successText);
    chdir($restoreDir);
    return 1;
}

#----------------------------------------------------------------------------
# Unregisters a system with sap hostagent. This only refreshs some cached hostagent info more timely.
# If it fails, hostagent does the refresh as soon it detects that the system is not there anymore. 
#

sub unregisterInstanceService {
    my (
            $msgHandler,    # a 'SDB::Install::BaseLegacy' object, used for info, warning and error messages. Mandatory
            $isProgressMsg, # AddProgressMsg is used to display start of installation/update. Optional.
            $restoreDir,    # we want to avoid Cwd::getcwd, but have to chdir to the 
                            # hostagent installation kit temporarily and then restore the cwd. Mandatory.
            $sid,           # the SID
            $instNumb,      # the instance number
            $saplocalhost,  # the host we (want to) run on
            $timeoutSeconds # saphostcontrol timeout ('mTimeout'), cf HostAgent documentation on asynchronous operations
                            # The default for this value is -1 (Synchrounous)
                            # asynchronous: >= 0
    ) = @_;
    if($isWin) {
        return 1;
    }
    my $msg;
    my $msgText = "Unregistering SAP Host Agent instance service for sid '$sid', instance '$instNumb', host '$saplocalhost'";
    if ($isProgressMsg) {
        $msg = $msgHandler->AddProgressMessage($msgText);
    }
    else {
        $msg = $msgHandler->AddMessage($msgText);
    }

    my $skipText = "Skipping instance service unregistration.";

    if (!isSapHostAgentRunning ($msg->getSubMsgLst (), $skipText, $restoreDir)){
        return 1;
    }

    my $mtimeout         = (defined $timeoutSeconds) ? $timeoutSeconds : -1;
    my $operationOptions = {'mTimeout' => $mtimeout};
    $msgHandler->AddMessage("Unregistering via SOAP over Unix Domain Socket...");
    my $hostcontrol = SDB::Install::SAPHostControl->new(); # ctor w/o params: HTTP over unix domain socket.
    my $result = $hostcontrol->UnregisterInstanceService($sid, $saplocalhost, $instNumb, $operationOptions);
    if (!$hostcontrol->getErrMsgLst()->isEmpty){
        my $msgstr = ${$hostcontrol->getErrMsgLst()->getMsgLstString()};
        $msgHandler->AddWarning("Saphostagent::unregisterInstanceService() failed: $msgstr");
        chdir($restoreDir);
        return 1;
    }
    my $successText = "Successfully unregistered SAP Host Agent instance service for sid '$sid', instance '$instNumb', host '$saplocalhost'.";
    $msgHandler->AddMessage($successText);
    chdir($restoreDir);
    return 1;
}

#----------------------------------------------------------------------------

#----------------------------------------------------------------------------
# Unregisters a system with sap hostagent. This only refreshs some cached hostagent info more timely.
# If it fails, hostagent does the refresh as soon it detects that the system is not there anymore.
#

sub setDatabaseProperties {
    my (
            $msglst,          # a 'SDB::Install::MsgLst' object, used for info, warning and error messages. Mandatory
            $sid,             # the SID
            $instNumb,        # the instance number
            $dbHost,          # host name
            $systemUserPasswd,# sql system user password
            $systemUser,      # sql system user name, optional
            $sdbctrlSqlPwd,   # sapdbctrl sql user password, optional
            $sdbctrlSqlName,  # sapdbctrl sql user name, optional
            $timeoutSeconds,  # saphostcontrol timeout ('mTimeout'), cf HostAgent documentation on asynchronous operations
                              # The default for this value is -1 (Synchrounous)
                              # asynchronous: >= 0
            $checkExistingProperties
    ) = @_;
    if($isWin) {
        return 1;
    }

    $checkExistingProperties = $checkExistingProperties // 0;
    my $mTimeout             = (defined $timeoutSeconds) ? $timeoutSeconds : -1;

    my $msg = $msglst->addMessage ("Setting database properties for SAPHostControl");
    $msglst = $msg->getSubMsgLst ();

    if (!isSapHostAgentRunning ($msglst, 'Skipping setting of database properties for SAPHostControl')){
        return 1;
    }

    my $hc = new SDB::Install::SAPHostControl ();
    $hc->setMsgLstContext ([$msglst]);

    if ($checkExistingProperties){
        $hc->GetDatabaseProperties(
            {
                'Database/Name'  => $sid,
                'Database/Type'  => 'HDB'
            },
            {
                'mTimeout'       => $mTimeout
            }
         );

        my $result = $hc->getLastResultHash();
        if (defined $result && defined $result->{'Database/DBCredentials'}){
            $msglst->addMessage ("Database properties already set");
            $msglst->addMessage ("Skipping 'SAPHostControl::SetDatabaseProperty()'");
            return 1;
        }
    }

    if (!defined $sdbctrlSqlPwd){
        require SDB::Install::Configuration::AnyConfig;
        my $anyconfig = new SDB::Install::Configuration::AnyConfig ();
        my @alphabet = ('a'..'z','A'..'Z',0..9,qw /% ! . : - + _ /);
        my $pwLength = 15;
        while (1){
            $sdbctrlSqlPwd = generatePassword ($pwLength, \@alphabet);
            if (!$sdbctrlSqlPwd){
                $msglst->addError ("Cannot generate password");
                return undef;
            }
            if ($anyconfig->complyDefaultSqlPasswordPolicy ($sdbctrlSqlPwd, new SDB::Install::MsgLst (), '')){
                last;
            }
        }
    }

    if (!defined $systemUser){
        $systemUser = 'SYSTEM';
    }

    $hc->SetDatabaseProperty (
        {
            'Database/Name'          => $sid,
            'Database/Type'          => 'HDB',
            'Database/InstanceName'  => sprintf ('HDB%02d', int ($instNumb)),
            'Database/Host'          => $dbHost,
            'Database/Username'      => $systemUser,
            'Database/Password'      => $systemUserPasswd,
            'Database/PropertyName'  => 'DBCredentials',
            'Database/PropertyValue' => 'SET',
            'Password'               => $sdbctrlSqlPwd
        },
        {
            'mTimeout'               => $mTimeout
        }
    );

    if (!$hc->getErrMsgLst()->isEmpty ()){
        my $msgstr = $hc->getErrorString ();
        $msglst->addWarning("Saphostagent::SetDatabaseProperty() failed: $msgstr");
        return 1;
    }
    return 1;
}


1;
