package SDB::Install::ServerVersionLinker;

use base qw (SDB::Install::BaseLegacy);
use SDB::Install::SysVars qw ($path_separator);
use SDB::Install::System qw(isSameFile deltree makedir);
use SAPDB::Install::System::Unix qw (lchown);
use File::Basename qw (basename);

use strict;

our $old_suffix = '_old';
our $new_suffix = '_new';

sub new{
    my $self = shift->SUPER::new ();
    ($self->{_baseDir},
     $self->{_activationDir},
     $self->{_componentKey},
     $self->{_newVersionDir},
     $self->{_uid},
     $self->{_gid}) = @_;
    if ($self->{_newVersionDir}){
        $self->setNewVersionDir ($self->{_newVersionDir});
    }
    return $self;
}

##########################################################################

#
# inline subs
#

sub fullpath ($$){
    $_[0]->{_baseDir} . $path_separator . $_[1];
}

sub createsymlink ($$$){
    $_[0]->AddMessage ("Creating symlink '$_[2]' => '$_[1]'");
    if (!symlink ($_[1], $_[2])){
        $_[0]->AddError ("Cannot create Symlink '$_[2]' => '$_[1]': $!");
        return undef;
    }
    if (!$> && defined $_[0]->{_uid} && defined $_[0]->{_gid}){
        lchown ($_[0]->{_uid}, $_[0]->{_gid}, $_[2]);
    }
    return 1;
}

#############################################################################


sub getActivationDir{
    return defined $_[0]->{_activationDir} ? $_[0]->{_activationDir} : $_[0]->{_baseDir};
}

sub getActiveLink{
    return $_[0]->getActivationDir () . $path_separator . $_[0]->{_componentKey};
}

sub getActiveVersion{
    my $activeLink = $_[0]->getActiveLink ();
    if (!-l $activeLink){
        return undef;
    }
    return basename (readlink($activeLink));
}

sub getOldLink{
    return fullpath ($_[0], $_[0]->{_componentKey} . $old_suffix);
}

sub getNewLink{
    return fullpath ($_[0], $_[0]->{_componentKey} . $new_suffix);
}

sub setActivationDir{
    $_[0]->{_activationDir} = $_[1];
}

sub setBaseDir{
    $_[0]->{_baseDir} = $_[1];
}

sub setUid{
    $_[0]->{_uid} = $_[1];
}

sub setGid{
    $_[0]->{_gid} = $_[1];
}

sub setComponentKey{
    $_[0]->{_componentKey} = $_[1];
}

sub setNewVersionDir{
    $_[0]->{_newVersionDir} = basename ($_[1]);
}

sub setRelativeBaseDir{
	my ($self, $relBaseDir) = @_;
	if (-d $self->{_activationDir}){
		if (!isSameFile ($self->{_baseDir}, $self->{_activationDir} . $path_separator . $relBaseDir)){
			$self->AddError ('Wrong relative ');
			return undef;
		}
	}
	$self->{_relativeBaseDir} = $relBaseDir;
	return 1;
}

sub setSyncLinksInDir{
    $_[0]->{_syncLinksInDir} = $_[1];
}


sub prepareNewVersion{
    my ($self, $versionDir) = @_;
    if (defined $versionDir){
        $self->setNewVersionDir($versionDir);
    }
    
    my $msg = $self->AddMessage ("Preparing version switch to '$self->{_newVersionDir}'");


    my $activeVersion = $self->getActiveVersion();
    
    if (defined $activeVersion){
        $self->AddSubMessage ($msg, "Active version is '$activeVersion'.");
    }
    else {
        $self->AddSubMessage ($msg, "There is no active version.");
    }
    my $newVersionDir = fullpath ($self, $self->{_newVersionDir});

    my @stat_newVersion = stat ($newVersionDir);

    if (!@stat_newVersion){
        $self->AddError ("New version directory '$newVersionDir' doesn't exist");
        return undef;
    }

    if (isSameFile (fullpath ($self, $activeVersion), undef, \@stat_newVersion)){
        $self->AddSubMessage ($msg, "New version directory '$self->{_newVersionDir}' is already activated");
        return 1;
    }
    $self->grantUserWritePermission ($self->{_baseDir});
    my $newLink = $self->getNewLink();
    if (-l $newLink){
        if (isSameFile ($newLink, undef, \@stat_newVersion)){
            $self->AddSubMessage ($msg, "New link '$newLink' already points to the right version");
            $self->restorePermission ($self->{_baseDir});
            return 1;
        }
        my $version = basename (readlink ($newLink));
        $self->AddWarning ("'$newLink' already exists and  points to $version");
        if (!unlink ($newLink)){
            $self->AddError ("Cannot remove old link '$newLink': $!");
            $self->restorePermission ($self->{_baseDir});
            return undef;
        }
    }
    elsif (-e $newLink){
        $self->AddError ("'$newLink' is no symlink");
        $self->restorePermission ($self->{_baseDir});
        return undef;
    }
   if (!defined createsymlink ($self,
        defined $self->{_activationDir} ?
        "$self->{_baseDir}$path_separator$self->{_newVersionDir}"
        :
        $self->{_newVersionDir}, $newLink)){
        $self->restorePermission ($self->{_baseDir});
        return undef;
    }
    $self->restorePermission ($self->{_baseDir});
    return 1;
}


sub syncSymlinks{
    my ($self, $src, $dst) = @_;
    if (!defined $self->{_syncLinksInDir}){
        return 1;
    }
    $src .= $path_separator . $self->{_syncLinksInDir};
    if (!-d $src){
        return 1;
    }
    $dst .= $path_separator . $self->{_syncLinksInDir};

    if (!opendir (DH, $src)){
        $self->AddError ("Cannot open directory '$src': $!");
        return undef;
    }
    my @links = grep {-l "$src/$_" && -e "$src/$_"} readdir (DH);
    closedir (DH);
    if (!@links){
        return 1;
    }
    if (!-d $dst){
        my $cfg = {'mode' => 0555, 'uid' => $self->{_uid}, 'gid' => $self->{_gid}};
        if (!defined makedir ($dst, $cfg)){
            $self->AddError ("Cannot create directory: $!");
            return undef;
        }
    }
    my $linkdest;
    my $newLink;
    $self->grantUserWritePermission ($dst);
    foreach my $link (@links){
        $linkdest = readlink ("$src/$link");
        $newLink = "$dst$path_separator$link";
        if (-l $newLink){
            if (-e $newLink){
                if (readlink ($newLink) eq $linkdest){
                    next;
                }
            }
            unlink ($newLink);
        }
        elsif (-e $newLink){
            $self->AddError ("'$newLink' already exists and is no symlink");
            $self->restorePermission ($dst);
            return undef;
        }
        if (!defined createsymlink($self, $linkdest, $newLink)){
            $self->restorePermission ($dst);
            return undef;
        }
    }
    $self->restorePermission ($dst);
    return 1;
}

#
# save original modes in this structure:
#   $restoreModes->{$deviceId}->{$inode} = $mode
#

sub setRestoreMode{
    my ($self,$statbuf) = @_;

    if (!defined $self->{restoreModes}){
        $self->{restoreModes} = {};
    }

    if (!defined $self->{restoreModes}->{$statbuf->[0]}){
        $self->{restoreModes}->{$statbuf->[0]} = {};
    }
    $self->{restoreModes}->{$statbuf->[0]}->{$statbuf->[1]} = $statbuf->[2];
}

#
# retrieve original mode by stat buffer reference
#

sub getRestoreMode{
    my ($self,$statbuf) = @_;
    if (!defined $self->{restoreModes}){
        return undef;
    }
    if (!defined $self->{restoreModes}->{$statbuf->[0]}){
        return undef;
    }
    return $self->{restoreModes}->{$statbuf->[0]}->{$statbuf->[1]};
}


sub grantUserWritePermission{
    my ($self, $dir) = @_;
    if ($>){ # not root user (euid != 0)
        my @statbuf = stat ($dir);
        if (!@statbuf){
            return undef;
        }
        if (defined $self->getRestoreMode (\@statbuf)){ # check whether already granted
            return 1;
        }
        if ($> == $statbuf[4]){ # check owner
            if (!($statbuf[2] & 0200)){ # check owner write bit
                if (chmod ($statbuf[2] | 0200, $dir)){
                    $self->setRestoreMode (\@statbuf); # save original mode
                    return 1;
                }
                return 0;
            }
        }
    }
    return 1;
}

sub restorePermission{
    my ($self, $dir) = @_;
    if (!defined $dir){
        return undef;
    }
    my @statbuf = stat ($dir);
    if (!@statbuf){
        return undef;
    }
    my $mode = $self->getRestoreMode (\@statbuf);

    if (defined $mode){
        if (chmod ($mode, $dir)){
            delete $self->{restoreModes}->{$statbuf[0]}->{$statbuf[1]};
            return 1;
        }
        return 0;
    }
    return 1;
}


sub activateNewVersion{
    my ($self) = @_;
    my $newLink = $self->getNewLink ();
    if (-e $newLink){
        if (!-l $newLink){
            $self->AddError ("'$newLink' is no symlink");
            return undef;
        }
    }
    else{
        if (-l $newLink){
            $self->AddError ("Symlink '$newLink' is broken");
            return undef;
        }
        $self->AddMessage ("No new version to activate ($self->{_componentKey})");
        return 1;
    }

    my $newVersion = basename (readlink ($newLink));
    
    
    $self->AddMessage ("Activating new version '$newVersion'");

    my $activeLink = $self->getActivationDir () . $path_separator . $self->{_componentKey};
    my $activeVersion;
    my @statbufActive;
    if (-e $activeLink){
        @statbufActive = stat (_);
        if (-l $activeLink){
            $activeVersion = basename (readlink ($activeLink));
        }
        else{
            $self->AddError ("'$activeLink' exists and is no symlink");
            return undef;
         }
    }
    elsif (-l $activeLink){
        my $pointsTo = readlink ($activeLink);
        $self->getMsgLst()->addWarning ("Dead symbolic link detected '$activeLink' => '$pointsTo'");
        $self->grantUserWritePermission ($self->{_baseDir});
        if (unlink ($activeLink)){
               $self->getMsgLst()->addMessage ("Dead link '$activeLink' deteted");
        }
        else{
            $self->setErrorMessage ("Cannot delete dead link '$activeLink': $!");
            $self->restorePermission ($self->{_baseDir});
            return undef;
        }
        $self->restorePermission ($self->{_baseDir});
    }
    $self->grantUserWritePermission ($self->{_baseDir});
    if (isSameFile (fullpath ($self, $newVersion), undef, \@statbufActive)){
        $self->AddMessage ("New version '$newVersion' is already activated");
        if (!unlink ($newLink)){
            $self->AddError ("Cannot remove newLink '$newLink': $!");
        }
        $self->restorePermission ($self->{_baseDir});
        return 1;
    }

    if ($activeVersion){
        my $oldLink = $self->getOldLink();
        my $oldVersion;
        my @statbufOld;
        if (-e $oldLink){
             @statbufOld = stat (_);
            if (! -l $oldLink){
                $self->AddError ("'$oldLink' exists and is no symlink");
                $self->restorePermission ($self->{_baseDir});
                return undef;
            }
           $oldVersion = basename (readlink ($oldLink));
        }
        elsif (-l $oldLink){
            if (!unlink ($oldLink)){
                $self->AddError ("Cannot remove old link '$oldLink': $!");
                $self->restorePermission ($self->{_baseDir});
                return undef;
            }
        }
        my $activeEquatesOld = 0;
        if (defined $oldVersion){
            $activeEquatesOld = isSameFile (fullpath($self,$activeVersion),
               undef, \@statbufOld);
            if (!$activeEquatesOld){
                if (!unlink ($oldLink)){
                    $self->AddError ("Cannot remove old link '$oldLink': $!");
                    $self->restorePermission ($self->{_baseDir});
                    return undef;
                }
                if(!isSameFile (fullpath ($self, $self->{_newVersionDir}),
                    undef, \@statbufOld)){
                    $self->AddMessage ("Deleting old version '$oldVersion'");
                    my $msglst = new SDB::Install::MsgLst;
                    if (deltree (fullpath ($self,$oldVersion), $msglst)){
                        $self->AddError (undef, $msglst);
                    }
                }
            }
        }
        if (!$activeEquatesOld && !defined createsymlink ($self,$activeVersion, $oldLink)){
           $self->restorePermission ($self->{_baseDir});
            return undef;
        }
        $self->grantUserWritePermission ($self->{_activationDir});
        if (!unlink ($activeLink)){
            $self->AddError ("Cannot remove active link '$activeLink': $!");
        }
    }
    $self->grantUserWritePermission ($self->{_activationDir});
    if (!defined $self->syncSymlinks (fullpath($self, $activeVersion), fullpath($self, $newVersion))){
        $self->restorePermission ($self->{_activationDir});
        $self->restorePermission ($self->{_baseDir});
        return undef;
    }

    if (defined $self->{_activationDir}){
        if (!-e $self->{_activationDir}){
            my $cfg = {'mode' => 0555, 'uid' => $self->{_uid}, 'gid' => $self->{_gid}};
            if (!defined makedir ($self->{_activationDir}, $cfg)){
                $self->AddError ("Cannot create directory '$self->{_activationDir}'", $cfg);
                $self->restorePermission ($self->{_baseDir});
                return undef;
            }
            $self->grantUserWritePermission ($self->{_activationDir});
        }
    }


	my $linkDest;
	if (defined $self->{_activationDir}){
		if (defined $self->{_relativeBaseDir}){
			if (!isSameFile ($self->{_baseDir}, $self->{_activationDir} . $path_separator . $self->{_relativeBaseDir})){
				$self->AddError ("Wrong relative base dir '$self->{_relativeBaseDir}'");
                $self->restorePermission ($self->{_activationDir});
                $self->restorePermission ($self->{_baseDir});
				return undef;
			}
			$linkDest = $self->{_relativeBaseDir} . $path_separator . $newVersion;
		}
		else{
			$linkDest = $self->{_baseDir} . $path_separator . $newVersion;
		}
	}
	else{
		$linkDest = $newVersion;
	}

    if (!defined createsymlink ($self, $linkDest, $activeLink)){
        $self->restorePermission ($self->{_activationDir});
        $self->restorePermission ($self->{_baseDir});
        return undef;
    }
    if (!unlink ($newLink)){
        $self->AddError ("Cannot remove newLink '$newLink': $!");
    }
    $self->restorePermission ($self->{_activationDir});
    $self->restorePermission ($self->{_baseDir});
    return 1;
}


1;
