#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#

package SDB::Install::SysInfo;

use SDB::Install::System qw (exec_program which);
use SDB::Install::SysVars qw ($isWin $isLinux $isPowerPC);
use SDB::Install::LoadSysInfoExtension;
use SDB::Install::Tools qw (trim rtrim ltrim);
use File::Basename qw (basename);
use File::stat;
use IO::File;
use strict;

use SDB::Install::BaseLegacy;
our @ISA = qw (SDB::Install::BaseLegacy);

our $dmidecodeCmd    = '/usr/sbin/dmidecode';
our $keyManufacturer = 'Manufacturer';
our $keyProductName  = 'Product Name';

our $sysInfo;


sub new{

	my $self = shift->SUPER::new ();
	my ($no_cache) = @_;

	if (!defined $no_cache){
		$no_cache = 0;
	}

	if (!$no_cache && defined $sysInfo){
		return $sysInfo;
	}

	$sysInfo = $self;

	if ($^O =~ /aix/i){
		$self->{is_pase} = readpipe("uname")=~/OS400/i;
	}

	my $sysinfo = SAPDB::Install::SysInfo::GetSystemInfo ();

	my $hwinfo;
	my $linux_distribution;
	
	if ($isLinux){
		$hwinfo = $self->hwinfo ();
		$linux_distribution = $self->findLinuxDistribution ();
	}

	if (exists $sysinfo->{'system'}) {
		
		if ($self->{is_pase}){
			$self->{system} = 'OS400';
			$self->AddMessage ("Operating System: OS400 (PASE)");
		}
		else{
			$self->{system} = $sysinfo->{'system'};
			$self->AddMessage ("Operating System: ".$sysinfo->{'system'});
		}
	}

	if (exists $sysinfo->{'architecture'}) {
		$self->{'architecture'} = $sysinfo->{'architecture'};
		$self->AddMessage ("Architecture    : ".$sysinfo->{'architecture'});
	}

	if (exists $sysinfo->{'version'}) {
		$self->{'version'} = $sysinfo->{'version'};
		my $version = $sysinfo->{'version'};
		if ($isWin){
			if ($version =~ /NT\ 5\.0/){
				$version = '2000';
			}
			elsif ($version =~ /NT\ 5\.1/){
				$version = 'XP';
			}
			elsif ($version =~ /NT\ 5\.2/){
				$version = 'Server 2003';
			}
			elsif ($version =~ /NT\ 6\.0/){
				$version = 'Vista / Server 2008';
			}
			elsif ($version =~ /NT\ 6\.1/){
				$version = 'Windows 7 / Server 2008 R2';
			}	
		}
		
		
		$self->AddMessage ("Version         : $version");
	}

	if (exists $sysinfo->{'subversion'}) {
		$self->{'subversion'} = $sysinfo->{'subversion'};
		$self->AddMessage ("Subversion      : ".$sysinfo->{'subversion'});
	}

	if (exists $sysinfo->{'c_runtime'}) {
		$self->{'c_runtime'} = $sysinfo->{'c_runtime'};
		$self->AddMessage ("C Runtime       : ".$sysinfo->{'c_runtime'});
	}

	if (exists $sysinfo->{'cpp_runtime'}) {
		$self->{'cpp_runtime'} = $sysinfo->{'cpp_runtime'};
		$self->AddMessage ("C++ Runtime     : ".$sysinfo->{'cpp_runtime'});
	}

	if (exists $sysinfo->{'cpp_runtime64'}) {
		$self->{'cpp_runtime64'} = $sysinfo->{'cpp_runtime64'};
		$self->AddMessage ("C++ Runtime 64  : ".$sysinfo->{'cpp_runtime64'});
	}
	
	if (exists $sysinfo->{processor_features}){
		$self->{processor_features} = $sysinfo->{processor_features};
		$self->AddMessage ("Cpu features    : ".$sysinfo->{processor_features});
	}
	
	if ($isWin){
		require SAPDB::Install::System::Win32::API;
		$self->{hostname} = SAPDB::Install::System::Win32::API::GetComputerName ();
	}
	else{
		require SAPDB::Install::System::Unix;
		my $uname = SAPDB::Install::System::Unix::uname ();
		$self->{hostname} = $uname->{nodename};	
	}
	$self->AddMessage ('Host Name       : '.$self->{hostname});

	if (defined $hwinfo){
		if (defined $hwinfo->{$keyManufacturer}){
			$self->AddMessage ('Manufacturer    : ' . $hwinfo->{$keyManufacturer});
		}
		if (defined $hwinfo->{$keyProductName}){
			$self->AddMessage ('Product         : ' . $hwinfo->{$keyProductName});
		}
	}
	
	if (defined $linux_distribution){
		$self->AddMessage ('Distribution    : ' .
			$linux_distribution->{distri} . 
			(defined $linux_distribution->{revision} ? 
				" $linux_distribution->{revision}" :
				''));
	}


	return $self;
}

sub findLinuxDistribution{
	my ($self) = @_;
	if (defined $self->{linux_distribution}){
		return $self->{linux_distribution};
	}
	my $release_file;
	my $distribution = '<unknown>';
	my $release;
	my @buffer;
	my $stat;
	my $ioFile;
	foreach my $file (qw (
		os-release
		SuSE-release
		redhat-release
		sles-release
		fedora-release
		lsb-release
		gentoo-release
		debian_version
		mandrake-release
		lfs-release
		turbolinux-release
		mandriva-release
		slackware-version
		nld-release
		knoppix_version)){
	
		$release_file = '/etc/' . $file;
		$stat = File::stat::stat($release_file);
		if (defined $stat){
			$ioFile = new IO::File($release_file);
			if (!defined $ioFile){
				$self->PushError ("Cannot open file '$release_file': $!");
				next;
			}
			@buffer = grep {chomp} $ioFile->getlines();
			$ioFile->close();
			if ($file eq 'os-release'){
				my ($key,$value);
				my %hashOsRelease;
				foreach my $line (@buffer){
					($key,$value) = ($line =~ /^([^=]+)=(.*)/);
					if (!defined $key){
						next;
					}
					trim (\$key);
					if (!$key){
						next;
					}
					trim (\$value);
					$value =~ s/^"(.*)"$/$1/;
					$hashOsRelease{$key} = $value;
				}
				$distribution = $hashOsRelease{'NAME'};
				$release = $hashOsRelease{'VERSION_ID'};
			}
			elsif ($file eq 'mandrake-release'){
				$distribution = 'Mandrake';
				$release = $buffer[0];
				$release =~ s/.*release\s//;
				$release =~ s/\s*$//;
			}
			elsif ($file eq 'debian_version'){
				$distribution = 'Debian '. $buffer[0];
			}
			elsif ($file eq 'lsb-release'){
				$distribution = $buffer[0];
				$distribution =~ s/.*=\s*//;
				$release = $buffer[1];
				$release =~ s/.*=\s*//;
				
			}
			else{
				if (scalar (@buffer) == 1){
					if ($buffer[0] =~ /release/){
						($distribution,$release) = ($buffer[0] =~ /(.*)\s+release\s+(.*)/)
					}
					else{
						$distribution = $buffer[0];
					}
				}
				else{
					$distribution = $buffer[0];
					$distribution =~ s/\s+[\d\.]+\s.*//;
					$release = $buffer[1];
					$release =~ s/.*=\s*//;
					if (scalar (@buffer) == 3){
						my $patch = $buffer[2];
						$patch =~ s/.*=\s*//;
						if ($patch){
							$release .= " SP$patch"
						}
					}
				}
			}
			last;
		}
	}
	$self->{linux_distribution} = {'distri' => $distribution, 'revision' => $release};
	
	return $self->{linux_distribution};
}


our $interfaces;
our $ifprog;

#
# interfaceInfo ()
# paramater:
#               $self           SDB::Install::MsgLst
#               $nocache        scalar boolean, optional, default = false
# return value:
#               $interfaces     hash reference, undef at error
#
#   $interfaces -> {<network if 1>} -> [0] = {'addr' => <ip address>, 'type' => inet|inet6}
#                       .               .
#                       .               .
#                       .              [n] = {...}
#                       .
#                  {<network if n>} -> [0] ....
#                                       .
#                                       .
#                                      [n] ....
#

sub interfaceInfoUx{
	my ($self, $nocache) = @_;

	if (defined $interfaces && !$nocache){
		return $interfaces;
	}
	
	if (!defined $ifprog){
		my $stat;
		foreach ('/sbin', '/bin', '/usr/sbin', '/usr/bin', split (':', $ENV{PATH})){
			$stat = File::stat::stat("$_/ip");
			if (defined $stat && -x $stat){
				$ifprog = "$_/ip";
				last;
			}
		}
		if (!$ifprog){
			$self->AddError ("Tool 'ip' not found");
			return undef;
		}
	}
	my $outbuffer;
	my $cfg = {'out' => \$outbuffer};
	my $rc;
	{
		local %ENV = %ENV;
		$ENV{LANG} = 'en_US.UTF-8';
		$ENV{LC_MESSAGES} = 'en_US.UTF-8';
		$rc = exec_program ($ifprog,['addr'],$cfg);
	}
	if (defined $rc && $rc == 0){
		my @buffer = split ("\n", $outbuffer);
		my ($interface, $ifoptions, $ip);
		my @columns;
		foreach my $i (0..$#buffer){
			if ($buffer[$i] =~ /^\S/){
				($interface, $ifoptions) = ($buffer[$i] =~ /^\d+:\s+(\S+):\s+(\S.*)/);
				if (!defined $interfaces->{$interface}){
					$interfaces->{$interface} = [];
				}
				next;
			}
			$buffer[$i] =~ s/^\s*//;
			@columns = split (/\s+/,$buffer[$i]);
			if ($columns[0] eq 'inet' || $columns[0] eq 'inet6'){
                $ip = {'type' => $columns[0]};
                push @{$interfaces->{$interface}}, $ip;
                ($ip->{addr},$ip->{netbits}) = ($columns[1] =~ /(.*)\/(\d+)/);
			}
            if ($columns[0] eq 'inet'){
                $ip->{netmask} = $columns[3];
            }
		}
	}
	else{
		$self->AddError ("Cannot get interface info", $cfg);
		return undef;
	}
	return $interfaces;
}

sub interfaceInfoWin{
	my ($self, $nocache) = @_;
	
	if (defined $interfaces && !$nocache){
		return $interfaces;
	}

	if (!defined $ifprog){
		$ifprog = which ('ipconfig');
		if (!$ifprog){
			$self->AddError ('Tool ipconfig not found: PATH=' . $ENV{PATH});
			return undef;
		}
	}
	
	my $outbuffer;
	my $cfg = {'out' => \$outbuffer};
	my $rc = exec_program ($ifprog,[],$cfg);
	
	#
	# add loop back interface
	#
	
	$interfaces->{Loopback} = [{'addr' => '127.0.0.1', 'type' => 'inet'}];
	
	if (defined $rc && $rc == 0){
		my @buffer = split ("\n", $outbuffer);
		my ($interface, $type, $i, $key, $value, $ipaddr, $addr);
		my @columns;
		foreach my $i (0..$#buffer){
			if ($buffer[$i] =~ /^\S/){
				($interface) = ($buffer[$i] =~ /^(.*)/);
				if (!defined $interfaces->{$interface}){
					$interfaces->{$interface} = [];
				}
				next;
			}
			$buffer[$i] =~ s/^\s*//;
            $ipaddr = undef;
            if ($buffer[$i] =~ /^IPv6/){
                $ipaddr = {'type' => 'inet6'};
            }elsif ($buffer[$i] =~ /^IP/){
                $ipaddr = {'type' => 'inet'};
            }
            else{
                next;
            }
            ($addr) = ($buffer[$i] =~ /\.\s:\s(.*)/);
            $ipaddr->{addr} = $addr;
            push @{$interfaces->{$interface}}, $ipaddr;
        }
	}
	else{
		$self->AddError ("Cannot get interface info", $cfg);
		return undef;
	}
	return $interfaces;
}

*interfaceInfo = $isWin ? \&interfaceInfoWin : \&interfaceInfoUx;


sub hwinfo{
	my ($self) = @_;

	if (defined $self->{hwinfo}){
		return $self->{hwinfo};
	}

	if ($isPowerPC){
		if (open (FD, '/proc/cpuinfo')){
			my ($model) = grep {/^model\s+:/} <FD>;
			close (FD);
			$model =~ s/^model\s+://;
			if ($model){
				$self->{hwinfo} = {};
				trim (\$model);
				($self->{hwinfo}->{$keyManufacturer},
					$self->{hwinfo}->{$keyProductName}) = split (',', $model);
				$self->{hwinfo_out} =
					"System Information\n" .
					"	$keyManufacturer: $self->{hwinfo}->{$keyManufacturer}\n" .
					"	$keyProductName: $self->{hwinfo}->{$keyProductName}\n";
			}
		}
		return $self->{hwinfo};
	}

	if ($>){
		# skip hwinfo for non-root users
		return undef;
	}
	

	my $hwi = $dmidecodeCmd;
	my $stat = File::stat::stat($hwi);
	if (!defined $stat || ! -x $stat){
		my $toolName = basename ($hwi);
		$hwi = which ($toolName);
		if (!$hwi){
			$self->AddWarning ("$toolName tool not found");
			return undef;
		}
	}

	$self->{hwinfo_out} = '';
	
	my $cfg =   {'out' => \$self->{hwinfo_out}};
	
	my $rc = exec_program ($hwi, [], $cfg);
	
	if (!defined $rc || $rc != 0){
		$self->AddError ("Performing hwinfo failed", $cfg);
		return undef;
	}
	
	my %hwinfo;
	
	$self->{hwinfo} = \%hwinfo;
	
	my ($key,$value);
	my $section;
	my $found = 0;
	foreach my $line (split ("\n", $self->{hwinfo_out})){
		if ($found and $line =~ /^\S/){
			last;
		}
		if ($line =~ /^System Information$/){
			$found = 1;
			next;
		}
		trim (\$line);
		($key,$value) = split (':', $line);
		if (!defined $key){
			next;
		}
		rtrim (\$key);
		ltrim (\$value) if defined $value;
		if ($found){
			$hwinfo{$key} = $value;
		}
	}
	return \%hwinfo;
}

our $redHatCppCompatLib = '/opt/rh/SAP/lib64/compat-sap-c++.so';
our $redHatCppCompatLibTempl = '/opt/rh/SAP/lib64/compat-sap-c++-%d.so';
our $redHatCppCompatLibPackageName  = 'compat-sap-c++';
our $redHatCppCompatLibPackageNameTempl = 'compat-sap-c++-%d';
our $redHatLibAtomicPackageName  = 'libatomic';


# SuSE names
our $libgcc_s1_package_name  = 'libgcc_s1';
our $libstdcpp6_package_name = 'libstdc++6';
our $libatomic_package_name  = 'libatomic1';

sub _gccVersion ($){
    if (!defined $_[0]){
        return undef;
    }
    return ($_[0]->getValue ('compiler-version') =~ /^GCC\s+(\d+)/)[0];
}

sub requiresRedHatCppCompatLib{
    my ($self, $manifest) = @_;
    if ($self->isRedHat ()){
        my ($version) = (split (/\./, $self->{linux_distribution}->{revision}));
        if ($version == 6){
            return 1;
        }
        my $gccVersion = _gccVersion ($manifest);
        if (!defined $gccVersion){
            $gccVersion = 4;
        }
        if ($version == 7 && $gccVersion > 4){
            return 1;
        }
    }
    return 0;
}

our $requiredGccRuntimeVersion = {
    '4' => [4, 8],
    '5' => [5, 3 , 1],
    '6' => [6, 2],
    '7' => [7, 2 , 1]
};

sub _getRHCppCompatLibPackageName ($){
    my $gccVersion = $_[0];
    if (!defined $gccVersion || $gccVersion < 5){
        return $redHatCppCompatLibPackageName;
    }
    return sprintf ($redHatCppCompatLibPackageNameTempl, $gccVersion);
}

our $requiredGccRuntimeLibsRedHat = {
    '4' => {_getRHCppCompatLibPackageName(4) => $requiredGccRuntimeVersion->{4}},
    '5' => {_getRHCppCompatLibPackageName(5) => $requiredGccRuntimeVersion->{5}},
    '6' => {_getRHCppCompatLibPackageName(6) => $requiredGccRuntimeVersion->{6}},
    '7' => {_getRHCppCompatLibPackageName(7) => $requiredGccRuntimeVersion->{7},
            $redHatLibAtomicPackageName      => [4,8,5]}
};

our $requiredGccRuntimeLibsSLES = {
    '4' => {$libgcc_s1_package_name  => $requiredGccRuntimeVersion->{4},
            $libstdcpp6_package_name => $requiredGccRuntimeVersion->{4}},
    '5' => {$libgcc_s1_package_name  => $requiredGccRuntimeVersion->{5},
            $libstdcpp6_package_name => $requiredGccRuntimeVersion->{5}},
    '6' => {$libgcc_s1_package_name  => $requiredGccRuntimeVersion->{6},
            $libstdcpp6_package_name => $requiredGccRuntimeVersion->{6}},
    '7' => {$libgcc_s1_package_name  => $requiredGccRuntimeVersion->{7},
            $libstdcpp6_package_name => $requiredGccRuntimeVersion->{7},
            $libatomic_package_name  => $requiredGccRuntimeVersion->{7}}
};

our $gccRuntimeSapNote = {
    '5' => '2338763',
    '6' => '2455582',
    '7' => '2593824'
};

sub _requiredGccRuntimeVersion ($){
    my $versionArray = $requiredGccRuntimeVersion->{$_[0]};
    if (!defined $versionArray){
        $versionArray = [$_[0]];
    }
    return $versionArray;
}

sub _getRHCppCompatLib ($){
    my $gccVersion = $_[0];
    if (!defined $gccVersion || $gccVersion < 5){
        return $redHatCppCompatLib;
    }
    return sprintf ($redHatCppCompatLibTempl, $gccVersion);
}


sub _checkGccRuntimeLibs{
    my ($self, $requiredGccRuntimeLibs, $gccVersion) = @_;
    my $msglst = $self->getMsgLst ();
    require SDB::Install::RPM;
    my $rpm = new SDB::Install::RPM ();
    my ($package, $version);
    my $checkVersion;
    my $versionOk = 1;
    my ($currentVersion);
    my $requiredVersionString;
    my $currentVersionString;
    my $packageOk;

    foreach my $packageName (sort keys %$requiredGccRuntimeLibs){
        $checkVersion = new SDB::Install::Version(@{$requiredGccRuntimeLibs->{$packageName}});
        $package = $rpm->getPackage ($packageName);
        if (!defined $package){
            $versionOk = 0;
            $self->appendErrorMessage ("rpm package '$packageName' is not installed");
            next;
        }
        $version = $package->getVersion();
        $version =~ s/_.*$//;
        $currentVersion = new SDB::Install::Version (split ('\.', $version));
        $currentVersionString = $currentVersion->asString ();
        $packageOk = 1;

        if ($checkVersion->isNewerThan ($currentVersion)){
            $packageOk = 0;
        }

        if ($packageOk){
            $msglst->addMessage ("rpm package '$packageName' $currentVersionString is already installed");
        }
        else{
            $requiredVersionString = $checkVersion->asString ();
            $self->appendErrorMessage ("rpm package '$packageName' needs at least version $requiredVersionString. (current version = $currentVersionString)");
            $versionOk = 0;
        }
    }
    if (!$versionOk){
        my $sapNote = $gccRuntimeSapNote->{$gccVersion};
        $self->appendErrorMessage ("The operating system is not ready to perform gcc $gccVersion assemblies");
        if (defined $sapNote){
            $self->appendErrorMessage ("For more information, see SAP Note $sapNote.");
        }
        return undef;
    }
    return 1;
}




sub hasRedHatCppCompatLib{
    my ($self, $manifest) = @_;

    my $gccVersion = _gccVersion ($manifest);

    if (!defined $gccVersion){
        $gccVersion = 4;
    }

    my $libFile = _getRHCppCompatLib ($gccVersion);
    my $packageName = _getRHCppCompatLibPackageName ($gccVersion);
    my $sapNote = $gccRuntimeSapNote->{$gccVersion};

    my $statLibFile = File::stat::stat ($libFile);
    if (!defined $statLibFile || ! -f $statLibFile){
        $self->setErrorMessage ("Cannot access required library '$libFile': $!");
        $self->appendErrorMessage ("Please install the rpm package '$packageName'!");
        if (defined $sapNote){
            $self->appendErrorMessage ("For more information, see SAP Note $sapNote.");
        }
        return 0;
    }
    return $self->_checkGccRuntimeLibs($requiredGccRuntimeLibsRedHat->{$gccVersion}, $gccVersion);
}

sub getRedHatCppCompatLib{
    return _getRHCppCompatLib (_gccVersion ($_[1]));
}

sub isSles{
    return (defined $_[0]->{linux_distribution} &&
        defined $_[0]->{linux_distribution}->{distri} &&
        $_[0]->{linux_distribution}->{distri} =~ /(SUSE\s+Linux\s+Enterprise\s+Server|SLES)/i);
}


sub _getSlesVersion{
    my ($self) = @_;
    my ($release, undef, $sp) =
        ($self->{linux_distribution}->{revision} =~ /^(\d+)(\s+SP|\.)(\d+)$/);
    if (!defined $release){
        ($release) = ($self->{linux_distribution}->{revision} =~ /^(\d+)$/);
    }
    if (!defined $release){
        return [];
    }
    return [$release,$sp];
}

sub isSles11{
    my ($self) = @_;
    if (!$self->isSles ()) {
        return 0;
    }
    my ($release, $sp) = @{$self->_getSlesVersion ()};
    if (!defined $release || $release ne '11') {
        return 0;
    }
    return 1;
}

sub isRedHat{
    return (defined $_[0]->{linux_distribution} &&
        defined $_[0]->{linux_distribution}->{distri} &&
        $_[0]->{linux_distribution}->{distri} =~ /Red\s+Hat\s+Enterprise\s+Linux\s+Server/i);
}

sub isUbuntu{
    return (defined $_[0]->{linux_distribution} &&
        defined $_[0]->{linux_distribution}->{distri} &&
        $_[0]->{linux_distribution}->{distri} =~ /Ubuntu/i);
}

sub isSles12SpSupported {
    my ($self) = @_;
    if (!$self->isSles()) {
        return 1;
    }
    my ($release, $sp) = @{$self->_getSlesVersion ()};
    if(!defined $release) {
        return 1;
    }
    if($release == 12) {
        if (defined $sp && $sp >= 1) {
            return 1;
        }
    }
    elsif($release > 12) {
        return 1;
    }

    $self->setErrorMessage ("Linux version not supported: SLES $self->{linux_distribution}->{revision}");
    $self->appendErrorMessage("Minimum SLES release required: SLES 12 SP1");
    return 0;
}

sub isRhel7SpSupported {
    my ($self) = @_;
    if (!$self->isRedHat()) {
        return 1;
    }
    my ($release, $sp) =
        (split (/\./, $self->{linux_distribution}->{revision}));
    if(!defined $release) {
        return 1;
    }
    if($release >= 7) {
        if (defined $sp && $sp >= 2) {
            return 1;
        }
    }

    $self->setErrorMessage ("Linux version not supported: RHEL $self->{linux_distribution}->{revision}");
    $self->appendErrorMessage("Minimal RHEL release required: RHEL 7.2");
    return 0;
}


sub isSlesReadyForGcc{
    my ($self, $manifest) = @_;

    if (!$self->isSles ()){
        return 1;
    }

    my $gccVersion = _gccVersion ($manifest);

    if (!defined $gccVersion){
        $gccVersion = 4;
    }
    return $self->_checkGccRuntimeLibs($requiredGccRuntimeLibsSLES->{$gccVersion}, $gccVersion);
}

sub _checkGccRuntime{
    my ($self, $config, $manifest) = @_;
    if (!$isLinux){
        return 1;
    }
    my $rc = 1;
    my $msg = $config->getMsgLst ()->addMessage ("Checking whether the operating system is ready to perform gcc assemblies");
    $self->setMsgLstContext ([$msg->getSubMsgLst(), $config->getErrMsgLst ()]);
    if ($self->isSles ()){
        if (!$self->isSlesReadyForGcc ($manifest)){
            $rc = undef;
        }

    }
    elsif ($self->isRedHat ()){
        if ($self->requiresRedHatCppCompatLib ($manifest)){
            if (!$self->hasRedHatCppCompatLib ($manifest)){
                $rc = undef;
            }
        }
    }
    else{
        $msg->getSubMsgLst()->addMessage ("Skipping gcc runtime check due to unknown distribution");
    }

    if (!defined $rc && $config->getIgnore ('check_platform')){
        $config->resetError ();
        $config->AddMessage ("Ignoring error due to command line switch '--ignore=check_platform'");
        $rc = 1;
    }

    return $rc;
}

1;
