#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: 


package SDB::Install::System;

use SAPDB::Install::FSUsage qw(GetFSSizeInBlocks);
use SAPDB::Install::System::Win32::Registry;
use SAPDB::Install::System::Win32::SecurityInfo;
use SAPDB::Install::System::Win32::API;
use SDB::Install::Globals qw ($gSapsysGroupName);
use SDB::Install::SysVars;
use Socket qw(getaddrinfo getnameinfo SOCK_RAW NI_NUMERICHOST);
use strict;
use SDB::Install::Tools qw (release2num);
use SDB::Common::Utils qw(createSysAdminUserName);
use File::Basename qw (basename dirname);
use SAPDB::Install::System::Unix qw (lchown);
use File::Spec;
use File::Temp;
use Time::HiRes qw (usleep);

use SDB::Install::BaseLegacy;
use Fcntl ':mode';
use experimental qw (smartmatch);

require Exporter;

our @ISA = qw (Exporter);

our @EXPORT = qw (
    exec_program
    copy_file
    copy_tree
    copySelectedFiles
    deltree
    enableWritePermissions
    makedir
    removeEmptyDirs
    removeEmptyDirsAtExit
    isRootFs
    changeOwn
	create_user
	create_group
	normalizeRelativePath
	normalizePath
	getDirectoryFilenames
	getFilesTimestampPID
	getMountPoint
	getFileSystemInfo
	canLockProcess
	check_dacl_operator
	check_dacl_admin
	set_dacl
	set_dacl_operator
	set_dacl_admin
	isPASE
	isAdmin
	isSidadmin
	check_services
	getRealPathName
	find
	getAllUsersAppDataPath
	getHomeDir
	getSysProgPath
	isSameFile
	isSameFile2
	getFileId
	which
	nslookup
	is_local_address
	$hostname_regex
	$plain_hostname_regex
	$ipv4_regex
	access_check
	R_OK
	W_OK
	X_OK
	getODBCDrivernamesByInstallationPath
	getSystemStartMenuPath
	getSystemDesktopPath
	readLink
	createShortcuts
	removeShortcuts
	createUninstallEntry
	removeUninstallEntry
	isEmptyDir
	listDir
	removeBusyFiles
	FIND_TYPE_FILE
	FIND_TYPE_DIR
	getSysPath
	isAbsolute
	is_nfsidmap_required
	nfsidmap
	renameRecursive
	loadSSLRequiringPackage
	getSAPDrive
	isLink
);

our $plain_hostname_regex = '([a-zA-Z0-9]|[a-zA-Z0-9][a-zA-Z0-9\-]{0,61}[a-zA-Z0-9])(\.([a-zA-Z0-9]|[a-zA-Z0-9][a-zA-Z0-9\-]{0,61}[a-zA-Z0-9]))*';
our $hostname_regex = '^' . $plain_hostname_regex .'$';
our $ipv4_regex = '^(([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])\.){3}([0-9]|[1-9][0-9]|1[0-9]{2}|2[0-4][0-9]|25[0-5])$';



our $is_pase = (($^O =~ /aix/i) && (readpipe('uname') =~ /OS400/i)) ? 1 : 0;


#----------------------------------------------------------------------------

sub isPASE () {$is_pase}

#----------------------------------------------------------------------------

sub deltree{
        my ($dir, $msglist, $errlst, $skipRoot) = @_;
        my $msg;
        if (!defined $skipRoot){
                $skipRoot = 0;
        }
        if (!-e $dir){
			if (defined $msglist){
				$msglist->AddMessage ("Directory '$dir' is already removed");
			}
			return 1; 
        };
        if (!-d _ ){
			if (defined $msglist){
				$msg = $msglist->PushError ("'$dir' is no directory");
                if (defined $errlst){
                    $errlst->appendMsg ($msg);
                }
			}
			return 0; 
        }
        if (!opendir(DH,$dir)){
			if (defined $msglist){
				$msg = $msglist->PushError ("Cannot open directory '$dir': $!");
                if (defined $errlst){
                    $errlst->appendMsg ($msg);
                }
			}
			return 0;
        }
        my @content = readdir(DH);
        closedir(DH);

		my $rc = 1;


	my (@dirs,@files);
	foreach my $entry (@content){
		
		if (-d "$dir/$entry" && !-l "$dir/$entry"){
			next if ($entry =~ /^\.{1,2}$/);
			push @dirs,$entry;
		}
		else{
			push @files,$entry;
		}
	}

	#
	#  enable write permission in directory to delete entries
	#

	my $restore_permissions;
	if (!$isWin){
		if($#files > -1 || $#dirs > -1){
			my @statbuf = stat($dir);
			my $mask;
			if($> == $statbuf[4]){
				$mask = 0200;
			}
			else{
				my $found;
				foreach my $gid (split(' ',$) )){
					if($gid == $statbuf[5]){
						$found = 1;
						last;
					}
				}
				if($found){
					$mask = 020;
				}
				else{
					$mask = 002;
				}
			}
			if(($statbuf[2] & $mask) == 0){
				if (chmod($statbuf[2] | $mask,$dir)){
					$restore_permissions = $statbuf[2] & 07777;
				}
				elsif (defined $msglist){
					$msg = $msglist->PushError ("Cannot enable write permissions for \"$dir\": $!");
                    if (defined $errlst){
                        $errlst->appendMsg ($msg);
                    }
				}
			}
		}
	}

	foreach (@dirs){
		deltree("$dir/$_", $msglist, $errlst) || ($rc = 0);
	}
	foreach (@files){
		if (!unlink("$dir/$_")){
			if (defined $msglist){
				$msg = $msglist->PushError ("Cannot delete file \"$dir/$_\": $!");
                if (defined $errlst){
                    $errlst->appendMsg ($msg);
                }
			}
			$rc = 0;
		} else {
			if (defined $msglist){
				$msglist->AddMessage ("File \"$dir/$_\" deleted");
			}
		}
	}

	if ($skipRoot){
		if (defined $restore_permissions){
			chmod($restore_permissions, $dir);
		}
		return $rc;
	}

	if ($rc && !rmdir($dir)){
		my $retries = 10;
		my @entries;
		my $sysmsg = $!;
		if ($!{ENOTEMPTY}){
			while ($retries){
				opendir (DH, $dir);
				@entries = grep {!/^\.{1,2}$/} readdir (DH);
				closedir (DH);
				if (!@entries){
					last;
				}
				if ($isWin){
					$retries = 0;
				}
				else{
					$retries--;
					foreach my $entry (@entries){
						if ($entry !~ /^\.nfs/){
							$retries = 0;
							last;
						}
						if (defined $msglist){
							$msglist->AddMessage ("Waiting for nfs files to disappear: '" . join ('\', \'', @entries) . '\'');
						}
						sleep (1);
					}
				}
			}
			if (!@entries){
				if (rmdir ($dir)){
					if (defined $msglist){
						$msglist->AddMessage ("Directory \"$dir\" removed");
					}
					return 1;
				}
			}
		}
		if (defined $msglist){
			$msg = $msglist->PushError ("Cannot remove directory \"$dir\": $sysmsg");
			if (@entries){
				$msglist->AddSubMessage ($msg, "Directory still contains: '" . join ('\', \'', @entries) . '\'');
			}
            if (defined $errlst){
                $errlst->appendMsg ($msg);
            }
		}
		$rc = 0;
	}
	elsif ($rc){
		if (defined $msglist){
			$msglist->AddMessage ("Directory \"$dir\" removed");
		}
	}

	if ($rc == 0){
		if (defined $restore_permissions){
			chmod ($restore_permissions, $dir);
		}
	}
	return $rc;
}


#----------------------------------------------------------------------------

sub exec_program{
    my ($prog,$args,$config) =  @_;

    if (!defined $config){
        $config = {};
    }

    require LCM::ProcessExecutor;

    my $executor = new LCM::ProcessExecutor (
            $prog,
            $args,
            $config->{stdin_buf},
            undef,
            undef,
            $config->{UID},
            $config->{GID},
            $config->{PTY_IO},
            $config->{TIMEOUT});

    if (defined $config->{fLogSubst}){
        $executor->setLogCmdFunction ($config->{fLogSubst});
    }

    if (defined $config->{callback}){
        $executor->setCallback ($config->{callback});
    }

    if (defined $config->{msg_lst}){
        $executor->getMsgLst ()->{msg_lst} = $config->{msg_lst};
    }
    else{
        $executor->getMsgLst ()->injectIntoConfigHash ($config);
    }

    my $rc = $executor->executeProgram (1);

    if (defined $config->{out}){
        my $lines = $executor->getOutputLines ();
        if (defined $lines){
            ${$config->{out}} = join ("\n",@$lines);
        }
    }
    if (defined $config->{outLines}){
        $config->{outLines} = $executor->getOutputLines ();
    }

    if (defined $config->{TIMEOUT} && $executor->wasATimeout()){
        $config->{WAS_A_TIMEOUT} = 1;
    }

    return $rc;
}

#----------------------------------------------------------------------------


sub set_dacl_operator{

	my ($path, $si, $group_name,$msglist) = @_;
	
	if (!$isWin){
		return 0;
	}

	my ($administrators_name,$administrators_sid) = 
				LookupBuiltinAccount (SECURITY_BUILTIN_DOMAIN_RID
									,DOMAIN_ALIAS_RID_ADMINS);
	my ($users_name,$users_sid) = 
				LookupBuiltinAccount (SECURITY_BUILTIN_DOMAIN_RID
									,DOMAIN_ALIAS_RID_USERS);
	
	my ($localsystem_name,$localsystem_sid) = 
				LookupBuiltinAccount (SECURITY_LOCAL_SYSTEM_RID);

	my $regex = $group_name;
	$regex =~ s/\s/\\s/g;
	$regex = '\\\\' . $regex . '$';

	my @list = (
		{'sid' => $administrators_sid, 'mask' => FILE_ALL_ACCESS},
		{'sid' => $localsystem_sid, 'mask' => FILE_ALL_ACCESS},
		{'sid' => $users_sid, 'mask' => GENERIC_READ},
		{'trustee' => $group_name, 'trustee_regex' => $regex, 'mask' => FILE_ALL_ACCESS},
	);
	return set_dacl ($path, $si,\@list,$msglist);
}

#----------------------------------------------------------------------------

sub set_dacl_admin{
	my ($path,$si,$group_name,$msglist) = @_;
	
	if (!$isWin){
		return 0;
	}

	my ($administrators_name,$administrators_sid) = 
				LookupBuiltinAccount (SECURITY_BUILTIN_DOMAIN_RID
									,DOMAIN_ALIAS_RID_ADMINS);

	my $regex = $group_name;
	$regex =~ s/\s/\\s/g;
	$regex = '\\\\' . $regex . '$';

	my @list = (
		{'sid' => $administrators_sid, 'mask' => &FILE_ALL_ACCESS},
		{'trustee' => $group_name, 'trustee_regex' => $regex, 'mask' => &GENERIC_READ},
	);
	
	return set_dacl ($path, $si, \@list,$msglist);
}


#----------------------------------------------------------------------------

sub set_dacl{
	my ($path, $si, $list,$msglist) = @_;
	
	if (!$isWin){
		return 0;
	}

	my $rc = 1;
	
	if (!defined $si){	
		$si = new SAPDB::Install::System::Win32::SecurityInfo($path);
	}	
	
	if (defined $si){
		my ($dacl,$defaulted) = $si->GetDACL();
		if (defined $dacl && @$dacl){
			my @new_dacl;
			my $found;
			my $inheritance;
			foreach my $e (@$list){
				$found = 0;
				if (defined $e->{inheritance}){
					$inheritance = $e->{inheritance};
				}
				else{
					$inheritance = &SUB_CONTAINERS_AND_OBJECTS_INHERIT;
				}
				
				foreach my $ace (@$dacl){
					if (((defined $e->{sid} && $ace->{sid} eq $e->{sid}) || 
					    (defined $e->{trustee_regex} && ($ace->{trustee} =~ /$e->{trustee_regex}/))) && 
						($ace->{inheritance} & $inheritance)  == $inheritance && 
						($ace->{mask} & $e->{mask}) == $e->{mask}){

							$found = 1;
							last;
						
					}
				}
			
			
			
				if (!$found){
					my %ace = ('mode' => &SET_ACCESS,
								'mask' => $e->{mask},
								'inheritance' => $inheritance);
					
					if (defined $e->{sid}){
						$ace{sid} = $e->{sid};   
					}
					else{
						$ace{trustee} = $e->{trustee};
					}
					
					push @new_dacl, \%ace;
				}
			}
			if (@new_dacl){
				push @$dacl, @new_dacl;
				if (!defined $si->SetDACL ($dacl)){
					if (defined $msglist){
						$msglist->AddError("Cannot change ACL of $path: ".$si->GetLastError);
					}
					$rc = 0;
				}
				elsif (defined $msglist){
					$msglist->AddMessage("ACL of $path changed"); 
				}
			}
			elsif (defined $msglist){
				$msglist->AddMessage("ACL of $path not changed");
				$rc = 2;
			}
		}
		else{
			my @dacl = ();
			foreach my $e (@$list){
				my %ace = ('mode' => &SET_ACCESS,
					'mask' => $e->{mask},
					'inheritance' => defined $e->{inheritance} ?
								$e->{inheritance} :
								&SUB_CONTAINERS_AND_OBJECTS_INHERIT);
					
				if (defined $e->{sid}){
					$ace{sid} = $e->{sid};   
				}
				else{
					$ace{trustee} = $e->{trustee};
				}
				push @dacl, \%ace;
			}
						
			if (!defined $si->SetDACL(\@dacl)){
				if (defined $msglist){
						$msglist->AddError("Cannot apply ACL for $path: ".$si->GetLastError);
				}		
				$rc = 0;
			}
			elsif (defined $msglist){
				$msglist->AddMessage("ACL of $path applied successfully"); 		
			}
		}
	}
	else{
		if (defined $msglist){
						$msglist->AddError("Cannot apply ACL for $path: can\'t get SecurityInfo :" . 
						SAPDB::Install::System::Win32::SecurityInfo::GetLastError());
		}				
		$rc = 0;
	}
	
	if ($rc){
		my ($acl,$defaulted) = $si->GetDACL();
		if (defined $acl){
			foreach my $ace (@$acl){
					my $account = $ace->{trustee};
					$account = $ace->{sid} unless $account =~ /\S/; 
					next unless $account =~ /\S/; 
					if (defined $msglist){
						$msglist->AddMessage("$account has rights ".
							$si->GetRightsString($ace->{mask})." (".sprintf("0x%x",$ace->{mask}).")");
					}
			}
		}
	}
	return $rc;
	
}
#----------------------------------------------------------------------------


our ($R_OK, $W_OK, $X_OK) = (4,2,1);

sub R_OK {$R_OK};

sub W_OK {$W_OK};

sub X_OK {$X_OK};

our @access_string = ('execute permission','write permission','read permission');

sub access_check{
    my ($path,$wanted,$uid,$gid, $msglst, $no_normalization) =  @_;
   my @stat;
    my $rc = 1;
    my $and_mask = 007;
    my $and_wanted = $wanted;
    my $level = 'others';
    my $shift = 0;
    if ($wanted){
        @stat = stat ($path);
        if (defined $uid && $stat[4] == $uid){
            $shift = 6;
            $level = 'owner';
        }
        elsif (defined $gid && $stat[5] == $gid){
            $shift = 3;
            $level = 'group';
        }

        if ($shift){
            $and_mask <<= $shift;
            $and_wanted <<= $shift;
        }
        if (($stat[2] & $and_wanted) != $and_wanted){
            my $missing = ((~$stat[2]) & ($and_wanted) & $and_mask) >> $shift;
            my $check = 1;
            foreach my $i (0..2){
                if ($check & $missing){
                    $msglst->PushError ("Missing $access_string[$i] for $level at $path")
                }
                $check <<= 1;
            }
            $rc = 0;
        }
    }

    if (!$no_normalization){
        my ($npath,$rc, $msg)  = normalizeRelativePath ('', $path);
        if ($rc){
            $path = $npath;
        }else{
            $msglst->AddError ("Cannot normalize path: $msg");
        }
    }
    #
    # checking r and x bits of parent directories
    #
    my ($directory,$basename) = ($path =~ /(.*)\/([^\/]+)$/);
    if ($directory){
        $rc = access_check ($directory,$R_OK | $X_OK, $uid, $gid, $msglst,1) && $rc;
    }
    return $rc;
}

#-------------------------------------------------------------------------------

#
# linux constants will probably not work on other oss (e.g. BSD)
#

our $nobodyUid      = 0xfffe;     # -2 short
our $nogroupGid     = 0xfffe;     # -2 short
our $nobodyGid      = 0xfffd;     # -3 short
our $nobodyLongUid  = 0xfffffffe; # -2 long
our $nogroupLongGid = 0xfffffffe; # -2 long
our $nobodyLongGid  = 0xfffffffd; # -3 long

sub ownerIsNobody ($) {
    my ($ownerName) = getpwuid($_[0]);
    return ($ownerName eq 'nobody');
}

sub groupIsNobody ($) {
	my ($groupName) = getgrgid($_[0]);
    return ($groupName eq 'nobody' || $groupName eq 'nogroup');
}

#-------------------------------------------------------------------------------
# Changes the owner/group ID of files, directories and links under
# the specified path.
#
# If an old ID filter is defined, the new ID is set also to files/directories
# belinging to 'nobody'.
#
# Returns int retCode

sub changeOwn{

    my ($oldUidFilter,       # user ID is set only for those files/directories where user ID matches $oldUidFilter
        $newUid,             # new user ID
        $oldGidFilter,       # group ID is set only for those files/directories where group ID matches $oldGidFilter
        $newGid,             # new group ID
        $path,
        $errlst,             # SDB::Install::MsgLst provided for errors
        $infoLst,            # SDB::Install::MsgLst provided for additional info
        $skipPathMap,        # skip directories contained in this hash
        $changeRootIDs,      # always change root user/group ID
        $neverChangeRootIds, # never  change root user/group ID
        $changeUidAndGid,    # change both UID & GID when match on UID occurs
       ) = @_;

    my $filterUIDs = undef;
    my $filterGIDs = undef;

    $errlst = $errlst // SDB::Install::MsgLst->new();
    $infoLst = $infoLst // SDB::Install::MsgLst->new();

    if (defined $oldUidFilter) {
        $filterUIDs = {$oldUidFilter => 1};
        $filterUIDs->{0} = 1 if $changeRootIDs;
        $filterUIDs->{$nobodyUid} = 1;
        $filterUIDs->{$nobodyLongUid} = 1;

    }

    if (defined $oldGidFilter) {
        $filterGIDs = {$oldGidFilter => 1};
        $filterGIDs->{0} = 1 if $changeRootIDs;
        $filterGIDs->{$nobodyGid} = 1;
        $filterGIDs->{$nogroupGid} = 1;
        $filterGIDs->{$nobodyLongGid} = 1;
        $filterGIDs->{$nogroupLongGid} = 1;
    }

    return _changeOwn ($filterUIDs,
                       $newUid,
                       $filterGIDs,
                       $newGid,
                       $path,
                       $errlst,
                       $infoLst,
                       $skipPathMap,
                       $neverChangeRootIds,
                       $changeUidAndGid);
}


#-------------------------------------------------------------------------------
# Internal subroutine that changes the owner IDs and group ID.
#
# Returns int retCode

sub _changeOwn {

    my ($filterUIDs,         # hash map containing UIDs
        $newUid,             # new user ID
        $filterGIDs,         # hash map containing GIDs
        $newGid,             # new group ID
        $path,
        $errlst,             # SDB::Install::MsgLst provided for errors
        $infoLst,            # SDB::Install::MsgLst provided for additional info
        $skipPathMap,        # skip directories contained in this hash
        $neverChangeRootIds, # never  change root user/group ID
        $changeUidAndGid,    # change both UID & GID when match on UID occurs
       ) = @_;

    my $rc = 1;

    if (-l $path){

        if (defined $skipPathMap) {

            my $linkName = basename ($path);

            if (exists $skipPathMap->{$linkName}) {
                $infoLst->addMessage ("Skipping link '$path'");
                return 1;
            }
        }

        lchown ($newUid, $newGid, $path);
        return 1; # do not follow links
    }

    my @statbuf = stat ($path);

    if (-d _){

        if (defined $skipPathMap) {

            my $directoryName = basename ($path);

            if (exists $skipPathMap->{$directoryName}) {
                $infoLst->addMessage ("Skipping directory '$path'");
                return 1;
            }
        }

        if (!opendir (DH, $path)){
            $errlst->PushError ("Cannot open directory '$path'");
            return undef;
        }
        my @entries = grep {!/^\.{1,2}$/} readdir (DH);
        closedir (DH);
        foreach my $entry (@entries){

            if (!defined _changeOwn ($filterUIDs,
                                     $newUid,
                                     $filterGIDs,
                                     $newGid,
                                     $path .$path_separator . $entry,
                                     $errlst,
                                     $infoLst,
                                     $skipPathMap,
                                     $neverChangeRootIds,
                                     $changeUidAndGid)) {
                $rc = undef;
            }
        }
    }
    my $uid = -1;
    my $gid = -1;

    if ($statbuf[4] != $newUid
            && !($neverChangeRootIds && ($statbuf[4] == 0))
            && (!defined $filterUIDs || $filterUIDs->{$statbuf[4]})) {
        $uid = $newUid;

        if($changeUidAndGid){
            $gid = $newGid;
        }
    }

    if ($statbuf[5] != $newGid
            && !($neverChangeRootIds && ($statbuf[5] == 0))
            && (!defined $filterGIDs || $filterGIDs->{$statbuf[5]})) {
        $gid = $newGid;
    }

    if ($uid != -1 || $gid != -1){
        if (!chown ($uid,$gid,$path)){
            $errlst->PushError ("Cannot change owner of '$path': $!");
            return undef;
        }
    }

    return $rc
}


#----------------------------------------------------------------------------

sub check_dacl_operator{

	my ($path,$group_name,$msglist) = @_;
	
	if (!$isWin){
		return 0;
	}

	my ($administrators_name,$administrators_sid) = 
				LookupBuiltinAccount (SECURITY_BUILTIN_DOMAIN_RID
									,DOMAIN_ALIAS_RID_ADMINS);
	my ($users_name,$users_sid) = 
				LookupBuiltinAccount (SECURITY_BUILTIN_DOMAIN_RID
									,DOMAIN_ALIAS_RID_USERS);
	
	my ($localsystem_name,$localsystem_sid) = 
				LookupBuiltinAccount (SECURITY_LOCAL_SYSTEM_RID);

	my $regex = $group_name;
	$regex =~ s/\s/\\s/g;
	$regex = '\\\\' . $regex . '$';

	my @list = (
		{'sid' => $administrators_sid, 'mask' => FILE_ALL_ACCESS, 'trustee' => $administrators_name},
		{'sid' => $localsystem_sid, 'mask' => FILE_ALL_ACCESS, 'trustee' => $localsystem_name},
		{'sid' => $users_sid, 'mask' => GENERIC_READ, 'trustee' => $users_name},
		{'trustee' => $group_name, 'trustee_regex' => $regex, 'mask' => FILE_ALL_ACCESS},
	);
	return check_dacl ($path,\@list,$msglist);
}


#----------------------------------------------------------------------------

sub check_dacl_admin{
	my ($path,$group_name,$msglist) = @_;
	
	if (!$isWin){
		return 0;
	}

	my ($administrators_name,$administrators_sid) = 
				LookupBuiltinAccount (SECURITY_BUILTIN_DOMAIN_RID
									,DOMAIN_ALIAS_RID_ADMINS);

	my $regex = $group_name;
	$regex =~ s/\s/\\s/g;
	$regex = '\\\\' . $regex . '$';

	my @list = (
		{'sid' => $administrators_sid, 'mask' => &FILE_ALL_ACCESS, 'trustee' => $administrators_name},
		{'trustee' => $group_name, 'trustee_regex' => $regex, 'mask' => &GENERIC_READ},
	);
	
	return check_dacl ($path,\@list,$msglist);
}


#----------------------------------------------------------------------------

sub check_dacl{
	my ($path, $list,$msglist) = @_;
	
	if (!$isWin){
		return 0;
	}

	my $rc = 1;
	
	my $si = new SAPDB::Install::System::Win32::SecurityInfo($path);
		
	if (defined $si){
		my ($dacl,$defaulted) = $si->GetDACL();
		if (defined $dacl && @$dacl){
			my @new_dacl;
			foreach my $e (@$list){
				my $found = 0;		
				foreach my $ace (@$dacl){
					if (((defined $e->{sid} && $ace->{sid} eq $e->{sid}) || 
					    (defined $e->{trustee_regex} && ($ace->{trustee} =~ /$e->{trustee_regex}/))) && 
						($ace->{inheritance} & &SUB_CONTAINERS_AND_OBJECTS_INHERIT)  == &SUB_CONTAINERS_AND_OBJECTS_INHERIT && 
						($ace->{mask} & $e->{mask}) == $e->{mask}){

							$found = 1;
							last;
						
					}
				}
					
				if (!$found){
					if (defined $msglist){
						$msglist->PushError ("ACL entry $e->{trustee} with rights " . $si->GetRightsString($e->{mask}) . ' not found');					
					}
					$rc = 0;
				}	
			}
		}
		else{
			if (defined $msglist){
				$msglist->AddError("No ACL found");
			}		
			$rc = 0;
		}
	}
	else{
		if (defined $msglist){
						$msglist->AddError("Can\'t get SecurityInfo :" . 
						SAPDB::Install::System::Win32::SecurityInfo::GetLastError());
		}				
		$rc = 0;
	}
	return $rc;
}


#----------------------------------------------------------------------------


sub check_services{
	my ($services,$msglst) = @_;
	my @unknown_services;
	foreach my $service (keys %$services){
		my $num = getservbyname ($service,'tcp');
		if (defined $num){
			$msglst->AddMessage ("Service \"$service\" already known as $num/tcp");
		}
		else{
			push @unknown_services, $service;
		}
	}
	
	if (@unknown_services){
		my $file;
		
		if ($isWin){
			my $submsglst = new SDB::Install::MsgLst ();
			my $hash = readValues(0,"SYSTEM\\CurrentControlSet\\Control\\Windows", undef, $submsglst);
			if (!defined $hash || !defined $hash->{SystemDirectory}){
				$msglst->AddError ("Cannot find systemdirectory in windows registry", $submsglst);
				return undef;
			}
			$file = "$hash->{SystemDirectory}\\drivers\\etc\\services";
		}
		elsif (!isPASE ()){
			$file = '/etc/services';
		}
		else{
			
			#
			# PASE
			#
			
			my $cmd = 'system';
			my $msg;
			my $args;
			my $config;
			my $rc;
			my $errstate = 0;
			
			foreach my $service (@unknown_services){
				$msg = $msglst->AddMessage ("[PASE] Append service \"$service\" as \"$services->{$service}/tcp\" to service table.");
				$args = ["ADDSRVTBLE SERVICE('$service') PORT($services->{$service}) PROTOCOL('tcp')"];
				$config = {};
				
				$rc = exec_program ($cmd,$args,$config);
				
				$msglst->AddSubMsgLst ($msg,$config);
				
				if ($rc != 0){
					$msglst->PushError ("[PASE] Cannot add  \"$service\" as \"$services->{$service}/tcp\" to service table.", $config);
					$errstate = 1;
					next;
				}
				
				# check again
			
				if (! getservbyname ($service, 'tcp')) {
					$msglst->PushError ("[PASE] Unable to append service \"$service\" as \"$services->{$service}/tcp\": $!");
					$errstate = 1;
				}
				
			}
						
			if ($errstate){
				return undef;
			}
			
			return 1;
		}
		
		if (!open (SRV,">>$file")){
			$msglst->AddError ("Cannot open \"$file\" to write: $!");
			return undef;	
		}
		
		foreach my $service (@unknown_services){
			print SRV "$service\t\t$services->{$service}/tcp\n";
			$msglst->AddMessage ("Append \"$service\" to \"$file\" as \"$services->{$service}/tcp\"");	
		}
		
		close(SRV);
	}
	return 1;
}


#----------------------------------------------------------------------------
sub copy_tree{
	my ($src_dir, $dest_dir, $msglst, $src_statbuf, $followSymlinks) = @_;
	$followSymlinks //= 1;
	my @statbuf;
	
	if (!defined $src_statbuf){
		@statbuf = stat ($src_dir);
		
		if (!@statbuf || !($statbuf[2] & 040000)){
			$msglst->AddError ("$src_dir: $!\n");
			return 0;
		}
		$src_statbuf = [@statbuf];
	}
	
	my $config = {};
	
	@statbuf = stat ($dest_dir);
		
	if (@statbuf && $statbuf[2] & 040000){
		$msglst->AddMessage ("Directory \"$dest_dir\" already exists");
	}
	else{
		$config->{mode} = $src_statbuf->[2] & 07777;
		if (!$isWin && ($> != $src_statbuf->[4] || $) != $src_statbuf->[5])){
			$config->{uid} = $src_statbuf->[4];
			$config->{gid} = $src_statbuf->[5];
		}
		if (!makedir ($dest_dir,$config)){
			$msglst->AddError ("Cannot create directory \"$dest_dir\"", $config);
			return 0;
		}
		$msglst->AddMessage ("Directory \"$dest_dir\" created");
	}
	
	if (!opendir (DH,$src_dir)){
			$msglst->AddError ('Cannot open directory \'' . $src_dir . '\': ' . $!);
			return 0;
		
	}
	
	my @entries = readdir (DH);
	closedir (DH);
	
		
	my ($file, $destfile);
	foreach my $entry (@entries){
		next if $entry =~ /^\.{1,2}$/;
		$file = "$src_dir$path_separator$entry";
		$destfile = "$dest_dir$path_separator$entry";
        if (isLink($file) && !$followSymlinks) {
            @statbuf = lstat($file);
        } else {
            @statbuf = stat($file);
        }
		if (!@statbuf){
			$msglst->AddError ("stat('$file') failed: $!");
			return 0;
		}
		if (!($statbuf[2] & 040000)){
			$config = {'binmode' => 1, 'chown' => 1, 'stat_buf' => \@statbuf};
			if (!defined copy_file ($file,$destfile, $config)){
				$msglst->AddError ('Cannot copy file' , $config);
				return 0;
			}
			$msglst->AddMessage ("Copying file \"$file\"", $config);
		}
		else {
			my $msg_list = new SDB::Install::MsgLst ();
			if (!copy_tree ($file,$destfile, $msg_list, \@statbuf, $followSymlinks)){
				$msglst->AddError ("Cannot copy directory \"$file\"",$msg_list);
				return 0;
			}
			$msglst->AddMessage ("Copying directory \"$file\"", $msg_list);			
		}	 
	}
	return 1;		
}

#----------------------------------------------------------------------------
sub enableWritePermissions{
    my ($path, $statbuf, $msglst) = @_;

    if (!$isWin && !$isApple && isAdmin()){
        # root can always write but on macosx and windows
        return undef;
    }

    if (!defined $msglst){
        $msglst = new SDB::Install::MsgLst ();
    }

    if (!defined $statbuf){
        $statbuf = [stat ($path)];
        if (!@$statbuf){
            $msglst->addError ("Cannot get mode of file '$path': $!");
            return undef;
        }
    }

    if (!$isWin && $> != $statbuf->[4]){
        # not owner
        $msglst->addError("Not owner of '$path'");
        return undef;
    }
    if ($statbuf->[2] & 0200){
        # owner has already write permissions
        $msglst->addMessage("Owner has already write permissions on '$path'");
        return undef;
    }

    if (!chmod (($statbuf->[2] & 07777) | 0200, $path)){
        $msglst->addError ("Cannot change mode of file '$path': $!");
        return undef;
    }
    $msglst->addMessage ("Mode of file '$path' changed.");
    return ($statbuf->[2] & 07777);
}


#----------------------------------------------------------------------------
sub copy_file{
	my ($file,$destination,$config,$keepSrcMode) = @_;
	if (!defined $keepSrcMode) {
		# default: transfer permissions of source file to destination file
		$keepSrcMode = 1;
	}
	my @stat_buf;
	if (defined $config->{stat_buf}){
		@stat_buf = @{$config->{stat_buf}};
	}
	else{
		@stat_buf = stat ($file);
		unless (@stat_buf){
			AddError ($config, 'Stat '.$file.': '.$!)
				if defined $config;
			return undef;
		}
	}
	if ($stat_buf[2] & 040000){
			AddError ($config, $file.' is not a regular file')
				if defined $config;
			return undef;
	}

	my @stat_buf_destination =  stat ($destination);
	
	my $filename = basename ($file);
	my $parentname = dirname ($file);

	my $dest_directory_missing = 1;	
	my $dest_directory;
	my $restore_parent_mode;
	if (@stat_buf_destination){
		if ($stat_buf_destination[2] & 040000){
			$destination .= "/$filename";
		}
		$dest_directory_missing = 0;
	}
	else{
		$dest_directory = dirname ($destination);
		if (-d $dest_directory){
			$dest_directory_missing = 0;
			$restore_parent_mode = enableWritePermissions ($dest_directory, [stat(_)]);
		}
	}
	
	if ($dest_directory_missing){
		if ($config->{createdir}){
			my $mkdir_config = {'uid' => $config->{uid},
				'gid' => $config->{gid}};
			if (defined $config->{dir_mode}){
				$mkdir_config->{mode} = $config->{dir_mode};
			}
			unless (defined makedir ($dest_directory,$mkdir_config)){
				if (defined $config){
					AddError($config, 'Cannot create directory '.$dest_directory, $mkdir_config);
				}	
				return undef;	
			}
		}
		else{
			AddError ($config, 'Destination directory '.$dest_directory.' not found: '.$!)
				if defined $config;
			return undef;
		}	
	
	}
	
	unless (open (SRC,$file)){
		AddError ($config, 'Cannot open file '. $file .':' . $!)
			if defined $config;
		if (defined $restore_parent_mode){
			if (!chmod ($restore_parent_mode, $dest_directory)){
				AddError ($config, "Cannot restore mode of parent directory '': $!")
					if defined $config;
			}
		}
		return undef;
	}
	
	binmode (SRC);	
	my $mode;
	if (-e $destination){
		$mode = enableWritePermissions ($destination, [stat (_)]);
	}
	my $rc = open(DST,">$destination");

	if (defined $restore_parent_mode){
		if (!chmod ($restore_parent_mode, $dest_directory)){
			AddError ($config, "Cannot restore mode of parent directory '': $!")
				if defined $config;
		}
	}

	if (!$rc){
		AddError ($config, 'Cannot create file '. $destination .':' . $!)
			if defined $config;
		close (SRC);
		if (defined $mode){
			chmod($mode, $destination);
		}
		return undef;
	}

	binmode (DST);	

	$config->{'binmode'} && binmode(DST);
	my $bufSize= $stat_buf [11] || 32768;
	my $buf;
	my $offset=0;
	my $written;
	while(my $len = sysread(SRC,$buf,$bufSize)){
		unless(defined $len){
			next if $! =~ /^Interrupted/;
			AddError ($config, 'Read failure: ' . $!)
				if defined $config;
			if (defined $mode){
				chmod($mode, $destination);
			}
			return undef;
		
		} 
		$offset=0;
		while($len){
			$written=syswrite(DST,$buf,$len,$offset);
			unless (defined $written){
				AddError ($config, 'Write failure: ' . $!)
					if defined $config;
				close(SRC);
				close(DST);
				if (defined $mode){
					chmod($mode, $destination);
				}
				return undef;
			}
			$len-=$written;
			$offset+=$written;
		}	
	}
	close(SRC);
	close(DST);
	my $atime = $stat_buf[8] <= 0 ? undef : $stat_buf[8];
	unless (utime ($atime,$stat_buf[9],$destination)){
		AddError ($config, "Cannot set time of copied file: $!\n" .
			"atime = $atime ($stat_buf[8]), mtime = $stat_buf[8], time = " . time ())
					if defined $config;
		if (defined $mode){
			chmod($mode, $destination);
		}
		return undef;
	}

	if (defined $mode){
		chmod($mode, $destination);
	}
	
	if(!$isWin){
		$stat_buf[2] &= 06777;
		if ($config->{'chown'} and $> != 0 and $> != $stat_buf[4]){
				unless (($stat_buf[2] & 06000) == 0){
					AddWarning ($config, 'WRN', 'Cannot set sbits of copied file: ' . $!)
						if defined $config;
					$stat_buf[2] &= 0777;
				}
		}
		
		if($keepSrcMode && (($stat_buf[2] & 06000) == 0)){
			unless (chmod ($stat_buf[2],$destination)){
				AddError ($config, 'Cannot change mode of copied file: ' . $!)
					if defined $config;
				return undef;
			}
		}
		if (!$> && $config->{'chown'}){
			unless (chown($stat_buf[4],$stat_buf[5],$destination)){
				AddError ($config, 'Cannot set uid/gid of copied file: ' . $!)
					if defined $config;
				if (!$config->{ignoreChownError}){
					return undef;
				}
			}
		}
		
		if (!$> && (defined $config->{uid} || defined $config->{gid})){
			my $uid = defined $config->{uid} ?  $config->{uid} : -1;
			my $gid = defined $config->{gid} ?  $config->{gid} : -1;
			unless (chown($uid,$gid,$destination)){
				AddError ($config, 'Cannot set uid/gid of copied file: ' . $!)
					if defined $config;
				if (!$config->{ignoreChownError}){
					return undef;
				}
			}
		}	
		
		unless(($stat_buf[2] & 06000) == 0 ){
			unless (chmod ($stat_buf[2],$destination)){
				AddError ($config, 'Cannot set sbits of copied file: ' . $!)
					if defined $config;
				return undef;
			}
		}
	}
	return 1;
}

#----------------------------------------------------------------------------

sub makedir{
	my ($dir,$config) = @_;
	
	unless ($dir =~ /\S/){
		AddError ($config, 'invalid path name') if defined $config;
		return undef;
	}
	
	#
	# checking root fs
	#
	
	if (isRootFs ($dir)){
		return 1;
	}

	#
	# remove trailing slashes / backslashes
	#
	
	$dir =~ s/[\\\/]*$//;
	
	if (-f $dir){
		AddError ($config, $dir . ' is a regular file')
				if defined $config;
		return undef;
	}
	
	unless(-d $dir){
		my ($parent) = ($dir =~ /(.*)[\/\\]+[^\/\\]+$/);
		if ($parent !~ /\S/ and $dir =~ /^[\/\\]/){
			$parent = '/';
		}
		unless (defined makedir($parent,$config)){
			return undef;
		}
		my $parent_mode = enableWritePermissions ($parent);
		
		if(mkdir ($dir)){
			if (defined $config && exists $config->{firstCreated} &&
				!defined $config->{firstCreated}){
				$config->{firstCreated} = $dir;
			}
			if (defined $config && defined $config->{listCreated}){
				push @{$config->{listCreated}}, $dir;
			}
			if (defined $parent_mode){
					chmod ($parent_mode, $parent);
			}
			if (!$isWin){
				if (defined $config){
					if (defined $config->{mode}){
						unless (chmod ($config->{mode}, $dir)){
							unless (defined makedir($parent,$config)){
								AddError ($config, sprintf ('cannot change mode of directory %s to 0%o: %s',$dir,$config->{mode},$!))
									if defined $config;
								return undef;
							}				
						}
					}
					
					if (!$> && defined $config->{uid}){
						if (!chown ($config->{uid},$config->{gid},$dir)){
								AddError ($config, sprintf ('cannot change owner of directory %s to uid %d and gid %d: %s',
															$dir,$config->{uid},$config->{gid}, $!))
									if defined $config;
								return undef;
						}
					}
				}	
			}
		}
		else{
			AddError ($config, 'Cannot create directory '.$dir.': '.$!)
				if defined $config;
			if (defined $parent_mode){
				chmod ($parent_mode, $parent);
			}
			return undef;
		}
	}
    return 1;
}

#----------------------------------------------------------------------------

sub removeEmptyDirs{
	my ($dir,$config)=@_;
	my @not_removed;
	my @removed;
	unless (defined $config){
		$config = {};
	}
	
	unless (-d $dir){
		AddError ($config, "Cannot remove directory\"$dir\": $!");
		return 0;
	}
	
	if (!$isWin && defined $config && 
 		(exists $config->{'uid'} || exists $config->{'uid'})){
 			
 		my @statbuf = stat(_);
 		if (exists $config->{'uid'}){
 			if ($statbuf[4] != $config->{'uid'}){
 				return (1,[],[$dir]);
 			}
 		}
 		if (exists $config->{'gid'}){
 			if ($statbuf[5] != $config->{'gid'}){
 				return (1,[],[$dir]);
 			}
 		}
 	}

	if (defined $config && exists $config->{filter_inodes}){
		my @statbuf = stat (_);
		if (exists $config->{filter_inodes}->{$statbuf[0]}){
			foreach my $inode (@{$config->{filter_inodes}->{$statbuf[0]}}){
				if ($inode == $statbuf[1]){
					AddMessage ($config ,"Skipping directory \"$dir\" due to inode filter", undef, undef, undef);
					return (1,[],[$dir]);
				}
			}
		}
	}


	unless (opendir(DH,$dir)){
		AddError ($config, "Cannot open directory \"$dir\": $!");
		return (0,[],[$dir]);
	}
	my @content=readdir(DH);
	closedir(DH);
	
	unless (@content){
		AddError ($config, "Cannot read directory \"$dir\": $!");
		return (0,[],[$dir]);
	}
	
	my $isEmpty = 1;
	my $aPath;
	if ($#content > 1){
		foreach my $element (@content){
			if(($element ne '.') and ($element ne '..')){
				$aPath = "$dir$path_separator$element";
				if(-d $aPath){
					my ($rc,$ref_removed,$ref_unremoved) = 	removeEmptyDirs($aPath,$config); 
					defined $ref_unremoved and push @not_removed,@$ref_unremoved;
					defined $ref_removed and push @removed,@$ref_removed;
					if (!$rc){
						$isEmpty &&= 0;
					}
				}
				else{
					$isEmpty &&= 0;
				}
			}		
		}
	}

	if ($isEmpty){
		if (!rmdir($dir)){
			if (!$isWin && $> && $!{'EACCES'}){
				my $parent = dirname ($dir);
				my $restore_mode = enableWritePermissions ($parent);
				if (defined $restore_mode){
					my @rc = removeEmptyDirs ($dir, $config);
					chmod ($restore_mode, $parent);
					return @rc;
				}
			}
			if (-l $dir){
				if (!unlink ($dir)){
					AddError ($config, "Cannot remove symbolic link \"$dir\": $!");
					return (0,\@removed,[@not_removed,$dir]);
				}
			}else{
				AddError ($config, "Cannot remove directory \"$dir\": $!");
				return (0,\@removed,[@not_removed,$dir]);
			}
		}
		push @removed,$dir;
		AddMessage ($config ,"Directory \"$dir\" removed", undef, undef, undef);
		return (1,\@removed,\@not_removed);
	}
	return (0,\@removed,\@not_removed);
}	


#----------------------------------------------------------------------------

#
# get rid of empty directories due to ephemeral .nfs* files caused by deleted
# busy installer files on a nfs mount
#
# use exec() to free all installer objects => .nfs* files are released
#
# Use this function at the end of your program, it will never return!
#

our $removeEmptyDirsAtExitScriptTemplate = << '__END_SCRIPT';
    # generated by SAP HANA installer
    sleep 1
    find "%s" -type d -empty -delete 2>/dev/null
    rm -rf %s
    exit %d
__END_SCRIPT

sub removeEmptyDirsAtExit{
    my ($rootPath, $rc) = @_;
    umask (077);
    my $tmpDir = File::Temp::newdir();
    if (!$tmpDir){
        print STDERR "Cannot get temporary directory: $!\n";
        exit $rc;
    }
    my $script = sprintf ($removeEmptyDirsAtExitScriptTemplate, $rootPath, $tmpDir, $rc);
    my $scriptName = "$tmpDir/script";
    if (open (FD, '>' . $scriptName)){
        print FD $script;
        close (FD);
        exec ("/bin/sh '$scriptName'");
    }
    exit $rc;
}

#----------------------------------------------------------------------------

sub create_user{
	my ($name,$group,$password, $config) = @_;
	
	my $path_string = $ENV{'PATH'};
	
	
	local %ENV = %ENV;
	$ENV{'PATH'} = '/sbin:/usr/sbin:'.$path_string;	
	
	my $cmd;
	my @cmd_args;	

	my $comment = 'SAP System Administrator';

	if($^O=~/aix/i){
		if (isPASE()) {
			my $cl = "CRTUSRPRF USRPRF($name) PASSWORD(SDBOFR) ";
			$cl .= "USRCLS(*PGMR) INLMNU(*SIGNOFF) LMTCPB(*YES) TEXT('$comment') ";
			$cl .= "SPCAUT(*JOBCTL) GRPPRF($group) OWNER(*USRPRF) GRPAUT(*NONE) GRPAUTTYP(*PRIVATE) ";
			$cl .= "UID(*GEN) HOMEDIR('/home/$name')";
			
			AddMessage ($config, "[PASE] System, createSDBUser($name, $group)", undef, undef, undef);
			
			$cmd = 'system';
			@cmd_args = ($cl); 
		}
		else {
			$cmd = 'mkuser';
			@cmd_args =  ("pgrp=$group", 'rlogin=true', 'login=true', 'account_locked=false', $name);
		}
	}
	else{
		$cmd = 'useradd';
		@cmd_args =  ('-s', '/bin/sh', '-c', $comment, '-g', $group, '-m');
		
		if (defined $password){
			my $crypt_password = crypt ($password,
				join '', ('.', '/', 0..9, 'A'..'Z', 'a'..'z')[rand 64, rand 64]);
			
			push @cmd_args, ('-p', $crypt_password);
		}
		push @cmd_args, $name;
	}

	my $cmd_out;

	my $exec_msglst = {out => \$cmd_out};

	
	my $msg = AddMessage ($config, "Creating user \"$name\"", undef, undef, undef);
	
	my $rc = exec_program ($cmd, \@cmd_args, $exec_msglst);
	
	AddSubMsgLst (undef, $msg, $exec_msglst);
	
	if (!defined $rc){
		AddError ($config, "Cannot create user \"$name\"", $exec_msglst);
		return undef;
	}
	
	if ($rc){
		AddError ($config, "$cmd exited with code $rc" . ($cmd_out =~ /\S/ ? "; $cmd_out" : ''));
		return undef;
	}


	#
	# if nscd is runnig, it's cache
	# takes max. 15 sec to invalidate
 	#
 	
	my $uid;
	my $tries = 15;
	
	while ($tries--){
		$uid = getpwnam ($name);
		last if defined $uid;
		sleep (1);
	}
		

	if (! defined $uid){
		AddError ($config, "Cannot create user \"$name\": $!");
		return undef;
	}

	return $uid;
}

#----------------------------------------------------------------------------

sub create_group{
	my ($name,$config) = @_;
	my $path_string = $ENV{'PATH'};
	local %ENV = %ENV;
	$ENV{'PATH'} = '/sbin:/usr/sbin:'.$path_string;
	
	my $cmd;
	my @cmd_args;
	
	if($^O=~/aix/i){
		if (isPASE()) {
			$cmd = 'system';
			my $cl = "CRTUSRPRF USRPRF($name) PASSWORD(SDBOFR) ";
			$cl .= "USRCLS(*PGMR) INLMNU(*SIGNOFF) LMTCPB(*YES) TEXT('MaxDB User Group') ";
			$cl .= "SPCAUT(*NONE) UID(*GEN) GID(*GEN)";
			AddMessage($config, "[PASE] System, createGroup($name)", undef, undef, undef);
			@cmd_args = ($cl);
		}
		else {
			$cmd = 'mkgroup';
			@cmd_args = ($name);
		}
	}
	else{
		$cmd = 'groupadd';
		@cmd_args = ($name);
	}
		
	
	my $cmd_out;
	my $exec_msglst = {'out' => \$cmd_out};

	my $msg = AddMessage ($config, "Creating group \"$name\"", undef, undef, undef);
	
	my $rc = exec_program ($cmd, \@cmd_args, $exec_msglst);
	
	AddSubMsgLst (undef, $msg, $exec_msglst);
	
	if (!defined $rc){
		AddError ($config, "Cannot create group \"$name\"", $exec_msglst);
		return undef;
	}
	
	if ($rc){
		AddError ($config, "$cmd exited with code $rc" . ($cmd_out =~ /\S/ ? "; $cmd_out" : ''));
		return undef;
	}
	
	my $gid;		
	my $tries = 15;
	
	while ($tries--){
		$gid = getgrnam ($name);
		last if defined $gid;
		sleep (1);
	}
	
	if (! defined $gid){
		AddError ($config, "Cannot create group \"$name\": $!");
		return undef;
	}
	
	return $gid;
}




#----------------------------------------------------------------------------

our %tmps;

sub getTmpDir{
	my ($drive) = @_;
	
	$drive = uc ($drive);
	
	defined $tmps{$drive} and return $tmps{$drive};
	
	my $tmp = $ENV{'TMP'};
	
	unless (defined $tmp){
		$tmp = $ENV{'TEMP'}; 
	}
	
	my $exp = '^'.$drive;
	
	if (defined $tmp && ($tmp =~ /$exp/i)){
		$tmps{$drive} = $tmp;
		return $tmp;
	}
	
	my $found = 0;
	foreach my $dir ('TEMP','TMP'){
		if( -d "$drive$path_separator$dir"){
			$tmps{$drive} = "$drive$path_separator$dir";
			return $tmps{$drive}
		}
	}

	# no tmp directory for drive found, make own one
	my $tmpdir = "$drive${path_separator}TEMP";
	
	if (!mkdir($tmpdir,0777)){
		return undef;
	}	 
	
	$tmps{$drive} = $tmpdir;
	return $tmpdir; 
}


sub removeBusyFiles{
	my ($files,$msglist) = @_;
	
	my $rc = 1;
	
	if ($isWin){
		my @renamedFiles;
		foreach my $file (@$files){
				my ($drive,$filename) = ($file =~ /^(..).*[\\\/]([^\\\/]+)$/);
				if ($drive != /[a-z]:/i){
					if (defined $msglist){
						$msglist->PushError ("File $file has no drive letter");
					}
					$rc = 0;
					next;
				}
				my $tmp = getTmpDir ($drive);
				my $destination = "$tmp\\$filename.del";
				if(-f $destination){
					my $count = 0;
					while(-f "$destination$count"){$count++;}
					$destination = "$destination$count";
				}
				if (rename($file,$destination)){
					if (defined $msglist){
						$msglist->AddMessage ("Renamed file $file => $destination");
					}
				}
				else{
					if (defined $msglist){
						$msglist->PushError ("Cannot rename file \"$file\" to \"$destination\": $!");
					}
					$rc = 0;
					next;
				}
				$destination =~ s/\//\\/g; 
				push @renamedFiles,$destination;
		}	
		if (@renamedFiles){
			require SAPDB::Install::System::Win32::API;
			foreach my $file (@renamedFiles){
				my ($lrc,$errtxt) = SAPDB::Install::System::Win32::API::MoveFileAfterReboot($file);	
				if ($lrc == 0){
					if (defined $msglist){
						$msglist->AddMessage ("Marked file \"$file\" for deletion after reboot");
					}
				}
				else{
					if (defined $msglist){
						$msglist->PushError ("Cannot mark \"$file\" for deletion after reboot: $errtxt");
						$rc = 0;
					}		
				}
			}	
		}
	}
	elsif ($^O =~ /hpux/i){
		require SDB::Install::System::Trashcan;
		my $trashcan = SDB::Install::System::Trashcan::getTrashcan();
		foreach my $file (@$files){
            if (!defined $trashcan->dropFile ($file)){
                if (defined $msglist){
                    $msglist->PushError ("Cannot remove busy file '$file'", $trashcan);
                }
                $rc = 0;
            }
            else{
                if (defined $msglist){
                    $msglist->AddMessage ("File \"$file\" deleted");
                }
            }
		}
	}
	return $rc;	
}


#----------------------------------------------------------------------------

=functionComment

To a given path which must be _relative_ to the installation root,
this function returns its normalization, i.e. prefixes the path
with the installation root, expands '.' and '..',
and collapses multiple adjacent separators.
The return value does not have trailing slashes.
Backslash separators will be replaced with slashes.

=cut

sub normalizeRelativePathWin {
    my (
        $root,
        $path
    ) = @_;
    my $success = 1;
    my $errmsg  = undef;
    my $prefix = $root;
    $prefix =~ s/^\s(.*)\s$/$1/;            # trim whitespace
    $path   =~ s/^\s(.*)\s$/$1/;            # trim whitespace
    $prefix =~ s/\\/\//g;                   # '\' -> '/'
    $path =~ s/\\/\//g;                     # '\' -> '/'
    my $retval = $prefix.'/'.$path;
    $retval =~ s/^([a-zA-Z]:\/)\/*\.\./$1/; # 'L:/..' equals 'L:/'
    $retval =~ s/^\.$//g;                   # resolve standalone '.'
    $retval =~ s/^\.\///g;                  # resolve './' at beginning
    $retval =~ s/\/\.$//g;                  # resolve '/.' at end
    my $substNum;
    do  {
        # resolve '/./'
        # this can probably be done w/o using a loop,
        $substNum = ($retval =~ s/\/\.\//\//g);
    }
    while($substNum);
    do {
        # resolve '..'
        # this can probably be done w/o using a loop,
        $substNum = ($retval =~ s/[^\/]+\/+\.\.//g);
    }
    while($substNum);
    $retval =~ s/([^\/])\/{2,}/$1\//g;     # collapse multiple adjacent slashes, except at beginning (unc network paths),
    $retval =~ s/^\/{3,}/\/\//g;           # collapse multiple adjacent slashes to a pair, at beginning (unc network paths).
    $retval =~ s/\/+$//g;                  # remove trailing slashes
    $retval = lc($retval);                 # translate to LC 
    return (
        $retval,
        $success,
        $errmsg
    );
}

#----------------------------------------------------------------------------

=functionComment

To a given path which must be _relative_ to the installation root,
this function returns its normalization, i.e. prefixes the path
with the installation root, resolves any symbolic links,
expands '.' and '..', and collapses multiple adjacent separators.
The return value does not have trailing slashes.
Any input containing backslashes yields a fatal error. 

=cut
sub normalizeRelativePathUnix {
    my (
        $root,
        $path,
        $maxSymlinks
    ) = @_;
    my $success = 1;
    my $errmsg  = undef;
    if (!defined $path){
        $path = '';
    }
    if($path =~ /\\/) {
        my $errmsg = "SDB::Install::System::normalizeRelativePathUnix: Malformed file path: \"".
                     $path.
                     "\"\n";
        $success = 0;
        return (undef, $success, $errmsg);
    }
    my $prefix = $root;
    my $pathElem;
    $prefix =~ s/^\s(.*)\s$/$1/;  # trim whitespace
    if($prefix =~ /^[^\/]+/) {
        $errmsg = "SDB::Install::System::normalizeRelativePathUnix: Installation root not absolute: \"".
                     $prefix.
                     "\"\n";
        $success = 0;
        return (undef, $success, $errmsg);
    }
    $path =~ s/^\s(.*)\s$/$1/;    # trim whitespace
    $path = $prefix.'/'.$path;    # prepend installation root
    my $refPath = $path;             # for reference in error message
    $path = conditionPathUnix($path);
    # resolve symlinks:
    my $retval = '';
    my $linkDest = undef;
    my $currentPath = $path;
    my $noOfEncounteredSymlinks = 0;
    while($currentPath) {
        if(-l $currentPath) {
            if($noOfEncounteredSymlinks  > $maxSymlinks) {
                $errmsg = "SDB::Install::System::normalizeRelativePathUnix: Too many (>".
                             $maxSymlinks.
                             ") symbolic links to follow in: \"".
                             $refPath.
                             "\"\n";
                $success = 0;
                return (undef, $success, $errmsg);
            }
            $noOfEncounteredSymlinks++;
            # we encountered a symlink:
            $linkDest = readlink($currentPath);
            $linkDest =~ s/^\s(.*)\s$/$1/;  # remove surrounding whitespace from what we get from the OS
            $linkDest =~ s/\/+$//;          # remove trailing slashes, if any
            if($linkDest =~ /^[^\/]+/) {
                # the symlink is relative:
                $currentPath =~ s/\/[^\/]*$//;
                $currentPath = $currentPath.'/'.$linkDest;
                $currentPath = conditionPathUnix($currentPath);
            }
            else {
                # the symlink is absolute:
                $currentPath = $linkDest;
                $currentPath = conditionPathUnix($currentPath);
            }
        }
        else {
            # not a symlink:
            ($currentPath, $pathElem) = ($currentPath =~ /(.*)(\/[^\/]*$)/);
            $retval = $pathElem.$retval;
        }
    }    
    return (
        $retval,
        $success,
        $errmsg
    );
}

#----------------------------------------------------------------------------

=functionComment

A helper function for unix path normalization, which does all
path modifications which can be accomplished with pattern
matching.

=cut
sub conditionPathUnix {
    my (
        $path
    ) = @_;
    $path =~ s/^\/+\.\./\//;      # '/..' (e.g. at the root) equals '/'
    $path =~ s/^\.$//g;           # resolve standalone '.'
    $path =~ s/^\.\///g;          # resolve './' at beginning
    $path =~ s/\/\.$//g;          # resolve '/.' at end
    my $substNum;
    do  {
        # resolve '/./'
        # this can probably be done w/o using a loop,
        $substNum = ($path =~ s/\/\.\//\//g);
    }
    while($substNum);
    do {
        # resolve '..'
        # this can probably be done w/o using a loop,
        $substNum = ($path =~ s/[^\/]+\/+\.\.//);
    }
    while($substNum);
    $path =~ s/\/+/\//g;          # resolve '//'
    $path =~ s/\/+$//g;           # remove trailing slashes
    return $path;
}

#----------------------------------------------------------------------------

=functionComment

To a given path which must be _relative_ to the installation root,
this function returns its normalization, i.e. prefixes the path
with the installation root, resolves any symbolic links,
expands '.' and '..', and collapses multiple adjacent separators.
Backslash separators will be replaced with slashes generally.
First  parameter: installation root
Second parameter: path

=cut

*normalizeRelativePath = $isWin ? 
                         sub { 
                             return normalizeRelativePathWin($_[0], $_[1]);
                         } :
                         sub {
                             return normalizeRelativePathUnix($_[0], $_[1], 100);
                         };

#----------------------------------------------------------------------------

=functionComment

To a given '$path' which is considered to be relative to '$root',
this function finds the file system it is in on a particular machine.
The mount point of this file system is returned, including a trailing slash.
The given path does not necessarily have to exist.
In the unix case, file systems (i.e. device ids) and their
mount points found by invocations of this function are optionally cached
for use in subsequent calls.
The cache may be provided as parameter '$fileSystemsByDeviceId' (below), or as 'undef', or omitted).
If defined, it must be a reference to a hash which has the device ids (as returned by 'stat')
as keys and the mount points as corresponding values.

=cut

sub getMountPoint_aux;
*getMountPoint_aux = $isWin ? \&getMountPointWin : \&getMountPointUnix;

our $fileSystemsByDeviceId = {};

sub getMountPoint {
    my (
        $root,
        $path,
        $nocache
    ) = @_;
    my $mountPoint     = undef;
    my $success        = 1;
    my $errmsg         = undef;
    my $instRoot       = $root;
    my $normalizedPath = undef;
    ($normalizedPath, $success, $errmsg) = normalizeRelativePath($instRoot, $path);
    if(!$success) {
        return (undef, $success, $errmsg);
    }
    ($mountPoint, $success, $errmsg) = getMountPoint_aux($normalizedPath, $nocache);
    return ($mountPoint, $success, $errmsg);
}

#----------------------------------------------------------------------------

=functionComment

To a given _absolute_and_normalized_ path
(note that this implies the path not being empty),
this function finds the file system it is in on a 
particular machine.
(If the path is not normalized, this function might
get into an endless loop!)
The mount point of this file system is returned, including a trailing slash.
The given path does not necessarily have to exist.
File systems (i.e. device ids) and their mount points found
by invocations of this function are optionally cached for use 
in subsequent calls.
The cache may be provided as parameter 
('$fileSystemsByDeviceId' below, may be given as 'undef', or omitted).
It is a reference to a hash which has the device ids (as returned by 'stat')
as keys and the mount points as corresponding values.

=cut
sub getMountPointUnix {
    my (
        $path,
        $nocache
    ) = @_;
    my $deviceId                      = undef;
    my $currentDeviceId               = undef;
    my $currentPath                   = undef;
    my $maxExistingPath               = undef;
    my $retval                        = undef;
    my $success                       = 1;
    my $errmsg                        = undef;
    # we ascend the path until we find an existing file/dir:
    for($currentPath = $path; $currentPath !~ /^\/*$/; $currentPath =~ s/[^\/]+\/*$//) {
        ($currentDeviceId) = stat($currentPath);
        if(defined $currentDeviceId) {
            $deviceId = $currentDeviceId;
            last;
        }
        elsif(!$!{'ENOENT'}) {
            $errmsg = "SDB::Install::System::getMountPointUnix: Fatal error during stat of \"".
                         $currentPath.
                         "\": \"".
                         $!.
                         "\"\n";
            $success = 0;
            return (undef, $success, $errmsg);
        }
    }
    if(!defined $deviceId) {
        ($deviceId) = stat('/');
    }
    if(!$nocache && defined $fileSystemsByDeviceId) {
        $retval = $fileSystemsByDeviceId->{$deviceId};
    }
    if(!defined $retval) {
        $maxExistingPath = $currentPath;
        if(-d $maxExistingPath && $maxExistingPath !~ /\/$/) {
            $maxExistingPath .= '/';
        }
        # now, we descend the path until we find the first file/dir with
        # matching device id. This is the mount point.
        for($currentPath = '/';
                length($currentPath) <= length($maxExistingPath);
                ($currentPath) = ($maxExistingPath =~ /^(${currentPath}[^\/]+\/).*$/)) {
            ($currentDeviceId) = stat($currentPath);
            if($currentDeviceId == $deviceId) {
                $retval = $currentPath;
                $retval =~ s/\/+$//;
                if(length($retval) == 0) {
                    $retval = '/';
                }
                if(defined $fileSystemsByDeviceId) {
                    $fileSystemsByDeviceId->{$deviceId} = $retval;
                }
                last;
            }
        }
        if(!defined $retval) {
            $errmsg = "SDB::Install::System::getMountPointUnix: Fatal error: could not determine file system of \"".
                         $path.
                         "\"\n";
            $success = 0;
            return (undef, $success, $errmsg);
        }
    }
    return ($retval, $success, $errmsg);
}

#----------------------------------------------------------------------------

=functionComment

To a given _absolute_and_normalized_ path 
(i.e. a UNC network path, or a local (drive letter prefixed) path),
this function finds the file system it is in
on a particular machine.
Note that 'normalized' implies here that backslashes have been 
replaced with slashes.
The mount point of this file system is returned, including a trailing slash.
The given path does not necessarily have to exist.
This windows variant of 'getMountPoint' uses pattern matching on
the starting characters of the given path and does not do any caching.

=cut
sub getMountPointWin {
    my (
        $path
    ) = @_;
    my $mountPoint = undef;
    my $success    = 1;
    my $errmsg     = undef;
    my ($drive) = ($path =~ /^([a-zA-Z]:\/)/);
    my ($host, $share) = ($path =~ /^(\/\/[^\/]+\/)([^\/]+\/)/);
    if($drive) {
        # we have a drive letter:
        $mountPoint = uc($drive);
        $mountPoint =~ s/\//\\/;
    }
    elsif(defined $host && defined $share) {
        # we have a UNC network path:
        $mountPoint = $host.$share;
    }
    else {
        $errmsg = "SDB::Install::System::getMountPointWin: Malformed file path normalization: \"".
                     $path.
                     "\"\n";
        $success = 0;
        return (undef, $success, $errmsg);
    }
    return ($mountPoint, $success, $errmsg);
}

#----------------------------------------------------------------------------

=functionComment

To a given _absolute_ path, this function returns informations about the file system
it is in on a particular machine.
The device id of this file system (as returned by 'stat'),
the total size of the file system in blocks,
the available space on the file system in blocks,
and the block size of the file system in bytes  is returned.
NOTE: if > 1, the scaling factor modifies the unit of fields
'fileSystemTotalSizeBlocks' and 'filesystemAvailableSizeBlocks'.
e.g. if scalingFactor is 1024 (1024*1024), the unit is kiloblocks
(megablocks). 
.

=cut
sub getFileSystemInfo {
    my (
        $path
    ) = @_;
    my $success    = 1;
    my $errmsg     = undef;
    my ($deviceId) = stat($path);
    if(!defined $deviceId) {
        my $errmsg = "SDB::Install::System::getFileSystemInfo: Fatal error during stat of \"".
                     $path.
                     "\": \"".
                     $!.
                     "\"\n";
        $success = 0;
        return (undef, undef, undef, undef, undef, undef, $success, $errmsg);
    }
    my (
        $fileSystemTotalSizeBlocks,
        $filesystemFreeSizeBlocks,
        $filesystemAvailableSizeBlocks,
        $filesystemBlockSizeBytes,
        $scalingFactor
    ) = GetFSSizeInBlocks($path);
    if(!defined $fileSystemTotalSizeBlocks ||
            !defined $filesystemFreeSizeBlocks ||
            !defined $filesystemAvailableSizeBlocks ||
            !defined $filesystemBlockSizeBytes ||
            !defined $scalingFactor) {
        my $errmsg = "SDB::Install::System::getFileSystemInfo: Fatal error during SAPDB::Install::FSUsage::GetFSSizeInBlocks on \"".
                     $path.
                     "\"\n";
        $success = 0;
        return (undef, undef, undef, undef, undef, undef, $success, $errmsg);
    }
    return (
        $deviceId,
        $fileSystemTotalSizeBlocks,
        $filesystemFreeSizeBlocks,
        $filesystemAvailableSizeBlocks,
        $filesystemBlockSizeBytes,
        $scalingFactor,
        $success,
        $errmsg
    );
}

#----------------------------------------------------------------------------

sub canLockProcessWin ($) {
	return ($_[0] =~ /\.exe$|\.dll$/i);
}

#----------------------------------------------------------------------------

our @negativesUnix = qw (htm html jar pm lst py war ti hif jpg jpeg gif ico tif tiff png);

our $negativesUnixRegex = '\.' . join ('$|\.', @negativesUnix) . '$';

sub canLockProcessUnix ($) {
    return ($_[0] !~ /$negativesUnixRegex/i);
 }

#----------------------------------------------------------------------------

=functionComment

For the given filename, this  function looks at its extension and determines
whether the file consists of object code and can lock a process which uses it.
The function may give false positives, but not false negatives.

=cut
sub canLockProcess ($);
*canLockProcess = $isWin ? \&canLockProcessWin : \&canLockProcessUnix;
    
#----------------------------------------------------------------------------

=functionComment

Creates start menu shortcuts according to the given first parameter which is an array reference.
Each item in the array defines one start menu shortcut and is a reference to a hash,
which e.g. looks like this:
{
    'caption' => 'StartMenuSubFolder1/Subfolder2/StartmenuEntry', # mandatory
    'source'  => 'path/to/some.exe',                              # mandatory
    'icon'    => 'path/to/some.ico',                              # optional
    'args'    => '-commandline -arguments',                       # optional
    'workdir' => 'the/working/directory'                          # optional
}
The optional second parameter controls whether the start menu entry
is system-wide or user-specific. Possible values are 'system' or 'user'.
undef or any value different from 'user' will be mapped to 'system'.
The function returns the number of successfully created shortcuts.

=cut
sub createShortcuts{
    my $returncode = 0;
    if ($isWin) {
        my ($defs, $startMenuContext,$msglst) = @_;
        my $regRoot = &HKEY_LOCAL_MACHINE;
        if($startMenuContext eq 'user') {
            $regRoot = &HKEY_CURRENT_USER;
        }
        
        my $regShellFolders = readValues ($regRoot, 'SOFTWARE\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders',undef, $msglst);
        
        if (!defined $regShellFolders){
            if (defined $msglst){
                $msglst->AddError ("Cannot find start menu folder (Shell Folders)");
			}
			return 0;
        }
        
        my $programs_path = $regShellFolders->{'Common Programs'};
        if($programs_path) {
            foreach my $def (@$defs) {
                my $prog = $def->{'caption'};
                my ($folder,$entry);
                if($prog =~ /[\\\/]/){
                    ($folder,$entry) = ($prog =~ /(.+)[\\\/]+([^\\\/]+)/);
                }
                else{
                    $entry = $prog;
                }
                
                if (defined $folder){
                    $folder = $programs_path.'/'.$folder;
                }
                else{
                    $folder = $programs_path;
                }
                unless (-d $folder){
                    makedir($folder,0775);
                }
                my $source  = $def->{'source'};
                unless (-e $source){
                    if (defined $msglst){
                           $msglst->AddError ("Program file '$source' not found");
                    }
                    return undef;
                }
                my $target = $programs_path.'/'.$prog;
                
                $source =~ s/\//\\/g;
                $target =~ s/\//\\/g;
                
                my $rc = SAPDB::Install::System::Win32::API::CreateShortcut(
                    $source,
                    $target.'.lnk',
                    $def->{'args'},
                    $def->{'comment'},
                    $def->{'workdir'},
                    $def->{'icon'},
                    $def->{'icon_index'});
                if ($rc == 0){
                    if (defined $msglst){
                       $msglst->AddError ("SAPDB::Install::System::Win32::API::CreateShortcut() failed");
                    }
                    $returncode += 1;
                }
            }
        }
    }
    return $returncode;
}


#----------------------------------------------------------------------------

sub getSystemStartMenuPathWin{
	my ($isUserMode, $errlst) = @_;

	if (!defined $errlst) {
		$errlst = new SDB::Install::MsgLst();
	}

	my $valueName;
	my $regRoot;
	my $keyName = 'SOFTWARE\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders';
	my $keyNameString;
	if($isUserMode) {
         $regRoot = &HKEY_CURRENT_USER;
		 $valueName = 'Programs';
		 $keyNameString = "HKEY_CURRENT_USER\\$keyName";
     }
	 else{
		$regRoot = &HKEY_LOCAL_MACHINE;
		$valueName = 'Common Programs';
		$keyNameString = "HKEY_LOCAL_MACHINE\\$keyName";
	 }
     my $msglst = new SDB::Install::MsgLst();
        my $regShellFolders = readValues($regRoot, $keyName, undef, $msglst);

        if (!defined $regShellFolders){
            $errlst->addError ("Cannot read registry values", $msglst);
			return undef;
        }
        my $programs_path = $regShellFolders->{$valueName};
		if (!$programs_path){
			$errlst->addError ("Registry value '$valueName' not found in key '$keyNameString'");
			return undef;
		}
		return $programs_path;
}


sub getSystemStartMenuPathUx{
	my ($isUserMode, $errlst) = @_;
	if (!defined $errlst) {
		$errlst = new SDB::Install::MsgLst();
	}
	$errlst->AddError ("Not yet implemented");
	return undef;
}

sub getSystemStartMenuPath;
*getSystemStartMenuPath = $isWin ? \&getSystemStartMenuPathWin : \&getSystemStartMenuPathUx;



#----------------------------------------------------------------------------

sub getSystemDesktopPathWin{
	my ($isUserMode, $errlst) = @_;
	if (!defined $errlst) {
		$errlst = new SDB::Install::MsgLst();
	}

	 my $regRoot = &HKEY_LOCAL_MACHINE;
     if($isUserMode) {
         $regRoot = &HKEY_CURRENT_USER;
     }
     my $msglst = new SDB::Install::MsgLst();
        my $regShellFolders = readValues($regRoot, 'SOFTWARE\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders', undef, $msglst);

        if (!defined $regShellFolders){
            $errlst->AddError ("Cannot read registry values", $msglst);
			return undef;
        }
        my $programs_path = $regShellFolders->{'Common Desktop'};
		if (!$programs_path){
			return undef;
		}

		return $programs_path;
}


sub getSystemDesktopPathUx{
	my ($isUserMode, $errlst) = @_;
	if (!defined $errlst) {
		$errlst = new SDB::Install::MsgLst();
	}
	$errlst->AddError ("Not yet implemented");
	return undef;
}

sub getSystemDesktopPath;
*getSystemDesktopPath = $isWin ? \&getSystemDesktopPathWin : \&getSystemDesktopPathUx;


#----------------------------------------------------------------------------

=functionComment

Removes start menu shortcuts according to the given first parameter which is an array reference.
Each item in the array defines one start menu shortcut and is a reference to a hash,
see function 'createShortcuts' (above) for an example.
The optional second parameter controls whether the start menu entry
is system-wide or user-specific. Possible values are 'system' or 'user'.
undef or any value different from 'user' will be mapped to 'system'.
The function returns the number of successfully removed shortcuts.

=cut
sub removeShortcuts{
    my $returncode = 0;
    if( $isWin ) {
        my ($defs, $startMenuContext, $msglst) = @_;
        my $regRoot = &HKEY_LOCAL_MACHINE;
        if($startMenuContext eq 'user') {
            $regRoot = &HKEY_CURRENT_USER;
        }
        my $regShellFolders = readValues($regRoot, 'SOFTWARE\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders', undef, $msglst);
        
        if (!defined $regShellFolders){
			return 0;
        }
        
        my $programs_path = $regShellFolders->{'Common Programs'};
        if($programs_path) {
            my %first_layers;
            foreach my $def (@$defs) {
                my $prog = $def->{'caption'};
                my ($folder,$entry);
                my ($first_layer) = ($prog =~ /^([^\\\/]+)[\\\/]/);
                if($prog =~ /[\\\/]/){
                    ($folder,$entry) = ($prog =~ /(.+)[\\\/]+([^\\\/]+)$/);
                }
                else{
                    $entry = $prog;
                }
                if (defined $folder){
                    $folder = $programs_path.'/'.$folder;
                }
                else{
                    $folder = $programs_path;
                }
                unless (-d $folder){
                    next;
                }
                if(!opendir(DH,$folder)) {
                    next;
                }
                my @shortcuts = grep {-f "$folder/$_" and /\.lnk$/} readdir(DH);
                closedir(DH);
                my $pattern = $entry;
                $pattern =~ s/\s/\\s/g;
                $pattern =~ s/\(/\\\(/g;
                $pattern =~ s/\)/\\\)/g;
                $pattern = '^'.$pattern.'\.lnk$';
                foreach my $shortcut (@shortcuts){
                    if($shortcut =~ /$pattern/i){
                        unless(unlink ($folder.'/'.$shortcut)){
                            next;
                        }
                    }
                }
                if (defined $first_layer and not exists $first_layers{$first_layer}){
                    $first_layers{$first_layer} = 1;
                }
                $returncode += 1;
            }
            foreach my $dir (keys (%first_layers)){
                if (-d "$programs_path/$dir"){
                    removeEmptyDirs("$programs_path/$dir");
                }
            }
        }
    }
    return $returncode;
}
    
#----------------------------------------------------------------------------

sub instWinSysFile {
    my (
        $file,
        $path,
        $tgz,
        $msglist,
        $versionString
    ) = @_;
    $path =~ s/\\/\//g;
    if(!$isWin){
        return 1;
    }
    $msglist->AddMessage("Checking system library...");
    if(not -d $path){
        $msglist->AddError("System path \"$path\" not found");
        return 0;
    } 
    my $check_update = 0; 
    my $install_files = 0;
    if (-f "$path/$file"){
        my $fname = "$path/$file";
        $fname =~ s/\//\\/g; 
        if(SAPDB::Install::System::Win32::API::SfcIsFileProtected($fname)){
            $msglist->AddMessage("File \"$fname\" is under system file protection");
        }
        else{
            $check_update = 1;
        }
    }  
    else{
        $install_files = 1;
    } 
    if($check_update){
        my $fname = "$path/$file";
        $msglist->AddMessage("Update test system file \"$fname\"");
        $msglist->AddMessage("System file \"$file\" inside archive has version $versionString");                 
        my $theirversion= SAPDB::Install::System::Win32::VersionInfo::SysVersionInfo($fname);
        if(not defined $theirversion){
            $msglist->AddError("Cannot get version of file \"$fname\"");
            return 0;
        }
        if(release2num($theirversion) < release2num($versionString)){
            $msglist->AddMessage("Installed system file \"$fname\" (version $theirversion) has to be updated");
            $install_files = 1;
        }
        else {
            $msglist->AddMessage("Installed system file \"$fname\" is up to date (version $theirversion)");
        }
    }
    $msglist->AddMessage(" ok");
    if($install_files){
        my $info = "$path/$file"; 
        $msglist->AddMessage("Installing system library...");
        if(-f "$path/$file"){                   
            if(open(tmpFD,'+<'."$path/$file" )){
                close(tmpFD);
            }
            else{
                $msglist->AddMessage("Cannot open \"$path/$file\" for writing: $! - moving it now");
		my $ml = new SDB::Install::MsgLst ();
                removeBusyFiles (["$path/$file"], $ml);
		$msglist->AddSubMsgLst ($msglist,$ml);
            }
        }
        $tgz->ExtractFileForce($info,$path,0);
        if($tgz->GetErr){
            $msglist->AddError("Cannot extract file \"$path/$file\": ".(join(': ',$tgz->GetErr)));
            return 0;
        }
        else{
            $msglist->AddMessage("$info extracted");
        }
        $msglist->AddMessage(" ok");
    }
    return 1;
}

#----------------------------------------------------------------------------

=functionComment

extracts version of archived system file only if it already exists in the file system 
returns undef on error, (possibly empty) version string on success:

=cut
sub getWinSysFileVersion {
    my (
        $file,
        $path,
        $tgz,
        $msglist
    ) = @_;
    $path =~ s/\\/\//g;
    if(!$isWin){
        return '';
    }
    if(not -d $path){
        $msglist->AddError("getWinSysFileVersion(): system path \"$path\" not found");
        return undef;
    } 
    my $myversion = '';
    my $fname = "$path/$file";
    $fname =~ s/\//\\/g;
    if (-f $fname) {
        $msglist->AddMessage("Extracting version info from archived system file \"$fname\"");
        $myversion = $tgz->ExtractSysVersionInfo();
        if($tgz->GetErr) {
            $msglist->AddError("Cannot get version of file \"$file\" inside archive: ".(join(': ',$tgz->GetErr)));
            return undef;
        }
        $msglist->AddMessage("System file \"$file\" inside archive has version $myversion");                 
    }
    $msglist->AddMessage(" ok");
    if(not defined $myversion) {
        $myversion = '';
    }
    return $myversion;
}

#----------------------------------------------------------------------------

sub getSysPath{
    my (
        $msglist
    ) = @_;
    if(!$isWin){
        $msglist->AddError("getSysPath() not implemented for this platform.");
        return undef;
    }
    my $submsglst = new SDB::Install::MsgLst ();
    my $hash = readValues (0,'SYSTEM\CurrentControlSet\Control\Windows', undef, $submsglst);
    if(!defined $hash || ! exists $hash->{'SystemDirectory'}){
        $msglist->AddError("Cannot find system directory inside windows registry.", $submsglst);
        return undef;
    }
    return $hash->{'SystemDirectory'};
}


#----------------------------------------------------------------------------

sub isAdmin{
	if($isWin){
		my $name;
		my $rc = SAPDB::Install::System::Win32::API::IsLocalAdmin ($name);
		if (defined $rc and $rc >= 0){
			return wantarray() ? ($rc,$name) : $rc;
		}
		return wantarray() ? (undef,$name) : undef;	
	}
	
	if (isPASE()){
		my $user = getpwuid($>);
		
		if ($user =~ /^QSECOFR$/i) {
			return 1;
		} 
		else {
			if (defined $SAPDB::Install::Values::log) {
				$SAPDB::Install::Values::log->SetMsg("WARN: [PASE] You are not logged on as user QSECOFR.\n");
			}
			return 0;
		}
		# In future, other users with *SECOFR rights should be able to perform 
		# installations as well, not just QSECOFR
		#my $usrprf = readpipe("system \"DSPUSRPRF $user\"");
		#if ( ($usrprf !~ /\*SECOFR/)   ||
		#     ($usrprf !~ /\*ALLOBJ/)   || ($usrprf !~ /\*AUDIT/) ||
		#     ($usrprf !~ /\*IOSYSCFG/) || ($usrprf !~ /\*JOBCTL/) ||
		#     ($usrprf !~ /\*SAVSYS/)   || ($usrprf !~ /\*SECADM/) ||
		#     ($usrprf !~ /\*SERVICE/)  || ($usrprf !~ /\*SPLCTL/) ) {
		#    print2stderr("[PASE] $> needs to be security officer (USRCLS (*SECOFR)) and ");
		#    print2stderr("has to have the following special authorities:\n");
		#    print2stderr("   SPCAUT(*ALLOBJ *AUDIT *IOSYSCFG *JOBCTL *SAVSYS *SECADM *SERVICE *SPLCTL)\n");
		#    return 0;
		#}
		#return 1;
	}
	
	if ($isApple){
		my @grent = getgrnam ('admin');
		my $name = getpwuid ($>);
		if (@grent){
			foreach my $user (split (/\s+/, $grent[3])){
				if ($name eq $user){
					return 1;
				}
			}
		}
		return 0;
	}


	if($> == 0){
		return 1;
	}
	return 0;		
}

#----------------------------------------------------------------------------

sub isSidadmin {
	my ($sid) = @_;
	my $currentUser = new SDB::Install::User();
	my $name = $currentUser->getname();

	if (not $sid) {
		require LCM::Installer;
		my $installer = new LCM::Installer();
		$sid = $installer->getSid();
	}

	if (!defined $sid){
		return undef;
	}

	my $sidAdm = createSysAdminUserName($sid);

	return $name eq $sidAdm && $currentUser->hasgroup($gSapsysGroupName);
}

#----------------------------------------------------------------------------

sub normalizePath{
	my ($srPath)=@_;
	if (!$isWin){
		if(ref($srPath)){
			ref($srPath) ne 'SCALAR' and return undef;
			return 0;
		} 
		return $srPath;
	}
	if(ref($srPath)){
		ref($srPath) ne 'SCALAR' and return;
		my $rc=0;
        $$srPath=~s/^\s*// and $rc=1;
        $$srPath=~s/\s*$// and $rc=1;
        $$srPath=~s/\\/\//g and $rc=1;
		$$srPath=~s/([^\/])\/{2,}/$1\//g and $rc=1; 
		$$srPath=~s/\/+$//g and $rc=1;
		$$srPath=~s/^\/{3,}/\/\//g and $rc=1;
		$$srPath = lc($$srPath) and $rc=1;		
		return $rc;
	}
	else{
		my $returnvalue = $srPath;
		$returnvalue =~ s/^\s*//;               # trim leading...
		$returnvalue =~ s/\s*$//;               # ... and trailing whitespace
		$returnvalue =~ s/\\/\//g;              # '\' -> '/'
		$returnvalue =~ s/([^\/])\/{2,}/$1\//g; # collapse multiple adjacent slashes, except at beginning (unc network paths)
		$returnvalue =~ s/\/+$//g;              # remove trailing slashes
		$returnvalue =~ s/^\/{3,}/\/\//g;       # collapse multiple adjacent slashes to a pair, at beginning (unc network paths).
        $returnvalue = lc($returnvalue);        # translate to LC
		return $returnvalue;
	}

}

#----------------------------------------------------------------------------

sub getRealPathName{
	my  ($path) = @_;
	return undef unless ($path =~ /\S/);
	return $path if !$isWin;
	
	my $rel_root;
	my $tmp_path;
	
	if ($path !~ /^[\\\/]|^[a-z]:/i){
		# relative path
		if (defined $SAPDB::Install::Config{CallerDir}){
			$rel_root = $SAPDB::Install::Config{CallerDir};
		}
		else{
			require Cwd;
			$rel_root = Cwd::getcwd();
		}
		$tmp_path = $rel_root.'\\'.$path;
	}
	else{
		$tmp_path = $path;
	}
	
	
	unless( -e $tmp_path){
		if ($path =~ /^[\\\/]{2}[^\\\/]+[\\\/]+[^\\\/]+[\\\/]*$/){
			#unc notation \\host\share
			my ($host,$share) = ($path =~ /^[\\\/]{2}([^\\\/]+)[\\\/]+([^\\\/]+)/);
			return '\\\\'.$host.'\\'.$share;
		}
		return $path unless $path =~ /[\\\/]/;
		my ($next_path,$dont_exist) = ($path =~ /(^.*)[\\\/]+([^\\\/]+)$/);
		return getRealPathName($next_path).'\\'.$dont_exist;
	}
	
	my $real_path = '';
	
	if ($path =~ /^[\\\/]{2}[^\\\/]+/){
		#unc notation  \\host\share
		my ($host,$share) = ($path =~ /^[\\\/]{2}([^\\\/]+)[\\\/]+([^\\\/]+)/);
		$real_path = '\\\\'.$host.'\\'.$share.'\\';
		$path =~ s/^[\\\/]{2}[^\\\/]+[\\\/]+[^\\\/]+[\\\/]*//; 		
	}
	elsif ($path =~ /^[a-z]:/i){
			# drive letter
			my ($drive) =  ($path =~ /^([a-z]:)/i);
			$drive =~ tr/[a-z]/[A-Z]/;
			$real_path = $drive.'\\';
			$path =~ s/^[a-z]:[\\\/]*//i; 
	}
		
	if ($rel_root !~ /\S/ and defined $real_path){
		$rel_root = $real_path;
	}	
	
	
	foreach my $fsnode (split(/[\\\/]+/,$path)){
		my $pattern = '^' . quotemeta ($fsnode) . '$';
		opendir(DH,$rel_root);
		foreach my $node (readdir(DH)){
			if($node =~ /$pattern/i){
				$real_path .= $node.'\\';
				$rel_root .= $node.'\\';
				last;
			}	
		}
		closedir(DH);
	}
	$real_path =~ s/\\$//; 
	return $real_path; 	
}

#----------------------------------------------------------------------------

our $FIND_TYPE_FILE = 1;
our $FIND_TYPE_DIR = 2;

sub FIND_TYPE_FILE {$FIND_TYPE_FILE}
sub FIND_TYPE_DIR {$FIND_TYPE_DIR}

sub find {
    my ($root, $errlst, $type,$dir,$pattern, $find_path_separator) = @_;
    if (!defined $type){
        $type = $FIND_TYPE_FILE;
    }
    if (!defined $find_path_separator){
        $find_path_separator = $path_separator
    }

    my $path = $root;
    if ($dir){
        $path .= $path_separator . $dir;
    }
    if (!opendir (DH, $path)){
        if (defined $errlst){
            $errlst->AddError ("Cannot open directory '$path': $!");
        }
        return undef;
    }
    my @entries = grep {!/^\.{1,2}$/} readdir (DH);
    closedir (DH);
    my @result;
    foreach my $entry (@entries){
        if (-d $path . $path_separator . $entry){
            if ($type & $FIND_TYPE_DIR){
                if (!defined $pattern || ($entry =~ /$pattern/)){
                    push @result, $dir ? $dir . $find_path_separator . $entry : $entry;
                }
            }
            my $rc = find ($root, $errlst, $type, $dir ? $dir . $find_path_separator . $entry : $entry, $pattern, $find_path_separator);
            if (!defined $rc){
                return undef;
            }
            push @result, @$rc;
        }
        else{
            if ($type & $FIND_TYPE_FILE){
                if (!defined $pattern || ($entry =~ /$pattern/)){
                    push @result, $dir ? $dir . $find_path_separator . $entry : $entry;
                }
            }
        }
    }
    return \@result;
}

#----------------------------------------------------------------------------

sub readLinkUx{
	my ($path) = @_;
	
	if (!-e $path){
		return $path;
	}
	
	if (!lstat ($path)){
		return $path;
	}
	my $dest = readlink ($path);
	
	if (!$dest){
		return $path;
	}
	
	if ($dest =~ /^\//){
		return $dest;
	}
	
	my ($dir) = ($path =~ /(.*)\//);
	return conditionPathUnix ("$dir/$dest");
	
}

#----------------------------------------------------------------------------

sub readValues{
	#sub retrun hash of values
	
	my ($root, $keyname, $notexpand, $msglst) = @_;
	
	if (!$isWin){
		if (defined $msglst){
			$msglst->AddError ("readValues() isn't implemented on this platform");
		}
		return undef;
	}
	if (!$root){
		 $root = &HKEY_LOCAL_MACHINE;
	}
	
	my ($key,$ValCount,$index,$sValName,$piType,$pValData);
	
	if (!RegOpenKeyEx($root, $keyname, 0, &KEY_READ , $key ) == 0){ 
		if (defined $msglst){
			$msglst->AddError ("Can't open registry key '$keyname': $!");
		}
		return undef;
	}
	my $i=0;
	my %returnvalue;
	RegQueryInfoKey ($key, 0, 0, 0, 0, 0, 0, $ValCount, 0, 0, 0, 0);
	for ($index = 0; $index < $ValCount; $index++){
		RegEnumValue( $key, $index ,$sValName, 0, 0, $piType, $pValData, 0 );
		if(Win32::GetLastError()==259){last;}
		if($piType==&REG_BINARY){
			$pValData=unpack("H*",$pValData);	
		}
		elsif($piType==&REG_EXPAND_SZ and not $notexpand){
			ExpandEnvironmentStrings($pValData,$pValData,0);
		}
		$sValName = '(Default)' unless $sValName =~ /\S/;
		$returnvalue{$sValName}=$pValData;
	}
	if (!RegCloseKey( $key ) == 0){
		if (defined $msglst){
			$msglst->AddWarning ("Can't close registry key '$keyname': $!");
		}
	}
	return \%returnvalue;
}

#----------------------------------------------------------------------------

sub readWinRegValue{
	my ($hKey, $keyname, $valuename,$msglst) = @_;
	my $hash = readValues($hKey, $keyname, $msglst);
	if (!defined $hash){
		if (defined $msglst){
			$msglst->AddError ('Cannot access windows registry', $msglst);
		}
		return undef;
	}
	elsif (!exists $hash->{$valuename}){
		if (defined $msglst){
			$msglst->AddError ("Value '$valuename' not found in 'HKEY_LOCAL_MACHINE\\$keyname'");
		}
		return undef;
	}
	return $hash->{$valuename};
}

#----------------------------------------------------------------------------

sub getAllUsersAppDataPath{
	if (!$isWin){
		return $isApple ? '/Applications/' : '/var/opt';
	}
	my ($msglst) = @_;
	my $valuename = 'Common AppData';
	my $keyname = 'SOFTWARE\Microsoft\Windows\CurrentVersion\Explorer\Shell Folders';
	return readWinRegValue (&HKEY_LOCAL_MACHINE, $keyname, $valuename, $msglst);
}

#----------------------------------------------------------------------------

sub getSysProgPath{
	if ($isWin){
		my ($msglst) = @_;
		my $keyname = 'SOFTWARE\Microsoft\Windows\CurrentVersion';
		my $valuename = 'ProgramFilesDir';
		return readWinRegValue (&HKEY_LOCAL_MACHINE, $keyname, $valuename, $msglst);
	}
	else{
		return $isApple ? '/Applications' : '/opt';
	}
}

#----------------------------------------------------------------------------

sub getHomeDir{
    return $isWin ? $ENV{USERPROFILE} : (getpwuid($>))[7];
}

#----------------------------------------------------------------------------

sub readLinkWin{
	my ($rc, $src, $comment) = 
		SAPDB::Install::System::Win32::API::ShortcutInfo ($_[0] . '.lnk');
	if ($rc == 0){
		return $src;
	}
	return $_[0];
}

#----------------------------------------------------------------------------

sub isSameFileUx{
	my ($file1, $file2, $statbuf2) =  @_;
	my @statbuf1 = stat $file1;
	if (!@statbuf1){
		return 0;
	}
	my @statbuf2 = defined $statbuf2 ? @$statbuf2 : stat $file2;
	if (!@statbuf2){
		return 0;
	}
	if ($statbuf1[1] != $statbuf2[1]){
		return 0;
	}
	
	if ($statbuf1[0] != $statbuf2[0]){
		return 0;
	}
	return 1;
}

#----------------------------------------------------------------------------

sub isSameFileWin{
	my ($file1, $file2, $file2_is_normalized) =  @_;
	return normalizePath ($file1) eq 
		(defined $file2_is_normalized ? $file2 : normalizePath ($file2));
}

#----------------------------------------------------------------------------

sub isSameFile2Win{
	my ($file1, $file2, $buf2) =  @_;
	my ($fs1, $low1, $high1) = SAPDB::Install::System::Win32::API::GetFileId ($file1);
	if (!defined $fs1){
		return undef;
	}
	my ($fs2, $low2, $high2);
	if (defined $buf2){
		($fs2, $low2, $high2) = @$buf2;
	}
	else{
		($fs2, $low2, $high2) = SAPDB::Install::System::Win32::API::GetFileId ($file2);
		if (!defined $fs1){
			return undef;
		}
	}

	if ($low1 != $low2){
		return 0;
	}
	if ($high1 != $high2){
		return 0;
	}

	if ($fs1 != $fs2){
		return 0;
	}
	return 1;
}

sub isSameFile;
*isSameFile = $isWin ? \&isSameFileWin : \&isSameFileUx;


#----------------------------------------------------------------------------
sub isSameFile2;
*isSameFile2 = $isWin ? \&isSameFile2Win : \&isSameFileUx;

sub readLink;
*readLink = $isWin ? \&readLinkWin : \&readLinkUx;

#----------------------------------------------------------------------------

sub getFileIdUx ($){
    return [stat (${$_[0]})];
}

sub getFileIdWin ($){
    return [SAPDB::Install::System::Win32::API::GetFileId (${$_[0]})];
}

sub getFileId ($);
*getFileId = $isWin ? \&getFileIdWin : \&getFileIdUx;

#----------------------------------------------------------------------------

sub which{
	my ($prog) = @_;
	my @progs;
	
	if ($isWin){
		my @exts = ('.exe','.com','.cmd','.bat');
		my @found;
		my $pattern;
		my $file;
		foreach my $path (split ($env_path_separator, $ENV{'PATH'})){
			opendir(DH,$path) or next;
			$pattern = '^'.$prog.'\.exe$|^'.$prog.'\.com$|^'.$prog.
						  '\.cmd$|^'.$prog.'\.bat$';
			
			if($prog =~ /\.exe$|\.com$|\.cmd$|\.bat$|/i){
				$pattern .= '|^'.$prog.'$';
			}
			@found = grep { /$pattern/i and -f $path.'\\'.$_} readdir(DH);
			closedir(DH);
			foreach $file (@found){
				if(wantarray){
					push @progs,$path.'\\'.$file;
				}
				else{
					return $path.'\\'.$file;
				}
			}
		}
	}
	else{
		foreach my $path (split ($env_path_separator, $ENV{'PATH'})){
			if(-x "$path/$prog"){
				if(wantarray){
					push @progs, "$path/$prog";
				}
				else{
					return "$path/$prog";
				}
			}
		}
	}
	return wantarray ? @progs : undef;
}

#----------------------------------------------------------------------------

our $nslookup;

sub nslookup_exe{
	my ($name,$msglst) = @_;
	my $path;
	if (!defined $nslookup){
		local %ENV = %ENV;
		if (!$isWin){
			$ENV{PATH} = join(':','/usr/bin', '/bin', '/usr/sbin', '/sbin', $ENV{PATH});
		}
		$path = $ENV{PATH};
		$nslookup = which ('nslookup');
	}
	
	if (!$nslookup){
		if (defined $msglst){
			$msglst->AddError ("Tool nslookup not found (PATH=$path)");
		}
		return undef;
	}
	
	my $out;
	
	my $cfg = {'out' => \$out};
	
	my $rc = exec_program ($nslookup,[$name], $cfg);

	if (!defined $rc || $rc != 0){
		if (defined $msglst){
			$msglst->AddMessage (undef, $cfg);
		}
		return undef;
	}
	if (defined $msglst){
		$msglst->AddMessage (undef, $cfg);
	}
	
	my @buffer = split ("\n", $out);
	
	my %result;
	my ($key);
	foreach my $i (0..$#buffer){
		if ($buffer[$i] !~ /\S/){
			next;
		}
		if ($buffer[$i] =~ /^\*/){
			($result{msg}) = ($buffer[$i] =~ /^\*+\s+(.*)/);
			next;
		}
	
		($key) = split (':', $buffer[$i]);
		if ($key eq 'Server'){
			($result{nameserver})= ($buffer[$i] =~ /\s+(.*)/);
		}
		elsif ($key eq 'Address'){
			if (!defined $result{name}){
				($result{nameserver_address})= ($buffer[$i] =~ /\s+(.*)/);
			}
			else{
				($result{address})= ($buffer[$i] =~ /\s+(.*)/);
			}
		}
		elsif ($key eq 'Name'){
			($result{name})= ($buffer[$i] =~ /\s+(.*)/);
		}
	}
	return \%result;
}

#----------------------------------------------------------------------------

sub nslookup_builtin{
    my ($name,$msglst) = @_;
    my ($err, @result) = getaddrinfo ($name, "", {socktype => SOCK_RAW});
    my ($ipaddr, $fname);
    if (!$err){
        ($err, $fname) = getnameinfo ($result[0]->{addr});
        foreach my $ai (@result){
            ($err, $ipaddr) = getnameinfo ($ai->{addr}, NI_NUMERICHOST);
            if ($err){
                last;
            }
        }
    }
    if (!$err){
        return {'address' => $ipaddr, 'name' => $fname};
    }
    if (defined $msglst){
        $msglst->AddError ("Cannot resolve host name '$name': $err");
    }
    return {};
}

#----------------------------------------------------------------------------

*nslookup = exists $ENV{USE_NSLOOKUP_EXECUTABLE} ? \&nslookup_exe :
	\&nslookup_builtin;

#----------------------------------------------------------------------------

sub is_local_address{
	my ($ip_address,$msglst) = @_;

	my @ipv4 = split (/\./, $ip_address);
	if (@ipv4 == 4){
		if ($ipv4[0] == 127){
			return 1;
		}
	}
	elsif ($ip_address eq '::1' || $ip_address eq '::'){
		return 1;
	}

	my $errlst = new SDB::Install::MsgLst ();
    require SDB::Install::SysInfo;
	my $interfaces = SDB::Install::SysInfo::interfaceInfo ($errlst);
	if (!defined $interfaces){
		if (defined $msglst){
			$msglst->AddError ('Cannot get network interfaces', $errlst);
		}
		return undef;
	}
    my $ipaddr;
	foreach my $ipaddrlist (values %$interfaces){
        foreach $ipaddr (@$ipaddrlist){
            if ($ipaddr->{addr} eq $ip_address){
                return 1;
            }
        }
    }
	return 0;
}

#----------------------------------------------------------------------------

our $regKeyUninstall = 'SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall';

sub createUninstallEntry{
    if (!$isWin){
        return 1;
    }
    my ($keyname, $entry, $isMultiple, $path, $msglst) = @_;

    if (!defined $msglst){
        $msglst = new SDB::Install::MsgLst ();
    }

    if ($isMultiple){
        $keyname = getNextUninstallKey ($keyname, $path);
        if (!$keyname){
            return undef;
        }
    }

    $keyname = $regKeyUninstall . '\\' . $keyname;

     if (!createRegistryKey (HKEY_LOCAL_MACHINE, $keyname,$msglst)){
        return undef;
    }

    my ($key, $rc);


    if (RegOpenKeyEx (HKEY_LOCAL_MACHINE, $keyname, 0, &KEY_SET_VALUE, $key ) != 0){
        $msglst->PushError ("Cannot open registry key 'HKEY_LOCAL_MACHINE\$keyname': $!");
        return undef;
    }

    foreach my $value ( ['DisplayName', $entry->{name}],
                        ['UninstallString', $entry->{cmd}],
                        ['Publisher', $entry->{publisher} ? $entry->{publisher} : 'SAP SE'],
                        ($entry->{version} ? ['DisplayVersion', $entry->{version}] : ()),
                        ($entry->{icon} ? ['DisplayIcon', $entry->{icon}] : ()),
                        ){
        if (RegSetValueEx ($key, $value->[0], 0, REG_SZ, $value->[1], 0) != 0){
            $msglst->PushError ("Cannot set registry value '$value->[0]' of key $keyname: $!");
            RegCloseKey ($key);
            return undef;
        }
    }

    foreach my $value ( ['NoModify', 1],
                        ['NoRepair', 1],
                       ($entry->{size} ? ['EstimatedSize', $entry->{size}] : ()),
                    ){
        if (RegSetValueEx ($key, $value->[0], 0, REG_DWORD, $value->[1], 0) != 0){
            $msglst->PushError ("Cannot set registry value '$value->[0]' of key $keyname: $!");
            RegCloseKey ($key);
            return undef;
        }
    }
    RegCloseKey ($key);
    return 1;
}

#----------------------------------------------------------------------------

sub findUninstallKey{
    my ($keyname,$path) = @_;
    my $subkeys = readRegistrySubKeys (HKEY_LOCAL_MACHINE, $regKeyUninstall);
    if (defined $subkeys && @$subkeys){
        my $values;
        my $pattern = '^' . quotemeta ($keyname) . '\d+$';
        my $path_pattern = '^' . quotemeta ($path);
        foreach my $key (grep {/$pattern/} @$subkeys){
            $values = readValues (HKEY_LOCAL_MACHINE, $regKeyUninstall . '\\' . $key);
            if (defined $values && $values->{UninstallString} =~ /$path_pattern/){
                return $key;
            }
        }
    }
    return '';
}

#----------------------------------------------------------------------------

sub getNextUninstallKey{
    my ($keyname,$path) = @_;
    my $subkeys = readRegistrySubKeys (HKEY_LOCAL_MACHINE, $regKeyUninstall);
    if (defined $subkeys && @$subkeys){
        my $values;
        my $found = 0;
        my $pattern = '^' . quotemeta ($keyname) . '\d*$';
        my $path_pattern = '^' . quotemeta ($path);
        foreach my $key (grep {/$pattern/} @$subkeys){
            $values = readValues (HKEY_LOCAL_MACHINE, $regKeyUninstall . '\\' . $key);
            if (defined $values && $values->{UninstallString} =~ /$path_pattern/i){
                return $key;
            }
        }
        my $test_keyname = $keyname;
        my $i = 0;
        foreach my $key (sort grep {/$pattern/} @$subkeys){
            if ($key ne $test_keyname){
                return $test_keyname;
            }
            $test_keyname = $keyname . $i++;
         }
         return $test_keyname;
    }
    return $keyname;
}

#----------------------------------------------------------------------------

sub removeUninstallEntry{
    if (!$isWin){
        return 1;
    }
    my ($keyname, $isMultiple, $path, $msglst) = @_;
    if (!defined $msglst){
        $msglst = new SDB::Install::MsgLst ();
    }
    if ($isMultiple){
        $keyname = findUninstallKey ($keyname, $path);
        if (!$keyname){
            return 1;
        }
    }
    $keyname = $regKeyUninstall . '\\' . $keyname;
    return removeRegistryKey(HKEY_LOCAL_MACHINE, $keyname, $msglst);
}

#----------------------------------------------------------------------------

sub createRegistryKey{
    if (!$isWin){
        return 1;
    }
    my ($hkey, $key, $msglst) = @_;

    if (!defined $msglst){
        $msglst = new SDB::Install::MsgLst ();
    }

    my ($parentkey) = ($key =~ /(.*)\\[^\\]*$/);

    if (!$parentkey){
        return 1;
    }

    if (!defined createRegistryKey($hkey,$parentkey, $msglst)){
        return undef;
    }

    my ($newkey,$piDisp);

    if (RegCreateKeyEx ($hkey,$key,0,0,REG_OPTION_NON_VOLATILE ,&KEY_CREATE_SUB_KEY  ,0,$newkey,$piDisp) != 0){
        $msglst->PushError ("Cannot create registry key '$key'");
        return undef;
    }

    if (RegCloseKey ($newkey) != 0){
        $msglst->AddWarning ("Cannot close registry key '$key'");
    }


    if ($piDisp == REG_CREATED_NEW_KEY){
        $msglst->AddMessage("Registry key '$key' successfully created");
    }
    elsif($piDisp == REG_OPENED_EXISTING_KEY){
        $msglst->AddMessage("Registry key '$key' already exists\n");
    }
    return 1;
}

#----------------------------------------------------------------------------

sub removeRegistryKey{

    if (!$isWin){
        return 1;
    }

    my ($hkey, $keyname, $msglst) = @_;

    if (!defined $msglst){
        $msglst = new SDB::Install::MsgLst ();
    }

    #
    # values
    #

    my $values = readValues($hkey, $keyname, 1, $msglst);
    my $key;

    if (RegOpenKeyEx($hkey, $keyname, 0, KEY_ALL_ACCESS , $key ) != 0){
        $msglst->AddError ("Can't open registry key '$keyname' : $!");
        return undef;
    }
    foreach my $value_name (keys(%$values)){
        if (RegDeleteValue ($key, $value_name) != 0){
            $msglst->AddError ("Cannot delete registry value '$value_name' of key '$keyname': $!");
            RegCloseKey ($key);
            return undef;
        }
    }

    RegCloseKey ($key);

    #
    # subkeys
    #

    my $subkeys = readRegistrySubKeys ($hkey,$keyname, $msglst);
    if (!defined $subkeys){
        return undef;
    }
    foreach my $subkey (@$subkeys){
        if (!defined removeRegistryKey ($hkey,"$keyname\\$subkey", $msglst)){
            return undef;
        }
    }


    my ($parent_keyname,$subkeyname) = ($keyname =~ /^(.*)\\([^\\]+)\\{0,1}$/);

    if (RegOpenKeyEx($hkey, $parent_keyname, 0, KEY_ALL_ACCESS , $key ) != 0){
        $msglst->AddError ("Can't open registry key '$parent_keyname': $!");
        return undef;
    }

    if (RegDeleteKey($key, $subkeyname) != 0){
        $msglst->AddError ("Cannot delete registry key '$keyname': " . Win32::GetLastError());
        RegCloseKey ($key);
        return undef;
    }
    RegCloseKey ($key);
    return 1;
}

#----------------------------------------------------------------------------

sub readRegistrySubKeys{
    if (!$isWin){
        return 1;
    }
    my (
        $hkey,
        $keyname,
        $msglst,
        $asHash # if set true, a hash ref is returned instead of an array ref, where key == value.
    ) = @_;

    if (!defined $msglst){
        $msglst = new SDB::Install::MsgLst ();
    }

    my ($key,$KeyCount, $sName);

    if (RegOpenKeyEx ($hkey, $keyname, 0, KEY_READ , $key ) != 0){
        $msglst->AddError ("Can't open registry key '$keyname': $!");
        return undef;
    }
    my $i=0;
    my @returnvalue;
    my %returnhash;
    RegQueryInfoKey ($key, 0, 0, 0, $KeyCount, 0, 0, 0, 0, 0, 0, 0);
    for (my $index = 0; $index < $KeyCount; $index++){
        RegEnumKeyEx ($key , $index, $sName, 0, 0, 0, 0, 0);
        if(Win32::GetLastError()==259){last;}
        if($asHash) {
            $returnhash{$sName} = $sName;
        }
        else {
            push(@returnvalue,$sName);
        }
    }
    RegCloseKey ($key);
    if($asHash) {
        return \%returnhash;
    }
    else {
        return \@returnvalue;
    }
    
}

#----------------------------------------------------------------------------

# returns hash: keys: the installation paths, values: arrays containing all reg'd driver names
# for that path.
# This method is DEPRECATED,
# use SDB::Install::System::Registration::ODBC::getDriverNamesAssignedToInstallation(...)
# It remains here for compatibility, since 'script.pm's of the odbc package in existing old
# installations (these are persisted in the INSTREG) still depend on it. (as of june 1st 2012) 
sub getODBCDrivernamesByInstallationPath {
    my (
        $odbcreg,   # absolute path to the appropriate odbcreg executable
        $msgHandler
    ) = @_;
    my $retval = {};
    my $msg = $msgHandler->AddMessage ("Looking for ODBC drivers registered by this installation.");
    my $out;
    my $cfg =  {'out' => \$out};
    my $rc = exec_program($odbcreg, ['-g1'], $cfg);
    if (!defined $rc || $rc != 0){
        $msgHandler->AddError("Could not enumerate ODBC drivers at this machine: rc=$rc.", $cfg);
        $msgHandler->AddSubMsgLst($msg, $cfg);
        return undef;
    }
    my @lines = split('\n', $out);
    foreach my $line (@lines) {
        if($line =~ /;/) {
            my ($driverName, $driverPath) = split(';', $line);
            $driverName =~ s/^\s(.*)\s$/$1/; #trim whitespace
            $driverPath =~ s/^\s(.*)\s$/$1/; #trim whitespace
            $driverPath =~ s/\\/\//g;        #fwd slashes
            ($driverPath,undef) = ($driverPath =~ /(.*)\/([^\/]+)$/);
            if($driverName !~ /^$/ and $driverPath !~ /^$/) {
                my $nameArray = $retval->{$driverPath};
                if(not defined $nameArray) {
                    $nameArray = [];
                    $retval->{$driverPath} = $nameArray;
                }
                push(@$nameArray, $driverName);
            }
        }
    }
    $msgHandler->AddSubMsgLst($msg, $cfg);
    return $retval;
}

#----------------------------------------------------------------------------

sub isEmptyDir {
    my ($path) = @_;
    my $retval = 0;
    if(-d $path) {
        $retval = 1;
        opendir DIR, $path;
        while(my $entry = readdir DIR) {
            if($entry =~ /^\.\.?$/) {
                next;
            }
            $retval = 0;
            last;
        }
        closedir DIR;
    }
    return $retval;
}

#----------------------------------------------------------------------------

sub listDir {
    my ($path) = @_;
    my @contents = ();
    if(-d $path) {
        opendir DIR, $path;
        while(my $entry = readdir DIR) {
            if($entry =~ /^\.\.?$/) {
                next;
            }
            push(@contents, $entry);
        }
        closedir DIR;
    }
    return \@contents;
}

#----------------------------------------------------------------------------

sub isRootFsWin{
    my ($path) = @_;

    #path with drive letter "<letter>:\"
    if ($path =~ /^[a-z]:[\\\/]*$/i){
        return 1;
    }

    #path without drive letter "\"
    if ($path =~ /^[\\\/]+$/i){
        return 1;
    }

    #unc path "\\<host>\<share>"
    if ($path =~ /^[\\\/]{2}[^\\\/]+[\\\/]+[^\\\/]+[\\\/]*$/i){
        return 1;
    }
    return 0;
}

sub isRootFsUx{
    return $_[0] eq '/';
}

sub isRootFs;
*isRootFs = $isWin ? \&isRootFsWin : \&isRootFsUx;

#-----------------------------------------------------------------------------

sub isAbsolute{
    my ($path) = @_;
    my $retval = 0;
    if($^O !~ /mswin/i) {
        if($path =~ /^\//) {
            $retval = 1;
        }
    }
    else {
        if($path =~ /^[a-z]:[\\\/]/i || $path =~ /^(\\\\|\/\/)/) {
            $retval = 1;
        }
    }
    return $retval;
}

#----------------------------------------------------------------------------

sub is_nfsidmap_required{
    if ($isWin){
        return 0;
    }
    my ($path) =  @_;
    my @statbuf = defined $path ? stat ($path) : stat (_);
    if (!@statbuf){
        return 0;
    }

    if (ownerIsNobody ($statbuf[4])){
        return 1;
    }

    if (groupIsNobody ($statbuf[5])){
        return 1;
    }
    return 0;
}

sub nfsidmap{
    my ($path, $timeout, $msglst, $errlst, $callback) = @_;

    if ($isWin){
        return undef;
    }

    if (!defined $msglst){
        $msglst = new SDB::Install::Msglst ();
    }

    if (!defined $errlst){
        $errlst = new SDB::Install::Msglst ();
    }

    if (!is_nfsidmap_required ($path)){
        $msglst->addMessage ("nfsidmap is not required for '$path'");
        return 1;
    }

    if (!defined $timeout){
        $timeout = 300; # default timeout 5min
    }

    my $nfsidmap;
    {
        local %ENV = %ENV;
        $ENV{PATH} = join (':', qw (/sbin /usr/sbin /bin /usr/bin), $ENV{PATH});
        $nfsidmap = which ('nfsidmap');
    }

    if (!defined $nfsidmap){
        $msglst->addProgressMessage ("Tool 'nfsidmap' not found, waiting up to 12 minutes for NFS user ID mapping on '$path'");
        $timeout = 720;
    }
    else{
        my $cfg = {};

        my $msg = $msglst->addProgressMessage ("Performing nfsidmap for '$path'...");
        $msg->getSubMsgLst ()->injectIntoConfigHash ($cfg);

        my $rc = exec_program ($nfsidmap, ['-c'], $cfg);

        if (!defined $rc || $rc != 0){
            my $msg = $errlst->addError ("nfsidmap for '$path' failed", $msg->getSubMsgLst);
            $msglst->appendMsg ($msg);
            return undef;
        }
    }

    my $endtime = time() + $timeout;

    while (is_nfsidmap_required ($path)){
        if (time() >= $endtime){
            my $msg = $errlst->addError ("nfsidmap for '$path': timeout reached (${timeout}s).");
            $msglst->appendMsg ($msg);
            $msg = $errlst->addError ("Please make sure, that 'idmapd' is configured properly.");
            $msglst->appendMsg ($msg);
            return 0;
        }
        if (defined $callback){
            eval{
                &$callback ();
            };
            if ($@){
                $msglst->addWarning ("nfsidmap for '$path': callback threw an exception: $@");
                $msglst->addWarning ("=> disable callback");
                $callback = undef;
            }
        }
        usleep (300);
    }
    return 1;
}


sub renameRecursive{
    my ($oldname, $newname, $msglst) = @_;
    if ($oldname eq $newname){
        return 1;
    }
    if (-e $newname){
        $msglst->addError ("target '$newname' already exists");
        return 0;
    }
    my @oldname = File::Spec->splitdir ($oldname);
    my @newname = File::Spec->splitdir ($newname);

    if (@oldname != @newname){
        $msglst->addError ("Cannot rename '$oldname' => '$newname' recusively");
        return 0;
    }

    my @curr;
    my $currPathOld;
    my $currPathNew;
    my @redoList;
    my $error = 0;
    foreach my $i (0..$#oldname){
        if ($oldname[$i] eq $newname[$i]){
            push @curr, $oldname[$i];
            next;
        }
        $currPathOld = File::Spec->join (@curr, $oldname[$i]);
        $currPathNew = File::Spec->join (@curr, $newname[$i]);
        if (!-e $currPathOld){
            if (!-e $currPathNew){
                $msglst->addError("Path '$currPathOld' doesn't exist");
                $error = 1;
                last;
            }
            $msglst->addMessage ("'$currPathOld' => '$currPathNew' is already renamed");
            push @curr, $newname[$i];
            next;
       }
       elsif (-e $currPathNew){
            $msglst->addError("Path '$currPathNew' already exist");
            $error = 1;
            last;
       }
       if (rename ($currPathOld,$currPathNew)){
            $msglst->addMessage ("'$currPathOld' => '$currPathNew' renamed");
            push @redoList, [$currPathOld,$currPathNew];
            push @curr, $newname[$i];
            next;
        }
        $msglst->addError("Cannot rename '$currPathOld' => '$currPathNew': $!");
        $error = 1;
        last;
    }

    if ($error){
        if (@redoList){
            my ($old,$new);
            foreach my $redoEntry (reverse @redoList){
                ($old,$new) = @$redoEntry;
                if(rename ($new,$old)){
                    $msglst->addError ("[REDO ERROR] Cannot rename '$new' => '$old': $!");
                }
                $msglst->addMessage ("[REDO] '$new' => '$old renamed");
            }
        }
        return 0;
    }
    return 1;
}


#-------------------------------------------------------------------------------
# Returns a reference to an array containing the filenames (without parent path)
# of the specified directory.
# In case of an error, a reference to an empty array is returned.

sub getDirectoryFilenames {
    my ($directory) = @_;
    my @filenames;
    if (opendir (DH, $directory)) {
        @filenames = grep {-f $directory . $path_separator . $_} readdir(DH);
        closedir (DH);
    }
    return \@filenames;
}


#-------------------------------------------------------------------------------
# Returns a reference to an array containing the filenames with the layout
# <programName>_<timestamp>_<PID><extension>

sub getFilesTimestampPID {
    my ($directory, $programName, $extension) = @_;
    my @filenames;
    my $pattern = '^' . $programName .
        '.*_\d\d\d\d\-\d\d\-\d\d_\d\d\.\d\d\.\d\d_\d+' . $extension . '$';
    if (opendir (DH, $directory)) {
        @filenames = grep {/$pattern/} readdir(DH);
        closedir (DH);
    }
    return \@filenames;
}

#-------------------------------------------------------------------------------
# Copies specific files of the source directory to the destination directory.

sub copySelectedFiles {

    my ($sourceDir,
        $destinDir,
        $msglst,
        $newPostfix,   # if specified, postfix is added to the filenames
        $oldFilenames, # basenames without path: if specified, these files are not copied
        $wantedUID,
        $wantedGID,
        $hostname,     # if specified, hostname is added if not already contained in the filename
        $pattern,      # if specified, only filenames containing this pattern are copied (without the pattern)
        $skipPattern,  # if specified, filenames containing this pattern are skipped
       ) = @_;

    if (! -d $sourceDir) {
        $msglst->addMessage
            ("Cannot copy new files: source directory '$sourceDir' not found");
        return undef;
    }

    if (! -d $destinDir) {
        $msglst->addMessage
            ("Cannot copy new files: destination directory '$destinDir' not found");
        return undef;
    }

    my $filenames = getDirectoryFilenames($sourceDir);
    my $rc = 1;
    foreach my $scrName (@$filenames) {

        if (defined $oldFilenames && ($scrName ~~ @$oldFilenames)) {
            next;
        }

        if (defined $skipPattern && ($scrName =~ /$skipPattern/)) {
            next;
        }

        my $destName = $scrName;

        if (defined $pattern && !($destName =~ s/$pattern//)) {
            next;
        }

        if (defined $hostname && ($destName !~ /$hostname/)) {
            $destName = $hostname . '_' . $destName;
        }

        if (defined $newPostfix) {
            my $idx       = rindex($destName, '.');
            my $extension = ($idx > 0) ? substr($destName, $idx)    : '';
            my $name      = ($idx > 0) ? substr($destName, 0, $idx) : $destName;
            if ($destName =~ s/\.check$//) {
                $extension = '.check' . $extension;
            }
            $destName = $name . '_' . $newPostfix . $extension;
        }

        my $source      = $sourceDir . $path_separator . $scrName;
        my $destin      = $destinDir . $path_separator . $destName;
        my $copyCfg     = {};
        $copyCfg->{uid} = $wantedUID if (defined $wantedUID);
        $copyCfg->{gid} = $wantedGID if (defined $wantedGID);
        $copyCfg->{ignoreChownError} = 1;
        if (!copy_file($source, $destin, $copyCfg)) {
            $msglst->addMessage("Cannot copy new file '$source' to '$destin'",
                                                                      $copyCfg);
            $rc = undef;
        }
    }
    return $rc;
}

sub _createLibSymlinks{
    my ($links, $msglst, $outlistFailedLinks) = @_;
    my $sslBaseName;
    my $symlinks_created = 0;
    foreach my $link (@$links){
        if (! -f $link->[1] && -f $link->[0]){
            $sslBaseName = basename ($link->[0]);
            $msglst->addMessage ("Creating symlink $link->[1] -> $sslBaseName");
            if (symlink ($sslBaseName, $link->[1])){
                $symlinks_created = 1;
            }
            else{
                $msglst->addError ("Cannot create symlink '$link->[1]': $!");
                if (defined $outlistFailedLinks){
                    push @$outlistFailedLinks, [$sslBaseName, $link->[1]];
                }
            }
        }
    }
    return $symlinks_created;
}


sub _isLibMissing{
    my ($exception, $links) = @_;
    my @rc;
    my $sslBaseName;
    my $symlinks_created = 0;
    my $basename;
    my $missing_pattern;
    foreach my $link (@$links){
        $basename = basename ($link->[1]);
        $missing_pattern = quotemeta ($basename);
        if ($exception =~ /$missing_pattern/){
             push @rc, $basename;
        }
    }
    return @rc;
}


sub loadSSLRequiringPackage{
    my ($package, $msglst, $errlst) = @_;

    #
    # checking existing symbol table
    #
    no strict 'refs';
    if (%{$package . '::'}){
        # package already loaded
        return 1;
    }
    use strict 'refs';

    my $distcompatDir = 'distcompat';

    my @rh7_ssl_links = (['/usr/lib64/libssl.so.10', '/usr/lib64/libssl.so.1.0.0'],
                     ['/usr/lib64/libcrypto.so.10', '/usr/lib64/libcrypto.so.1.0.0']);

    my @rh6_ssl_links = (['/usr/lib64/libssl.so.0.9.8e', '/usr/lib64/libssl.so.0.9.8'],
                     ['/usr/lib64/libcrypto.so.0.9.8e', '/usr/lib64/libcrypto.so.0.9.8']);

    my $absDistcompatDir = $SAPDB::Install::Config{RuntimeDir} . $path_separator . $distcompatDir;

    require SDB::Install::SysInfo;
    my $sysinfo = new SDB::Install::SysInfo ();
    my %saved_INC = %INC;
    if (!$isPPC64 && ($package ne 'Net::SSH2' || !$sysinfo->isSles11())){
        if (-d $absDistcompatDir){
            #
            # try to load extension based on openssl 1.0
            #
            $msglst->addMessage ("Trying to load $package ($distcompatDir)...");
            local @INC = ($absDistcompatDir, grep {/lcm_pm_ext/} @INC);
            eval ("require $package;");
            my $symlinks_created = 0;
            if (!$@){
                # success
                $msglst->addMessage ("$package ($distcompatDir) successfully loaded");
                return 1;
            }
            %INC = %saved_INC;
            my @missingLib = _isLibMissing ($@, \@rh7_ssl_links);
            my @failed_symlinks;
            if (@missingLib){
                $symlinks_created = _createLibSymlinks (\@rh7_ssl_links, $msglst, \@failed_symlinks);
            }
            if ($symlinks_created){
                eval ("require $package;");
                if (!$@){
                    # success
                    $msglst->addMessage ("$package ($distcompatDir) successfully loaded");
                    return 1;
                }
            }
            $msglst->addMessage ("Cannot load $package ($distcompatDir): $@");
            foreach my $symlink (@failed_symlinks){
                $msglst->addError ("Could not create symlink $symlink->[1] => $symlink->[0]");
                $msglst->addError ("Please try to create it by hand: 'ln -s $symlink->[0] $symlink->[1]'");
            }
        }
        else{
            $msglst->addMessage ("Loading openssl: extension directory '$absDistcompatDir' not found");
        }
    }

    #
    # try to load extension based on openssl 0.9.8
    # built on SLES11 on x86_64 and ppc big endian
    #
    # try to load extension based on openssl 1.0.0
    # built on SLES12 on ppc little endian
    #
    $msglst->addMessage ("Trying to load $package...");
    eval ("require $package;");
    if (!$@){
        # success
        $msglst->addMessage ("$package successfully loaded");
        return 1;
    }
    %INC = %saved_INC;
    $msglst->addMessage ("Cannot load $package: $@");

    my $symlinks_created = 0;
    my $symlinks = $isPPC64LE ? \@rh7_ssl_links : \@rh6_ssl_links;
    my @missingLib = _isLibMissing ($@, $symlinks);
    my @failed_symlinks;
    if (@missingLib){
        $symlinks_created = _createLibSymlinks ($symlinks, $msglst, \@failed_symlinks);
    }
    if ($symlinks_created){
        eval ("require $package;");
        if (!$@){
            # success
            $msglst->addMessage ("$package successfully loaded");
            return 1;
        }
        %INC = %saved_INC;
    }
    $msglst->addMessage ("Cannot load $package: $@");
    foreach my $symlink (@failed_symlinks){
        $msglst->addError ("Could not create symlink $symlink->[1] => $symlink->[0]");
        $msglst->addError ("Please try to create it by hand: 'ln -s $symlink->[0] $symlink->[1]'");
    }

    if ($sysinfo->isRedHat ()){
        $errlst->addMessage ("For more information, see SAP Note 2009879.");
    }
    else{
        $errlst->addMessage ("For more information, see SAP Note 1984787.");
    }

    return 0;
}


our $sapDrive;
sub getSAPDrive{
    my ($errMsgLst, $nocache) = @_;
    $nocache //= 0;
    if (defined $sapDrive && !$nocache){
        return $sapDrive;
    }
    $errMsgLst //= new SDB::Install::MsgLst ();
    if (!$isWin){
        $errMsgLst->addError ("Win32::API::ShareInfo(): unsupported platform");
        return undef;
    }
    require SAPDB::Install::System::Win32::API;
    my ($rc,$path) = SAPDB::Install::System::Win32::API::ShareInfo('saploc');
    if($rc != 0){
        $errMsgLst->addError("Could not determine 'saploc' share (rc=$rc)");
        return undef;
    }
    ($sapDrive) = ($path =~ /^([a-z]:)/i);
    return $sapDrive;
}

sub isLink{
    my($path) = @_;
    return (readLink($path) ne $path);
}

1;
