package SDB::Install::System::VCRedistCheck;

use SDB::Install::BaseLegacy;
use strict;
use SDB::Install::System::Win32::Msinst;
use SAPDB::Install::System::Win32::VersionInfo;
use SDB::Install::Version;

our @ISA = qw (SDB::Install::BaseLegacy);


our $vs_pattern = '^Microsoft\sVisual\sC\+\+\s';
our $runtime_pattern = '\sRuntime\s';
our $redistributable_pattern = '\sRedistributable';
our $platform_x86_pattern = 'x86';
our $platform_x64_pattern = 'x64';
our $platform_ia64_pattern = 'ia64';
our $version_2015_pattern = '\s2015\s';
our $version_2013_pattern = '\s2013\s';
our $version_2010_pattern = '\s2010\s';
our $version_2008_pattern = '\s2008\s';
our $version_2005_pattern = '\s2005\s';

#
# search msi database for vs studio c++ runtimes
# creates such a data structure
#
# $self->{runtimes} -> {x86} -> {2005} ->[0] -> {name => <productName>, rawversion=><DWORD>, version=><versionString>}
#                                         .
#                                         .
#                                         .
#                                      ->[n] -> {name => <productName>, rawversion=><DWORD>, version=><versionString>}
#
#                            -> {2008} ->[i] -> [...]
#
#                            -> {2010} ->[i] -> [...]
#
#                   -> {x64} -> {2005} ->[i] -> [...]
#                            -> {2008} ->[i] -> [...]
#                            -> {2010} ->[i] -> [...]
#                            -> {2013} ->[i] -> [...]

sub scan{
    my ($self) = @_;
    my $rc = 0;
    my $productIndex = 0;
    my ($productCode, $productName, $state, $version, $versionString, $vcr_hash);
    my ($platform_hash, $version_list);
    my ($platform_key, $version_key);
    my $msg = $self->AddMessage ("Scanning for installed Visual C++ runtimes");
    $self->{runtimes} = {};
    while (1){
        $rc = MsiEnumProducts ($productIndex++, $productCode);
        if (!defined $productCode){
            $self->AddError ('MsiEnumProducts() API error');
            return undef;
        }
        if ($rc != 0){
            last;
        }
        if (MsiGetProductInfo ($productCode, INSTALLPROPERTY_INSTALLEDPRODUCTNAME, $productName) == 0){
            if ($productName !~ /$vs_pattern/){
                # no visual studio c++ product
                next;
            }
            if ($productName !~ /$runtime_pattern|$redistributable_pattern/){
                # not a runtime or redistributable
                next;
            }
            $state = MsiQueryProductState ($productCode);
            if ($state != INSTALLSTATE_DEFAULT){
                # product is not properly installed
                next;
            }
            $vcr_hash = {};
            $vcr_hash->{name} = $productName;
            $vcr_hash->{code} = $productCode;
            if (MsiGetProductInfo ($productCode, INSTALLPROPERTY_VERSION, $version) == 0){
                $vcr_hash->{rawversion} = $version;
            }
            if (MsiGetProductInfo ($productCode, INSTALLPROPERTY_VERSIONSTRING, $versionString) == 0){
                $vcr_hash->{version} = $versionString;
            }
            if ($productName =~ /$platform_x86_pattern/){
                $platform_key = 'x86';
            }
            elsif ($productName =~ /$platform_x64_pattern/){
                $platform_key = 'x64';
            }
            elsif ($productName =~ /$platform_ia64_pattern/){
                $platform_key = 'ia64';
            }
            elsif ($productName =~ /$version_2005_pattern/){
                # vs 2005 x86 has no platform identifier in productName
                $platform_key = 'x86';
                # add platform id to the msglist entry
                $productName .=  ' (x86)';
            }
            else{
                next;
            }
            if (!defined $self->{runtimes}->{$platform_key}){
                $self->{runtimes}->{$platform_key} = {};
            }
            $platform_hash = $self->{runtimes}->{$platform_key};
            if ($productName =~ /$version_2005_pattern/){
                $version_key = '2005';
                # vs 2005 has no version in productName
                # => add version string to the msglist entry
                $productName .=  " - $versionString";
            }
            elsif ($productName =~ /$version_2008_pattern/){
                $version_key = '2008';
            }
            elsif ($productName =~ /$version_2010_pattern/){
                $version_key = '2010';
            }
            elsif ($productName =~ /$version_2013_pattern/){
                $version_key = '2013';
            }
            elsif ($productName =~ /$version_2015_pattern/){
                $version_key = '2015';
            }

            else{
                next;
            }
            if (!defined $platform_hash->{$version_key}){
                $platform_hash->{$version_key} = [];
            }

            $self->AddSubMessage ($msg, "Found $productName");

            $version_list = $platform_hash->{$version_key};
            push @$version_list, $vcr_hash;
        }
    }
    return $productIndex;
}


#
# method gets a file name of a redistributable as parameter
# it returns true if a proper version is already installed.
# it returns false if there is no version, or an older version installed
#

our $hdb_vcredist_pattern = '^msdev(\d{4}).*vcredist_(\w+)\.';

sub checkRedistributable{
    my ($self, $file) = @_;
    my $archive_version = SAPDB::Install::System::Win32::VersionInfo::SysVersionInfo ($file);
    my ($basename) = ($file =~ /([^\\\/]+)$/);
    my ($studio_version, $platform) = ($basename =~ /$hdb_vcredist_pattern/i);
    $self->AddMessage ("Found Microsoft C/C++ runtime redistributable '$basename'" . (defined $archive_version ? " ($archive_version)" : ''));

    if ($studio_version eq '2005'){
        $self->AddMessage ("Skipping '$basename'");
        return 1;
    }

    $platform = lc ($platform);
    if (!defined  $self->{runtimes}->{$platform} || ! defined $self->{runtimes}->{$platform}->{$studio_version}){
        $self->AddMessage ("No runtime for Visual $studio_version $platform installed");
        return 0;
    }

    my $version = new SDB::Install::Version ($archive_version);
    foreach my $vcr (@{$self->{runtimes}->{$platform}->{$studio_version}}){
        if ($version->isNewerThan (new SDB::Install::Version ($vcr->{version}))){
            $self->AddMessage ("$vcr->{name} is out-dated");
            next;
        }
        $self->AddMessage ("$vcr->{name} is already installed");
        return 1;
    }
    $self->AddMessage ("'$basename' has to be installed");
    return 0;
}

1;
