#!/usr/bin/perl
#
# $Header$
# $DateTime$
# $Change$
#
# Desc: 

package SDB::Install::Tools;

use strict;
use SDB::Install::SysVars qw ($isWin $path_separator);

require Exporter;
our @ISA = qw (Exporter);

our @EXPORT = qw (
	arrayToHash
	askConfirmation
    callstack
	divideIntegrally
    isMountPoint
	iso8601DateTime
    ltrim
    print_callstack
    printSortedTableToArrayOfLines
    printTableToArrayOfLines
    readini
    release2num
    longVersionRegex
    standardVersionRegex
    shortVersionRegex
	rtrim
    trim
    repeat
	URLencode
	escapeCommandLineArgument
    generatePassword
	getFilenameTimeStamp
	getFilenameTimeStampPattern
    formatSecondsIntoHhMmSs
);

#----------------------------------------------------------------------------
our $cor_digit_num = 5;
our $patch_digit_num = 10;
#revMajor.revMinor.revNumber.revPatchLevel.revChangelist
our $longVersionRegex = qr/^(\d)+\.(\d+)\.(\d+)\.(\d+)\.(\d+)$/;
#revMajor.revNumber.revPatchLevel.revChangelist
our $standardVersionRegex = qr/^(\d)+\.(\d+)\.(\d+)\.(\d+)$/;
#revMajor.revNumber.revChangelist
our $shortVersionRegex = qr/^(\d)+\.(\d+)\.(\d+)$/;

sub release2num{
    #sub return a normalized decimal string to compare release strings
    my $version=$_[0]; # release in format like this 7.2.4.9 (without leading nulls)
	#if(/^MSG\s\d+\.\d/){
	#	# found version of MaxDB messages
	#	my ($seq) = (/^MSG\s\d+\.(\d+)/); 
	#	return  $seq;
	#}
    my $releaseMajor=undef;
    my $releaseMinor=00;
    my $revNumber= undef;
    my $revPatch=00;
    my $revChnage=undef;
    if( $version =~ m/$longVersionRegex/ ){
        $releaseMajor=$1;
        $releaseMinor=$2;
        $revNumber=$3;
        $revPatch=$4;    
        $revChnage=$5;	
    } elsif ( $version =~ m/$standardVersionRegex/ ){
        $releaseMajor=$1;
        $revNumber=$2;
        $revPatch=$3;    
        $revChnage=$4;  
    } elsif ( $version =~ m/$shortVersionRegex/ ){
        $releaseMajor=$1;
        $revNumber=$2;
        $revChnage=$3;  
    } else{
    	return undef;
    }
		
	if (length ($releaseMajor) > 3){
		return undef;
	}

	if (length ($releaseMinor) > 3) {
		return undef;
	}
	
	if (length ($revNumber) > 3){
		return undef;
    }
	
	if($cor_digit_num < length($revPatch)){
		return undef;
	}
	
	if ($patch_digit_num < length($revChnage)){
		return undef;
	}

	return sprintf ("%03d%03d%03d%0${cor_digit_num}d%0${patch_digit_num}d",
			int ($releaseMajor), int($releaseMinor), int ($revNumber), int ($revPatch), int ($revChnage));
}

sub readini{
	# sub return such a hash : $hash{$SECTION}{$KEY}=$VALUE;
	# $SECTION - in ini file something like this: [my section]
	# $KEY - in ini file  something like this: my key = xxxxx
	# $VALUE - in ini file something like this: xxxx = my value
	my $file=$_[0]; # full path to ini file
	my $msglst = $_[1];
	my $multipleValues = $_[2];
	if (!defined $multipleValues){
		$multipleValues = 0;
	}
	my %returnvalue;
	unless(-f $file){
		if ($msglst){
			$msglst->AddError ("File \"$file\" not found");
		}
		return undef;
	}
	unless (open(INI,$file)){
		if ($msglst){
                        $msglst->AddError ("Cannot open file \"$file\": $!");
                }
		return undef;
	}
	my $section = '';
    my ($key,$value);
	while(my $line=<INI>){
		chomp($line);
        trim (\$line);
		$_=$line;
        next if (/^#/);
		if (/^\[(.*)\]$/){
            $section = $1;
            trim (\$section);
            next;
        }
		($key,$value) = (/([^=]*)=(.*)/);
        if (!$key){
            next;
        }
        rtrim (\$key);
        ltrim (\$value);
        if ($multipleValues){
            if (defined $returnvalue{$section}{$key}){
                unshift @{$returnvalue{$section}{$key}}, $value;
            }
            else{
                $returnvalue{$section}{$key} = [$value];
            }
        }
        else{
            $returnvalue{$section}{$key}=$value;
        }
	}
	close(INI);
	unless(%returnvalue){
		return undef;
	}
	return \%returnvalue;
}

#----------------------------------------------------------------------------

=functionComment

'arrayToHash' converts an array to a hash where all keys are equal to
their associated values and the array of the keys 
(hence the array of values as well) equals the input array 
(possibly mod permutations).
Doing this is useful to avoid array scans when merging lists (given as arrays),
and duplicate entries must be eliminated.

=cut
sub arrayToHash {
    my ($arryRef) = @_;
    my %retval;
    foreach my $entry (@$arryRef) {
        $retval{$entry} = $entry;
    }
    return \%retval;
}

#----------------------------------------------------------------------------

=functionComment

Sorts the sub array of a two-dimensional array '$rrows'
by the specified column '$sortIdx'.
Calls printTableToArrayOfLines to create the sorted table.
The individual fields of a row are considered to be separated with '$separator'.
On success, the function returns a string containing the table
(w/o trailing newline) and a returncode of 1.

If the array parameter '$colLimits' is specified, the alignment of
each column can be limited (0 - without limit). Additional lines are created
if colunmns exceed the limit.

If the parameter '$indentPrefix' is specified, each line including overflow
lines is prefixed with this string.

If the parameter '$constraintIdx' is specified, rows with a defined constraint
(has to be the last column) are grouped under the contraint name
after the normal output without constraint.


On error, '(undef, 0)' is returned.

=cut
sub printSortedTableToArrayOfLines{

    my ($rrows, $sortIdx, $separator, $colLimits, $indentPrefix, $constraintIdx) = @_;

    my %constraints;
    my %hashRows;
    my @sortedRows;

    foreach my $currRow (@$rrows) {

        if (defined $constraintIdx && scalar (@$currRow) == $constraintIdx+1) {

            my $constr = pop @$currRow;

            if (defined $constr) {
                $constraints{$constr}->{$currRow->[$sortIdx]} = $currRow;
            }
            else {
                $hashRows{$currRow->[$sortIdx]} = $currRow;
            }
        }
        else {
            $hashRows{$currRow->[$sortIdx]} = $currRow;
        }
    }

    foreach my $key (sort keys %hashRows) {
        push @sortedRows, $hashRows{$key};
    }

    my $result = printTableToArrayOfLines(\@sortedRows,
                                          $separator,
                                          0,
                                          $colLimits,
                                          $indentPrefix);

    foreach my $currConstraint (sort keys(%constraints)) {
        push @$result, '';
        push @$result, "$currConstraint:";
        @sortedRows        = ();
        my $constraintRows = $constraints{$currConstraint};

        foreach my $rowKey (sort keys (%$constraintRows)) {
            push @sortedRows, $constraintRows->{$rowKey};
        }

        my $sectionResult = printTableToArrayOfLines(\@sortedRows,
                                                     $separator,
                                                     0,
                                                     $colLimits,
                                                     $indentPrefix);
        $result = [@$result, @$sectionResult];
    }

    return $result
}


#----------------------------------------------------------------------------

=functionComment

Takes a reference '$rrows' to a two-dimensional array of cells to be formatted
into a table which is returned as an array strings, one for each row (w/o trailing newline).
The individual fields of a row are considered to be separated with '$separator'.
On success, the function returns a string containing the table (w/o trailing newline)
and a returncode of 1.

If the array parameter '$colLimits' is specified, the alignment of
each column can be limited (0 - without limit). Additional lines are created
if colunmns exceed the limit.

If the parameter '$indentPrefix' is specified, each line including overflow
lines is prefixed with this string.

=cut
sub printTableToArrayOfLines{
    my (
        $rrows,
        $separator, # column separator: e.g. '  ' (2 blanks) or ' | '
        $with_head,
        $colLimits,
        $indentPrefix,
        $isSpaceBetweenRows, # e.g. "\n"
    ) = @_;

    my $retval = undef;
    defined $rrows and ref($rrows) eq 'ARRAY'  or return $retval;
    my @rows = @$rrows;
    my @width; # array of column widths

    $separator       = '' if (!defined $separator);
    my $separatorLen = length($separator);

    # find column widths
    foreach (@rows){
        if(!defined $_) {
            next;
        }
        ref eq 'ARRAY' or return $retval;
        my @columns  = @$_;
        my $colIndex = 0;
        my $subColLen;
        my $len;
        foreach my $col (@columns){

            $width[$colIndex] = 0 if (!defined $width[$colIndex]);

            if (!defined $col) {
                $colIndex++;
                next;
            }

            my $currLimit = (defined $colLimits) ? $colLimits->[$colIndex] : 0;
            $currLimit    = 0 if (!defined $currLimit);

            my $nextLimit = (defined $colLimits) ? $colLimits->[$colIndex+1]: 0;
            $nextLimit    = 0 if (!defined $nextLimit);

            my $nextWidth = $width[$colIndex+1];
            $nextWidth    = 0 if (!defined $nextWidth);

            $len = 0;
            foreach my $subCol (split("\n", $col)) {
                $subColLen = length($subCol);
                $len = $subColLen if ($subColLen > $len);
            }

            my $limitLen = ($currLimit == 0)
                           ? $len
                           : ($len <= $currLimit) ? $len : $currLimit;

            if (!defined $width[$colIndex] || ($limitLen > $width[$colIndex])) {

                $width[$colIndex] = $limitLen;
            }
            if (($colIndex == 0)
                    && ($currLimit > 0) && ($len > $currLimit)
                    && defined $colLimits
                    && ($len < $currLimit + $nextLimit)
                    && ($len > $width[$colIndex] + $nextWidth)) {

                my $lenCurrCol = ($len - $nextWidth > $currLimit)
                              ? $currLimit
                              : $len - $nextWidth;

                my $lenNextCol = $len - $lenCurrCol;

                $width[$colIndex  ] = $lenCurrCol if ($lenCurrCol > $width[$colIndex]);
                $width[$colIndex+1] = $lenNextCol if ($lenNextCol > $nextWidth);
            }
            $colIndex++;
        }
    }

    my $limitLastCol = 0;
    my $lineEmptyCols= '';

        # set width to limit, if width is not set before
        my $colIndex = 0;
        foreach my $currWidth (@width) {
            if (!defined $currWidth) {
                $width[$colIndex] = (defined $colLimits)
                                    ? @$colLimits[$colIndex] : 1;
            }
            if ($colIndex != $#width) {
                $lineEmptyCols .= (' ' x ($width[$colIndex])) . $separator;
            }
            $colIndex++;
        }
        $limitLastCol = (defined $colLimits) ? @$colLimits[$#width] : 0;

    my $rowlen;
    if ($with_head){
        foreach my $currWidth (@width){
            $rowlen += $currWidth;
        }
        $rowlen += ($#width * $separatorLen);
    }
    # print table
    $retval = [];
    my $row = 0;

    foreach (@rows){

        my @columns     = @$_;
        my $colIndex    = 0;
        my $expectedLen = 0;
        my $lineLen     = 0;
        my $printBuf    = (defined $isSpaceBetweenRows && ($row > 0))
                          ? $lineEmptyCols . "\n" : '';
        $printBuf      .= $indentPrefix if (defined $indentPrefix);

        foreach my $col (@columns) {

            my $colLen    = (defined $col) ? length($col) : 0;

            if ($colLen > 0) {

                if ($expectedLen < $lineLen) {
                   push @$retval, $printBuf;
                   $printBuf  = (defined $indentPrefix)
                                ? $indentPrefix . $lineEmptyCols
                                : $lineEmptyCols;
                   $lineLen   = $expectedLen;
                }
            }

            if ($colIndex != $#columns) {
                # not last column
                $printBuf    .= $col if ($colLen > 0);
                $lineLen     += $colLen;
                $expectedLen += $width[$colIndex];
                my $fillLen   = $expectedLen - $lineLen;

                if ($fillLen > 0) {
                    $printBuf .= ' ' x ($fillLen);
                    $lineLen  += $fillLen;
                }

                if ($expectedLen == $lineLen || $colLen > 0) {

                    # do not add a separator to an empty col in case of overflow
                    $printBuf .= $separator;
                    $lineLen  += $separatorLen;
                }
                $expectedLen += $separatorLen;
            }
            else {
                # print last column (may split over several lines)
                # last column may contain additional lines

                my $lineFeedIdx      = index ($col, "\n", 0);
                my $addedLines       = undef;
                my $splitDefaultList = 0;

                if ($lineFeedIdx >= 0) {
                    $addedLines = substr($col, $lineFeedIdx+1);
                    $col        = substr($col, 0, $lineFeedIdx);
                    $colLen     = length($col);
                }

                while (($limitLastCol > 0) && ($colLen > $limitLastCol)) {
                    # last column (without added lines) exceeds limit

                    my $truncIdx = rindex($col, ' ', $limitLastCol);
                    my $truncLen = $truncIdx;

                    if (($truncIdx <= 0) && $splitDefaultList) {
                        $truncIdx = rindex($col, ',', $limitLastCol);
                        $truncLen = $truncIdx+1;
                    }

                    if (($truncIdx > 0) &&
                        ($truncIdx+2 < $colLen)) { # +2 in case of trailing blank

                        if (!$splitDefaultList) {

                            my $truncCol   = substr($col, 0, $truncIdx);
                            my ($foundStr) = $truncCol =~
                                                 /.+\[interactive, (default:)$/;

                            if (!defined $foundStr) {
                                ($foundStr) = $truncCol =~
                                   /.+(\[default|\[default value:|\[default:)$/;
                            }

                            if (defined $foundStr) {

                                $truncIdx = rindex($col, ' ',
                                                 $truncIdx - length($foundStr));
                                $truncLen = $truncIdx;
                                if ($truncIdx == 0) {
                                    last;
                                }

                                my $defaultValue = substr($col,
                                             $truncIdx + 2 + length($foundStr));

                                if ((length($defaultValue) > $limitLastCol)
                                    && (rindex($defaultValue, ',') > 0)) {
                                    $splitDefaultList = 1;
                                }
                            }
                        }

                        $printBuf .= substr($col, 0, $truncLen);
                        push @$retval, $printBuf;
                        $printBuf  = (defined $indentPrefix)
                                     ? $indentPrefix . $lineEmptyCols
                                     : $lineEmptyCols;
                        $col       = substr($col, $truncIdx+1);
                        $colLen    = length($col);
                    }
                    else {
                        last;
                    }
                }

                $printBuf .= $col if ($colLen > 0);

                if (defined $addedLines) {

                    my $off = 0;
                    my $len = length($addedLines);

                    while ($off < $len) {
                        my $endIdx = index ($addedLines, "\n", $off);
                        $endIdx    = $len if ($endIdx < 0);
                        push @$retval, $printBuf;
                        $printBuf  = (defined $indentPrefix)
                                     ? $indentPrefix . $lineEmptyCols
                                     : $lineEmptyCols;
                        $printBuf .= substr($addedLines, $off, $endIdx-$off);
                        $off = $endIdx+1;
                    }
                }
            }
            $colIndex++;
        }
        push @$retval, ($printBuf);
        if ($row == 0 && $with_head){
            push @$retval, "-" x $rowlen;
        }
        $row++;
    }

    return $retval;
}

#----------------------------------------------------------------------------

=functionComment

Unfortunately, there is no simple and implementation independent way to perform
integer divisions in perl (with "use integer", it is _not_guaranteed_ that the
interpreter actually maps this to integer division in C, according to perldoc).
Only the '%' operator is a genuine integer operator in perl.
For the result, the relation 

    '$numerator == divideIntegrally($numerator, $denominator) * $denominator + $numerator % $denominator'

always holds, and 

    '$numerator % $denominator' is negative iff '$denominator' is negative and the division has a nonzero remainder.

(where '%' is the mod operator of perl, _not_  the mod operator of C
 (which one would get with "use integer")).

=cut
sub divideIntegrally {
    my (
        $numerator,
        $denominator
    ) = @_;
    my $retval = undef;
    if($denominator != 0) {
        $retval = ($numerator - $numerator % $denominator) / $denominator;
    }
    return $retval;
}

#----------------------------------------------------------------------------

=functionComment

returns a String containing the current local datetime with offset to UTC.
The Format is as given by ISO 8601.
example :  
'2011-04-19T11:25:42+0200' is a local datetime in timezone CEST.

TODO: cache the offset to UTC before using this function with high frequency,
e.g. for log file decoration.

=cut
sub iso8601DateTime {
        my ($time) = @_;
        my ($Lsec, $Lmin, $Lhour, $Lmday, $Lmon, $Lyear);
        my ($Gsec, $Gmin, $Ghour);
        if(defined $time) {
            ($Lsec, $Lmin, $Lhour, $Lmday, $Lmon, $Lyear, undef, undef, undef) = localtime($time);
            ($Gsec, $Gmin, $Ghour, undef,  undef, undef,  undef, undef, undef) = gmtime($time);
        }
        else {
            # localtime() != localtime(undef)  !!!!
            ($Lsec, $Lmin, $Lhour, $Lmday, $Lmon, $Lyear, undef, undef, undef) = localtime();
            ($Gsec, $Gmin, $Ghour, undef,  undef, undef,  undef, undef, undef) = gmtime();
        }
        if($Gsec < $Lsec) {
            # seconds wraparound occurred between the two time function calls
            # (we also assume that the lapse is always < 60s)
            $Gmin -= 1;
            if($Gmin == -1) {
                $Gmin = 59;
                $Ghour -=1;
                if($Ghour == -1) {
                    $Ghour = 23;
                }
            }
        }
        my $OffMin = ($Ghour-$Lhour)*60+($Gmin-$Lmin);
        my $OffSgn = '-';
        if($OffMin <= 0) {
            $OffMin *= -1;
            $OffSgn = '+';
        }
        my $OffHour = divideIntegrally($OffMin, 60);
        $OffMin = $OffMin%60;;
        my $retval = sprintf("%4d-%02d-%02dT%02d:%02d:%02d$OffSgn%02d%02d",
                                      $Lyear+1900, $Lmon+1, $Lmday, $Lhour, $Lmin, $Lsec, $OffHour, $OffMin);
        return $retval;
}

#----------------------------------------------------------------------------

sub getFilenameTimeStamp {
	# changes in the returned timestamp format must be reflected in getFilenameTimeStampPattern()
	my ($time) = @_;
	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(defined $time ? $time : time ());
	$mon++;
	$year += 1900;
	return sprintf ("%d-%02d-%02d_%02d.%02d.%02d", $year, $mon, $mday, $hour, $min, $sec);
}

#----------------------------------------------------------------------------

sub getFilenameTimeStampPattern {
	return '\d{4}-\d{2}-\d{2}_\d{2}.\d{2}.\d{2}';
}

#----------------------------------------------------------------------------

sub callstack{
    my ($msglst, $level) = @_;
    if (!defined $level){
        $level = 1;
    }
    my @frames;
    my @context;
    my $i = 0;
    while (@context = caller ($level++)){
        if (defined $msglst){
            $msglst->AddMessage (($i++) . ": $context[3] in source $context[1] line $context[2]");
        }
        push @frames, \@context;
    }
    return \@frames;
}

#----------------------------------------------------------------------------

sub print_callstack{
	my $msglst = new SDB::Install::MsgLst ();
	callstack ($msglst,2);
	print ${$msglst->GetMsgListString()} . "\n";
}

#----------------------------------------------------------------------------

sub ltrim ($){
   ${$_[0]} =~ s/^\s*//;
}

#--------------------------------------------------------------------------------------------------

sub rtrim ($){
   ${$_[0]} =~ s/\s*$//;
}

#--------------------------------------------------------------------------------------------------

sub trim ($){
    ltrim ($_[0]);
    rtrim ($_[0]);
}

#--------------------------------------------------------------------------------------------------

sub repeat {
    my ($function, $n, $operand) = @_;
    return $n == 0 ? $operand : repeat($function, $n - 1, $function->($operand));
}

#--------------------------------------------------------------------------------------------------

sub URLencode {
    my (
        $buf
    ) = @_;
    if(not defined $buf) {
        return undef;
    }
    my $retval = $buf;
    $retval =~ s/([^a-zA-Z0-9_.-\/])/uc sprintf("%%%02x",ord($1))/eg;
    return $retval;
}

#--------------------------------------------------------------------------------------------------
# Returns 1 if the specified directory is a mount point
# otherwise 0 or undef if the directory does not exist.

sub isMountPoint{
    my ($dir)   = @_;
    my @statbuf = stat ($dir);
    if (!@statbuf){
        return undef;
    }
    # stat of '..'
    my @parentStatbuf = stat ($dir . $path_separator . '..');
    if (!@parentStatbuf){
        return undef;
    }
    # compare device id
    if ($statbuf[0] != $parentStatbuf[0]){
        return 1;
    }
    return 0;
}


#-------------------------------------------------------------------------------
# Returns 1 if the user wants to continue. Uses standard in/output.

sub askConfirmation {
    my ($question, $default) = @_;
    $question = 'Do you want to continue?' if (!defined $question);
    print $question . ' (y/n)' . ( defined $default ? ' [' . $default . ']' : '' ) . ': ';
    my $answer = undef;
    while () {
        $answer = <STDIN>;
        if ($isWin){
            local $/ = "\r\n";
            chomp ($answer);
        }
        chomp ($answer);
        $answer = lc $answer;
        if ( $answer eq "y" or ( $answer eq "" and $default eq "y" )) {
            return 1;
        }
        elsif ( $answer eq "n" or ( $answer eq "" and $default eq "n" )) {
            return 0;
        }
        else {            
            print "Invalid input \"" . $answer . "\". Please type \"y\" or \"n\": ";
        }
    }
}

sub escapeCommandLineArgument{
	my($arg) = @_;
	if(!defined $arg){
	   return undef;
	}
	my $escapedValue = $arg;
    $escapedValue =~ s/"/\\"/g;
    $escapedValue = "\"$escapedValue\"";
	return $escapedValue;
}

sub generatePassword{
    my ($length, $alphabet) = @_;
    if (!defined $alphabet){
        $alphabet = ['a'..'z','A'..'Z',0..9];
    }
    else{
        if (ref($alphabet) ne 'ARRAY'){
            return '';
        }
    }
    if (!defined $length){
        $length = 8;
    }
    return join ('', map {$alphabet->[rand @$alphabet]} 1..$length);
}

sub formatSecondsIntoHhMmSs {
    my ($seconds) = @_;
    return undef if (int($seconds) < 0);
    my $hours = int ($seconds / 3600);
    my $rest = $seconds % 3600;
    my $minutes = int ($rest / 60);
    $rest = $rest % 60;
    return sprintf ("%02d:%02d:%02d", $hours,$minutes,$rest);
}

1;
