package SDB::Install::Utils::DatabaseStudio::DatabaseStudioFeatureDetector;
use SDB::Install::SysVars;
use SDB::Install::System qw ( makedir copy_file );
use SDB::Install::SysVars   qw( $isWin $isApple );
use SDB::Install::Archive;

use File::Spec;
use File::Basename qw (dirname);

use strict;

our @ISA = qw (SDB::Install::BaseLegacy Exporter);
our @EXPORT = qw(LCM_FEATURE_ID NEW_PRODUCT_ID OLD_PRODUCT_ID);

sub LCM_FEATURE_ID  {'com.sap.ndb.studio.lcm.feature.feature.group'};
sub NEW_PRODUCT_ID  {'com.sap.ndb.studio.product.base'};
sub OLD_PRODUCT_ID  {'com.sap.ndb.studio.product'};

our $EQUINOX_APP_ID = 'org.eclipse.equinox.frameworkadmin.equinox';
our $REQUIRED_EQUINOX_VERSION_FOR_APPLE_APP_LAYOUT = '1.0.500.v20150304-1709';

sub new {
	my ( $package, $kit, $path, $repository, $javaVm ) = @_;
	my $self = {};
	bless( $self, $package );
	$self->{kit} = $kit;
	$self->{path} = $path;
	$self->{repository} = $repository;
	$self->{javaVm} = $javaVm;	
	return $self;
}

sub getRepositoryFeatures {
	my ( $self ) = @_;
	return $self->{repositoryFeatures};
}

sub getRepositoryFeatureIds {
	my ( $self ) = @_;
	my $result   = [];
	my $features = $self->{repositoryFeatures};
	foreach my $feature ( @$features ) {
		push @$result, $feature->{id};
	}
	return $result;
}

sub getRepositoryFeatureById {
	my ( $self, $featureId) = @_;
	my $features = $self->{repositoryFeatures};
	foreach my $feature ( @$features ) {
		if($feature->{id} eq $featureId){
			return $feature;
		}
	}
	return undef;
}

sub isInstalledFeature {
    my ($self,$featureId) = @_;
	return defined $self->getInstalledFeatureById($featureId)
}

sub getInstalledFeatureById {
    my ($self, $featureId) = @_;
    my $installedFeatures = $self->{installedFeatures};
    foreach my $feature (@$installedFeatures) {
    	if ($featureId eq $feature->{id}) {
    		return $feature;
    	}
    }
    return undef;
}

sub isInRepository {
	my ( $self, $featureId ) = @_;
	my $repositoryFeatures = $self->getRepositoryFeatures();
	foreach my $feature (@$repositoryFeatures) {
		if ( $featureId eq $feature->{id} ) {
			return 1;
		}
	}
	
	my $repositoryProducts = $self->{repositoryProducts};
	foreach my $product (@$repositoryProducts) {
		if ( $featureId eq $product->{id} ) {
			return 1;
		}
	}
	return 0;
}

sub getNewestRepositoryProductFeature {
	my ( $self ) = @_;
	my $productId = $self->getNewestProductIdInRepository();
	my $repositoryProducts = $self->{repositoryProducts};
	foreach my $product (@$repositoryProducts) {
		if ( $productId eq $product->{id} ) {
			return $product;
		}
	}
	return undef;
}

sub isOldFeaturesSetInRepository {
	my ($self) = @_;
	#if LCM feature is in repo, the repo is rev <70, the features are not split
	#if LCM feature is not in repo, it is rev 70 or 71, where the features are spti but still contained in the old product and product.base is not extracted
	return $self->isInRepository(LCM_FEATURE_ID) || ( ! $self->isInRepository(NEW_PRODUCT_ID) );
}

sub isOldFeaturesSetInstalled {
	my ($self) = @_;
	#Check weather the new product id is iinstalled because of a a p2 director issue: When commanded to uninstall the LCM Feature, along with replacing the studio product with product base + separate features, it fails to uninstall the LCM feature
	return $self->isInstalledFeature(LCM_FEATURE_ID) && ( ! $self->isInstalledFeature(NEW_PRODUCT_ID) );
}


sub repositoryCanAppleAppLayout {
    my ($self) = @_;
    if (!$isApple){
        return 0;
    }
    if (defined $self->{_repositoryCanAppleAppLayout}){
        return $self->{_repositoryCanAppleAppLayout};
    }
    my $errlst = new SDB::Install::MsgLst ();
    my $tmp_dir = $self->extractTmpDirector ('/var/tmp', $errlst);
    if (!defined $tmp_dir){
        return undef;
    }
    my $repository = $self->{repository};
    my $kit = $self->getKit ();
    my $cfg = {};
    my $out;
    $cfg->{out} = \$out;
    my $javaVm = $self->_getJavaVm();
    my $args = [
        ($javaVm ? ('-vm', $javaVm) : ()),
        '-repository', "$repository",
        '-list', $EQUINOX_APP_ID
    ];

    my $rc = $kit->callDirector( $tmp_dir, $args, $errlst, $cfg );

    my @lines = split /\n/, $out;
    my $newLines = _removeUnnecessaryLines(\@lines);

    my $pattern = quotemeta ($EQUINOX_APP_ID);
    my $equinoxVersionString;

    foreach my $line (@lines){
        if ($line !~ /$pattern/){
            next;
        }
        $equinoxVersionString = (split ('=', $line))[1];
        if ($equinoxVersionString){
            last;
        }
    }

    $self->{_repositoryCanAppleAppLayout} = undef;

    if (defined $equinoxVersionString){
        require SDB::Install::Version;
        my $equinoxVersion = new SDB::Install::Version (split('\.', $equinoxVersionString));
        my $checkVersion = new SDB::Install::Version (
            split ('\.',$REQUIRED_EQUINOX_VERSION_FOR_APPLE_APP_LAYOUT));
        $self->{_repositoryCanAppleAppLayout} =
            !$checkVersion->isNewerThan ($equinoxVersion);
    }

    $self->AddMessage( "Deleting directory $tmp_dir..." );
    if ( ! SDB::Install::System::deltree ( $tmp_dir ) ) {
        $self->AddWarning ( "Cannot delete directory $tmp_dir: $!" );
    }
    return $self->{_repositoryCanAppleAppLayout};
}

sub isOldProductInstalled {
    my ($self) = @_;
    return $self->isInstalledFeature(OLD_PRODUCT_ID);
}

sub getNewestProductIdInRepository {
	my ($self) = @_;
	if($self->isOldFeaturesSetInRepository()){
		return OLD_PRODUCT_ID;
	}
	if($self->isInRepository(NEW_PRODUCT_ID)){
		return NEW_PRODUCT_ID;
	}
#	in case of revision 70, where new features with old product id coexist
	return OLD_PRODUCT_ID;
}

#todo rename to getInstalledProductID
sub getInstalledProduct {
   	my ($self) = @_;
   	if ($self->isInstalledFeature(OLD_PRODUCT_ID)) {
   			return OLD_PRODUCT_ID;
   	}
    return NEW_PRODUCT_ID;    	
}


sub extractTmpDirector{
    my ($self, $path, $errlst) = @_;
    my $kit = $self->getKit ();

    my $archivePath = $kit->{archive_dir} . $path_separator . 'DIRECTOR.TGZ';
    my $archive = new SDB::Install::Archive ( $archivePath );

    my ( $sec, $min, $hour, $mday, $mon, $year, $wday, $yday, $isdst ) = localtime( time );

    my $now = sprintf( "%d.%02d.%02d_%02d.%02d.%02d", $year + 1900, $mon + 1, $mday, $hour, $min, $sec );
    my $tmp_dir = $path . $path_separator . 'tmp_p2director' . $now;

    if ( !makedir ( $tmp_dir ) ) {
        $self->PushError ( "Cannot create temporary p2 director directory:" . $tmp_dir );
        return undef;
    }

    $self->AddMessage( "Extracting '$archivePath'..." );
    my $rc = $archive->Extract ( $tmp_dir );

    if ( !$rc ) {
        $self->PushError ( "Cannot retrieve features from repository", $errlst );
        return undef;
    }

    if( $isWin ) {
        my $instDir = dirname(new SDB::Install::Installer()->GetRuntimeDir());
        my $msvcr100File = File::Spec->catfile($instDir,'msvcr100.dll');
        if(-e $msvcr100File){
            my $targetMsvcr100File = File::Spec->catfile($tmp_dir,'director','msvcr100.dll');
            copy_file($msvcr100File,$targetMsvcr100File);
        }
    }
    return $tmp_dir;
}

sub retrieveFeaturesFromNewRepository {
	my ( $self) = @_;
	my $errlst = new SDB::Install::MsgLst ();
	my $tmp_dir = $self->extractTmpDirector ($self->{path}, $errlst);
	if (!defined $tmp_dir){
		return undef;
	}
	my $repository = $self->{repository};
	$self->AddMessage( "Searching for repository features..." );
	if( ! $self->findRepositoryFeatures($tmp_dir, $errlst) ){
		return undef;
	}
	
	$self->AddMessage( "Searching for repository products..." );
	if( ! $self->findRepositoryProducts($tmp_dir, $errlst) ){
		return undef;
	}
	
	$self->AddMessage( "Deleting directory $tmp_dir..." );
	if ( ! SDB::Install::System::deltree ( $tmp_dir ) ) {
		$self->AddWarning ( "Cannot delete directory $tmp_dir: $!" );
	}
	
	return 1;
}

sub findRepositoryFeatures{
	my ( $self, $tmp_dir, $errlst) = @_ ;
	my $kit = $self->getKit ();
	my $repository = $self->{repository};
	my $cfg = {};
	my $out;
	$cfg->{out} = \$out;
	my $query = "Q:everything.select(y | everything.select(x | x.properties ~= filter(\'(org.eclipse.equinox.p2.type.category=true)\')).collect(x | x.requirements).flatten().exists(r | y ~= r))";
	my $javaVm = $self->_getJavaVm();
	my $getIdVersionArgs = [ ($javaVm ? ('-vm', $javaVm) : ()), '-list', $query, '-repository', "$repository", ];

	# Get id, version and description of installable units. 
	# Works on P2 Director version from SP8 and later
	my @getIdVersionDescriptionArgs = (@$getIdVersionArgs, '-listFormat' , '${id}=${version}=${org.eclipse.equinox.p2.name}');

	my $rc = $kit->callDirector( $tmp_dir, \@getIdVersionDescriptionArgs, $errlst, $cfg );
	if ( !defined $rc || $rc !=0 ) {
		$rc = $kit->callDirector( $tmp_dir, $getIdVersionArgs, $errlst, $cfg );
		if ( !defined $rc || $rc !=0 ) {
			$self->PushError ( "Eclipse Equinox Director failed", $errlst );
			return undef;
		}
	}

	my @lines = split /\n/, $out;
	my $newLines = _removeUnnecessaryLines(\@lines);
	my $features = [];

	for my $line ( @$newLines ) {
		push @$features, _createFeature ( split( /=/, $line ) );
	}

	$self->{repositoryFeatures} = $features;
	return 1;
}

sub findRepositoryProducts{
	my ( $self, $tmp_dir, $errlst) = @_ ;
	my $kit = $self->getKit ();
	my $repository = $self->{repository};
	my $cfg = {};
	my $out;
	$cfg->{out} = \$out;
    my $javaVm = $self->_getJavaVm();
    my $rc = $kit->callDirector (
    			$tmp_dir, [ ($javaVm ? ('-vm', $javaVm) : ()), '-list', '-profile', 'hdbstudio', '-repository', "$repository", ],
				$errlst, $cfg );	
	if ( !defined $rc || $rc !=0 ) {
		$self->PushError ( "Eclipse Equinox Director failed", $errlst );
		return undef;
	}

	my @lines = split /\n/, $out;
	my $newLines = _removeUnnecessaryLines(\@lines);
	my $products = [];

	for my $line ( @$newLines ) {
		if( $line =~ /product/) {
			push @$products, _createFeature ( split( /=/, $line ) );
		}
	}

	$self->{repositoryProducts} = $products;
	return 1;
}

sub retrieveInstalledFeatures {
	my ( $self ) = @_;
	my $path = $self->{path};
	my $cfg = {};
	my $errlst = new SDB::Install::MsgLst ();
	my $out;
	$cfg->{out} = \$out;
    my $javaVm = $self->_getJavaVm();	
	my $res = $self->getKit ()->callDirector (
		$path,
		[
		    ($javaVm ? ('-vm', $javaVm) : ()),
			'-listInstalledRoots', '-profile',
			'hdbstudio',           '-destination',
			"file:$path",
		],
		$errlst, $cfg
	);
	if ( !defined $res || $res != 0 ) {
		$self->PushError ( "Eclipse Equinox Director failed", $errlst );
		return undef;
	}

	my @lines = split /\n/, $out;
	my $newLines = _removeUnnecessaryLines(\@lines);
	my $result = [];

	for my $line ( @$newLines ) {
		push @$result, _createFeature ( split( /\//, $line ) );
	}
	
	$self->{installedFeatures} = $result;
	return 1;
}

sub _createFeature {
	my ( $featureId, $version, $description ) = @_;

	my %feature = (
		'id'      => $featureId,
		'version' => $version,
		'desc' => $description,
	);
	return \%feature;
}

sub _removeUnnecessaryLines{
	my ($lines) = @_;

	my $newLines = [];

	# Remove lines containing:
	# 'Operation completed in xxx ms.' and
	# 'Java HotSpot(TM) 64-Bit Server VM warning: ignoring option MaxPermSize=512m; support was removed in 8.0'
	# 'Picked up JAVA_TOOL_OPTIONS: -agentlib:jvmhook' env variable etc..
	for my $line ( @$lines ) {
		if ( $line !~ m/.*[=\/]\d.*/ ){
			next;
		}
		if ( $line =~ /Operation completed|MaxPermSize|Picked up/ ) {
			next;
		}
		push (@$newLines, $line);
	}

	return $newLines;
}

sub getKit {
	return $_[0]->{kit};
}

sub _getJavaVm {
    return $_[0]->{javaVm};
}

1;
