package SDB::Web::REST;
use strict;
use base 'Exporter';
use Encode qw(encode_utf8);

use feature qw(switch);
use Data::Dumper;
use CGI;
use SDB::Web::TRACE qw( TRACE_VAR_TO_FILE );
use SDB::Common::Utils qw(createXMLParser);
use experimental qw (smartmatch);

our @EXPORT_OK = qw(get post put );
our %EXPORT_TAGS = ( http => [qw(get post put)] );

my $XML_ELEMENT_NODE	= 1;
my $XML_TEXT_NODE 		= 3;

# Hashes to hold REST resources
my %httpGetHash;
my %httpPostHash;
my %httpPutHash;

sub new {
	my $class   = shift();
	my $self  = bless({}, $class );
	$self->{Scenario} = shift();
	$self->{PARAMS} = shift();
	$self->{SLPPExecutionHandler} = shift();

	return $self;
}

sub setRequestContext {
	my ($self, $request) = @_;
	my ($in, $out, $err) = $request->GetHandles();

	$self->{ENV}    = $request->GetEnvironment();
	$self->{IN}     = $in;
	$self->{OUT}    = $out;
	$self->{ERR}    = $err;
	{
		local %ENV = %{$self->{ENV}};
		$self->{CGI}    = CGI->new($in);
	}
}

#=== FUNCTION ==================================================================
#

=pod

=head1 put

Adds REST resource for the HTTP PUT method.
PARAMETERS:
    - URL for the REST resource
    - sobroutine that must be called when the URL is accessed

USAGE:

put '/path/{variable-path}[query-parameter]' => sub {
    respondText(200, "Called");
}

=cut

#
#===============================================================================
sub put {
	my ( $url, $subroutine ) = @_;
	$httpPutHash{$url} = $subroutine;
}

#=== FUNCTION ==================================================================
#

=pod

=head1 get

Adds REST resource for the HTTP GET method.
PARAMETERS:
    - URL for the REST resource
    - sobroutine that must be called when the URL is accessed

USAGE:

get '/path/{variable-path}[query-parameter]' => sub {
    respondText(200, "Called");
}

=cut

#
#===============================================================================
sub get {
	my ( $url, $subroutine ) = @_;
	$httpGetHash{$url} = $subroutine;
}

#=== FUNCTION ==================================================================
#

=pod

=head1 put

Adds REST resource for the HTTP POST method.
PARAMETERS:
    - URL for the REST resource
    - sobroutine that must be called when the URL is accessed

USAGE:

post '/path/{variable-path}[query-parameter]' => sub {
    respondText(200, "Called");
}

=cut

#
#===============================================================================
sub post {
	my ( $url, $subroutine ) = @_;
	$httpPostHash{$url} = $subroutine;
}

sub respond {
	my ( $self, $http_code, $respondString ) = @_;
	
	if ( $self->isXmlAcceptedType() ) {
		respondXml ( $self, $http_code, $respondString );
	} elsif ( $self->isJsonAcceptedType() ) {
		respondJson ( $self, $http_code, $respondString );
	} else {
		respondText ( $self, $http_code, $respondString );
	}
}

sub respondText {
	my ( $self, $http_code, $http_message ) = @_;
	$self->_respond ( $http_code, 'text/plain', $http_message );
}

sub respondXml {
	my ( $self, $http_code, $xml_string ) = @_;
	$self->_respond ( $http_code, 'application/xml; charset="utf-8"', $xml_string );
}

sub respondJson {
	my ( $self, $http_code, $jsonString ) = @_;
	$self->_respond ( $http_code, 'application/json', $jsonString );
}

sub _respond {
	my ( $self, $http_code, $http_content_type, $http_message ) = @_;

	# determine the HTTP status message by it's code
	my $http_status = undef;
	given ( $http_code ) {
		when ( 200 ) { $http_status = $http_code . " " . "OK"; }
		when ( 201 ) { $http_status = $http_code . " " . "Created"; }
		when ( 202 ) { $http_status = $http_code . " " . "Accepted"; }
		when ( 400 ) { $http_status = $http_code . " " . "Bad Request"; }
		when ( 401 ) { $http_status = $http_code . " " . "Unauthorized"; }
		when ( 403 ) { $http_status = $http_code . " " . "Forbidden"; }
		when ( 404 ) { $http_status = $http_code . " " . "Not Found"; }
		when ( 407 ) { $http_status = $http_code . " " . "Proxy Authentication Required"; }
		when ( 408 ) { $http_status = $http_code . " " . "Request Timeout"; }
		when ( 500 ) { $http_status = $http_code . " " . "Internal Server Error"; }
		when ( 503 ) { $http_status = $http_code . " " . "Service Unavailable"; }
		default      { $http_status = $http_code; }
	}

	my $outFH =  $self->{OUT};
    eval {
	print $outFH "Status: $http_status\n";
	{
		use bytes;
		print $outFH "Content-Length: " . length($http_message) . "\r\n";
	}
	print $outFH "Content-Type: $http_content_type\r\n\r\n";
	print $outFH $http_message;
    }
}

sub getPathParams {
	my $self = shift();
	return _extractPathParams ( $self->{MAPPING_KEY}, $self->{ENV}->{PATH_INFO} );
}

sub getParam {
	my $self   = shift();
	my $search = shift();
	local %ENV = %{$self->{ENV}};
	return encode_utf8 ($self->{CGI}->param( $search ));
}

sub getPostdata {
	my $self   = shift();
	
	my $postdata;
	if( $self->{ENV}->{CONTENT_TYPE} eq 'application/xml' ) {
		$postdata = $self->getParam( 'XForms:Model' );
	} else {
		$postdata = $self->getParam( 'POSTDATA' );
	}
	
	return $postdata;
}

sub isJsonAcceptedType {
	my $self   = shift();
	{
		local %ENV = %{$self->{ENV}};
		return $self->{CGI}->Accept( 'application/json' );
	}
}

sub isXmlAcceptedType {
	my $self   = shift();
	{
		local %ENV = %{$self->{ENV}};
		return $self->{CGI}->Accept( 'application/xml' );
	}
}

sub canHandleRequest {
	my $self = shift();
	# get defined REST resources for the current HTTP method
	my $restMethodMapping = _getRestMethodMapping ( $self->{ENV}->{REQUEST_METHOD} );

	$self->{REST_METHOD_MAPPING} = $restMethodMapping;    # cache
	return ( defined $restMethodMapping ) ? 1 : 0;
}

sub canHandleUrl {
	my $self = shift();

	my $mappingKey = _getMappingKeyByUrl ( $self->{REST_METHOD_MAPPING}, $self->{ENV}->{PATH_INFO} )
		;    # find which REST method corresponds with the asked URL
	my $executable = _getExecutableByMappingKey ( $self->{REST_METHOD_MAPPING}, $mappingKey );

	$self->{MAPPING_KEY}     = $mappingKey;    # cache
	$self->{REST_EXECUTABLE} = $executable;    # cache

	return ( defined $executable ) ? 1 : 0;
}

sub handleUrl {
	my $self        = shift();
	my $executable  = $self->{REST_EXECUTABLE};
	my $returnValue = undef;

	do {
		local $@;

		# Execute the defined REST resource.
		# The returned value will be used only in case of an error.
		# REST resource must define by itself any HTTP response to the client.
		eval {
			$returnValue = $executable->( $self );
			1;
		} or $self->respondOnFailedExecutable ( $returnValue, $@ );
	};
}

sub respondOnFailedExecutable {
	my ( $self, $returnValue, $error ) = @_;

	my $dumper = Data::Dumper->new ( [ $self->{MAPPING_KEY} ], ["REST resource"] );
	$dumper->Deparse ( 0 );    # do NOT resolve references as text
	$dumper->Indent  ( 0 );    # no intendation
	my $trace = $dumper->Dump ();

	$self->respondText ( 500,
		      "Error occurred while trying to execute $trace\n"
			. "Error message: $error"
			. "Rest resource return value: $returnValue\n" );
}

sub _getRestMethodMapping {
	my $httpMethod = shift();
	my $mapping;
	given ( $httpMethod ) {
		when ( 'PUT' )  { $mapping = \%httpPutHash; }
		when ( 'GET' )  { $mapping = \%httpGetHash; }
		when ( 'POST' ) { $mapping = \%httpPostHash; }
		default         { 1; }
	}
	return $mapping;
}

#=== FUNCTION ==================================================================
#

=pod

=head1 getMappingKeyByUrl

Determines which REST resource responds for the given URL
=cut

#	respondText(200, "id called with $path1 $path2");
#
#===============================================================================
sub _getMappingKeyByUrl {
	my ( $mapping, $path ) = @_;
	my $executable = undef;
	foreach my $key ( keys %$mapping ) {
		my $normalizedKey = $key;
		$normalizedKey =~ s/\[.*\]//g;           # remove query parameters
		$normalizedKey =~ s/\{.*\}\//\.\*\//;    # replace "{something}/" with ".*"
		$normalizedKey =~ s/\{.*\}/\.\*/;        # replace "{something} with ".*"

		if ( $path =~ m/^$normalizedKey$/ ) {

			# if the requested url matches some defined REST resource
			return $key;
		}
	}
	return undef;
}

sub _getExecutableByMappingKey {
	my ( $mapping, $key ) = @_;
	return ${$mapping}{$key};
}

sub _extractPathParams {
	my ( $mappingKey, $url ) = @_;
	$mappingKey =~ s/\[.*\]//g;    # remove query parameters
	$mappingKey =~ s/^\///g;       # remove root slash
	$url =~ s/^\///g;              # remove root slash

	my @mappingKeys = split( "/", $mappingKey );
	my @paths       = split( "/", $url );
	my %pathParams;
	my $counter = 0;
	foreach my $key ( @mappingKeys ) {
		my $path = $paths[ $counter++ ];
		next if $key eq $path;     # skip if the key is not "{something}"
		next if $key eq '';
		$key =~ s/\{//g;
		$key =~ s/\}//g;
		$pathParams{$key} = $path;
	}
	return \%pathParams;
}

sub _checkXml {
	my $self	 = shift();
	my $node	 = shift();
	my $metadata = shift();
	my $name	 = shift();
	
	my $node_name = $node->nodeName;
	if( defined $name && $name ne $node_name ) {
		return undef;
	}
	
	my @children = $node->childNodes();
	foreach ( @children ) {
		if ( $_->nodeType == $XML_TEXT_NODE ) {
			next;
		}
		unless ( defined $self->_checkXml ( $_ ) ) {
			return undef;
		}
	}
	
	return 1;
}

sub checkXml {
	my $self	 = shift();
	my $xml	 	 = shift();
	my $metadata = shift();
	
    unless ( defined $xml and length $xml ) {
		require Carp;
        Carp::croak("Empty string");
    }
	
	my $parser = createXMLParser();
	my $tree = $parser->parse_string($xml);
	my $root = $tree->getDocumentElement();
	
	unless ( defined $self->_checkXml($root, $metadata, shift @$metadata) ) {
		require Carp;
		Carp::croak("Unexpected xml structure.");
	}
}

sub copyHash {
	my $self = shift();
	my $hashVar = shift();
	
	my %newHash = ();
	foreach my $key ( keys %$hashVar ) {
		$newHash{$key} = $hashVar->{$key};
	}
	return \%newHash;
}

1;
