import re
import time
import datetime
import decimal
import sys
import gc

import pyhdbcli
from pyhdbcli import LOB
from .resultrow import ResultRow

if sys.version_info >= (3,):
    long = int
    buffer = memoryview
    unicode = str
    xrange = range

#
# globals
#
apilevel = '2.0'
threadsafety = 1
paramstyle = ('qmark', 'named')


#
# connection class
#
class Connection(object):
    def __init__ (self,
                  address='',
                  port=0,
                  user='',
                  password='',
                  autocommit=True,
                  packetsize=None,
                  userkey=None,
                  **properties):
        """Initialize Connection object

Parameters:
    address : ip address or hostname of DB instance
    port : port number of the DB instance
    user : user name
    password : password for user name
    autocommit : auto commit mode
    packetsize : communication packet size
    userkey : user store key
    properties : additional dict object with special properties
Default parameter values:
    address : ''
    port : 0
    user : ''
    password : ''
    autocommit : True
    packetsize : None
    userkey : None
Alias:
    dbapi.connect

Example:
    dbapi.connect(address='localhost',
                  port=8565,
                  user='system',
                  password='manager')
"""
        self.__isconnected = False
        self.__connection = None
        self.__properties = dict()
        self.__clientinfo = dict()

        #set special connect properties
        self.__properties.update(properties)
        #set some known & supported connect properties
        if packetsize != None:
            self.__properties['PACKETSIZE'] = packetsize
        # property is named KEY for historical SQLDBC reasons
        if userkey != None:
            self.__properties['KEY'] = userkey

        #connect
        self.__connection = pyhdbcli.connect("%s:%d" % (address, port), 'HDB', user, password, self.__properties)
        self.__isconnected = self.__connection.checkconnection()

        self.__address = address
        self.__id = 'Connection object : %s,%s,%s,%s,%s' % (address,str(port),
                                                            user,password,
                                                            str(autocommit))
        #autocommit
        self.__connection.setautocommit(autocommit)

        #set unicode build
        self.__connection.setucs4build(sys.maxunicode > 0xFFFF)
        return

    def __del__(self):
        self.close()
        return

    def __str__(self):
        return '<dbapi.Connection ' + self.__id + '>'

    def close(self):
        if self.__isconnected is True:
            try:
                self.__connection.close()
            except:
                pass # ignore exception on closing
            finally:
                self.__isconnected = False

    def commit(self):
        if self.__isconnected is True:
            self.__connection.commit()
        else:
            raise ProgrammingError(0,"Connection closed")

    def rollback(self):
        if self.__isconnected is True:
            self.__connection.rollback()
        else:
            raise ProgrammingError(0,"Connection closed")

    def cursor(self):
        if self.__isconnected is True:
            return Cursor(self)
        else:
            raise ProgrammingError(0,"Connection closed")

    def setautocommit(self, auto=True):
        if self.__isconnected is True:
            self.__connection.setautocommit(auto)
        else:
            raise ProgrammingError(0,"Connection closed")

    def getautocommit(self):
        if self.__isconnected is True:
            return self.__connection.getautocommit()
        else:
            raise ProgrammingError(0,"Connection closed")

    def cancel(self):
        if self.__isconnected is True:
            return self.__connection.cancel()
        else:
            raise ProgrammingError(0,"Connection closed")

    def setclientinfo(self, key, value=None):
        if self.__isconnected is True:
            if value is None:
                self.__clientinfo.pop(key, None) # remove item
            else:
                self.__clientinfo[key] = value
            return self.__connection.setclientinfo(key, value)
        else:
            raise ProgrammingError(0,"Connection closed")

    def getclientinfo(self, key=None):
        if self.__isconnected is True:
            if key is None:
                return self.__clientinfo
            else:
                return self.__clientinfo.get(key)
        else:
            raise ProgrammingError(0,"Connection closed")

    def getproperty(self, key):
        if self.__isconnected is True:
            return self.__connection.getproperty(key)
        else:
            raise ProgrammingError(0,"Connection closed")

    def isconnected(self):
        self.__isconnected = self.__connection.checkconnection()
        return self.__isconnected

    def getaddress(self):
        return self.__address

# Alias dbapi.connect
connect = Connection

#
# cursor class
#
class Cursor(object):
    def __init__(self, connection):
        if not isinstance(connection, Connection):
            raise ProgrammingError(0, "Connection object is required to initialize Cursor object")
        self.connection = connection
        self.__cursor = connection._Connection__connection.cursor() # access like friend
        self.__column_labels = None
        self.description = None
        self.rowcount = -1
        self.arraysize = 32
        self.maxage = None
        self.refreshts = None
        self.__call_pattern = re.compile("^[\s]*{?[\s]*((CALL)|(DO))[\s]*", re.IGNORECASE)
        self.__call_pattern_unicode = re.compile(u"^[\s]*{?[\s]*((CALL)|(DO))[\s]*", re.UNICODE|re.IGNORECASE)
        self._scrollable = False

    def __enter__(self):
        return self

    def __exit__(self, type, value, traceback):
        self.close()

    def __del__(self):
        self.__cursor = None
        return

    def __iter__(self):
        while True:
            row = self.fetchone()
            if row is None:
                break
            yield row

    def __str__(self):
        return '<dbapi.Cursor instance>'

    def __parsenamedquery(self, operation, parameters):
        return self.__cursor.parsenamedquery(operation, parameters)

    def __iscalloperation(self, operation):
        if isinstance(operation, unicode):
            return self.__call_pattern_unicode.search(operation) is not None
        else:
            return self.__call_pattern.search(operation) is not None

    def __execute(self, operation, parameters = None):
        iscall = self.__iscalloperation(operation)
        # parameters is already checked as None or Tuple type.
        ret = self.__cursor.execute(operation, parameters=parameters, iscall=iscall, scrollable=self._scrollable)
        self.__cursor.setfetchsize(self.arraysize)
        return ret

    def __callproc(self, operation, parameters = None):
        if parameters is None:
            return self.__cursor.callproc(operation)
        elif isinstance(parameters, tuple):
            return self.__cursor.callproc(operation, parameters)
        elif isinstance(parameters, list):
            return self.__cursor.callproc(operation, tuple(parameters))
        else:
            raise ProgrammingError(0,"%s is not acceptable as parameters" % str(type(parameters)))

    def __metadata(self):
        if self.__cursor.has_result_set():
            self.rowcount = -1
            self.description = self.__cursor.description()
            self.__column_labels = []
            for column in self.description:
                self.__column_labels.append(column[0])
            self.maxage = self.__cursor.maxage
            self.refreshts = self.__cursor.refreshts
        else:
            self.rowcount = self.__cursor.get_rows_affected()
            self.description = None
            self.__column_labels = None
            self.maxage = None
            self.refreshts = None

    def __last_param_types(self, params):
        self.paramtypes = []
        for param in params:
            self.paramtypes.append(type(param))

    def __can_batch(self, params):
        if len(self.paramtypes) != len(params):
            return False
        for i in xrange(0, len(params)):
            paramtype = type(params[i])
            if self.paramtypes[i] != paramtype:
                if paramtype is type(None):
                    pass
                elif self.paramtypes[i] is type(None):
                    self.paramtypes[i] = paramtype
                elif paramtype in (int, long) and self.paramtypes[i] in (int, long): # allow mixed use of int and long
                    pass
                else:
                    return False
        return True

    def __executemany_in_batch(self, operation, list_of_parameterset):
        try:
            self.description = None
            self.rowcount = -1
            self.maxage = None
            self.refreshts = None
            batch_rowcount = []
            batch_ret = []

            batch_start_index= 0
            self.__last_param_types(list_of_parameterset[batch_start_index]) # save last param types
            for batch_index in xrange(1, len(list_of_parameterset)):
                if self.__can_batch(list_of_parameterset[batch_index]) is False:
                    try:
                        ret = self.__executemany(operation, list_of_parameterset[batch_start_index:batch_index])
                        batch_ret.extend(ret)
                    finally:
                        if self.rowcount >= 0:
                            batch_rowcount.append(self.rowcount)

                    batch_start_index = batch_index
                    self.__last_param_types(list_of_parameterset[batch_start_index]) # reset last param types

            try:
                ret = self.__executemany(operation, list_of_parameterset[batch_start_index:]) # use all remaining parameter sets
                batch_ret.extend(ret)
            finally:
                if self.rowcount >= 0:
                    batch_rowcount.append(self.rowcount)
            return batch_ret
        finally:
            if len(batch_rowcount) > 0:
                self.rowcount = sum(batch_rowcount)

    def __executemany(self, operation, paramsets=None):
        try:
            if paramsets is None:
                if isinstance(operation, (tuple, list)):
                    return self.__cursor.executemany(tuple(operation))
                else:
                    raise ProgrammingError(0, "First parameter must be a sequence of strings")
            else:
                if isinstance(operation, (str, unicode)):
                    return self.__cursor.executemany(operation, tuple(paramsets))
                else:
                    raise ProgrammingError(0, "First parameter must be a string")
        finally:
            self.rowcount = self.__cursor.get_rows_affected()

    def close(self):
        if self.__cursor.isconnected() is True:
            self.__cursor.close()

    def execute(self, operation, parameters=None, **keywords):
        if self.__cursor.isconnected() is False:
            raise ProgrammingError(0, "Connection closed")
        self.rowcount = -1 # reset
        self.description = None # reset
        self.maxage = None # reset
        self.refreshts = None # reset
        if not isinstance(operation, (str, unicode)):
            raise ProgrammingError(0, "First parameter must be a string")
        if parameters is None and len(keywords) == 0:
            ret = self.__execute(operation)
            self.__metadata()
            return ret
        elif parameters is None and len(keywords) > 0:
            qmark_sql, param_values = self.__parsenamedquery(operation, keywords)
            ret = self.__execute(qmark_sql, param_values)
            self.__metadata()
            return ret
        elif isinstance(parameters, dict):
            parameters.update(keywords)
            qmark_sql, param_values = self.__parsenamedquery(operation, parameters)
            ret = self.__execute(qmark_sql, param_values)
            self.__metadata()
            return ret
        elif isinstance(parameters, tuple):
            ret = self.__execute(operation, parameters)
            self.__metadata()
            return ret
        elif isinstance(parameters, list):
            ret = self.__execute(operation, tuple(parameters))
            self.__metadata()
            return ret
        elif operation.count('?') == 1:
            ret = self.__execute(operation, (parameters,))
            self.__metadata()
            return ret
        else:
            raise ProgrammingError(0,"Invalid parameters : execute(%s, %s, %s)"
                                   % (operation, str(parameters), str(keywords)))

    def executemany(self, operation, list_of_parameters = None):
        if self.__cursor.isconnected() is False:
            raise ProgrammingError(0,"Connection closed")
        self.rowcount = -1 # reset
        self.description = None # reset
        self.maxage = None #reset
        self.refreshts = None #reset
        if isinstance(operation, (str, unicode)):
            if list_of_parameters is None or len(list_of_parameters) == 0:
                return self.execute(operation)
            elif isinstance(list_of_parameters, (tuple, list)):
                qmark_sql = operation
                paramsets = []

                for parameters in list_of_parameters:
                    if isinstance(parameters, tuple):
                        paramsets.append(parameters)
                    elif isinstance(parameters, list):
                        paramsets.append(tuple(parameters))
                    elif isinstance(parameters, dict):
                        qmark_sql, param_values = self.__parsenamedquery(operation, parameters)
                        paramsets.append(param_values)
                    else:
                        raise ProgrammingError(0,"A tuple, a list or a dictionary is allowed in the sequence(tuple, list) of parameters.")

                # execute in batch
                ret = self.__executemany_in_batch(qmark_sql, paramsets)
                return ret
            else:
                raise ProgrammingError(0,"Second parameter should be a tuple or a list of parameters")
        elif list_of_parameters is None:
            ret = self.__executemany(operation)
            return ret

        else:
            raise ProgrammingError(0,"Invalid parameter : Cursor.executemany(operation[s][, list of parameters])")

    def fetchone(self, uselob = False):
        if self.__cursor.isconnected() is False:
            raise ProgrammingError(0,"Connection closed")
        if not self.__cursor.has_result_set():
            raise ProgrammingError(0,"No result set")
        ret = self.__cursor.fetchone(uselob)
        if ret is not None:
            return ResultRow(self.__column_labels, ret)
        else:
            return None

    def fetchmany(self, size = None):
        gc_on = gc.isenabled()
        if gc_on:
            gc.disable()
        try:
            if size == None:
                size = self.arraysize
            if self.__cursor.isconnected() is False:
                raise ProgrammingError(0,"Connection closed")
            if not self.__cursor.has_result_set():
                raise ProgrammingError(0,"No result set")
            ret = self.__cursor.fetchmany(size)
            if ret is not None:
                length = len(ret)
                for i in range(length):
                    ret[i] = ResultRow(self.__column_labels, ret[i])
                return ret
            else:
                return None
        finally:
            if gc_on:
                gc.enable()

    def fetchall(self):
        gc_on = gc.isenabled()
        if gc_on:
            gc.disable()
        try:
            if self.__cursor.isconnected() is False:
                raise ProgrammingError(0,"Connection closed")
            if not self.__cursor.has_result_set():
                raise ProgrammingError(0,"No result set")
            ret = self.__cursor.fetchall()
            if ret is not None:
                length = len(ret)
                for i in range(length):
                    ret[i] = ResultRow(self.__column_labels, ret[i])
                return ret
            else:
                return None
        finally:
            if gc_on:
                gc.enable()

    def setfetchsize(self, value):
        self.arraysize = value
        self.__cursor.setfetchsize(value)

    def scroll(self, value, mode="relative"):
        if self._scrollable == False:
            raise ProgrammingError(0,"should be a scrollable-enabled cursor")
        ret = None;
        if( mode == "relative" ):
            ret = self.__cursor.relative(value)
        else:
            ret = self.__cursor.absolute(value)
        if ret == None :
            raise IndexError(0, "cursor position out of range")
        return ret

    def setinputsizes(self, sizes):
        pass

    def setoutputsize(self, size, column = None):
        pass

    def callproc(self, procname, parameters = (), overview = False):
        if self.__cursor.isconnected() is False:
            raise ProgrammingError(0,"Connection closed")
        self.rowcount = -1 # reset
        self.description = None # reset
        self.maxage = None #reset
        self.refreshts = None #reset
        if isinstance(parameters, tuple) :
            qmarks = ','.join(tuple('?' * len(parameters)))
            callsql = "CALL %s(%s)" % (procname, qmarks)
            if overview:
                callsql = callsql + " WITH OVERVIEW"
            callproc = "{ %s }" % callsql
            ret = self.__callproc(callproc, parameters)
            self.__metadata()
            return ret
        else:
            raise ProgrammingError(0,"Second parameter should be a tuple")

    def nextset(self):
        if self.__cursor.isconnected() is True:
            ret = self.__cursor.nextset()
            self.__metadata()
            return ret
        else:
            raise ProgrammingError(0,"Connection closed")

    def get_resultset_holdability(self):
        if self.__cursor.isconnected() is True:
            return self.__cursor.get_resultset_holdability()
        else:
            raise ProgrammingError(0,"Connection closed")

    def set_resultset_holdability(self, holdability):
        if self.__cursor.isconnected() is True:
            self.__cursor.set_resultset_holdability(holdability)
        else:
            raise ProgrammingError(0,"Connection closed")

    def description_ext(self):
        if self.description is None:
            return None
        else:
            return self.__cursor.description_ext()

    def parameter_description(self):
        return self.__cursor.parameter_description()

    def haswarning(self):
        return self.__cursor.haswarning()

    def getwarning(self):
        return self.__cursor.getwarning()





    def setquerytimeout(self, timeout):
        return self.__cursor.setquerytimeout(timeout)

#
# exceptions
#
from pyhdbcli import Warning
Warning.__module__ = __name__
from pyhdbcli import Error
Error.__module__ = __name__
def __errorinit(self, *args):
    super(Error, self).__init__(*args)
    argc = len(args)
    if argc == 1:
        if isinstance(args[0], Error):
            self.errorcode = args[0].errorcode
            self.errortext = args[0].errortext
        elif isinstance(args[0], (str, unicode)):
            self.errorcode = 0
            self.errortext = args[0]
    elif argc >= 2 and isinstance(args[0], (int, long)) and isinstance(args[1], (str, unicode)):
        self.errorcode = args[0]
        self.errortext = args[1]
Error.__init__ = __errorinit
from pyhdbcli import DatabaseError
DatabaseError.__module__ = __name__
from pyhdbcli import OperationalError
OperationalError.__module__ = __name__
from pyhdbcli import ProgrammingError
ProgrammingError.__module__ = __name__
from pyhdbcli import IntegrityError
IntegrityError.__module__ = __name__
from pyhdbcli import InterfaceError
InterfaceError.__module__ = __name__
from pyhdbcli import InternalError
InternalError.__module__ = __name__
from pyhdbcli import DataError
DataError.__module__ = __name__
from pyhdbcli import NotSupportedError
NotSupportedError.__module__ = __name__


#
# input conversions
#

def Date(year, month, day):
    return datetime.date(year, month, day)

def Time(hour, minute, second, millisecond = 0):
    return datetime.time(hour, minute, second, millisecond * 1000)

def Timestamp(year, month, day, hour, minute, second, millisecond = 0):
    return datetime.datetime(year, month, day, hour, minute, second, millisecond * 1000)

def DateFromTicks(ticks):
    localtime = time.localtime(ticks)
    year = localtime[0]
    month = localtime[1]
    day = localtime[2]
    return Date(year, month, day)

def TimeFromTicks(ticks):
    localtime = time.localtime(ticks)
    hour = localtime[3]
    minute = localtime[4]
    second = localtime[5]
    return Time(hour, minute, second)

def TimestampFromTicks(ticks):
    localtime = time.localtime(ticks)
    year = localtime[0]
    month = localtime[1]
    day = localtime[2]
    hour = localtime[3]
    minute = localtime[4]
    second = localtime[5]
    return Timestamp(year, month, day, hour, minute, second)

def Binary(data):
    return buffer(data)

#
# Decimal
#
Decimal = decimal.Decimal

#
# type objects
#
class _AbstractType:
    def __init__(self, name, typeobjects):
        self.name = name
        self.typeobjects = typeobjects

    def __str__(self):
        return self.name

    def __cmp__(self, other):
        if other in self.typeobjects:
            return 0
        else:
            return -1

    def __eq__(self, other):
        return (other in self.typeobjects)

    def __hash__(self):
        return hash(self.name)

NUMBER = _AbstractType('NUMBER', (int, long, float, complex))
DATETIME = _AbstractType('DATETIME', (type(datetime.time(0)), type(datetime.date(1,1,1)), type(datetime.datetime(1,1,1))))
STRING = str
BINARY = buffer
ROWID = int
