package LCM::ActionsDetector;

use strict;
use LCM::HostsUtility;
use LCM::Installer;
use LCM::ActionsDetector::Action::Uninstall;
use LCM::ActionsDetector::Action::RenameSystem;
use LCM::ActionsDetector::Action::UnregisterSystem;
use LCM::ActionsDetector::Action::UpdateComponents;
use LCM::ActionsDetector::Action::Update;
use LCM::ActionsDetector::Action::ConfigureInternalNetwork;
use LCM::ActionsDetector::Action::AddHosts;
use LCM::ActionsDetector::Action::AddHostRoles;
use LCM::ActionsDetector::Action::RemoveHostRoles;
use LCM::ActionsDetector::Action::ConfigureSLD;
use LCM::ActionsDetector::Action::UpdateComponentList;
use LCM::ActionsDetector::Action::ExtractComponents;
use LCM::ActionsDetector::Action::UpdateHost;
use LCM::ActionsDetector::Action::CheckInstallation;
use LCM::ActionsDetector::Action::PrintComponentList;
use LCM::ActionsDetector::Action::RegisterRenameSystem;
use LCM::ActionsDetector::Action::RemoveHosts;
use LCM::ActionsDetector::Action::UnregisterInstance;
use LCM::ActionsDetector::Action::PrintDetectedComponents;
use LCM::PersistencyUtils qw (getAllPendingUpdateInfo);
use SDB::Install::Globals qw($gFlavourPlatform $gFlavourExpress);

sub new {
    my ($class, $config, $isGui) = @_;
    my $self = bless({}, $class);
    
    $self->{config}  = $config;
    $self->{isGui}    = $isGui;

    $self->{numberOfSystemHosts} = LCM::HostsUtility::GetNumberOfSystemHosts();
    $self->{isHostPartOfSystem}  = LCM::HostsUtility::IsHostPartOfSystem();
    $self->{systemFlavour}       = $self->_detectFlavour();

    $self->createActions();

    return $self;
}

sub getConfiguration {
    my ($self) = @_;
    return $self->{config};
}

sub isGui {
    my ($self) = @_;
    return $self->{isGui};
}

sub getNumberOfHosts {
    my ($self) = @_;
    return $self->{numberOfSystemHosts};
}

sub isHostPartOfSystem {
    my ($self) = @_;
    return $self->{isHostPartOfSystem};
}

sub getSystemFlavour {
    my ($self) = @_;
    return $self->{systemFlavour};
}

sub _detectFlavour {
    my ($self) = @_;
    my $installer = LCM::Installer->new();
    my $sapmnt = File::Spec->catfile($installer->getRuntimeDir(), '..', '..', '..');
    my $scm = $installer->getOwnSystemComponentManager($sapmnt);
    return $scm ? $scm->getHANAFlavour() : $gFlavourPlatform;
}

sub _getPendingInfo {
	my ($self, $isResident) = @_;
    my $config = $self->getConfiguration();
	my $installer = new LCM::Installer();
	my $sapSys = $installer->getInstallation($config->getMsgLst());
	my $pendingInfo = '';
	my $updatePendingInfo = getAllPendingUpdateInfo($sapSys, '  ', '', undef, $config->getMsgLst(), $isResident);

	if ($updatePendingInfo && !$self->isGui()) {
		$pendingInfo = "\n" . $updatePendingInfo;
	}

	return $pendingInfo;
}

sub addAction {
    my ($self, $action) = @_;
    my $key = $action->getKey();
    $self->{actions}->{$key} = $action;
}

sub getActions {
    my ($self) = @_;
    return [ sort { $a->getKey() cmp $b->getKey() } values %{$self->{actions}} ];
}

sub getActionByKey {
    my ($self, $key) = @_;
    return $self->{actions}->{$key};
}

sub getActionByBatchKey {
    my ($self, $batchKey) = @_;
    return (grep { $_->getBatchKey() eq $batchKey } @{$self->getActions()})[0];
}

sub createActions {
    my ($self) = @_;

    $self->addAction(LCM::ActionsDetector::Action::Uninstall->new());
    $self->addAction(LCM::ActionsDetector::Action::RenameSystem->new());
    $self->addAction(LCM::ActionsDetector::Action::UnregisterSystem->new());
    $self->addAction(LCM::ActionsDetector::Action::UpdateComponents->new($self->_getPendingInfo(1)));
    $self->addAction(LCM::ActionsDetector::Action::Update->new($self->_getPendingInfo(0)));
    $self->addAction(LCM::ActionsDetector::Action::ConfigureInternalNetwork->new());
    $self->addAction(LCM::ActionsDetector::Action::AddHosts->new());
    $self->addAction(LCM::ActionsDetector::Action::AddHostRoles->new());
    $self->addAction(LCM::ActionsDetector::Action::RemoveHostRoles->new());
    $self->addAction(LCM::ActionsDetector::Action::ConfigureSLD->new());
    $self->addAction(LCM::ActionsDetector::Action::UpdateComponentList->new());
    $self->addAction(LCM::ActionsDetector::Action::UpdateHost->new());
    $self->addAction(LCM::ActionsDetector::Action::ExtractComponents->new());
    $self->addAction(LCM::ActionsDetector::Action::CheckInstallation->new());
    $self->addAction(LCM::ActionsDetector::Action::PrintComponentList->new());
    $self->addAction(LCM::ActionsDetector::Action::RegisterRenameSystem->new());
    $self->addAction(LCM::ActionsDetector::Action::AddHosts->new());
    $self->addAction(LCM::ActionsDetector::Action::RemoveHosts->new());
    $self->addAction(LCM::ActionsDetector::Action::UnregisterInstance->new());
    $self->addAction(LCM::ActionsDetector::Action::PrintDetectedComponents->new());
}

sub getApplicableActions {
    my ($self) = @_;
    my $actions = $self->getActions();
    return [ grep { $_->isApplicable($self) } @{$actions} ];
}

sub getNotApplicableActions {
    my ($self) = @_;
    my $actions = $self->getActions();
    return [ grep { !$_->isApplicable($self) } @{$actions} ];
}

sub _getActionsUIValues {
    my ($self, $actions) = @_;
    my @validActions = ();
    my @validUiActions = ();

    for my $action (@{$actions}) {
        push (@validActions, $action->getBatchKey());
        push (@validUiActions, $action->getUIString());
    }

    return (\@validActions, \@validUiActions);
}

sub getValidActionsUIValues {
	my ($self) = @_;
    my $applicableActions = $self->getApplicableActions();
    return $self->_getActionsUIValues($applicableActions);
}

sub getInvalidActionsUIValues {
    my ($self) = @_;
    my @notApplicableActions = grep { $_->isVisibleIfNotApplicable() } @{$self->getNotApplicableActions()};
    return $self->_getActionsUIValues(\@notApplicableActions);
}

sub getRootRestrictedActionsMessage {
    my ($self) = @_;
    my @relevantActions = grep { $_->isVisibleIfNotApplicable() } @{$self->getNotApplicableActions()};
    return undef if !@relevantActions;

    my $actionsString = join "\n", map { $_->asString() } @relevantActions;
    return sprintf("The following actions cannot be selected because the tool was not started as root user:\n%s", $actionsString);
}

sub isActionApplicable {
    my ($self, $batchKey) = @_;
    my $action = (grep { $_->getBatchKey() eq $batchKey } @{$self->getApplicableActions()})[0];
    return defined($action);
}

1;