package LCM::ActionsDetector::Action;

use strict;
use warnings;
use LCM::App::ApplicationContext;
use LCM::ActionsDetector::NotApplicableActionReason;
use SDB::Install::Globals qw(
    $gFlavourPlatform
    $gFlavourExpress
    $gFlavourCockpit
    $gProductName
);
use SDB::Install::System qw();
use File::Spec;


sub new {
    my ($class) = @_;
    my $self = bless({}, $class);
    return $self;
}

sub asString {
    my ($self) = @_;
    my $indent = ' ' x 3;
    return sprintf("%s - %s", $indent, $self->getUIString());
}

sub getKey {
    my ($self) = @_;
    ...
}

sub getBatchKey {
    my ($self) = @_;
    ...
}

sub getUIString {
    my ($self) = @_;
    ...
}

sub _setNotApplicableReason {
    my ($self, $message, $isVisible) = @_;
    $self->{_notApplicableReason} = LCM::ActionsDetector::NotApplicableActionReason->new($message, $isVisible);
}

sub _getNotApplicableReason {
    my ($self) = @_;
    return $self->{_notApplicableReason};
}

sub isVisibleIfNotApplicable {
    my ($self) = @_;
    my $reason = $self->_getNotApplicableReason();
    return $reason && $reason->isVisible();
}

sub requiresRoot {
    return 0;
}

sub supportsGUI {
    return 1;
}

sub _getProductName {
    my ($self) = @_;
    if (!$self->{productName}) {
        $self->{productName} = LCM::App::ApplicationContext::getFlavourProductName();
    }
    return $self->{productName};
}

sub executeSpecificChecks {
    my ($self, $actionsManager) = @_;
    return 1;
}

sub getSupportedFlavours {
    my ($self) = @_;
    return [ $gFlavourPlatform, $gFlavourExpress, $gFlavourCockpit ];
}

sub requiresExistingInstance {
    my ($self, $actionsManager) = @_;
    return 1;
}

sub checkRoot {
    my ($self) = @_;
    if ($self->requiresRoot()) {
        return SDB::Install::System::isAdmin();
    }
    return 1;
}

sub _checkExistingInstance {
    my ($self, $actionsManager) = @_;
    if ($self->requiresExistingInstance($actionsManager)) {
        return $actionsManager->isHostPartOfSystem();
    }
    return 1;
}

sub _checkFlavour {
    my ($self, $actionsManager) = @_;
    my $systemFlavour = $actionsManager->getSystemFlavour();
    return grep { $_ eq $systemFlavour } @{$self->getSupportedFlavours()};
}

sub isApplicable {
    my ($self, $actionsManager) = @_;
    return 0 if !$self->supportsGUI() && $actionsManager->isGui();
    return 0 if !$self->executeSpecificChecks($actionsManager);

    if (!$self->_checkFlavour($actionsManager)) {
        $self->_setNotApplicableReason("the current system flavour is not supported");
        return 0;
    }
    if (!$self->_checkExistingInstance($actionsManager)) {
        $self->_setNotApplicableReason("the local host is not part of a $gProductName system");
        return 0;
    }
    if (!$self->checkRoot()) {
        $self->_setNotApplicableReason("the tool wasn't started as root user", 1); # 1 - reason should be visualized in the console and in the GUI
        return 0;
    }

    return 1;
}

sub getNotApplicableMessage {
    my ($self) = @_;
    my $reason = $self->_getNotApplicableReason();
    return sprintf("Action '%s' cannot be started because %s.", $self->getBatchKey(), $reason->getMessage());
}

1;