#======================================================
# Specifies the custom_input methods for
# the RemoveRoles parameter
# This module is alsmost identical to AddHostRolesUtils
#======================================================
package LCM::App::RemoveHostRolesUtils;
use base qw(Exporter);
use strict;

use SDB::Install::Configuration::AnyMultiHostConfig qw($validHostRoles);
use SDB::Install::Globals qw(getHumanReadableRolesString GetHostRoleProperties $gProductNameSystem);
use SDB::Install::MsgLst;
use SDB::Install::Tools qw(printTableToArrayOfLines);

our @EXPORT = qw (readInteractiveRemoveHostRolesInput getRemovableRolesForExistingHosts);

# Returns undef if an error occurs
sub getRemovableRolesForExistingHosts {
    my ($config) = @_;
    my $instance = $config->getOwnInstance();
    return undef if (!defined $instance);

    my %hostToRemovableRolesMap = ();
    my $allHosts = $instance->get_allhosts();
    for my $host (@$allHosts) {
        my $hostRoles = $instance->getHostRolesByIniFile($host);
        return undef if(!defined $hostRoles);
        next if (scalar(@$hostRoles) == 1); # A host with only one role cannot have it removed
        for my $hostRole (@$hostRoles) {
            if ($validHostRoles->{$hostRole}->{removable}) {
                push(@{$hostToRemovableRolesMap{$host}}, $hostRole);
                $hostToRemovableRolesMap{$host} = [] if (! exists $hostToRemovableRolesMap{$host});
            }
        }
    }
    return \%hostToRemovableRolesMap;
}

sub readInteractiveRemoveHostRolesInput {
    my ($console, $config) = @_;
    my $existingHostToValidRolesMap = getRemovableRolesForExistingHosts($config);
    if (! scalar(keys %$existingHostToValidRolesMap)) {
        $config->PushError("There aren't any $gProductNameSystem hosts with removable roles.");
        return undef;
    }
    my $invalidHostsMessage = $config->getInvalidHostsForSelectionMessage($existingHostToValidRolesMap);

    my $oldContext = $config->setMsgLstContext([new SDB::Install::MsgLst(), new SDB::Install::MsgLst()]);
    my @hostsValidForRemoveRoles = sort(keys(%$existingHostToValidRolesMap));

    my $shouldUseDefaultsOnly = 0;
    my @hostsSelected = ();
    my $TABLE_LENGTH_WITH_ONE_VALUE = 3;
    while (1) {
        my $table = _buildExistingHostsTable($config, \@hostsValidForRemoveRoles);
        my $hasOnlySingleEntry = (scalar @$table == $TABLE_LENGTH_WITH_ONE_VALUE) ? 1 : 0;
        my $defaultIndex = $hasOnlySingleEntry ? "[1]" : "";
        my $action = sprintf("Enter comma-separated list of selected indices %s: ", $defaultIndex);

        my $dummyParam = {type => 'csv', valid_values => \@hostsValidForRemoveRoles};
        if ($config->getUseDefault('RemoveRoles') && $hasOnlySingleEntry) {
            my $defaultHost = $console->convertFromIndexToValue($dummyParam, 1);
            my $numberOfRolesToRemove = scalar(@{$existingHostToValidRolesMap->{$defaultHost}});
            $shouldUseDefaultsOnly = ($numberOfRolesToRemove == 1);
        }

        my $input = '';
        if ($shouldUseDefaultsOnly) {
            $input = 1;
        } else {
            print("$invalidHostsMessage\n") if (defined $invalidHostsMessage);
            print "Select hosts from which you would like to remove any existing roles\n\n";
            _printTable($table);
            print($action);
            chomp($input = <STDIN>);
        }
        $input = 1 if ($input eq "" && $hasOnlySingleEntry);
        my $csvHosts = $console->convertFromIndexToValue($dummyParam, $input);
        next if (!defined $csvHosts);
        if ($csvHosts eq '') {
            print "Please select at least one host.\n";
            next;
        }
        @hostsSelected = split(',', $csvHosts);
        print "\n";
        last;
    }

    for my $host (@hostsSelected) {
        while (1) {
            my $table = _buildSelectRemovableRolesTable($config, $existingHostToValidRolesMap->{$host});
            my $hasOnlySingleEntry = (scalar @$table == $TABLE_LENGTH_WITH_ONE_VALUE) ? 1 : 0;
            my $defaultIndex = $hasOnlySingleEntry ? "[1]" : "";
            my $action = sprintf("Enter comma-separated list of roles to remove for host '$host' %s: ", $defaultIndex);

            my $input = '';
            if ($shouldUseDefaultsOnly) {
                $input = 1;
            } else {
                print "\nSpecify roles to remove for host '$host'\n\n";
                _printTable($table);
                print($action);
                chomp($input = <STDIN>);
            }
            my $dummyParam = {type => 'csv', valid_values => $existingHostToValidRolesMap->{$host}};
            $input = 1 if ($input eq "" && $hasOnlySingleEntry);
            my $csvRoles = $console->convertFromIndexToValue($dummyParam, $input);
            next if (!defined $csvRoles);
            if ($csvRoles eq '') {
                print "Please select at least one host role.\n";
                next;
            }
            if (!$config->setMapValueItem('RemoveRoles', $host, $csvRoles)) {
                print($config->getErrorString() . "\n");
                $config->resetError();
                next;
            }
            last;
        }
    }
    $config->setMsgLstContext($oldContext);
    return 1;
}

sub _buildExistingHostsTable {
    my ($config, $hostsValidForRemoveRoles) = @_;
    my $roleProperties = GetHostRoleProperties();
    my $instance = $config->getOwnInstance();
    my $hostsWithRolesHash = $instance->getHostRolesInfo();
    my @hostRolesColumn = ();
    for my $host (@$hostsValidForRemoveRoles) {
        my $hostRoles = $hostsWithRolesHash->{$host};
        my $humanReadableRoles = getHumanReadableRolesString(split(' ', $hostRoles));
        push(@hostRolesColumn, $humanReadableRoles);
    }
    my $columnsArray= [$hostsValidForRemoveRoles, \@hostRolesColumn];
    return _buildTable($columnsArray, ['Index', 'System host', 'Roles']);
}

sub _buildSelectRemovableRolesTable {
    my ($config, $validRoles) = @_;
    my $roleProperties = GetHostRoleProperties();

    my @roles = ();
    my @roleDescription = ();
    for my $role (@$validRoles) {
        push @roles, $role;
        push @roleDescription, $roleProperties->{$role}->{str};
    }
    my $columnsArray = [\@roles, \@roleDescription];
    return _buildTable($columnsArray, ['Index', 'Removable Role', 'Role Description']);
}

# Args:
# $colmunsArray => [[entry1.1, entry2.1], [entry1.2, entry2.2], ...]
#   where in 'entryX.Y' 'X' means 'row' and 'Y' means 'column'.
#
# $columnNames => ['HeaderForColumn1', 'HeaderForColmun2' ...]
#
sub _buildTable {
    my ($columnsArray, $columnNames) = @_;

    my @table = ();
    push @table, $columnNames;
    my $numOfColumns = @$columnsArray;
    my $numOfEntries = @{$columnsArray->[0]};

    my $index = 1;
    while ($index <= $numOfEntries) {
        my $row = [$index];
        for (my $columnNumber = 0; $columnNumber < $numOfColumns; $columnNumber++) {
            my $currentColumn = $columnsArray->[$columnNumber];
            push (@$row, $currentColumn->[$index - 1]);
        }
        push @table, $row;
        $index++;
    }
    return printTableToArrayOfLines(\@table, ' | ', 1);
}

sub _printTable {
    my ($indexTable) = @_;
    print '  ' . join ("\n  ", @$indexTable) . "\n\n";
}

1;