package LCM::App::UpdateSystemUtils;
use strict;
use warnings;

use File::Spec;
use File::Basename;
use SDB::Install::Globals qw ($gKeynameEngine $gProductNameEngine $gSAPDSignerNotDetectedMessage $gUpgradeRestrictionsMessage GetNewerServerAlreadyInstalledMessage);
use LCM::Manifests::ManifestFactory;
use SDB::Install::System qw(copy_tree deltree isAdmin);
use SDB::Install::MsgLst;
use LCM::Installer;
use LCM::SAPDSigner;
use LCM::FileUtils;
use LCM::Component::Installed;
use LCM::Component::Installable;
use SDB::Install::Globals qw(getFlavourProductName $gProductNameInstaller);
use SDB::Install::SysVars qw($currentPlatformName $isPowerPC);
use SDB::Install::User;
use File::stat;

use Exporter;
our @ISA = qw (Exporter);
our @EXPORT = qw(getServerPathFromUserInput
                 getLCMFromServerPath
                 updateNonResidentWebResources
                 isPathToServerBinaries
                 isInstalledUi5UpToDate
                 isInstalledAdminHelpToDate
                 validateExternalInstaller
                 validateResidentInstaller
                 getInstalledServerManifest
                 getServerComponentFromPath
                 getLCMComponentFromPath);


sub getServerPathFromUserInput {
	my ($serverPath, $errorMessageList, $ignoreVersion) = @_;
	$ignoreVersion //= 0;
	my $pathCandidateValidation;
	my $rootPathValidation = isServerPathValid($serverPath, $errorMessageList, $ignoreVersion);

	return $serverPath if($rootPathValidation == 0);
	
	for my $pathCandidate (@{_getServerPathCandidates()}){
		my $serverPathCandidate = File::Spec->catdir($serverPath, $pathCandidate);
		my $msgLst = SDB::Install::MsgLst->new();
		my $retCode = isServerPathValid($serverPathCandidate, $msgLst, $ignoreVersion);
		return $serverPathCandidate if($retCode == 0);

		if ($retCode > 2) {
			$pathCandidateValidation = $retCode;
			$errorMessageList->initWithMsgLst($msgLst);
		}
	}

# retCode > 2 means that a kit was found but it has other problems
	my $detectedKitErrorForPathCandidate = ($pathCandidateValidation > 2);
	my $detectedKitErrorForRootPath      = ($rootPathValidation > 2);
	my $hasDetectedKitErrors             = ($detectedKitErrorForPathCandidate || $detectedKitErrorForRootPath);

	my $subdir = _getSingleSubdirPath($serverPath);
	if (!$hasDetectedKitErrors && defined $subdir && $subdir !~ /^(SAP_HANA_DATABASE|HDB_SERVER_LINUX_(\w+?)|DATA_UNITS)$/) {
		return (getServerPathFromUserInput($subdir, $errorMessageList));
	}
	if (!$hasDetectedKitErrors) {
		$errorMessageList->initMsgLst();
		$errorMessageList->addError("Unable to detect an installation kit in '$serverPath'");
	}
	return undef;
}

sub getLCMFromServerPath {
	my ($serverPath, $executable) = @_; # Server path should point to installation kit location
	$executable //= 'hdblcm';

	my $serverParentDir = File::Basename::dirname($serverPath);
	if ($serverParentDir eq File::Spec->rootdir()) {
# Do not look for HDB_(_)?LCM_* directories under /
		return $serverPath;
	}

	for my $lcmCandidatePath (@{_getLCMPathCandidates()}) {
		my $fullPath = File::Spec->catfile($serverParentDir, $lcmCandidatePath);
		my $msgLst = SDB::Install::MsgLst->new();
		return $fullPath if _checkInstaller($fullPath, $executable, $msgLst);
	}

	return $serverPath;
}

sub isServerPathValid {
	my ($targetServerPath, $errorMessageList, $ignoreVersion) = @_;
	my $serverPathObject = File::stat::stat($targetServerPath);
	$errorMessageList->initMsgLst();

	if(!$serverPathObject || ! -d $serverPathObject){
		$errorMessageList->addError("The specified directory '$targetServerPath' was not found");
		return 1;
	}
	
	if(isPathToServerBinaries($targetServerPath)){
		$targetServerPath = File::Basename::dirname($targetServerPath);
	}

	if(!_isPathToManifest($targetServerPath)) {
		$errorMessageList->addError("Could not find a sub-directory 'server' containing a 'manifest' file in the specified directory '$targetServerPath'");
		return 2;
	}
	
	if(!_isServerUpdatable($targetServerPath, $errorMessageList, $ignoreVersion)){
		return 3;
	}
	if(!_checkInstaller($targetServerPath, 'hdblcmweb', $errorMessageList)){
		return 4;
	}
	return 0;
}

sub validateExternalInstaller {
	my ($installerComponent, $errorMessageList) = @_;
	$errorMessageList->initMsgLst();

	my $sapdsigner = LCM::SAPDSigner->getInstance();
	if (!defined($sapdsigner)) {
		$errorMessageList->addError($gSAPDSignerNotDetectedMessage);
		return 0;
	}

	$sapdsigner->setMsgLstContext([ SDB::Install::MsgLst->new(), $errorMessageList ]);
	if (!$sapdsigner->authenticateInstaller($installerComponent)) {
		return 0;
	}

	return 1;
}

sub getServerComponentFromPath {
	my ($serverPath, $errorMessageList) = @_;
	$errorMessageList->initMsgLst();

	my $manifestDir = File::Spec->canonpath(File::Spec->catfile($serverPath, 'server'));
	return _createComponentByManifestDir($manifestDir, 0, $errorMessageList);
}

sub getLCMComponentFromPath {
	my ($lcmPath, $errorMessageList) = @_;
	$errorMessageList->initMsgLst();

	my $manifestDir = File::Spec->catfile($lcmPath, 'instruntime');
	return _createComponentByManifestDir($manifestDir, 0, $errorMessageList);
}

sub validateResidentInstaller {
	my ($errorMessageList) = @_;
	$errorMessageList->initMsgLst();

	my $instruntimeDir = LCM::Installer->new()->getRuntimeDir();
	my $residentInstaller = _createComponentByManifestDir($instruntimeDir, 1, $errorMessageList);
	my $sapdsigner = LCM::SAPDSigner->getInstance();
	return $sapdsigner ? $sapdsigner->authenticateInstaller($residentInstaller) : 0;
}

sub updateNonResidentWebResources {
	my ($installerPath, $errlst) = @_;
	
	my $exec_dir = LCM::Installer->new()->getRuntimeDir(); # <installation_dir>/<SID>/hdblcm/instruntime	
	my $hdblcm_dir = File::Basename::dirname($exec_dir);
	
	$errlst->initMsgLst();
	
	my $currentNonResidentResourcesDir = File::Spec->catdir($hdblcm_dir, 'resources', 'nonresident');
	if (!deltree($currentNonResidentResourcesDir, $errlst)) {
		return undef;
	}	
	
	my $targetResidentResourcesDir = File::Spec->catdir($installerPath, 'resources', 'resident');
	if (!copy_tree($targetResidentResourcesDir,  File::Spec->catdir($currentNonResidentResourcesDir, 'resident'), $errlst)) {
		return undef;
	}
	
	my $targetNonResidentResourcesDir = File::Spec->catdir($installerPath, 'resources', 'nonresident');
	if (!copy_tree($targetNonResidentResourcesDir, $currentNonResidentResourcesDir, $errlst)) {
		return undef;
	}
	
	my $isUi5Available = 0;
	my $currentResidentResourcesDir = File::Spec->catdir($hdblcm_dir, 'resources', 'resident');
	if(isInstalledUi5UpToDate($targetResidentResourcesDir, $currentResidentResourcesDir) &&  eval {symlink("",""); 1} ){
		my $curentResidentUi5Directory = File::Spec->catdir($currentResidentResourcesDir, 'sapui5libs');
		my $curentNonResidentUi5Directory = File::Spec->catdir($hdblcm_dir, 'resources', 'nonresident', 'resident', 'sapui5libs');
		$isUi5Available = symlink($curentResidentUi5Directory, $curentNonResidentUi5Directory);
	}
	if(!$isUi5Available) {
		require SDB::Install::Archive;
		my $archive = new SDB::Install::Archive ( File::Spec->catfile($hdblcm_dir, 'resources', 'nonresident', 'resident', 'sapui5libs.tgz') );
		my $returnCode = $archive->Extract(File::Spec->catfile($hdblcm_dir, 'resources', 'nonresident', 'resident'));
		
		if(!$returnCode){
			$errlst->initWithMsgLst($archive->getErrMsgLst());
			return undef;
		}
	}
	
	chmod(0755, $currentNonResidentResourcesDir);
	return 1;
}

sub isInstalledUi5UpToDate {
	my ($targetDir, $currentDir) = @_;
	my $targetFile = File::Spec->catfile($targetDir, 'ui5version');
	my $currentFile = File::Spec->catfile($currentDir, 'ui5version');
	my ($targetLines, $currentLines) = (readFile($targetFile), readFile($currentFile)); 

	return 0 if(!defined($targetLines) || !defined($currentLines));
	
	chomp(my $targetVersion = $targetLines->[0]);
	chomp(my $currentVersion = $currentLines->[0]);

	return $targetVersion eq $currentVersion;
}

sub isInstalledAdminHelpToDate {
	my ($targetDir, $currentDir) = @_;
	my $targetFile = File::Spec->catfile($targetDir, 'adminhelpversion');
	my $currentFile = File::Spec->catfile($currentDir, 'adminhelpversion');
	my ($targetLines, $currentLines) = (readFile($targetFile), readFile($currentFile));

	return 0 if(!defined($targetLines) || !defined($currentLines));

	chomp(my $targetVersion = $targetLines->[0]);
	chomp(my $currentVersion = $currentLines->[0]);

	return $targetVersion eq $currentVersion;
}

sub isPathToServerBinaries{
	my ($targetServerPath) = @_;
	my $serverManifestFile = File::Spec->catfile($targetServerPath, 'manifest');
	my $serverManifest = LCM::Manifests::ManifestFactory::createComponentManifest($serverManifestFile);

	return $serverManifest->isServer();
}

sub getInstalledServerManifest {
	my ($errorMessageList, $sapmnt) = @_;
	require LCM::Component;
	require LCM::Installer;

	my $installer = new LCM::Installer();
	my $systemComponentManager = $installer->getOwnSystemComponentManager($sapmnt);
	return undef if(! defined $systemComponentManager);
	my $installedServerComponent = $systemComponentManager->getComponentByKeyName($gKeynameEngine);

	if(!defined $installedServerComponent){
		$errorMessageList->addError("Unable to detect installed $gProductNameEngine on the system");
		return undef;
	}

	return $installedServerComponent->getManifest();
}


sub _isPathToManifest {
	my $serverPath = shift();
	my $manifestCandidatePath = File::Spec->catfile($serverPath, 'server', 'manifest');
	my $manifestPathObject = File::stat::stat($manifestCandidatePath);
	return ($manifestPathObject && -f $manifestPathObject) ? 1 : 0;
}

sub _checkInstaller {
	my ($targetServerPath, $executable, $errorMessageList) = @_;
	my $executablePath = File::Spec->catfile($targetServerPath, $executable);
	my $executableObject = File::stat::stat($executablePath);
	my $hdblcmManifestPath = File::Spec->catfile($targetServerPath, 'instruntime', 'manifest');
	
	my $hdblcmManifest = LCM::Manifests::ManifestFactory::createComponentManifest($hdblcmManifestPath);
	my $isHdblcmManifestOk = defined($hdblcmManifest->read(undef, 1));
	my $isHdblcmExececutableOk = ($executableObject && (-f $executableObject) && (-x $executableObject));
	
	if(!$isHdblcmManifestOk){
		$errorMessageList->addError("Unable to detect hdblcm in the specified location", $hdblcmManifest->getErrMsgLst());
		return undef;
	}
	if(!$isHdblcmExececutableOk){
		$errorMessageList->addError("$executable executable does not exist or does not have the right permissions");
		return undef;
	}
	
	my $isPlatformSupported = $hdblcmManifest->isPlatformSupported($currentPlatformName);
	if(!$isPlatformSupported){
		$errorMessageList->addError("$executable does not support current OS platform '$currentPlatformName'");
		return undef;
	}

	return 1;
}

sub _isServerUpdatable {
	my ($targetServerPath, $errorMessageList, $ignoreVersion) = @_;
	my $installedServerManifest = getInstalledServerManifest($errorMessageList);
	
	if(! defined $installedServerManifest){
		$errorMessageList->addError('Unable to detect SAP HANA Database installation');
		return (undef, $errorMessageList);
	}

	my $installedServerVersion = $installedServerManifest->getVersionObject();
	my $installedVersionString = $installedServerVersion->asString();
    my $installedFlavour = $installedServerManifest->getHANAFlavour();
	my $targetServerManifestPath = File::Spec->catfile($targetServerPath, 'server', 'manifest');
	my $targetServerManifest = LCM::Manifests::ManifestFactory::createComponentManifest($targetServerManifestPath);
	my $isTargetServerManifestOk = defined($targetServerManifest->read(undef, 1));
	
    if(!$isTargetServerManifestOk || !$targetServerManifest->isServer() || $installedFlavour ne $targetServerManifest->getHANAFlavour()){   
        my $flavourProductName = getFlavourProductName($installedFlavour);
        $errorMessageList->addError("Error detecting $flavourProductName installation kit in '$targetServerPath'", $targetServerManifest->getErrMsgLst());
        return undef;
    }

	my $targetServerVersion = $targetServerManifest->getVersionObject();
	my $targetVersionString = $targetServerVersion->asString();
	my $isTargetOlderThanSource = $installedServerVersion->isNewerThan($targetServerVersion);
	my $isUpdatePossible = $targetServerManifest->checkSPSUpgradeLimitation($installedServerManifest);

	if($isTargetOlderThanSource && !$ignoreVersion){
		$errorMessageList->addError(GetNewerServerAlreadyInstalledMessage($targetVersionString, $installedVersionString));
		return undef;
	}
	if(! $isUpdatePossible ){
		$errorMessageList->addError($gUpgradeRestrictionsMessage, $targetServerManifest->getErrMsgLst());
		return undef;
	}

	return 1;
}

sub _createComponentByManifestDir {
	my ($manifestDir, $isInstalled, $errorMessageList) = @_;
	my $manifestFile = File::Spec->catfile($manifestDir, 'manifest');
	my $manifest = LCM::Manifests::ManifestFactory::createComponentManifest($manifestFile);
	if ($manifest->errorState()) {
		$errorMessageList->addError("Invalid manifest file '$manifestFile'", $manifest->getErrMsgLst());
		return undef;
	}

	my $componentClass = $isInstalled ? 'LCM::Component::Installed' : 'LCM::Component::Installable';
	my $component = $componentClass->new($manifestDir, $manifest, undef, undef, 1);
	if ($component->errorState()) {
		my $componentName = $component->getComponentName();
		$errorMessageList->addError("Failed to create component '".($componentName // '<unkown component>')."'", $component->getErrMsgLst());
		return undef;
	}

	return $component;
}

sub _getLCMPathCandidates {
	my @platformSpecificFolders = (
		$isPowerPC ?
		('HDB_LCM_LINUX_PPC64', 'HDB_LCM_LINUX_PPC64LE', 'HDB__LCM_LINUX_PPC64', 'HDB__LCM_LINUX_PPC64LE') :
		('HDB_LCM_LINUX_X86_64', 'HDB__LCM_LINUX_X86_64')
	);
	return [ map { ($_, File::Spec->catfile('DATA_UNITS', $_)) } @platformSpecificFolders ];
}

sub _getServerPathCandidates {
	my @platformSpecificFolders = $isPowerPC ? ('HDB_SERVER_LINUX_PPC64', 'HDB_SERVER_LINUX_PPC64LE') : ('HDB_SERVER_LINUX_X86_64');
	my @candidateSubdirs = ('SAP_HANA_DATABASE');
	for my $platformSpecificDir (@platformSpecificFolders) {
		push @candidateSubdirs, $platformSpecificDir;
		push @candidateSubdirs, File::Spec->catfile('DATA_UNITS', $platformSpecificDir),
	}
	my @candidates = map { File::Spec->catfile($_, 'server') } @candidateSubdirs;
	unshift @candidates, 'server'; # Handle the case of standalone 'server' dir

	return \@candidates;
}

sub _getSingleSubdirPath {
	my ($path) = @_;
	my $contents = listDirectory($path);
	my $subdirPath;
	for my $content (@$contents) {
		next if ($content =~ /^(\.|\.\.)$/);
		my $fullPath = File::Spec->catdir($path, $content);
		if (-d $fullPath && defined $subdirPath) {
			return undef;
		} elsif (! -l $fullPath && -d $fullPath && !defined $subdirPath) {
			$subdirPath = $fullPath;
		}
	}
	return $subdirPath;
}

1;