package LCM::Component::Installable::Accelerator;

use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::Globals qw ($gProductNameAccelerator
                              $gShortProductNameAccelerator
                              $gHostRoleAcceleratorWorker
                              $gHostRoleAcceleratorStandby
                              $gTmpDir);
use LCM::ProcessExecutor;
use LCM::Installer;
use LCM::FileUtils;
use File::Basename qw (dirname);
use strict;

use base 'LCM::Component::Installable';

our $installScript = 'aseinst';
our $updateScript  = 'aseupd';

if ($isWin){
    $installScript .= '.exe';
    $updateScript  .= '.exe';
} else {
    $installScript .= '.sh';
    $updateScript  .= '.sh';
}

sub installComponent {
    my ( $self, $instconfig) = @_;

    my $msg               = $self->getMsgLst()->addProgressMessage($self->getProgressMsg() . '...');

    my $saveCntxt = $self->setMsgLstContext( [ $msg->getSubMsgLst() ] );

    my $sapSystem = $self->getSAPSystem ( $instconfig, $msg, $saveCntxt);
    if (not defined $sapSystem) {
        return undef;
    }

	my $returnCode = $self->_installAcceleratorComponent($instconfig, $sapSystem);
    my $endMessage = $self->_getEndMessage($returnCode);
    $msg->endMessage (undef, $endMessage);
    $self->setMsgLstContext($saveCntxt);
    if ($returnCode && !$self->isUpdate()) {
        my $instance  = $sapSystem->getNewDBInstances()->[0];
        if (defined $instance) {
            $returnCode = $instance->setAcceleratorBasePaths
                                ($instconfig->getValue('AcceleratorDataPath'),
                                 $instconfig->getValue('AcceleratorLogPath'));
        }
    }
    return $returnCode;
}


sub _getEndMessage {
    my ($self, $returnCode) = @_;

    my $message  = 'Installation of ' . $self->getComponentName();
	$message .= ($returnCode == 1) ? ' finished' : ' failed';
    return $message;
}

sub _installAcceleratorComponent {
    my ($self, $instconfig, $sapSystem) = @_;

    my $exer = $self->createProcessExecutor($instconfig, $sapSystem);

    $self->initProgressHandler ();
    $exer->setOutputHandler($self->getProgressHandler ());
    my $exitCode = $exer->executeProgram();

    $self->getMsgLst ()->addMessage(undef, $exer->getMsgLst());
    $self->setLogLocation($self->parseLogFileLocation($exer->getOutputLines()));

    if (!defined $exitCode || $exitCode){
        my $errMsgLst    = $self->getHdbInstallerErrorMessages
                                                      ($exer->getOutputLines());
        my $actionErr    = $self->isUpdate() ?'Update ' : 'Installation ';
        my $errorMessage = "$actionErr of " . $self->getComponentName() . ' failed';
        $self->setErrorMessage ($errorMessage,
                                ($errMsgLst->isEmpty() ? $exer->getErrMsgLst()
                                                       : $errMsgLst));
        return undef;
    }
    return 1;
}

sub parseLogFileLocation {
	my ($self, $outputLines) = @_;
	my $pattern = "Log file written to \'(.*)\'";

	return undef if (!defined($outputLines));

	for my $line (reverse(@{$outputLines})){
		return $1 if($line =~ /$pattern/i);
	}
}

sub getSAPSystem {
    my ( $self, $instconfig, $msg, $saveCntxt) = @_;

    my $systems   = $instconfig->CollectSAPSystems (undef, 1);
    my $sid       = $instconfig->getValue("SID");
    my $sapSystem = $systems->{$sid};

    if (not defined $sapSystem) {
        $self->setErrorMessage ("No SAP system '$sid' found");
        $msg->endMessage( undef, 'Install ' . $self->getComponentName());
        $self->setMsgLstContext($saveCntxt);
        return undef;
    }

    return $sapSystem;
}

sub _buildArgs {
    my ($self, $instconfig, $installDir) = @_;

    my @arguments = ();
    push( @arguments, '-path');
    push( @arguments, $installDir);
    push( @arguments, '-sid');
    push( @arguments, $instconfig->getSID ());

    my $instanceNr = undef;
    my $hostname   = undef;
    my $instance   = $instconfig->getOwnInstance();

    if (defined $instance) {
        $instanceNr = $instance->get_nr();
        $hostname   = $instance->get_host();
    }

    $instanceNr = $instconfig->getValue('InstanceNumber') if (!defined $instanceNr);
    $hostname   = $instconfig->getValue('HostName')       if (!defined $hostname);

    if (defined $instanceNr) {
        push( @arguments, '-inst');
        push( @arguments, $instanceNr);
    }

    if (defined $hostname) {
        push( @arguments, '-hdbhost');
        push( @arguments, $hostname);
    }

    my $user = $instconfig->getValue ('SystemUser');
    if (defined $user ) {
        push( @arguments, '-hdbuser');
        push( @arguments, $user);
    }

    my $aseUser = $instconfig->getValue('AcceleratorUser');
    if (defined $aseUser && $instconfig->shouldProvideAseUserCredentials() ) {
        push( @arguments, '-aseuser');
        push( @arguments, $aseUser);
    }

    if(!$self->isUpdate()) {
        push( @arguments, '-data_ase');
        push( @arguments, $instconfig->getValue('AcceleratorDataPath'));

        push( @arguments, '-log_ase');
        push( @arguments, $instconfig->getValue('AcceleratorLogPath'));
    }

    push @arguments, @{$self->SUPER::_buildArgs($instconfig)};
    return \@arguments;
}

sub createProcessExecutor{
    my ($self, $instconfig, $sapSystem) = @_;

    my $installDir = $sapSystem->get_globalSidDir();
    my $user = $sapSystem->getUser();
    my $uid = $user->uid();
    my $gid = $user->gid();

    my $instance = $sapSystem->getNewDBInstances ()->[0];
    my $stdinLine = $instconfig->getValue ('SQLSysPasswd');
    if($instconfig->shouldProvideAseUserCredentials()){
        $stdinLine = join (" ", $stdinLine, $instconfig->getValue ('AcceleratorPassword'));
    }
    #! all passwords should be given on one stdin line to aseupd
    my $stdinPasswords = [
            $stdinLine
        ];

    my $command = $self->getExecutable();
    my $args = $self->_buildArgs($instconfig, $installDir);
    if ($self->isSidAdmUserExecution()) {
        return new LCM::ProcessExecutor($command, $args, $stdinPasswords);
    }

    return new LCM::ProcessExecutor($command, $args, $stdinPasswords,
                                         dirname ($command), undef, $uid, $gid);
}

sub getExecutable {
    my ($self) = @_;
    my $script = $self->isUpdate() ? $updateScript : $installScript;
    return File::Spec->catfile($self->getInstallerDir(), $script);
}

sub getNumberOfExpectedOutputLines{
    return 75;
}

sub getDefaultSelection{
    my ($self, $stackUpdate) = @_;
    if ($stackUpdate){
        return $self->SUPER::getDefaultSelection ($stackUpdate);
    }
    return 0;
}

sub getComponentName {
    return $gProductNameAccelerator;
}

sub getSlppLogFileName {
    return 'ase.log'
}

sub requireSystemRestart {
    return 0;
}

sub updateComponent;
*updateComponent =  \&installComponent;

sub getHostRoles{
	return [$gHostRoleAcceleratorWorker, $gHostRoleAcceleratorStandby];
}

sub requiresSqlSysPasswd{
    return 1;
}

1;
