package LCM::Component::Installable::CockpitStack;

use strict;
use base 'LCM::Component::InternallyPersistedInstallable';
use SDB::Install::System qw(copy_file);
use SDB::Install::SysVars qw($isWin);
use SDB::Install::Globals qw ($gDirNameCockpitStack $gProductNameCockpit $gShortProductNameCockpit $gShortProductNameXS2);

my $CREATE_TENANT_STATEMENT = 'CREATE DATABASE %s SYSTEM USER PASSWORD "%s"';

sub _install {
	my ($self, $configuration) = @_;
    my $msg = $self->getMsgLst()->addProgressMessage($self->getProgressMsg() . '...');
    my $saveContext = $self->setMsgLstContext([$msg->getSubMsgLst()]);
    $self->getMsgLst()->getProgressHandler()->incrementIndentationDepth();

    my $rc = 1;
    if ($configuration->isCockpitOnPlatformAction() && !$self->isUpdate()) {
        if (!$self->_prepareDB($configuration)) {
            $self->setErrorMessage("Failed to prepare database for $gProductNameCockpit", $self->getErrMsgLst());
            $rc = 0;
        }
    }

    if ($rc && !$self->_copyCockpitStack($configuration)) {
        $self->setErrorMessage("Failed to copy $gShortProductNameCockpit Stack manifest.", $self->getErrMsgLst());
        $rc = 0;
    }

	my $type = $self->isUpdate() ? "Update" : "Installation";
    my $endMessage = $self->_getEndMessage($rc, $type);
    $self->getMsgLst()->getProgressHandler()->decrementIndentationDepth();
    $msg->endMessage (undef, $endMessage);
    $self->setMsgLstContext($saveContext);
	return $rc;
}

sub _copyCockpitStack {
    my ($self, $configuration) = @_;
    my $manifestPath = File::Spec->catfile($self->getPath(), 'manifest');
    my $instPath = $self->getInstSidPath($configuration);
    my $config = $self->_createSidAdmFileCfg($configuration);
    my $targetFile = File::Spec->catfile($instPath, $gDirNameCockpitStack, 'manifest');
    $self->getMsgLst()->addMessage("Copying cockpit stack manifest from '$manifestPath' to '$targetFile'");

    return copy_file($manifestPath, $targetFile, $config);
}

sub _getEndMessage {
    my ($self, $returnCode, $type) = @_;
    my $message = "$type of " . $self->getComponentName();
    $message .= (defined $returnCode && ($returnCode == 1)) ? ' finished' : ' failed';
    return $message;
}

sub _createSidAdmFileCfg{
	my($self, $configuration) = @_;
	my $fileCfg;
	if (!$isWin){
		my $user = new SDB::Install::NewDBUser($configuration->getValue('SID'));
        $fileCfg->{uid} = $user->uid();
        $fileCfg->{gid} = $user->gid();
        $fileCfg->{createdir} = 1;
	}
	return $fileCfg;
}

sub _prepareDB {
    my ($self, $configuration) = @_;
    my $space = $configuration->getValue('CockpitXSSpace');
    my $xsClient = $configuration->getXSClient(1);
    $xsClient->setMsgLstContext($self->getMsgLstContext());

    if (!$xsClient->login()) {
        $self->appendErrorMessage("Failed to login into $gShortProductNameXS2");
        return 0;
    }

    $self->getMsgLst()->addMessage("Checking whether $gShortProductNameXS2 space '$space' exists...");
    if ($xsClient->spaceExists($space)) {
        my $message = "$gShortProductNameXS2 space '$space' already exists. It will not be changed and will be used for the installation of $gProductNameCockpit";
        $self->getMsgLst()->addProgressMessage("$gShortProductNameXS2 space '$space' already exists. It will be used for the instalaltion of $gProductNameCockpit.");
        return 1;
    }

    return 0 if !$self->_handleCockpitTenant($configuration);
    return 0 if !$self->_enableTenantForXSA($configuration, $xsClient);
    return 0 if !$self->_createCockpitSpace($configuration, $xsClient);
    return 1;
}

sub _persistCockpitTenant {
    my ($self, $configuration) = @_;
    my $sapSys = $configuration->getSAPSystem();
    my $configFile = $sapSys->getUserConfigFile();
    my $cockpitTenant = $configuration->getValue('CockpitTenant');

    if (!$sapSys->saveConfig({ 'cockpit_tenant_db' => $cockpitTenant })) {
        $self->getMsgLst()->addWarning("Failed to persist the $gShortProductNameCockpit tenant database name in the user config file '$configFile'.");
    }

    return 1;
}

sub _createCockpitSpace {
    my ($self, $configuration, $xsClient) = @_;
    my $space = $configuration->getValue('CockpitXSSpace');
    my $tenant = $configuration->getValue('CockpitTenant');

    if (!$xsClient->createSpace($space, $tenant)) {
        $self->appendErrorMessage("Failed to create $gShortProductNameXS2 space '$space'...");
        return 0;
    }

    return 1;
}

sub _enableTenantForXSA {
    my ($self, $configuration, $xsClient) = @_;
    my $tenantName = $configuration->getValue('CockpitTenant');
    my $tenantUser = $configuration->getValue('CockpitTenantUser') // 'SYSTEM';
    my $tenantPasswd = $configuration->getValue('CockpitTenantUserPassword');
    my $systemUser = $configuration->getValue('SystemUser');
    my $systemPasswd = $configuration->getValue('SQLSysPasswd');

    if (!$xsClient->enableTenantDB($tenantName, $tenantUser, $tenantPasswd, $systemUser, $systemPasswd)) {
        $self->appendErrorMessage("Failed to enable tenant database '$tenantName' for $gShortProductNameXS2 usage.");
        return 0;
    }

    return 1;
}

sub _handleCockpitTenant {
    my ($self, $configuration) = @_;
    my $instance = $configuration->getOwnInstance();
    my $cockpitTenant = $configuration->getValue('CockpitTenant');
    if ($instance->tenantExists($cockpitTenant)) {
        $self->getMsgLst()->addProgressMessage("Tenant database '$cockpitTenant' already exists. Skipping creation...");
        return 1;
    }

    return $self->_createCockpitTenant($configuration);
}

sub _createCockpitTenant {
    my ($self, $configuration) = @_;
    my $systemUsername = $configuration->getValue('SystemUser');
    my $systemPassword = $configuration->getValue('SQLSysPasswd');
    my $tenantName = $configuration->getValue('CockpitTenant');
    my $tenantPasswd = $configuration->getValue('CockpitTenantUserPassword');
    $self->getMsgLst()->addProgressMessage("Creating tenant database '$tenantName'...");

    my ($rc, $sql) = $configuration->createSQLConnection($systemUsername, $systemPassword, 'SYSTEMDB');
    if (!$rc) {
        $self->appendErrorMessage("Failed to connect to SYSTEMDB.", $sql->getErrMsgLst());
        return 0;
    }

# The double quotes characters in the password must be escaped because if they're not, they interfere with the
# double quotes enclosing the whole password (see $CREATE_TENANT_STATEMENT) thus making the whole SQL statement invalid.
    my $escapedPassword = $tenantPasswd =~ s/"/""/gr;
    my $statement = sprintf($CREATE_TENANT_STATEMENT, $tenantName, $escapedPassword);
    my $logStatement = sprintf($CREATE_TENANT_STATEMENT, $tenantName, '***');
    $self->getMsgLst()->addMessage("Executing '$logStatement'...");

    if (!$sql->execute($statement)) {
        $self->appendErrorMessage("Failed to execute '$logStatement'.", $sql->getErrMsgLst());
        return 0;
    }
    $self->_persistCockpitTenant($configuration);
    return 1;
}

sub _update;
*_update =  \&_install;

sub getComponentName { return "$gProductNameCockpit Stack"; }

sub isInternal { return 0; }

sub getInstallerDir{ return undef; }

sub getHdbInstallExecutable{ return undef; }

sub getHdbUpdateExecutable{ return undef; }

sub getSlppLogFileName { return 'cockpit_stack.log'; }

sub getNumberOfExpectedOutputLines{ return 10; }

sub isPostAddHostsComponent {return 1}

1;
