package LCM::Component::Installable::HDBServer::HDBServerBase;

use File::Spec;
use SDB::Install::Globals qw($gHostRoleStandby
                             $gHostRoleWorker
                             $gProductNameEngine
                             $gPhasePrepareOption
                             $gKeynameEngine
                             $gKeynameRDSync
                             $gLogDir);
use SDB::Install::Configuration::Upgrade;
use LCM::Configuration::GenericStackAny qw ($ini_section_server);
use LCM::PersistencyUtils qw (existsHdbupdPendingUpdate getHdbupdPersistenceFilename existsHdbinstPendingInstall getHdbinstPersistenceFilename);
use LCM::Component qw ( PREPARE_PHASE OFFLINE_PHASE CONFIGURE_PHASE ONLINE_PHASE );
use LCM::Persistence::ServerUpdatePersistenceIniToXMLAdapter;
use LCM::Persistence::ServerInstallPersistenceIniToXMLAdapter;
use LCM::Task::ConvertToMultiDBTask::ConvertServerStateTask;

use strict;

use base qw ( LCM::Component::Installable );

sub getNumberOfExpectedOutputLines{
    return 60;
}

sub _buildArgs {
    my ($self, $instconfig) = @_;
    my $args = $self->SUPER::_buildArgs($instconfig);
    push @{$args}, sprintf("--archive_dir=%s", $self->getPath());
    push @{$args}, sprintf("--instlog_dir=%s", $gLogDir);

    if ($self->isUpdate()) {
        # Build hdbupd args
        push @{$args}, @{$self->_buildArgsUpdate($instconfig)};
    } else {
        # Build hdbinst args
        push @{$args}, @{$self->_buildArgsInstall($instconfig)};
    }

    return $args;
}

sub _buildArgsUpdate {
    my ($self, $instconfig) = @_;
    if($instconfig->isOptimizedExecution()  && $self->getPhase() eq ONLINE_PHASE && $self->_isConvertToMDCStarted($instconfig)){
        $self->_changeOnlinePhaseParameterValue($instconfig, 'SystemUser', LCM::Task::ConvertToMultiDBTask::ConvertServerStateTask::SYSTEM_USER);
        $self->_changeOnlinePhaseParameterValue($instconfig, 'SQLSysPasswd', $instconfig->getValue('SystemDBSQLPassword'));
    }
    $self->_handleIgnorePendingUpdate($instconfig);
    my $skippedParams = $self->_getHdbupdSkippedParams($instconfig);
    my $updateConfig = SDB::Install::Configuration::Upgrade->new();
    my $args = $instconfig->getCmdLineArgsFromInstconfig($updateConfig, $skippedParams, $ini_section_server, 'hdbupd');
    my $existsSelectedServerPlugins = $instconfig->getMediumComponentManager()->existsSelectedServerPluginComponents();
    if ($instconfig->isServerIgnoreCheckComponentDependencies()) {
        $self->_addIgnoreArgument($args, 'check_plugin_dependencies');
    } elsif ($existsSelectedServerPlugins &&
            $instconfig->isOptimizedExecution() &&
            $self->getPhase() eq OFFLINE_PHASE) {
        # Bypass hdbupd check for server plugin dependencies in offline phase
        $self->_addIgnoreArgument($args, 'check_plugin_dependencies');
    }

    if ($self->getIgnorePersistency() &&
        !$instconfig->getIgnore ('check_pending_upgrade') &&
        (!$instconfig->isOptimizedExecution() ||
        $self->getPhase() eq PREPARE_PHASE)) {
        $self->_addIgnoreArgument($args, 'check_pending_upgrade');
    }

    if($instconfig->hasValue('ISCMode')){
        push @{$args}, sprintf('--isc_mode=%s', $instconfig->getValue('ISCMode'));
    }

    return $args;
}

sub _buildArgsInstall {
	my ($self, $instconfig) = @_;
    my $installConfig = SDB::Install::Configuration::NewDB->new();
    my $args = $instconfig->getCmdLineArgsFromInstconfig($installConfig, ['Phase', 'AddHosts'], $ini_section_server, 'hdbinst');

    if ($self->_willHdbinstAddHosts($instconfig)) {
        my @addHosts = split(/\s*,\s*/, $instconfig->getValue('AddHosts'));
        my @hosts = grep(!/extended_storage_worker|extended_storage_standby|ets_worker|ets_standby|streaming|rdsync|xs/, @addHosts);
        if (@hosts) {
            my $addhostsOpt = $instconfig->getOpt('AddHosts');
            my $addhostsVal = join ',', @hosts;
            push @{$args}, sprintf("%s=%s", $addhostsOpt, $addhostsVal);
        }
    }

    return $args;
}

sub _willHdbinstAddHosts {
    my ($self, $instconfig) = @_;
# No need to pass --addhosts to hdbinst if this is phased installation because
# HDBLCM will add all hosts, including the DB ones
    my $isOptimizedExecution = $instconfig->isOptimizedExecution();
    my $isAutoAddEnabled = $instconfig->getValue('AutoAddXS2Roles') && !$instconfig->isSkipped('AutoAddXS2Roles');
    my $addHosts = $instconfig->getValue('AddHosts');
    return !$isOptimizedExecution && !$isAutoAddEnabled && $addHosts;
}

sub _getHdbupdSkippedParams {
    my ($self, $configuration) = @_;
    my $skippedParams = ['Phase'];
    if ($configuration->isConvertToMultiDbRequired()) {
# In case of Update + Convert, --secure_store should not be passed down to hdbupd because the secure store migration
# depends on HANA already being converted to MDC. In this case the migration to LSS is done by hdblcm before the online
# phase - see MigrateSecureStoreTask.pm. For details - Bug: 220403
        push @{$skippedParams}, 'SecureStore';
    }
    return $skippedParams;
}

sub _isConvertToMDCStarted{
    my($self,$configuration) =@_;
    if($self->isUpdate()){
        my $server = $configuration->getSystemComponentManager()->getComponentByKeyName($gKeynameEngine);
        if (defined $server && $server->isConvertToMDCStarted()){
           return 1;
        }
	}
	return 0;
}

sub _handleIgnorePendingUpdate {
	my ($self, $instconfig) = @_;

	if (!$instconfig->getIgnore ('check_pending_upgrade')) {
		return;
	}

	my $handlerIgnore = $instconfig->getOptionIgnore();
	if (!defined $handlerIgnore) {
		return;
	}

	if ($instconfig->isOptimizedExecution() && $self->getPhase() ne PREPARE_PHASE) {
		$handlerIgnore->deleteArgument('check_pending_upgrade');
	}
}

sub _addIgnoreArgument {
	my ($self, $args, $ignoreArgument) = @_;
	my $isFoundIgnoreArg;

	for my $arg (@$args) {
		if ($arg !~ /ignore/) {
			next;
		}

		$isFoundIgnoreArg = 1;
		my $lastChar = substr($arg, -1);

		if ($lastChar ne ',') {
			$arg .= ',';
		}
		$arg .= $ignoreArgument;
	}

	if (!$isFoundIgnoreArg) {
		push @$args, "--ignore=$ignoreArgument";
	}
}

sub getComponentName {
    return $gProductNameEngine;
}

sub requireSystemRestart {
	return 1;
}

sub getSlppLogFileName {
	return 'server.log';
}

sub isInPendingState {
	my ($self, $configuration) = @_;
	if (!defined $configuration) {
		return undef;
	}
	return existsHdbupdPendingUpdate($configuration->getValue('SID'));
}

sub getDefaultPhase {
	return OFFLINE_PHASE;
}

sub getHostRoles {
    return [$gHostRoleStandby,$gHostRoleWorker];
}

sub hasPersistenceFile {
	my ($self, $configuration, $checkCustomFile) = @_;

    if($checkCustomFile){
        return $self->SUPER::hasPersistenceFile($configuration);
    }

	my $sid = $configuration->getValue('SID');
	$sid = $configuration->{'current_sid'} if (!defined $sid);
	my $target = $configuration->getValue('Target');

	return existsHdbupdPendingUpdate($sid) || existsHdbinstPendingInstall($sid, $target);
}

sub getPersistenceXMLObject {
	my ($self, $configuration) = @_;
	my $sid = $configuration->getValue('SID');
	$sid = $configuration->{'current_sid'} if (!defined $sid);
	my $target = $configuration->getValue('Target');
	my $filename;

	if (existsHdbupdPendingUpdate($sid)) {
		$filename = getHdbupdPersistenceFilename($sid);
		my $persistenceXML = new LCM::Persistence::ServerUpdatePersistenceIniToXMLAdapter($filename, $configuration);
		$self->_addWarningMsg($configuration, $persistenceXML);
		return $persistenceXML;
	}

	if (existsHdbinstPendingInstall($sid, $target)) {
		$filename = getHdbinstPersistenceFilename($sid, $target);
		my $persistenceXML = new LCM::Persistence::ServerInstallPersistenceIniToXMLAdapter($filename, $configuration);
		$self->_addWarningMsg($configuration, $persistenceXML);
		return $persistenceXML;
	}

	return undef;
}

sub _addWarningMsg {
	my ($self, $configuration, $persistenceXML) = @_;

	if ($persistenceXML->errorState()) {
		$configuration->getMsgLst()->addWarning($persistenceXML->getErrorString());
	}
}

sub getDeprecatedComponentKeynames {
	my ($self) = @_;
	return [$gKeynameRDSync];
}

sub getDeprecatedPlugins {
	my ($self) = @_;
	if ( ! defined $self->{_parsedDeprecatedPlugins}) {
		$self->{deprecatedPlugins} = $self->parseDeprecatedPlugins();
	}
	return $self->{deprecatedPlugins};
}

sub parseDeprecatedPlugins {
	my ($self) = @_;

	$self->{_parsedDeprecatedPlugins} = 1;
	my $deprecatedPluginsXmlFile = File::Spec->catfile($self->{manifestDir}, 'deprecatedPlugins.xml');
	return undef if (! -e $deprecatedPluginsXmlFile);
	my $deprecatedPlugins = {};
	require XML::LibXML;
	my $parser = XML::LibXML->new();
	my $doc = $parser->parse_file($deprecatedPluginsXmlFile);
	my $root = $doc->getDocumentElement();
	for my $plugin ($root->getElementsByTagName('plugin')) {
		my $key = $plugin->getAttribute('key');
		my $pluginData = {};
		$pluginData->{message} = $plugin->getAttribute('message');
		my ($replacedByElement) = $plugin->getElementsByTagName('replacedby');
		if (defined $replacedByElement) {
			$pluginData->{replacedByKey} = $replacedByElement->getAttribute('key');
			$pluginData->{replacedByMinVersion} = $replacedByElement->getAttribute('min-version');
			$pluginData->{replacedByCaption} = $replacedByElement->getAttribute('caption');
		}
		$deprecatedPlugins->{$key} = $pluginData;
	}
	return $deprecatedPlugins;
}

sub _changeOnlinePhaseParameterValue{
    my ($self,$config,$paramId,$value) = @_;
    my $paramName = $config->getParamName($paramId);
    $config->{params}->{$paramId}->{value} = $value;
    $value = '***' if $config->getType($paramId) =~ qr/.*passwd/;

    $self->getMsgLst()->addMessage( "$paramName changed in online phase, value = '".$value."'");
    return 1;
}

sub canBeCalledInPhase {
    my($self, $phase, $configuration) = @_;
    if ($phase eq PREPARE_PHASE && !$self->isUpdate()) {
# hdbinst doesn't support prepare phase, so skip the prepare phase during installation
        return 0;
    }
    return $self->SUPER::canBeCalledInPhase($phase, $configuration);
}

sub requiresSqlSysPasswd{
    return 1;
}

sub isSigned {
	return 1;
}

sub requiresRootPrivilege {
    return 1;
}

sub isSupportedByLSS {
    return 1;
}

sub getLSSRegistrationPhase {
    my ($self) = @_;
# Register HDB in the configure phase as it is still not installed
# during the offline phase
    return CONFIGURE_PHASE;
}

sub hasOverwrittenComponentPersFile {
    return 0;
}

sub getFlavour {
	my ($self) = @_;
	return $self->getManifest()->getHANAFlavour();
}

sub registerInLss {
    my ($self, $config) = @_;
    my $rc = $self->SUPER::registerInLss($config);
    if($rc && $config->getValue("LSSTrustUnsignedServer")){
        $rc = $self->modifyTrustLevelInLss($config);
    }
    return $rc;
}

1;
