package LCM::Component::Installable::HDBServerPlugin;

use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::System qw(isAdmin);
use SDB::Install::Configuration qw($bool_true_pattern);
use SDB::Install::Configuration::GenericServerPlugin;
use SDB::Install::Globals qw ($gLogDir $gKeynameEngine);
use LCM::Configuration::GenericStackAny qw ($ini_section_plugin);
use LCM::ProcessExecutor;
use LCM::Component qw ( ONLINE_PHASE OFFLINE_PHASE );

use strict;

use base 'LCM::Component::Installable';

sub updateComponent;
*updateComponent =  \&installComponent;

sub installComponent {
    my ( $self, $instconfig, ) = @_;

    my $msg = $self->getMsgLst ()->addProgressMessage ($self->getProgressMsg() . '...');
    my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst ()]);
    $self->initProgressHandler ();

    my $rc = 1;
    my $processExecutor = $self->_constructProcessExecutor($instconfig);
    if (!defined $processExecutor){
        return undef;
    }
    my $environment = $self->prepareHdbInstallerEnvironment ();
    $processExecutor->setOutputHandler($self->getProgressHandler ());
    $processExecutor->setProcessEnvironment ($environment);
    my $exitCode = $processExecutor->executeProgram();
    $self->getMsgLst ()->addMessage(undef, $processExecutor->getMsgLst());
    $self->setLogLocation($self->parseLogFileLocation($processExecutor->getOutputLines()));

    if (!defined $exitCode || $exitCode){
        my $errMsgLst = $self->getHdbInstallerErrorMessages ($processExecutor->getOutputLines());
        my $action = $self->isUpdate ? 'Update' : 'Installation';
        $self->setErrorMessage ($action.' of ' . $self->getComponentName() . ' failed',
            $errMsgLst->isEmpty ? $processExecutor->getErrMsgLst() : $errMsgLst);
        $rc = undef;
    }

    my $action = $self->isUpdate ? 'Update' : 'Install';
    $msg->endMessage (undef, $action. ' ' . $self->getComponentName());
    $self->setMsgLstContext($saveCntxt);
    return $rc;
}

sub _constructProcessExecutor {
	my ($self, $instconfig) = @_;
	my $command = $self->getHdbInstallExecutable();
	my $args = $self->_buildArgs($instconfig);
	if (!defined $args){
		return undef;
	}
	my ($uid, $gid) = (undef, undef);
	my $stdinLines = $instconfig->getXmlPasswordStream();

	push(@$args, ('-b', "--instlog_dir=${gLogDir}"));
	push(@$args, '--read_password_from_stdin=xml') if (defined $stdinLines);

	if (!$isWin && isAdmin()) {
		my $user = new SDB::Install::NewDBUser($instconfig->getSID());
		($uid, $gid) = ($user->uid(), $user->gid());
	}
	return new LCM::ProcessExecutor($command, $args, $stdinLines, undef, undef, $uid, $gid);
}

sub getNumberOfExpectedOutputLines{
    return 30;
}

sub getDefaultSelection{
    my ($self, $stackUpdate) = @_;

	if (!$stackUpdate) {
		return 0;
	}

	my $installedComponent = $self->getInstalledComponent();
	if (!$installedComponent) {
		return 0;
	}

	my $newVersion = $self->getManifest()->getVersionObject();
	my $installedVersion = $installedComponent->getManifest()->getVersionObject();
	return ($installedVersion->isNewerThan($newVersion)) ? 0 : 1;
}

sub _buildArgs {
	my ($self, $instconfig) = @_;

    my $pluginConfig = $self->getExternalHdbInstallerConfiguration ();
    return undef if (!defined $pluginConfig);

	my $paramsToSkip =['NoStart'];
	my $args = $instconfig->getCmdLineArgsFromInstconfig($pluginConfig, $paramsToSkip, $ini_section_plugin);
	if(!$self->_shouldStart($instconfig)){
        if ($instconfig->usePhases() && $self->getCurrentPhase() eq OFFLINE_PHASE) {
            push(@$args, "--system_is_offline");
        }
		push(@$args, "--nostart=on");

	}
	push(@$args, "--archive_dir=$self->{manifestDir}");
	push(@$args, '--ignore_unknown_option');
	push @$args, @{$self->SUPER::_buildArgs($instconfig)};
	return $args;
}

sub getConfigParser {
    my ($self) = @_;
    require SDB::Install::ConfigXMLParser;
    my $path = $self->getInstallerDir() . $path_separator . 'packages';
    my $xmlFile = $path . $path_separator . 'InstallParams.xml';
    my $dtdFile = $path . $path_separator . 'installcfg.dtd';
    if (!-f $xmlFile){
        $self->getErrMsgLst()->addError ("Configuration file '$xmlFile' not found");
        return undef;
    }
    if (!-f $dtdFile){
       $self->getMsgLst()->addWarning ("Configuration DTD file '$dtdFile' not found");
       $dtdFile = undef;
    }
    my $parser = new SDB::Install::ConfigXMLParser ();
    eval{
        $parser->parsefile ($xmlFile, $dtdFile);
    };
    if ($@){
       $self->getErrMsgLst()->addError ("Parsing configuration file '$xmlFile' failed: $@");
       return undef;
    }
    return $parser;
}

sub shouldImportContent {
    my ($self) = @_;
    my $configparser = $self->getConfigParser();
    return 0 if(!$configparser);
    return $configparser->getImportContent() =~ /$bool_true_pattern/i;
}

sub requireSystemRestart {
    return 1;
}

# Override
sub isSystemRestartRequired {
    my ($self) = @_;
# Usually this method checks in the manifest of the component for requires-restart entry
# and the result is used in the ComponentTasklist factory classes to determine whether
# hdblcm should stop/start the system. The plugins do not support phases, but still
# require system restart in the optimized execution cases if they do not support online
# installation/update
    return $self->supportsOnlineOperations() ? 0 : 1;
}

sub requiresSqlSysPasswd{
    my ($self) = @_;
    my $shouldImportContent = $self->shouldImportContent();
    my $shallRestartSystem = $self->_shallSystemBeRestarted();
    my $supportsOnlineOperations = $self->supportsOnlineOperations();

    return $shouldImportContent || ($supportsOnlineOperations && !$shallRestartSystem);
}

sub _shouldStart{
    my ($self, $instconfig) = @_;

    return 0 if $self->_isUpdatingServer();

    my $passThroughPrefix = $instconfig->getPassThroughParamPrefix($ini_section_plugin, 'hdbinst');
    my $passThroughParamIdNoStart = sprintf('%sNoStart', $passThroughPrefix);
    my $isNoStart = $instconfig->getValue($passThroughParamIdNoStart);
    if(!$instconfig->isSkipped($passThroughParamIdNoStart) && ($isNoStart =~ /$bool_true_pattern/)){
        return 0;
    }

    my $isUsingPhases = $instconfig->usePhases();
    if($isUsingPhases) {
        my $currentPhase = $self->getCurrentPhase();
	    return 0 if $currentPhase ne ONLINE_PHASE;
    }

    my $keyname = $self->getComponentKeyName();
    my @remainingPluginsForUpdate = grep{$_ ne $keyname} @{$instconfig->{plugins_list}};
    $instconfig->{plugins_list} = \@remainingPluginsForUpdate;
    my $remainingPluginsCount = scalar(@remainingPluginsForUpdate);
    return ($remainingPluginsCount == 0 || $self->shouldImportContent()) ? 1 : 0;
}

sub getCurrentPhase {
    my ($self) = @_;
    my $phase = $self->getPhase();
    return $phase ? $phase : $self->getDefaultPhase();
}

sub _shallSystemBeRestarted {
    my ($self) = @_;
    return 1 if !$self->supportsOnlineOperations();

    my $mcm = $self->getComponentManager();
    my @restartSystemComponents = grep { $_->isSystemRestartRequired() } @{$mcm->getSelectedComponents()};
    return @restartSystemComponents > 0 ? 1 : 0;
}

sub _isUpdatingServer {
    my ($self) = @_;
    my $componentManager = $self->getComponentManager();
    return 0 if !$componentManager->isHDBServerComponentSelected();

    my $serverComponent = $componentManager->getComponentByKeyName($gKeynameEngine);
    return (defined $serverComponent && $serverComponent->isUpdateToNewerVersion());
}

sub _isServerInstall {
    my ($self) = @_;
    my $componentManager = $self->getComponentManager();
    return 0 if !$componentManager->isHDBServerComponentSelected();

    my $serverComponent = $componentManager->getComponentByKeyName($gKeynameEngine);
    return (defined $serverComponent && !$serverComponent->isUpdate());
}

sub getSlppLogFileName {
    my ($self) = @_;
    my $manifest = $self->{manifest};
    my $componentKey = $manifest->getComponentKey();
    return "$componentKey.log"
}

sub getDefaultPhase {
    my ($self) = @_;
    return OFFLINE_PHASE if $self->_isServerInstall();  # For the sake of phased installation
    return OFFLINE_PHASE if $self->_isUpdatingServer(); # For the sake of phased update
    return OFFLINE_PHASE if $self->_shallSystemBeRestarted();

    my $shouldImportContent = $self->shouldImportContent();
    my $supportsOnlineOperations = $self->supportsOnlineOperations();
    return ($shouldImportContent || $supportsOnlineOperations) ? ONLINE_PHASE : OFFLINE_PHASE;
}

sub supportsOnlineOperations {
    my ($self) = @_;
    my $manifest = $self->getManifest();
    if ($self->isUpdate()) {
        if($manifest->isNonAflServerPluginOnlineUpdatable()) {
            # this is for online updateable server plugins which dont
            # follow the online update protocol of the AFL server plugins.
            # these plugins (eg epmmds) provide the name of a db procedure
            # to call in their manifest. if we have this kind of plugin,
            # we ignore all AFL related settings, if present:
            return 1;
        }
        # For online update of an AFL server plugin to be possible,
        # both the installed AFL plugin and the one, which is
        # being installed, have to support it:
        my $installedPlugin = $self->getInstalledComponent();
        my $installedManifest = $installedPlugin->getManifest();
        return $manifest->isServerPluginOnlineUpdatable() && $installedManifest->isServerPluginOnlineUpdatable();
    }
    else {
        return $manifest->isServerPluginOnlineInstallable();
    }
}

sub getFallbackHdbInstallerConfiguration {
    return SDB::Install::Configuration::GenericServerPlugin->new();
}

sub isSigned {
    return 1;
}

sub isSupportedByLSS {
    return 1;
}

1;
