package LCM::Component::Installable::HDBStudio;

use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::Globals qw($gProductNameStudio $gKeynameEngine);
use SDB::Install::System qw (makedir);
use SDB::Install::Configuration::DatabaseStudio;
use SDB::Install::MsgLst;
use SDB::Install::DirectoryWalker;
use SDB::Install::NewDBUser;
use LCM::Configuration::GenericStackAny qw ($ini_section_studio $ini_section_server);
use LCM::HDBComponentsConfigurator qw($action_register $action_unregister);
use LCM::Component::Registrable qw($RETURN_CODE_SUCCESS $RETURN_CODE_ERROR);
use LCM::ProcessExecutor;
use LCM::ExecutionWarningsObservable;
use File::Copy;
use SDB::Install::TextFile;

use strict;

use base 'LCM::Component::Installable','LCM::Component::Registrable';

sub installComponent {
    my ( $self, $instconfig ) = @_;

    my $actionPrefix = $self->isUpdate ? 'Updat' : 'Install';    
    my $msg = $self->getMsgLst ()->addProgressMessage ($self->getProgressMsg() . '...');
    my $saveCntxt = $self->setMsgLstContext([$msg->getSubMsgLst ()]);
    $self->initProgressHandler ();

    my $rc = 1;
    my $args = $self->_buildArgs($instconfig, 0);
    if (!defined $args){

		return undef;
	}
	my $path = $self->getStudioPath($instconfig);
	my $command = $self->getHdbInstallExecutable();
	my $exer = new LCM::ProcessExecutor($command, $args);
    $exer->setOutputHandler($self->getProgressHandler ());
    my $environment = $self->prepareHdbInstallerEnvironment ();
    $exer->setProcessEnvironment ($environment);
    my $exitCode = $exer->executeProgram();
    $self->getMsgLst ()->addMessage(undef, $exer->getMsgLst());
    $self->setLogLocation($self->parseLogFileLocation($exer->getOutputLines()));
    if ( ! defined $exitCode || $exitCode ) {
        my $errMsgLst = $self->getHdbInstallerErrorMessages ($exer->getOutputLines());
        my $action = $self->isUpdate ? 'Update' : 'Installation';
        $self->setErrorMessage ($action.' of ' . $self->getComponentName() . ' failed',
            $errMsgLst->isEmpty ? $exer->getErrMsgLst() : $errMsgLst);
       	$rc = undef;
  	}

    if ( $rc  && ! $isWin) {
	    if ( $self->_shallDeployStudioRepository($instconfig)) {
	    	my $actionPrefix = $self->isUpdate ? 'Updat' : 'Install';	
			$self->getMsgLst ()->addProgressMessage ($actionPrefix."ing " . $self->getComponentName() . " Update repository...");
            $args = $self->_buildArgs($instconfig, 1);
	        my $warningsDuringDeploy = $self->_deployStudioRepository($instconfig, $args, $command, $exer, $path);
            if ( ! $warningsDuringDeploy->isEmpty() ) {
		    	my $operation = $self->isUpdate() ? 'update' : 'installation';
		    	$self->getMsgLst ()->AddWarning($self->getComponentName()." Update repository ".$operation." failed", $warningsDuringDeploy);
                LCM::ExecutionWarningsObservable->getInstance()->notifyWarning(
                    "Could not deploy the $gProductNameStudio repository to the XS engine. See the hdblcm log file for error details.\n"
                    . "To retry the failed deployment, update the $gProductNameStudio component." );
		    }
	    }

        my $defaultPath = $instconfig->getDefault("StudioPath");	    
	    if($path ne $defaultPath) {
	    	if(!symlink ($path, $defaultPath)){
        		$self->getMsgLst ()->addWarning("Cannot create symbolic link from the default to the custom studio location: $defaultPath -> $path.");
           	}
	    }
	}

    my $action = $self->isUpdate ? 'Update' : 'Install';    
    $msg->endMessage (undef, $action. ' ' . $self->getComponentName());
    $self->setMsgLstContext($saveCntxt);

    return $rc;
}

sub _deployStudioRepository{
	my ($self ,$instconfig, $args, $command, $exer, $path ) = @_;
	
    my $deployRepositoryErrorMessages = new SDB::Install::MsgLst();
    
    my $repository_dir = $instconfig->getValue ('Target') . $path_separator
                             . $instconfig->getValue ('SID') . $path_separator . 'hdbstudio_update';
	my $user = new SDB::Install::NewDBUser ($instconfig->getValue ('SID'));
    my $uid = $user->uid();
    my $gid = $user->gid();
	my $newDirOwnership = {'uid' => $uid,
                      'gid' => $gid,
                      'mode' => 0750};    		
    if (!-d $repository_dir) {
        $self->getMsgLst()->addMessage ("Creating directory '$repository_dir'...");
        if (!makedir($repository_dir, $newDirOwnership)) {
            $deployRepositoryErrorMessages->addMessage( "Cannot create directory '$repository_dir'" );
        	return $deployRepositoryErrorMessages;
        }
    }
    if (!-d $repository_dir) {
    	$deployRepositoryErrorMessages->addMessage( "Directory '$repository_dir' does not exist" );
    	return $deployRepositoryErrorMessages;
    }    
    $args = ["-a", $self->{manifestDir}, "--copy_repository=$repository_dir", "-b"];
    $exer = new LCM::ProcessExecutor($command, $args);
    $exer->setProcessEnvironment ($self->prepareHdbInstallerEnvironment (
    $self->getComponentBatchKey () . '_copy_repository'));
    my $exitCode = $exer->executeProgram();
    $self->getMsgLst ()->addMessage(undef, $exer->getMsgLst());
    if (!defined $exitCode || $exitCode){
        my $errMsgLst = $self->getHdbInstallerErrorMessages ($exer->getOutputLines());
        $deployRepositoryErrorMessages->addMessage( 'Copying repository failed' );
        return $deployRepositoryErrorMessages;
    }

    if ( ! defined $user->possess($repository_dir, 1, 0750)) {
        $deployRepositoryErrorMessages->addMessage( 'Changing permissions of repository directory failed' );
        return $deployRepositoryErrorMessages;
    };

    # moving studio to internal 'studio' directory, see bug 67996
    #/hana/shared/ABC/hdbstudio_update/repository/* -> /hana/shared/ABC/hdbstudio_update/repository/studio/*
    my $repository_copy_from_dir = $repository_dir. $path_separator ."repository";
    my $repository_copy_to_dir = $repository_copy_from_dir. $path_separator ."studio";
    my $tempDir = $repository_copy_from_dir."_";
    
    if ( ! move( $repository_copy_from_dir, $tempDir)){
    	    $deployRepositoryErrorMessages->addMessage( "Cannot move to directory '$tempDir'" );
    	    return $deployRepositoryErrorMessages;
    }
    if ( ! makedir($repository_copy_from_dir, $newDirOwnership)) {
        $deployRepositoryErrorMessages->addMessage( "Cannot create directory '$repository_copy_from_dir'" );
    	return $deployRepositoryErrorMessages;
    }
    if ( ! move( $tempDir, $repository_copy_to_dir)){
    	    $deployRepositoryErrorMessages->addMessage( "Cannot move to directory '$repository_copy_to_dir'" );
    	    return $deployRepositoryErrorMessages;
    }
    #Generate the .xsapp & .xsaccess metafiles hdbeuspack will fail to.
    my $xsappFile=  SDB::Install::TextFile->new($repository_copy_to_dir.$path_separator.".xsapp");
    $xsappFile->addToBuffer("{}\n");
    if (!$xsappFile->write()) {
    	$deployRepositoryErrorMessages->addWarning("Generation of .xsapp deployment metafile failed", $xsappFile);
    }
    my $xsaccess = new SDB::Install::TextFile($repository_copy_to_dir.$path_separator.".xsaccess");
    $xsaccess->addToBuffer("{\n");
    $xsaccess->addToBuffer("\"exposed\":true,\n");
    $xsaccess->addToBuffer("\"authentication\":null\n");
    $xsaccess->addToBuffer("}\n");
    
    if (!$xsaccess->write()) {
		$deployRepositoryErrorMessages->addWarning("Generation of .xsaccess deployment metafile failed", $xsaccess);
	}
    
    if(!$isWin) {
    	my $rc = 1;
    	#TODO ownership change of just installed studio should not happen when(if) the repo is deploywed! Extract and reuse!
        $rc = chmod(0755, $path);
        if(!$rc) {
            $deployRepositoryErrorMessages->addMessage( "Could not chmod $path: $!" );
            return $deployRepositoryErrorMessages;
        }
        $rc = chown($uid, $gid, $path);
        if(!$rc) {
        	$deployRepositoryErrorMessages->addMessage( "Could not chown $path: $!" );
        	return $deployRepositoryErrorMessages;	
       	}
       	# donate everything installed in the file system to <SID>adm and sapsys, and make some directories writable:
        my $dirWalker = SDB::Install::DirectoryWalker->new(
                        undef, undef, undef, undef, undef, undef, # no action matcher, i.e. all inodes encountered are processed; no pruning
                        sub{
		                    # action callback
		                    my $fullentry = $_[3].$path_separator.$_[4];
		                    my $retcode = chown($uid, $gid, $fullentry);
		                    if(!$retcode) {
		                        $deployRepositoryErrorMessages->addMessage( "Could not chown $fullentry: $!" );
		                        return undef;
		                    }
		                    if($_[6] && $_[3] eq $path) {
		                        # we are a directory, under the inst path (not repository).
		                        $retcode = chmod(0755, $fullentry);
		                        if(!$retcode) {
		                            $deployRepositoryErrorMessages->addMessage( "Could not chmod $fullentry: $!" );
		                            return undef;
		                        }
		                    }
		                    return 1;
                        }, undef, undef, # we can use closure here.
		                0,    # we do it depth-first.
		                1,    # dont collect list
		                1     # follow symlinks
                    );
        # dirwalker behaves like unix find: it does not process the root itself:
       	$rc = $dirWalker->findAndProcess($path); #TODO ownership change of just installed studio should not happen when(if) the repo is deploywed! Extract and reuse!
        if(!$rc) {
    		return $deployRepositoryErrorMessages;
        }
        #TODO ownership change of just installed studio should not happen when(if) the repo is deploywed! Extract and reuse!
	    $rc = $dirWalker->findAndProcess($repository_dir);
	    if ( !$rc )  {
    		return $deployRepositoryErrorMessages;
    	}
    }

    if( ! $self->_isDbConnected($instconfig) ){
        $deployRepositoryErrorMessages->addMessage( "Could not deploy studio repository. The System is not started." );
        return $deployRepositoryErrorMessages;
    }

    my $trexInstance = $instconfig->getRefreshedInstance();
    return undef unless $trexInstance;

    my $secondary = $trexInstance->isSecondarySystem($self->getMsgLst(), $deployRepositoryErrorMessages);
	if (defined $secondary && $secondary) {
		$self->getMsgLst ()->addProgressMessage ("Skipping deploy of studio repository because this is a secondary system");
	} 
	elsif( ! $self->_deployRepository($instconfig, $repository_dir, $uid, $gid)  ) {
       	$deployRepositoryErrorMessages->addMessage( "Could not deploy repository." );
    }
    return $deployRepositoryErrorMessages;
}

sub _isDbConnected {
    my ($self, $instconfig)  = @_;

    my $password = $instconfig->getValue ('Password');

    my $sid = $instconfig->getValue ('SID');
    my $sidadmUser = $instconfig->getSysAdminUserName ($sid);
    my $hdbInstance = $instconfig->getOwnInstance (1);
    my $hostname = $isWin ? $hdbInstance->get_host() : undef;
    # $hostname == undef => use unix domain socket, no credentials required
    my $sapControl = new SDB::Install::SAPControl ($hostname,
                                                   $hdbInstance->get_nr(),
                                                   $sidadmUser,
                                                   $password,
                                                   $instconfig->isUseHttps(),
                                                   $instconfig->getValue ('SSOCertificate')
                                                   );
    my $isRunning = $sapControl->isRunning();

    if ( ! defined $isRunning ) {
        $self->getMsgLst()->addWarning ( "Could not check running service.", $sapControl->getErrMsgLst() );
    }
    return $isRunning;
}

sub _deployRepository {
    my ($self, $instconfig, $repository_dir, $uid, $gid) = @_;
    my $msg = $self->getMsgLst ()->addMessage ("Deploying repository...");

    my $rc = $self->_generateRepositoryArchive($instconfig, $repository_dir, $uid, $gid);
    if ( $rc ) {
        $rc = $self->_deployRepositoryArchive($instconfig, $repository_dir, $uid, $gid);    	
    }
    return $rc;	
}	

sub _generateRepositoryArchive {
    my ($self, $instconfig, $repository_dir, $uid, $gid) = @_;
    my $trexInstance = $instconfig->getRefreshedInstance();
    return undef unless $trexInstance;
    my $args = $self->_createHdbeuspackArgs($instconfig, $repository_dir);
    my $deployRepositoryMsgLst = $self->getDeployRepositoryMsgLst();
    my $rc = $trexInstance->runUtilityInHDBEnv('hdbeuspack', $args, $deployRepositoryMsgLst, undef, undef, undef);
    $self->getMsgLst()->appendMsgLst($deployRepositoryMsgLst);
    if (!defined $rc || !$rc) {
        $self->getMsgLst()->addWarning('Generating of repository archive failed', $self->getMsgLst());
        $rc = undef;
    }
    return $rc; 
}

sub _createHdbeuspackArgs{
    my ($self, $instconfig, $repository_dir) = @_;
    my $sid = $instconfig->getValue ('SID');
    
    my @args;
    push (@args, "-o");
    push (@args, $repository_dir);
    push (@args, "-m");
    push (@args, sprintf("sap.hana"));
    push (@args, sprintf("HANA_STUDIO_%s", $sid));
    push (@args, sprintf("%s/repository", $repository_dir)); 
           
    return \@args;
}

sub _deployRepositoryArchive {
    my ($self, $instconfig, $repository_dir, $uid, $gid) = @_;
    my $rc = 1; 
    my $trexInstance = $instconfig->getRefreshedInstance();
    return undef unless $instconfig;
    my $executable = $self->_createHdbupdrepExecutable($instconfig);
    my $args = $self->_createHdbupdrepArgs($instconfig, $repository_dir);
    my $stdinLines = undef;
    my $sqlSysPw   = $instconfig->getValue ('SQLSysPasswd');
    if (defined $sqlSysPw) {
        push (@$args, '--read_password_from_stdin=xml');
        $stdinLines = $instconfig->getXmlPasswordStream(['SQLSysPasswd']);
    }
    my $installerEnv = $self->prepareHdbInstallerEnvironment ( $self->getComponentBatchKey () . '_deploy_archive');
    $rc = $trexInstance->runUtilityInEnv( $executable, $args, $self->getMsgLst(), undef, $stdinLines, $installerEnv, undef, undef, undef, undef, 1);
    if (!defined $rc || $rc!=0){
        $self->getMsgLst()->addWarning('Deploying of repository archive failed',  $self->getMsgLst());
        $rc = undef;
    } else {
        $rc = 1;
    }
    return $rc;      
}

sub _createHdbupdrepExecutable{
    my ($self, $instconfig) = @_;
    my $sid = $instconfig->getValue ('SID');
    my $hdbupdrepDest = sprintf("/usr/sap/%s/SYS/global/hdb/install/bin/hdbupdrep", $sid);
    $hdbupdrepDest .= $isWin ? '.exe' : '';
    return $hdbupdrepDest;
}

sub _createHdbupdrepArgs{
    my ($self, $instconfig, $repository_dir) = @_;
    my $sid = $instconfig->getValue ('SID');
    my $systemUser = $instconfig->getValue ('SystemUser');
    my $systemUserOption = ($systemUser ? "--system_user=$systemUser" : '');
    
    my @args;
    push (@args, "-s");
    push (@args, $sid);
        push (@args, sprintf("--delivery_unit=%s/HANA_STUDIO_%s.tgz", $repository_dir, $sid));
    push (@args, $systemUserOption) if $systemUserOption;
    push (@args, "-b");
    
    return \@args;
}

sub updateComponent;
*updateComponent =  \&installComponent;

sub getNumberOfExpectedOutputLines{
    return $isWin ? 18 : 30;
}

sub _buildArgs {
    my ($self, $instconfig, $isDeployRepository) = @_;

    my $studioConfig = $self->getExternalHdbInstallerConfiguration();
    return undef if (!defined $studioConfig);

    my $hasVMBatchValue = defined $instconfig->getBatchValue('JavaVm');
    my $skipParamIds = $isDeployRepository ? [] : ['TargetRepositoryPath'];
    push(@{$skipParamIds}, 'JavaVm') if !$hasVMBatchValue;
    my $args = $instconfig->getCmdLineArgsFromInstconfig($studioConfig,
                                                    $skipParamIds, $ini_section_studio);

    my $path = $self->getStudioPath($instconfig);
    push @$args, '--path='. $path;
    push @$args, "--archive_dir=$self->{manifestDir}";
    push @$args, @{$self->SUPER::_buildArgs($instconfig)};
    push @$args, '-b';
    return $args;
}

sub getStudioPath {
	my ($self, $instconfig) = @_;

	my $path;
    my $installedComponent = $self->getInstalledComponent();
    if (defined $installedComponent){
        $path = $installedComponent->getPath ();
    } else {
        $path = $instconfig->getValue ('StudioPath')
    }

    return $path;
}

sub getSlppLogFileName {
	return 'studio.log'
}

sub getComponentName {
    return $gProductNameStudio;
}

sub getDefaultSelection {
	return 0; #Disable default selection of the studio component, for legal reasons regarding the contained sap jvm.
}

sub register{
    my($self,$logger,$instconfig, $sharedDir,$sid) = @_;
    my $componentName = $self->getComponentName();
    my $rc = $self->configureHdbComponents ($logger,['hdbstudio'],
        $sharedDir, $sid, $action_register);
    return (!defined $rc) ? $RETURN_CODE_ERROR : $RETURN_CODE_SUCCESS;
}

sub getDeployRepositoryMsgLst(){
	my $parentThis = shift;
	return eval{
		package HDBStudio::DeployRepositoryMsgLst;
		our @ISA=qw(SDB::Install::MsgLst);
		
		sub addError{
		    my ($self,$text,$list,$level) = @_;
		    my $wrnList = $self->convertToWarnings($list);
		    return $self->addWarning($text,$list, $level);
		}
		sub convertToWarnings{
			my ($self,$list) = @_;
			my $result = new SDB::Install::MsgLst();
			foreach my $msg (@{$list->{msg_lst}}){
				if( $msg->{type} eq 'ERR' ){
					$msg->{type}='WRN';
				}
				$result->addMessage($msg->{text},undef,$msg->{type},$msg->{line});
			}
			return $result;
		}
		__PACKAGE__;
	}->new();
}

sub _shallDeployStudioRepository{
	my($self,$configuration) = @_;
	my $isServerSelected = $configuration->getComponentManager()->isComponentSelected($gKeynameEngine);
	my $isServerUpdate = $configuration->isUpdate() && $isServerSelected;
	if($isServerUpdate){
		my $passThroughPrefix = $configuration->getPassThroughParamPrefix($ini_section_server, 'hdbupd');
		my $passThroughParamIdNoStart = sprintf('%sNoStart', $passThroughPrefix);
		my $isNoStart = $configuration->getValue($passThroughParamIdNoStart);
		return !$isNoStart;
	}
	return $configuration->getValue('StudioRepository') ;
	
}

sub getFallbackHdbInstallerConfiguration {
    return SDB::Install::Configuration::DatabaseStudio->new();
}

sub requiresSqlSysPasswd{
    return 1;
}

sub isSigned {
    return 1;
}

1;
