package LCM::Component::Installable::SDA;

use SDB::Install::SysVars qw($isWin $path_separator);
use SDB::Install::System qw(copy_tree deltree);
use SDB::Install::Configuration::NewDB;
use SDB::Install::SAPSystem qw(CollectSAPSystems);
use SDB::Install::Globals qw($gProductName $gSapsysGroupName);
use LCM::ProcessExecutor;
use File::Spec;
use File::Basename qw (dirname basename);
use strict;

use base 'LCM::Component::Installable';

sub getDefaultSelection{
    my ($self, $stackUpdate) = @_;

	if (!$stackUpdate) {
		return 0;
	}

	my $installedComponent = $self->getInstalledComponent();
	if (!$installedComponent) {
		return 0;
	}

	my $newVersion = new SDB::Install::Version(split('\.', $self->getVersion()));
	my $installedVersion = new SDB::Install::Version(split('\.', $installedComponent->getVersion()));
	return ($installedVersion->isNewerThan($newVersion)) ? 0 : 1;
}

sub getNumberOfExpectedOutputLines {
    return 5;
}

sub installComponent {
	my ( $self, $instconfig ) = @_;

	my $rc  = 1;
	my $msg = $self->getMsgLst()->addProgressMessage($self->getProgressMsg() . '...');
	my $saveCntxt = $self->setMsgLstContext( [ $msg->getSubMsgLst() ] );
	my $systems = CollectSAPSystems ($self->getMsgLst(), 1);
	
	#create federation directory if it doesn't exist
	my $sid = $instconfig->getValue("SID");
    my $sapSystem = $systems->{$sid};
    if (not defined $sapSystem) {
        $self->setErrorMessage ("No SAP system '$sid' found");
        $msg->endMessage( undef, 'Installation failed ');
		$self->setMsgLstContext($saveCntxt);
        return undef;
    }
	
	my $federationDir = File::Spec->join($sapSystem->get_globalSidDir(), "federation");
	
	if (not -d $federationDir) {
		$rc = mkdir $federationDir;
		
		if (not defined $rc) {
			$self->setErrorMessage ("Unable to create directory '$federationDir'");
			$msg->endMessage( undef, 'Installation failed ');
			$self->setMsgLstContext($saveCntxt);
			return undef;
		}
		
		if (!$isWin) {
			my $sidAdm = $instconfig->getSysAdminUserName($sid);
			my (undef, undef, $uid, $gid) = getpwnam($sidAdm);
			chown $uid, $gid, $federationDir or return undef; 
		} 
	}
	
	#call install script
	my $exer = $self->createProcessExecutor( $instconfig );
    $self->initProgressHandler ();
    $exer->setOutputHandler($self->getProgressHandler ());
    
    my $exitCode = $exer->executeProgram();
	$self->getMsgLst()->addMessage( undef, $exer->getMsgLst() );
	$self->setLogLocation($self->parseLogFileLocation($exer->getOutputLines()));
	if ( !defined $exitCode || $exitCode ) {
        my $errMsgLst = $self->getHdbInstallerErrorMessages ($exer->getOutputLines());
        $self->setErrorMessage ('Installation of ' . $self->getComponentName() . ' failed',
            $errMsgLst->isEmpty ? $exer->getErrMsgLst() : $errMsgLst);
        $rc = undef;
	}
	
	$msg->endMessage( undef, 'Install ' . $self->getComponentName() );
	$self->setMsgLstContext($saveCntxt);
	return $rc;
}

sub _buildArgs {
	my ($self, $instconfig) = @_;
	my $args = [];
	
	push @$args, "-s" , $instconfig->getValue("SID");
	push @$args, @{$self->SUPER::_buildArgs($instconfig)};
	return $args;
}

sub updateComponent;
*updateComponent =  \&installComponent;

sub getComponentName {
	return "$gProductName Smart Data Access";
}

sub getExecutable{
    my ($self) = @_;
    return $self->getInstallerDir() . $path_separator . ($isWin ? 'install.bat' : 'install.sh');
}

sub createProcessExecutor{
	my ($self, $instconfig, $installDir) = @_;
	my $args = $self->_buildArgs($instconfig);
	my $command = $self->getExecutable();
	
	if ($isWin || $self->isSidAdmUserExecution()) {
		return new LCM::ProcessExecutor($command, $args);
	}

	#sda instalation requires sidadm user
	my $sidAdm = $instconfig->getSysAdminUserName($instconfig->getValue('SID'));
	my $uid = getpwnam($sidAdm);
	my $gid = getgrnam($gSapsysGroupName);

	return new LCM::ProcessExecutor($command, $args, undef, undef, undef, $uid, $gid);
}

sub getSlppLogFileName {
	return 'sda.log'
}

sub isSigned {
	return 1;
}

1;
