package LCM::Component::Installable::Streaming;

use SDB::Install::SysVars qw ($path_separator);
use SDB::Install::System qw (makedir);
use LCM::ProcessExecutor;
use SDB::Install::SAPSystem;
use File::Spec;
use File::Basename qw (dirname);
use SDB::Install::Globals qw ($gDirNameStreaming
                              $gHostRoleStreaming
                              $gProductNameStreaming
                              $gLogDir);
use strict;

use base 'LCM::Component::Installable';

our $installScript = 'strinst.sh';

sub installComponent {
    my ( $self, $instconfig) = @_;
    return $self->_executeAction( $instconfig );
}

sub updateComponent {
    my ( $self, $instconfig) = @_;
    return $self->_executeAction( $instconfig );
}

sub _executeAction {
	my ( $self, $instconfig) = @_;
	$self->_startMessages( $instconfig );
	my $executor = $self->createProcessExecutor( $instconfig );
	return undef if ( ! defined $executor );
    $self->initProgressHandler();

    my $exitCode;
    {	# due to a bug in InstallAnywhere $DISPLAY has to be unset
        local %ENV = %ENV;
        delete $ENV{DISPLAY};
        $exitCode = $executor->executeProgram();
    }

	return $self->_handleExecutionResult($executor, $exitCode);
}

sub _startMessages {
    my ( $self, $instconfig) = @_;
    my $msg = $self->getMsgLst()->addProgressMessage($self->getProgressMsg() . '...');
    my $saveCntxt = $self->setMsgLstContext( [ $msg->getSubMsgLst() ] );
    $self->_setMsg( $msg );
    $self->_setSaveContext( $saveCntxt );
}

sub createProcessExecutor{
    my ($self, $instconfig, ) = @_;
    my $sid = $instconfig->getValue("SID");
	my $systems = CollectSAPSystems (undef, 1);
    my $sapSystem = $systems->{$sid};
    if (not defined $sapSystem) {
        $self->setErrorMessage ("No SAP system '$sid' found");
        $self->_getMsg()->endMessage( undef, $self->getAction() . ' ' . $self->getComponentName());
        $self->setMsgLstContext($self->_getSaveContext());
        return undef;
    }

    $self->{streamingDir} = join($path_separator, $sapSystem->get_globalSidDir(), $gDirNameStreaming);
    my $streamingDir = $self->{streamingDir};

    my $user = $sapSystem->getUser();
    my $uid = $user->uid();
    my $gid = $user->gid();
    my $cfg = {'uid' => $uid, 'gid' => $gid};

    if ( ! $self->isUpdate() && ! -d $streamingDir ) {
        if ( ! defined makedir($streamingDir, $cfg) ) {
            $self->setErrorMessage ("Unable to create directory '$streamingDir'", $cfg);
            $self->setMsgLstContext($self->_getSaveContext());
            $self->_getMsg()->endMessage( undef, $self->getAction() . ' ' . $self->getComponentName());
            return undef;
        }
    }

    my $inst_number = $sapSystem->getNewDBInstanceNr();
    my $instance = $sapSystem->getNewDBInstances ()->[0];
    my $basepath_streaming = $instconfig->getValue('BasepathStreaming');
    $instance->createStreamingBasepath($basepath_streaming);

    my $hostname = $instance->get_host();
    my $command = $self->getExecutable ();
    my $args = $self->_buildArgs($instconfig, $sapSystem->get_globalSidDir() , $inst_number, $hostname, $basepath_streaming);
    my $databaseUserPassword = $instconfig->getValue ('SQLSysPasswd');
    my $stdinPassword = [$databaseUserPassword];

    if (!$self->isUpdate() && !$instconfig->isSkipped('StreamingClusterManagerPassword')){
        push(@{$stdinPassword}, $instconfig->getValue ('StreamingClusterManagerPassword'));
    }
    if ($self->isSidAdmUserExecution()) {
    	($uid, $gid) = (undef, undef); # no need to switch to sidadmin's uid/gid
    }

    my $executor = new LCM::ProcessExecutor($command, $args, $stdinPassword, undef, undef, $uid, $gid);
    $executor->setMsgLstContext ([$self->_getMsg()->getSubMsgLst()]);
    $executor->setOutputHandler($self->getProgressHandler ());

    return $executor;
}

sub getExecutable {
    my ($self) = @_;
    return File::Spec->catfile($self->getInstallerDir(), $installScript);
}

sub _buildArgs {
    my ($self, $instconfig, $installDir, $inst_number, $hostname, $basepath_streaming) = @_;

    my @args = ('-path', $installDir, '-sid', $instconfig->getSID (), '-inst', $inst_number, '-hdbhost', $hostname, '-log_dir', $gLogDir);
    my $user = $instconfig->getValue ('SystemUser');
    if (defined $user && ($user ne "SYSTEM")) {
        push @args, ('-hdbuser', $user);
    }
    if ($self->isUpdate ()){
        push @args, ('-action', 'update');
    }else{
        push @args, ('-basepath_streaming', $basepath_streaming);
    }
    push @args, @{$self->SUPER::_buildArgs($instconfig)};
    return \@args;
}

sub _handleExecutionResult {
    my ($self, $executor, $exitCode) = @_;
    $self->setLogLocation($self->parseLogFileLocation($executor->getOutputLines()));
    
    if (!defined $exitCode){
        $self->setErrorMessage ("Cannot execute " . $self->getComponentName() . " installer", $executor->getErrMsgLst ());
        $self->_getMsg()->endMessage( undef, $self->getAction() . ' ' . $self->getComponentName());
        $self->setMsgLstContext($self->_getSaveContext());
        return undef;
    }

    if ($exitCode != 0){
        $self->setErrorMessage ("Error performing ".$self->getComponentName()." installer: exit code = $exitCode", $executor->getMsgLst ());
        $self->_getMsg()->endMessage( undef, $self->getAction() . ' ' . $self->getComponentName());
        $self->setMsgLstContext($self->_getSaveContext());
        return undef;
    }
    my $msglst = $self->_getMsg()->getSubMsgLst;
    my $streamingDir = $self->{streamingDir};
    $msglst->addMessage ("For further details please see the ".$self->getComponentName()." installer log files in '$streamingDir/log'."),
    $self->_getMsg()->endMessage( undef, $self->getAction() . ' ' . $self->getComponentName() );
    $self->setMsgLstContext($self->_getSaveContext());

    return 1;
}

sub parseLogFileLocation {
    my ($self, $outputLines) = @_;
    if (!defined $outputLines){
        return undef;
    }
    my $log_location;
    my $pattern = "Log file written to [\']*([^\']*)[\']*";
    for my $line (reverse @$outputLines){
        if (substr ($line, -1) eq '.') {
           chop $line;
        }
        ( $log_location ) = $line =~ /$pattern/i;
        
        if (defined $log_location) {
            return $log_location;
        }
    }
}

sub getNumberOfExpectedOutputLines{
    return 20;
}

sub getDefaultSelection{
    my ($self, $stackUpdate) = @_;
    return $stackUpdate ? $self->SUPER::getDefaultSelection ($stackUpdate) : 0;
}

sub getSlppLogFileName {
	return 'streaming.log'
}

sub getComponentName {
	return $gProductNameStreaming;
}

sub requireSystemRestart {
    return 0;
}

sub getHostRoles {
    return [$gHostRoleStreaming];
}

sub requiresSqlSysPasswd{
    return 1;
}

sub isSigned {
    return 1;
}

1;
