package LCM::Configuration::BaseConfigurationSystemCollection;

use strict;
use warnings;

use LCM::PersistencyUtils qw( existsHdblcmPendingUpdate
                              existsHdblcmPendingInstall
                              existsHdbinstPendingInstall
                              existsPendingComponentsUpdate
                              getAllPendingUpdateInfo) ;
use SDB::Install::System qw(isSidadmin isAdmin);
use SDB::Install::SAPSystem qw (@STEP_NAMES getPendingInfoFromPersFile);
use SDB::Install::Configuration::NewDB;
use LCM::App::ApplicationContext;
use SDB::Install::Globals qw($gFlavourPlatform  $gFlavourExpress $gFlavourCockpit $gSAPLocation $gKeynameLSS);
use LCM::Component qw($componentKeynameToBatchKey);
use SDB::Install::System qw(listDir);
use File::stat qw(stat);
use File::Spec;
use Fcntl qw(:mode);

sub new{
    my($class,$config) = @_;
    my $self = bless({},$class);
    $self->setConfiguration($config);
    return $self;
}

sub getHdbLcmPendingInstallations {
    my ($self) = @_;
    my @pendingHdbLcmInstallations = ();
    my @pendingServerInstallations = ();

    my $newDBConfig = SDB::Install::Configuration::NewDB->new();
    my $allSystems = $newDBConfig->getCollectSAPSystems(1);
    foreach my $currSID (sort keys %{$allSystems}) {
        next if (!SDB::Install::System::isAdmin() && !SDB::Install::System::isSidadmin($currSID));
        my $currSystem = $allSystems->{$currSID};
        $newDBConfig->{current_sid} = $currSID;
        $newDBConfig->{params}->{Target}->{value} = $currSystem->get_target();

        if ( $newDBConfig->pers_exists() && LCM::PersistencyUtils::existsHdblcmPendingInstall( $currSID, $currSystem->get_target() )) { # Server pending installation
            push(@pendingServerInstallations, $currSystem);
            push(@pendingHdbLcmInstallations, $currSystem);
            next;
        }

        next if ! LCM::PersistencyUtils::existsHdblcmPendingInstall( $currSID, $currSystem->get_target() ) ||
                ! LCM::PersistencyUtils::existsPendingComponentsUpdate( $currSID );

        # Component pending installation
        push(@pendingHdbLcmInstallations, $currSystem);
    }

    return (\@pendingHdbLcmInstallations, \@pendingServerInstallations); #TODO make it more normal
}

sub detectPendingInstallations{
    my $self = shift;
    my $flavour =  LCM::App::ApplicationContext::getFlavour();
    return if(!defined $flavour );
    die "Non supported flavour: $flavour" if($flavour ne $gFlavourPlatform && $flavour ne $gFlavourExpress && $flavour ne $gFlavourCockpit);
    my ($pendingInstallations) = $self->getHdbLcmPendingInstallations();
    foreach my $currPending (@$pendingInstallations) {
        my $currSid = $currPending->get_sid();
        my $systemFlavour =  $currPending->getHANAFlavour();
        next if(!defined $systemFlavour || $flavour ne $systemFlavour);

        my $info = $self->getProductInfo($currPending,1);
        my $instconfig = new LCM::Configuration::GenericStackInstallConfiguration();
        $instconfig->{sapSys}                    = $currPending;
        $instconfig->{current_sid}               = $currSid;
        $instconfig->{params}->{SID}->{value}    = $currSid;
        $instconfig->{params}->{Target}->{value} = $currPending->get_target();
        my $instPendingInfo =  LCM::PersistencyUtils::getAllPendingInstallInfo($instconfig, 0, $instconfig->getMsgLst());
        $info .= "\n$instPendingInfo" if($instPendingInfo);
        $self->{pendingInstallSystems}->{$currSid} = {
           'sapSys' => $currPending,
           'info' => $info,
           'instanceNr' => LCM::PersistencyUtils::getPersistenceEntryValue($instconfig,'InstanceNumber'),
        };
    }

}

sub getPendingInstallations{
    my $self = shift;
    $self->detectPendingInstallations();
    return () if !defined $self->{pendingInstallSystems};
    my @sids  = keys %{$self->{pendingInstallSystems}};
    return sort(@sids);
}

sub getPendingInstallSystemInfo{
    my($self,$sid) = @_;
    return undef if(!defined $self->{pendingInstallSystems}->{$sid});
    return $self->{pendingInstallSystems}->{$sid}->{'info'};
}

sub getPendingInstallSystem{
    my($self,$sid) = @_;
    return undef if(!defined $self->{pendingInstallSystems}->{$sid});
    return $self->{pendingInstallSystems}->{$sid}->{'sapSys'};
}

sub detectInstalledSystems{
    my ($self,$flavour) = @_;

    my $getHandler = new SDB::Install::Configuration::NewDB();
    my $installedSystems     = {};
    my @installedSIDs = $getHandler->getNewDBInstallations( $installedSystems, [] );
    foreach my $currSID (@installedSIDs) {
        next if (!SDB::Install::System::isAdmin() && !SDB::Install::System::isSidadmin($currSID));
        my $sapSys = $installedSystems->{$currSID};
        my $hanaFlavour = $sapSys->getHANAFlavour();
        next if !defined $hanaFlavour || $flavour ne $hanaFlavour;
        next if LCM::PersistencyUtils::existsHdblcmPendingInstall($currSID, $sapSys->get_target()) &&
            (LCM::PersistencyUtils::existsHdbinstPendingInstall($currSID, $sapSys->get_target()) || LCM::PersistencyUtils::existsPendingComponentsUpdate($currSID));

        my $isPending = LCM::PersistencyUtils::existsHdblcmPendingUpdate($currSID);
        my $info   = $self->getProductInfo($sapSys,$isPending);
        my $updPendingInfo = LCM::PersistencyUtils::getAllPendingUpdateInfo($sapSys, '', '', 1, $self->getConfiguration()->getMsgLst());
        $self->addInstalledSystem($currSID,$sapSys,$info,$isPending,$updPendingInfo);
    }
}

sub addInstalledSystem{
    my ($self,$sid,$sapSys,$info,$isPendingUpdate,$updPendingInfo) = @_;
    $self->{installedSystems}->{$sid} = {
           'info' => $info,
           'sapSys' => $sapSys,
           'isPendingUpdate' =>$isPendingUpdate,
           'pendingUpdateInfo' => $updPendingInfo
    };
}

sub getInstalledSystems{
    my ($self,$flavour) = @_;
    my $currentFlavour = $flavour // LCM::App::ApplicationContext::getFlavour();
    $self->detectInstalledSystems($currentFlavour);
    return () if ( !defined $self->{installedSystems});
    my @sids  = keys %{$self->{installedSystems}};
    return sort( grep{ $self->getInstalledSystemFlavour($_) eq $currentFlavour } @sids); 
}

sub getInstalledSystemInfo{
    my($self,$sid,$withPendingInfo) = @_;
    return undef if(!defined $self->{installedSystems}->{$sid});
    my $info =  $self->{installedSystems}->{$sid}->{'info'};
    $withPendingInfo //= 1;
    my $updPendingInfo = $self->getPendingUpdateInfo($sid);
    if($withPendingInfo && $updPendingInfo) {
            $info .= "\n";
            $info .= "$updPendingInfo";
    };
    return $info;
}

sub getConfiguration{
    my $self = shift;
    return $self->{_config};
}

sub setConfiguration{
    my ($self,$config) = @_;
    $self->{_config} = $config;
}

sub getLSSPendingInstallations {
    my ($self) = @_;
    if(!defined $self->{pendingLSSInstallations}){
        if (!File::stat::stat($gSAPLocation)){
            return ();
        }
        my $fileList = SDB::Install::System::listDir($gSAPLocation);
        my $cmpBatchKey = $componentKeynameToBatchKey->{$gKeynameLSS};
        my $lssPersFilePattern = 'pending_'.$cmpBatchKey.'\.xml$';

       my @pendingLSSInstSids = ();
       foreach my $file(@$fileList){
           if($file =~ /$lssPersFilePattern/){
               my $fullFilename = File::Spec->catfile($gSAPLocation,$file);
               my $fileStat = File::stat::stat($fullFilename);
               next if(!S_ISREG($fileStat->mode));
               my $sid = substr $file, 0, 3;
               push(@pendingLSSInstSids,$sid);
           }
       }
        foreach my $sid(@pendingLSSInstSids){
            my $instconfig = new LCM::Configuration::GenericStackInstallConfiguration();
            $instconfig->{current_sid} = $sid;
            $instconfig->setValue('SID', $sid);
            my $instPendingInfo = LCM::PersistencyUtils::getAllPendingInstallInfo($instconfig, 0, $instconfig->getMsgLst());
            my $info .= "";
            $info .= "$instPendingInfo" if($instPendingInfo);
            my $pendingInfo = $info. "\n";
            my $instanceNr = LCM::PersistencyUtils::getPersistenceEntryValue($instconfig,'InstanceNumber');
            my $target = LCM::PersistencyUtils::getPersistenceEntryValue($instconfig,'Target');
            $self->addPendingLSSInstallation($sid,$pendingInfo,$instanceNr,$target);
        }
    }
    my @sids  = keys %{$self->{pendingLSSInstallations}};
    return sort(@sids);
}

sub addPendingLSSInstallation{
    my($self,$sid,$pendingInfo,$instanceNr,$target) = @_;
    $self->{pendingLSSInstallations}->{$sid} = {
        'info' => $pendingInfo,
        'instanceNr' => $instanceNr,
        'target' => $target
    };
}

sub getInstalledSystemFlavour{
    my($self,$sid) = @_;
    my $sapSys = $self->getInstalledSystem($sid);
    return undef if(!defined $sapSys);
    return $sapSys->getHANAFlavour();
}

sub isInstalledSystemPendingUpdate{
    my($self,$sid) = @_;
    return undef if(!defined $self->{installedSystems}->{$sid});
    return $self->{installedSystems}->{$sid}->{'isPendingUpdate'};
}

sub isInstalledSystem{
    my($self,$sid) = @_;
    return defined $self->{installedSystems}->{$sid};
}

sub isPendingInstallation{
    my($self,$sid) = @_;
    return defined $self->{pendingInstallSystems}->{$sid};
}

sub isPendingLSSInstallation{
    my($self,$sid) = @_;
    my @pendingLSS = $self->getLSSPendingInstallations();
    return grep{$_ eq $sid} @pendingLSS;
}

sub getInstalledSystem{
    my($self,$sid) = @_;
    return undef if(!defined $self->{installedSystems}->{$sid});
    return $self->{installedSystems}->{$sid}->{'sapSys'};
}

sub getSystemVersion{
    my($self,$sid) = @_;
    my $system = $self->isPendingInstallation($sid) ? $self->getPendingInstallSystem($sid) : $self->getInstalledSystem($sid);
    return undef if(!defined $system);
    my $version = $system->GetVersion();
    return $version // 'Version is unknown';
}

sub getPendingInstallationInstanceNr{
    my($self,$sid) = @_;
    return undef if(!defined $self->{pendingInstallSystems}->{$sid});
    return $self->{pendingInstallSystems}->{$sid}->{'instanceNr'};
}

sub getPendingLSSInstallationInfo{
    my($self,$sid) = @_;
    return undef if(!defined $self->{pendingLSSInstallations}->{$sid});
    return $self->{pendingLSSInstallations}->{$sid}->{'info'};
}

sub getLSSPendingInstallationInstanceNr{
    my($self,$sid) = @_;
    return undef if(!defined $self->{pendingLSSInstallations}->{$sid});
    return $self->{pendingLSSInstallations}->{$sid}->{'instanceNr'};
}

sub getPendingLSSInstallationTarget{
    my($self,$sid) = @_;
    return undef if(!defined $self->{pendingLSSInstallations}->{$sid});
    return $self->{pendingLSSInstallations}->{$sid}->{'target'};
}

sub getPendingUpdateInfo{
    my($self,$sid) = @_;
    return undef if(!defined $self->{installedSystems}->{$sid});
    return $self->{installedSystems}->{$sid}->{'pendingUpdateInfo'};
}

sub getProductInfo{
    ...
}

1;