package LCM::Configuration::Cockpit::CockpitOnHANAConfiguration;

use strict;
use parent 'LCM::Configuration::GenericStackUpdateConfiguration';
use SDB::Install::Globals qw($gCockpitStackParametersConstraint
                             $gKeynameEngine
                             $gKeynameInstaller
                             $gKeynameXS2
                             $gKeynameCockpitStack
                             $gXSASpaceSAP
                             $gFlavourPlatform
                             $gProductNamePlatform
                             $gProductNameCockpit
                             $gShortProductNameCockpit
                             $gShortProductNameXS2
                             $gEvenShorterProductNameXSA
                             $gOptionOptimizedExecutionMode);
use SDB::Install::MsgLst;
use SDB::Install::Version;
use SDB::Install::Configuration qw($bool_false_pattern);
use LCM::Configuration::GenericStackAny qw($ini_section_cockpit);
use LCM::Configuration::ValueChangeListeners::Cockpit::CockpitTenantListener;
use LCM::Configuration::ValueChangeListeners::Cockpit::CockpitSpaceListener;
use LCM::Configuration::ValueChangeListeners::Cockpit::CockpitParamsListener;
use LCM::Configuration::ValueChangeListeners::Cockpit::CockpitSelectionListener;
use LCM::Configuration::ValueChangeListeners::Cockpit::DetectXSSpaceListener;
use LCM::Configuration::ValueChangeListeners::Cockpit::CockpitSpaceGUIDListener;
use LCM::Configuration::ValueChangeListeners::InstallUpdate::SelectedXs2ComponentsListener;
use LCM::Configuration::ParametersCreator qw(GetParamCockpitTenant GetParamCockpitXSSpace GetParamCockpitSpaceGUID);


sub new {
    my $self = shift->SUPER::new(@_);
    $self->defineCockpitStackParams();
    $self->getPersistenceManager()->registerParamsToPersist();
    $self->_addListeners();

    return $self;
}

sub checkExecutionMode {
    my ($self, $value) = @_;
    if ($value ne $gOptionOptimizedExecutionMode) {
        $self->setErrorMessage("$gProductNameCockpit installation on top of $gProductNamePlatform is only possible in $gOptionOptimizedExecutionMode update execution mode.");
        return 0;
    }
    return 1;
}

sub isCockpitOnPlatformAction {
    return 1;
}

sub checkFlavourCompatibility {
# Always return 1 as the flavours will be always different
    return 1;
}

sub handleSelectionOfServerComponent {
    my ($self) = @_;
    my $mcm = $self->getComponentManager();
    my $componentsToRestrict = [ $gKeynameEngine, $gKeynameInstaller ];
    for my $cmpKey (@{$componentsToRestrict}) {
        my $component = $mcm->getComponentByKeyName($cmpKey);
        next if(!defined($component));
        $component->selectComponent(0);
        $component->setCanSelectComponent(0);
    }
}

sub CheckParams {
    my ($self, $batchMode) = @_;
    if (!$self->SUPER::CheckParams($batchMode)) {
        return 0;
    }

    if (!$self->isXSWorkerHostAvailable()) {
# If neither xs worker host exists in the system nor one will be added during this execution,
# the Cockpit installation cannot continue
        $self->setErrorMessage("$gProductNameCockpit installation requires $gShortProductNameXS2 worker role.");
        return 0;
    }

    return 1;
}

sub _recomputeRequiredParameters {
    my ($self) = @_;
    $self->SUPER::_recomputeRequiredParameters();

    my $mcm = $self->getComponentManager();
    if ($self->hasXSA() && !$mcm->willInstallComponent($gKeynameXS2, $self)) {
        $self->enableRequiredXSAParameters();
    }
}

sub enableRequiredXSAParameters {
    my ($self) = @_;
    my @relevantParams = ('SystemUser', 'SQLSysPasswd', 'OrgName', 'OrgManagerUser', 'OrgManagerPassword');
    $self->setSkip($_, 0) for @relevantParams;
    $self->setType('OrgManagerPassword', 'passwd');
}

sub setCockpitTenantUser {
    my ($self, $value) = @_;
    $self->setUsernameToSQLUserPassword('CockpitTenantUserPassword', $value);
    $self->{params}->{CockpitTenantUser}->{value} = $value;
    return 1;
}

sub checkCockpitXSSpace {
    my ($self, $space) = @_;
    if ($space eq $gXSASpaceSAP) {
        my $string = $self->getString('CockpitXSSpace');
        $self->setErrorMessage("$string '$space' is not allowed");
        return 0;
    }
    return 1;
}

sub checkCockpitTenant {
    my ($self, $tenantName) = @_;
    if ($tenantName !~ /^\w+$/) {
        $self->setErrorMessage("The name of the tenant must consist of uppercase letters, lowercase letters, numbers and _ (underscore).");
        return 0;
    }
    return 1;
}

sub checkCockpitTenantUserPassword {
    my ($self, $value) = @_;
    my $msgLst = SDB::Install::MsgLst->new();
    my $rc = undef;
    if ($self->getType('CockpitTenantUserPassword') =~ /initial/) {
        $rc = $self->complyDefaultSqlPasswordPolicy($value,  $self->getErrMsgLst(), 'CockpitTenantUserPassword');
    } else {
        $rc = $self->_checkExistingTenantCredentials($value);
    }
    return $rc;
}

sub getCannotBeSelectedMessage {
    return '';
}

sub _checkExistingTenantCredentials {
    my ($self, $tenantDbPassword) = @_;
    my $tenantName = $self->getValue('CockpitTenant');
    my $instance = $self->getOwnInstance();
    if (!$instance || !$instance->tenantExists($tenantName)) {
        $self->getErrMsgLst()->addError("Failed to retrieve information about tenant db '$tenantName'");
        return 0;
    }
    my $user = $self->getValue('CockpitTenantUser');
    return $self->checkDatabaseUserPassword($user, $tenantDbPassword, 'CockpitTenantUserPassword', undef, undef, $tenantName);
}

sub defineCockpitStackParams {
    my ($self) = @_;
    my $order = scalar keys %{ $self->{params} };
    my $cockpitStackParams = {
        'CockpitXSSpace'            => GetParamCockpitXSSpace($order++, $ini_section_cockpit, $gCockpitStackParametersConstraint),
        'CockpitSpaceGUID'          => GetParamCockpitSpaceGUID($order++, $ini_section_cockpit, $gCockpitStackParametersConstraint), # Internal usage
        'CockpitTenant'             => GetParamCockpitTenant($order++, $ini_section_cockpit, $gCockpitStackParametersConstraint),
        'CockpitTenantUser'         => $self->getParamCockpitTenantUser($order++, $ini_section_cockpit, $gCockpitStackParametersConstraint),
        'CockpitTenantUserPassword' => $self->getParamCockpitTenantUserPassword($order++, $ini_section_cockpit, $gCockpitStackParametersConstraint, 'CockpitTenantUser'),
    };
    map { $self->{params}->{$_} = $cockpitStackParams->{$_} } keys %$cockpitStackParams;

# Hide the Tenant User parameter for now as it does not exist in the XSA client command
# For further info - check Bug 186671
    $self->setHidden('CockpitTenantUser', 1);
    $self->setInteractive('CockpitTenantUser', 0);

# This redistribution of the XS2 parameters is necessary because we need to know
# the CockpitXSSpace (or its GUID) before the SelectedXS2Appliacations so that we 
# are able to detect the apps in this space and tell which one is available and
# will be updated and which one must be installed
    $self->defineXSParamsAfterSelectedXSApps();
    return 1;
}

sub defineXS2Params {
    my ($self) = @_;
    $self->defineXSParamsBeforeSelectedXSApps();
}

sub hasXSA {
    my ($self) = @_;
    my $scm = $self->getSystemComponentManager();
    if (defined($scm) && $scm->isComponentAvailable($gKeynameXS2)) {
        return 1;
    }
    return 0;
}

sub isCockpitStackDetected {
    my ($self) = @_;
    my $mcm = $self->getComponentManager();
    my $cockpitStack = $mcm->getComponentByKeyName($gKeynameCockpitStack);
    return defined($cockpitStack) ? 1 : 0;
}

sub _getXS2AppsEnablingComponentKeys {
    my ($self) = @_;
# In the Cockpit on Platform case we can install XS apps without installing/updating XSA
    return [ @{$self->SUPER::_getXS2AppsEnablingComponentKeys()}, $gKeynameCockpitStack ];
}

sub _addListeners {
    my ($self) = @_;
    $self->addParameterListener('SID', LCM::Configuration::ValueChangeListeners::Cockpit::CockpitParamsListener->new());
    $self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::Cockpit::CockpitSelectionListener->new());
    $self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::Cockpit::DetectXSSpaceListener->new());
    $self->addParameterListener('CockpitXSSpace', LCM::Configuration::ValueChangeListeners::Cockpit::CockpitSpaceListener->new());
    $self->addParameterListener('CockpitSpaceGUID', LCM::Configuration::ValueChangeListeners::Cockpit::CockpitSpaceGUIDListener->new());
    $self->addParameterListener('CockpitTenant', LCM::Configuration::ValueChangeListeners::Cockpit::CockpitTenantListener->new());
    $self->addParameterListener('SelectedXs2Applications', LCM::Configuration::ValueChangeListeners::InstallUpdate::SelectedXs2ComponentsListener->new());
}

sub getParamCockpitTenantUser {
    my ($self, $order, $section, $constraint) = @_;
    return {
        'order'             => $order,
        'opt'               => 'cockpit_tenant_user',
        'opt_arg'           => '<name>',
        'type'              => 'string',
        'section'           => $section,
        'default'           => 'SYSTEM',
        'init_with_default' => 1,
        'str'               => "$gShortProductNameCockpit Tenant Database User Name",
        'desc'              => "The name of the user for the $gProductNameCockpit tenant",
        'skip'              => 1,
        'mandatory'         => 1,
        'set_interactive'   => 1,
        'constraint'        => $constraint,
    };
}

sub getParamCockpitTenantUserPassword {
    my ($self, $order, $section, $constraint, $userParamId) = @_;
    my $defaultTenantUser = "SYSTEM";
    return {
        'order'             => $order,
        'opt'               => 'cockpit_tenant_user_password',
        'type'              => 'passwd',
        'section'           => $section,
        'str'               => "$gShortProductNameCockpit Tenant Database User ($defaultTenantUser) Password",
        'str_templ'         => "$gShortProductNameCockpit Tenant Database User (%s) Password",
        'desc'              => "The password of the user for the $gProductNameCockpit tenant",
        'skip'              => 1,
        'mandatory'         => 1,
        'set_interactive'   => 1,
        'constraint'        => $constraint,
        'user_paramid'      => $userParamId,
    };
}

sub isCockpitStackUpdate{
    my $self = shift;
    my $cockpitStack = $self->getComponentManager()->getComponentByKeyName($gKeynameCockpitStack);
    return (defined $cockpitStack && $cockpitStack->isUpdate()) ? 1 : 0;
}

1;