package LCM::Configuration::Cockpit::CockpitStackSystemCollection;
use parent qw(LCM::Configuration::BaseConfigurationSystemCollection);

use SDB::Install::Globals qw($gFlavourCockpit $gFlavourPlatform $gProductNameCockpit $gProductNameEngine $gDirNameCockpitStack $gKeynameEngine $gKeynameCockpitStack);
use LCM::Manifests::CockpitStackManifest;
use File::stat qw(stat);
use File::Spec;
use LCM::Utils::CommonUtils qw(getCockpitProductVersion);
use SDB::Install::DependencyParser;

sub getProductInfo {
    my ($self, $sapSys,$isPending) = @_;
    my $productInfo = '';
    if (!$isPending && $sapSys->getHANAFlavour() eq $gFlavourCockpit) {
        my $incompleteMessage = $self->_getSystemMessageIncompleteCockpit($sapSys);
        $productInfo.= "\n".$incompleteMessage if ($incompleteMessage);
    }

    return $productInfo;
}

sub _getSystemMessageIncompleteCockpit {
    my ($self, $sapSys) = @_;

    my $message = 'System is incomplete due to failed installation or update (resume not possible)';
    my $isIncomplete = 0;

    my $manifestDir = File::Spec->catdir($sapSys->get_globalSidDir(), $gDirNameCockpitStack);
    my $manifestPath = File::Spec->catfile($manifestDir, 'manifest');
    my $manifest = LCM::Manifests::CockpitStackManifest->new($manifestPath);
    my $parser = new SDB::Install::DependencyParser();
    my $dependencyString = $manifest->getValue('required-components');
    return '' unless($dependencyString && $parser->parse($dependencyString));

    my $scm = new LCM::ComponentManager::SystemComponentManager($self->getConfiguration());
    my $sid = $sapSys->get_sid();
    my $rc = $scm->detectComponentsBySid($sid);
    if ( not defined $rc ) {
        $self->getConfiguration()->PushError( "Cannot detect installed components", $scm->getErrMsgLst() );
        return undef;
    }

    for my $vendor (sort(keys %{$parser->{parseTree}})) {
        for my $keyname (sort(keys %{$parser->{parseTree}->{$vendor}})) {
            $isIncomplete = 0;
            my $minDependency = $parser->getMinVersion($vendor, $keyname);
            my $maxDependency = $parser->getMaxVersion($vendor, $keyname);
            # If the component is undefined, try lower-casing the name because XS applications' keynames
            # are stored with lowercase but are described with uppercase in the Cockpit stack's manifest
            my $component = $scm->getComponentByKeyName($keyname) || $scm->getComponentByKeyName(lc($keyname));
            if (defined $component) {
                my $compName = $component->getComponentName();
                my $version = $component->getVersion();
                my $manifest = $component->getManifest();
                my $incompatibleComponentMessage = $self->_getIncompatibleComponentMessage($compName, $version, $minDependency, $maxDependency, 
                                            $manifest->getValue('release'), $manifest->getValue('rev-number'), $manifest->getValue('rev-patchlevel'), $manifest->getValue('rev-changelist'));
                if (defined $incompatibleComponentMessage) {
                    $isIncomplete = 1;
                    $message .= "\n  ".$incompatibleComponentMessage;
                }
            } else {
                $isIncomplete = 1;
                $message .= "\n  $keyname is not installed";
            }
        }
    }

    return $isIncomplete? $message : '';
}

sub _getIncompatibleComponentMessage {
    my ($self, $compName, $version, $minDependency, $maxDependency, $release, $sp, $pl, $cl) = @_;

    if ((defined $minDependency) && (!$minDependency->isCompatible($release, $sp, $pl, $cl)) ||
        (defined $maxDependency) && (!$maxDependency->isCompatible($release, $sp, $pl, $cl))) {
            my $minVersionString = $minDependency->getVersionString().' '.($minDependency->{inclusive} ? "inclusive" : "exclusive");
            my $maxVersionString = $maxDependency->getVersionString().' '.($maxDependency->{inclusive} ? "inclusive" : "exclusive");

            my $versionsString = $minDependency->getVersionString();
            if ($minVersionString ne $maxVersionString) {
                $versionsString = "between " . $minVersionString ." and " . $maxVersionString;
            }

        return "$compName is version $version, but should be $versionsString";
    } else {
        return undef;
    }
}

sub hasCockpitStack{
    my ($self,$sid) = @_;
    my $sapSys = $self->getInstalledSystem($sid);
    my $cockpitDir =  File::Spec->catfile($sapSys->get_globalSidDir(),$gDirNameCockpitStack);
    return 1 if (File::stat::stat($cockpitDir));
    return 0;
}

sub getHanaInstallationsWithoutCockpit{
    my $self = shift;
    my @installedHanaSystems = $self->getInstalledSystems($gFlavourPlatform);
    my @result = ();
    foreach my $sid (@installedHanaSystems){
        next if($self->hasCockpitStack($sid));
        push(@result,$sid);
    }
    return sort(@result);
}

sub getHanaInstallationsWithCockpit{
    my $self = shift;
    my @installedHanaSystems = $self->getInstalledSystems($gFlavourPlatform);
    my @result = ();
    foreach my $sid (@installedHanaSystems){
        next if(!$self->hasCockpitStack($sid));
        push(@result,$sid);
    }
    return sort(@result);
}

sub getSystemVersion{
    my($self,$sid) = @_;
    my $configuration;
    my $isPending = $self->isPendingInstallation($sid) || $self->isPendingLSSInstallation($sid) || $self->isInstalledSystemPendingUpdate($sid);
    my $sapSystem;
    if($self->isPendingInstallation($sid)|| $self->isPendingLSSInstallation($sid)){
        $sapSystem = $self->getPendingInstallSystem($sid);
    } else {
        $sapSystem = $self->getInstalledSystem($sid);
    }
    return LCM::Utils::CommonUtils::getCockpitProductVersion($sapSystem) // 'Version is unknown';

}

1;