package LCM::Configuration::ComponentValidation::InstallableDependencyCheck;

use strict;
use SDB::Install::Globals qw($gKeynameInstaller $gProductNameEngine);
use LCM::Component qw( getComponentKeyCaptionByKeyName );

sub new{
	my ($class) = @_;
	my $self = {};
	bless($self,$class);
	return $self;
}

sub check{
	my($self,$config) = @_;
	my $targetComponents = $self->getTargetComponents($config);
    my $targetSetOfManifests = [ map { $_->getManifest() } @{$targetComponents} ];
    my @selectedComponents = grep { $_->isComponentSelected() } @{$targetComponents};

    for my $component (@{$targetComponents}) {
        if ($config->isUpdate() && $component->isDeprecated($config)) {
            return 0 if !$config->validateDeprecatedComponent($component);
            next;
        }
        my $isComponentSelected = $component->isComponentSelected();
        my $isComponentWithDependenciesSelected = $isComponentSelected && $self->checkForExistingDependencies($component);
        my $isComponentWithDependenciesInSelectedSet = grep { $self->checkForExistingDependenciesWithComponent($component,$_) } @selectedComponents;
        my $isSelectedForCheckOnly = $isComponentSelected ? 0 : 1;
        next if( !$isComponentWithDependenciesSelected && !$isComponentWithDependenciesInSelectedSet);
        if(!$self->checkComponentDependencies($config,$component, $targetSetOfManifests, $isSelectedForCheckOnly)){
            return 0;
        }
    }

    if ($self->_shouldCheckNonResidentHdblcmDependenciesExplicitly($config)) {
        my $mcm = $config->getMediumComponentManager();
        my $nonResidentHdblcmComponent = $mcm->getComponentByKeyName($gKeynameInstaller);
        if(!$self->_checkHdblcmServerDependency($config, $nonResidentHdblcmComponent, $targetSetOfManifests)){
            return 0;
        }
    }

    return 1;
}

sub _checkHdblcmServerDependency {
    my ($self, $config, $nonResidentHdblcmComponent, $targetSetOfManifests) = @_;
    my $manifest = $nonResidentHdblcmComponent->getManifest();
    my $parser = $manifest->getDependencyParser();
    return 1 if(! defined $parser || $parser == 1);
    for my $targetManifest (@$targetSetOfManifests) {
        next if (!$targetManifest->isServer());
        if (! $manifest->checkComponentDependencyForParser($targetManifest, undef, $parser)) {
            my $serverVersion = $targetManifest->getVersion();
            my $nonResidentHdblcmVersion = $manifest->getVersion();
            my $msg = "HDBLCM version '$nonResidentHdblcmVersion' cannot be used to install or update additional components of $gProductNameEngine version '$serverVersion'.";
            my $mcm = $config->getMediumComponentManager();
            if (!$mcm->_isOnlyHdblcmSelected()) {
                my $scm = $config->getSystemComponentManager();
                my $residentHdblcm = (defined $scm) ? $scm->getComponentByKeyName($gKeynameInstaller) : undef;
                if (defined $residentHdblcm) {
                    my $sidPath = $residentHdblcm->getInstSidPath($config);
                    $msg .= " Use the action 'update_components' of the resident HDBLCM under '$sidPath'.";
                }
            }
            $self->notifyMessageHandlers($msg);
            return 0;
        }
    }
    return 1;
}

sub checkForExistingDependencies{
	my ($self, $component) = @_;
	my $manifest = $component->getManifest();
  	my $parser = $manifest->getDependencyParser();
    return 1 if(defined $parser && $parser != 1);
    $parser = $manifest->getSupportedComponentsParser();
    return 0 if(!defined $parser || $parser == 1);
    return 1;
}

sub checkForExistingDependenciesWithComponent {
    my ($self, $sourceComponent,$targetComponent) = @_;
    my $srcManifest = $sourceComponent->getManifest();
    my $sourceDependencyParser = $srcManifest->getDependencyParser();
    return 0 if(!defined $sourceDependencyParser || $sourceDependencyParser == 1 );
    
    my $targetManifest = $targetComponent->getManifest();
    my $targetComponentVendor = $targetManifest->getValue('keyvendor');
	 my $targetComponentKeyName = $targetManifest->getValue('keyname');

	my $minDependency = $sourceDependencyParser->getMinVersion($targetComponentVendor, $targetComponentKeyName);
	my $maxDependency = $sourceDependencyParser->getMaxVersion($targetComponentVendor, $targetComponentKeyName);

    return 0 if(!defined $minDependency || !defined $maxDependency);
    return 1;
}



sub getTargetComponents{
	my($self,$config) = @_;
	$config->gatherTargetSetOfComponents();
}

sub addMessageHandler{
	my($self,$handler) = @_;
	if(!defined $self->{_messageHandler}){
		$self->{_messageHandler} = [];
	}
	push(@{$self->{_messageHandler}},$handler);
}

sub notifyMessageHandlers{
	my($self,$message) = @_;
	return if(!defined $self->{_messageHandler});
	foreach my $handler(@{$self->{_messageHandler}}){
		$handler->onComponentValidationMessage($message);
	}
}

sub checkComponentDependencies {
    my ($self, $config, $component, $manifests, $isCheckOnly) = @_;
    my $manifest = $component->getManifest();
    my $parser = $manifest->getDependencyParser();
    if(defined $parser && $parser != 1) {
       my $rc = $self->_checkComponentDependenciesForParser($config, $component, $manifests, $isCheckOnly, $parser, 1);
       return 0 if (!$rc);
    }
    $parser = $manifest->getSupportedComponentsParser();
    return 1 if(!defined $parser || $parser == 1);
    return $self->_checkComponentDependenciesForParser($config, $component, $manifests, $isCheckOnly, $parser, 0);
}

sub _checkComponentDependenciesForParser {
	my ($self, $config, $component, $manifests, $isCheckOnly, $parser, $mandatoryDependencies) = @_;
 	my $manifest = $component->getManifest();
	my $detectedXSAComponentsKeynames = $config->getDetectedXSAComponentsKeyNameList();
	my $dependenciesVendors = $parser->getDependenciesVendors();
	foreach my $vendor (@$dependenciesVendors) {
		my $dependenciesKeyNames = $parser->getDependenciesKeyNamesForVendor($vendor);
		foreach my $keyname (@$dependenciesKeyNames) {
			my $found = 0;
# Workaround: In the Cockpit use case the database can have dependencies
# to XSA Applications. At this stage of the code we do not know if the
# needed Apps are selected for installation and we can only check that
# they are detected.
			if ( grep { lc($keyname) eq lc($_) } @$detectedXSAComponentsKeynames) {
				$found = 1;
				next;
			}
			foreach my $targetManifest (@$manifests) {
				my $targetVendor  = $targetManifest->getValue('keyvendor');
				my $targetKeyname = $targetManifest->getValue('keyname');
				if (defined $targetVendor && defined $targetKeyname
					&& ($vendor eq $targetVendor) && ($keyname eq $targetKeyname)) {
					$found = 1;
					my $rc = $manifest->checkComponentDependencyForParser($targetManifest, undef, $parser);
					if (!$rc) {
						my $msg;
						my $caption = $component->getComponentKeyCaptionByKeyName($targetKeyname);
						my $requiresMessage = $config->_getRequiresComponentMessage($component, $keyname,
							$parser->getMinVersion($vendor,$keyname),
							$parser->getMaxVersion($vendor,$keyname));
						if ($component->isInstalled()) {
							$msg = "An already installed component " . $component->getComponentName() . " is not compatible with " .
								$caption . " version '" . $targetManifest->getVersion() . "'. " . $requiresMessage;
								if($component->getComponentBatchKey() eq 'lcapps'){
									$msg.=" To proceed with the update, follow the instructions in SAP Note 2223318.";
								} elsif ($manifest->isServerPlugin()) {
                                    $msg.=" The update of " . $caption . " to version '" . $targetManifest->getVersion() . "'" .
                                    " requires also the matching " . $component->getComponentName() .
                                    " to be updated. Updates for plugins are available on the SAP Support Portal. For withdrawn plugins, follow SAP Note 2293092.";
								}
						} else {
							my $selectedStr = ($component->getComponentKeyName eq $gKeynameInstaller) ? 'Auto selected' : 'Selected';
							$msg = "$selectedStr component " . $component->getComponentName() . " is not compatible with " .
								$caption . " version '" . $targetManifest->getVersion() . "'. " . $requiresMessage;
						}
						$self->notifyMessageHandlers($msg);
						if (!$config->isPrepareUpdate()) {
							return 0;
						}

						return 1;
					}
				}
			}

			if (!$isCheckOnly && !$found && $mandatoryDependencies) {
				my $caption = $component->getComponentKeyCaptionByKeyName($keyname);
				my $requiresMessage = $config->_getRequiresComponentMessage($component, $keyname,
					$parser->getMinVersion($vendor, $keyname),
					$parser->getMaxVersion($vendor, $keyname));
				$self->notifyMessageHandlers($requiresMessage);
				if (!$config->isPrepareUpdate()) {
					return 0;
				}
			}
		}
	}
	return 1;
}

sub _shouldCheckNonResidentHdblcmDependenciesExplicitly {
    my ($self, $config) = @_;
    return 0 if (!$config->isUpdate());

    my $mcm = $config->getMediumComponentManager();
    return 0 if ($mcm->isResidentComponentManager());

    my $nonResidentHdblcmComponent = $mcm->getComponentByKeyName($gKeynameInstaller);
    return 0 if ($nonResidentHdblcmComponent->isComponentSelected());

    return 1;
}

1;