package LCM::Configuration::GenericStackUpdateConfiguration;

use strict;
use LCM::Configuration::GenericStackAny;
use LCM::Configuration::GenericStackConfiguration;
use LCM::Component;
use LCM::Component::Installable::ReferenceData qw (INSTALLATION_PATH_MODE);
use LCM::Configuration::ValueChangeListeners::Update::RemoteExecutionListener;
use LCM::Configuration::ValueChangeListeners::Update::ConsoleWriterSIDListener;
use LCM::Configuration::ValueChangeListeners::Update::SelectedComponentsListener;
use LCM::Configuration::ValueChangeListeners::Update::DeprecatedComponentsListener;
use LCM::Configuration::ValueChangeListeners::Update::MetadataSeparationCheckListener;
use LCM::Configuration::ValueChangeListeners::Update::XS::OrgNameDetectionListener;
use LCM::Configuration::ValueChangeListeners::Update::XS::OrgManagerUserDetectionListener;
use LCM::Configuration::ValueChangeListeners::Update::XS::XsTenantUserDetectionListener;
use LCM::Configuration::ValueChangeListeners::Update::XsTenantInstallationListener;
use LCM::Configuration::ValueChangeListeners::Update::XsTenantDatabaseListener;
use LCM::Configuration::ValueChangeListeners::Update::InstallHostagentListener;
use LCM::Configuration::ValueChangeListeners::Update::NonRootRemoteHostsAuthenticationListener;
use LCM::Configuration::ValueChangeListeners::Update::SystemDBUseSingleDBUserPasswordListener;
use LCM::Configuration::ValueChangeListeners::Update::SystemUserListener;
use LCM::Configuration::ValueChangeListeners::Update::RootUserListener;
use LCM::Configuration::ValueChangeListeners::FillingCertificatesListener;
use LCM::Configuration::ValueChangeListeners::FillingRolesHostmapListener;
use LCM::Configuration::ValueChangeListeners::AddHosts::RemoteCredentialsListener;
use LCM::Configuration::ValueChangeListeners::Update::ScopeListener;
use LCM::Configuration::ValueChangeListeners::Update::AdditionalHostsListener;
use LCM::Configuration::ValueChangeListeners::Update::AdditionalRolesListener;
use LCM::Configuration::ValueChangeListeners::InstallUpdate::RecomputeXs2ApplicationParametersListener;
use LCM::Configuration::ValueChangeListeners::InstallUpdate::RecomputeXs2SpaceIsolationParameters;
use LCM::Configuration::ValueChangeListeners::InstallUpdate::XSSpaceIsolationStringListener;
use SDB::Install::Configuration::LSSUserStringListener;
use LCM::Configuration::ValueChangeListeners::InstallUpdate::InstallXSInDefaultTenantListener;
use LCM::Configuration::ValueChangeListeners::InstallUpdate::HandleLSSUserAndGroupListener;
use LCM::Configuration::ValueChangeListeners::Update::UpdateExecutionModeListener;
use LCM::Configuration::ValueChangeListeners::Update::ContinueUpdateListener;
use LCM::Configuration::ValueChangeListeners::XsInitTenantDatabaseListener;
use LCM::Configuration::ValueChangeListeners::SHAVersionListener;
use LCM::Configuration::ValueChangeListeners::ReservedIDsHandler;
use LCM::Configuration::ValueChangeListeners::SignatureVerificationListenerFactory;
use LCM::Configuration::ValueChangeListeners::AddComponentsLocationListener;
use SDB::Install::Configuration::TenantUserListener;
use LCM::Configuration::ValueChangeListeners::XsSpaceIsolationListener;
use LCM::Configuration::Validators::FileModeValidator;
use LCM::FileUtils;
use LCM::Utils::CommonUtils qw(getLocalHostSidadmUid removeElementFromArray getCockpitProductVersion isSystemReplication);
use LCM::HostsParser;
use LCM::App::ConsoleUtils qw(printPendingInfo printMsg);
use SDB::Install::SysVars;
use SDB::Install::Globals qw ($gProductNameStudio
                              $gProductNameClient
                              $gProductNameInstaller
                              $gProductNameSHA
                              $gProductNameEngine
							  $gHostRoleAcceleratorWorker
							  $gHostRoleAcceleratorStandby
							  $gHostRoleEsWorker
							  $gHostRoleEsStandby
							  $gHostRoleStandby
							  $gHostRoleXS2Standby
							  $gHostRoleXS2Worker
							  $gHostRoleWorker
							  $gKeynameAccelerator
							  $gKeynameEngine
							  $gKeynameInstaller
							  $gKeynameXS2
							  $gXSParametersConstraint
							  $gSapsysGroupName
							  $gFlavourPlatform
							  $gFlavourExpress
							  $gFlavourCockpit
							  $gKeynameClient
							  $gKeynameStudio
							  $gKeynameCockpitStack
							  $gDirNameCockpitStack
                              $gOperationHdblcmCollectInstallerInfo
							  getHumanReadableRoleByHostRole
                              GetHostRoleProperties
                              determineSignedManifestRelativePath
                              $gKeynameStreaming
                              $gKeynameLSS);
use SDB::Install::Saphostagent qw(getSaphostexecPath);
use SDB::Install::System qw (getSysProgPath 
							 isSidadmin 
							 isAdmin 
							 $hostname_regex 
							 $ipv4_regex 
							 getAllUsersAppDataPath
							 getSAPDrive);
use SDB::Install::SAPSystem qw ( CollectSAPSystems );
use SDB::Install::Configuration::AnyConfig;
use SDB::Install::RemoteHostctrlHosts;
use SDB::Install::Installer;
use SDB::Install::Configuration::NewServerConfig;
use SAPDB::Install::Hostname;
use SDB::Install::NewDBUser;
use SDB::Install::User;
use LCM::Configuration::ParametersCreator;
use LCM::App::AddHostsUtils;
use LCM::App::ApplicationContext;
use File::stat;
use File::Spec;
use File::Basename qw (basename dirname);
use experimental qw (smartmatch);
use LCM::PersistencyUtils qw (existsHdblcmPendingUpdate
							  existsHdblcmPendingInstall
							  existsHdbinstPendingInstall
							  existsPendingComponentsUpdate
							  getAllPendingUpdateInfo
							  getPendingUpdateSIDs) ;
use LCM::Configuration::PersistenceUpdateManager;
use LCM::Hosts::HostsDetector;
use SDB::Install::Configuration qw( $bool_false_pattern $bool_true_pattern );
use LCM::Component::Installed::HDBServer;
use LCM::Configuration::ValueChangeListeners::Update::SystemDBSQLPasswordBatchHandler;
use LCM::Configuration::ValueChangeListeners::Update::EnableTenantSystemUserCredentials;
use LCM::Configuration::Cockpit::CockpitStackSystemCollection;
use LCM::Configuration::GenericStackSystemCollection;
use SDB::Install::Configuration::LSSPasswordListener;
use LCM::Configuration::ValueChangeListeners::InstallUpdate::LocalSecureStoreListener;


our $progpath          = $isWin || $isApple ? getSysProgPath() : '/usr';

our @ISA = qw (LCM::Configuration::GenericStackConfiguration SDB::Install::Configuration::AnyConfig SDB::Install::Configuration::AnyModifyConfig Exporter);

our @EXPORT = qw ($unableToContinueUpdateMsg);

our $unableToContinueUpdateMsg =  "Pending update cannot be resumed with the persisted parameters. New update process will be started.";


sub new {
    my ($class, $options, $configfile) = @_;
    my $self = $class->SUPER::new($options, $configfile);

    $self->defineUpdateParams();
	$self->defineLSSParams();
	$self->defineAcceleratorParams();
    $self->defineStreamingParams();
    $self->defineEsParams();
	$self->defineXS2Params();
	$self->addListeners();
    $self->addEventHandlers();

	if($self->getValueFromConfigfile('UseSimpleUI')){
		$self->setValue('UseSimpleUI', 1);
		$self->setInteractive('XsEaDataPath', 0);
	}

	$self->setPersistenceManager(new LCM::Configuration::PersistenceUpdateManager($self));
	$self->getPersistenceManager()->registerParamsToPersist();

	$self->SDB::Install::Configuration::AnyModifyConfig::InitDefaults(@_);

	return $self;
}

sub CheckParams{
	my ($self, $batchMode) = @_;
	my $saphostexecPath = getSaphostexecPath();
	if (!isAdmin() && !defined $saphostexecPath) {
		my $action = $self->getAction();
		my $currentUser = new SDB::Install::User();
		my $username = $currentUser->getname();
		my $message = "SAP Host Agent is required to perform $action as '$username' user. ";
		$message .=  "Either install SAP Host Agent or start hdblcm as root user.";
		$self->appendErrorMessage($message);
		return undef;
	}
	my $rc = $self->SUPER::CheckParams( $batchMode );

	return $rc if (!$rc);

	my $sid = $self->getValueOrBatchValue("SID");
	if(defined $sid){
		my $isPendingUpdate = existsHdblcmPendingUpdate($sid);
		$self->handleHostsPropertiesSkipValue($isPendingUpdate);
	}
    $self->_addIgnoreCheckBusyFiles();
	return $rc;
}

sub handleHostsPropertiesSkipValue {
    my ($self, $isPendingUpdate) = @_;
    my $addHostsBatchValue = $self->getBatchValue('AddHosts');
    my $application = LCM::App::ApplicationContext::getInstance()->getApplication();
    my $isGui = $application->isGUIApplication();
    my $batchMode = $application->isBatchMode();
    
    if ( (!$isGui)
         && !$batchMode && (!defined $addHostsBatchValue || (length($addHostsBatchValue) == 0))
         && !$isPendingUpdate ) {
        return if(!exists($self->{params}->{AddHosts}) && exists($self->{params}->{HostsProperties}));

        $self->{params}->{AddHosts}->{set_interactive} = 1;
        $self->{params}->{HostsProperties}->{set_interactive} = 1;
        $self->{params}->{HostsProperties}->{skip} = 0;
    }
}

sub checkSelectedComponents {
    my ($self, $selectedComponentsCsv) = @_;

    if (!$selectedComponentsCsv && !$self->isBootstrapFromResident()) {
        $self->PushError("There are no components selected. Select at least one component.");
        return 0;
    }

    return 1 if($selectedComponentsCsv eq 'hdblcm');
    return 1 if($selectedComponentsCsv eq 'all');

    my $scm = $self->getSystemComponentManager();
    my $mcm = $self->getMediumComponentManager();
    my $mandatorySelection = [];

    return 1 if ($scm->getHANAFlavour() eq $gFlavourPlatform);

    for my $targetComponent (@{$mcm->getAllComponents()}) {
        next if ($targetComponent->isInternal());
        next if (!$self->isComponentInstalled($targetComponent->getComponentKeyName()));
        next if (index($selectedComponentsCsv, $targetComponent->getComponentBatchKey()) > -1);

        my $sourceComponent = $scm->getComponentByKeyName($targetComponent->getComponentKeyName());
        my $sourceVersion = $sourceComponent->getManifest()->getVersionObject();
        my $targetVersion = $targetComponent->getManifest()->getVersionObject();

        if ($targetVersion->isNewerThan($sourceVersion)){
            push(@{$mandatorySelection}, $targetComponent->getComponentName());
        }
    }

    if (scalar(@{$mandatorySelection}) > 0) {
        my $mandatorySelectionList = join (',', @$mandatorySelection );
        $self->PushError("Some components are mandatory for update. The following components must be selected: $mandatorySelectionList\n");
        return 0;
    }
    return 1;
}

sub isDistributedSystem {
	my $self = shift();

	return 1 if($self->SUPER::isDistributedSystem(@_));

	my $isSkippedAddHosts = $self->isSkipped('AddHosts');
	my $addHostsValue = $self->getValue('AddHosts');
	return ( $addHostsValue && (! $isSkippedAddHosts) ) ? 1 : 0;
}

sub handleSelectionOfResidentInstaller {
	my ( $self ) = @_;

	if (! $self->SUPER::handleSelectionOfResidentInstaller()) {
		return 0;
	}

	my $mcm = $self->getMediumComponentManager();
	my $installationHdblcmCmp = $mcm->getComponentByKeyName( $gKeynameInstaller );
	if (defined $installationHdblcmCmp && $installationHdblcmCmp->isComponentSelected()) {
		return 1;
	}

	my $scm = $self->getSystemComponentManager();
	if (! defined $scm) {
		return undef;
	}

	if ($scm->isComponentAvailable($gKeynameInstaller) ) {
		return 1;
	}

	if (! $mcm->isExternalComponentSelected()) {
		return 1;
	}

    my $installedServerCmp =  $scm->getComponentByKeyName($gKeynameEngine);
    my $installedServerManifest = $installedServerCmp->getManifest();

	if ($self->validateComponent($installationHdblcmCmp, [$installedServerManifest])) {
		$installationHdblcmCmp->selectComponent();
		$self->getMsgLst()->addMessage("'hdblcm' is not installed. Selecting it for installation.");
		return 1;
	}
    return 0;
}

sub fillCertificatesHostmapDefaultValuesAndLocalData {
    my ($self, $useSHAOperation) = @_;

    $self->{'params'}->{'CertificatesHostmap'}->{'default_map'} = undef;
    $self->fillCertificatesHostmapDataForLocalHost($useSHAOperation);

    my $systemHosts = $self->getSystemHosts() || [];
    my $additionalHosts = $self->getAdditionalRemoteHosts();
    my $additionalHostNames = defined($additionalHosts) ? $additionalHosts->getHostNames() : [];

    $self->setOriginHosts([@{$systemHosts}, @{$additionalHostNames}]);
    $self->fillCertificatesHostmapDefaultValues();
    return 1;
}

sub _handleAllComponentsSelected {
    my ($self, $numberOfSelectedComponents) = @_;
    if ( $numberOfSelectedComponents > 1 ) {
        $self->AddError ("Some of the provided values are conflicting with each other");
        return undef;
    }
    my $app = LCM::App::ApplicationContext::getInstance()->getApplication();
    if (! defined $app) {
        $self->AddError("Could not get application context.");
        return undef;
    }
    my $isBatchMode = 0;
    if (!$app->isGUIApplication()){
        $isBatchMode = $app->isBatchMode();
    }
    my $selectAllDetected = ($ENV{HDBLCM_SELECT_ALL_DETECTED_COMPONENTS}) ? $ENV{HDBLCM_SELECT_ALL_DETECTED_COMPONENTS} : 0;
    if ($isBatchMode && $selectAllDetected =~ /$bool_true_pattern/i && !$self->_checkUpgradeOfRestrictedComponents()) {
        return undef;
    }
    $self->selectAllComponents(1);
    return 1;
}

sub _checkUpgradeOfRestrictedComponents {
    my ($self) = @_;
    my $cannotBeSelectedMessage = $self->getCannotBeSelectedMessage();

    if (defined $cannotBeSelectedMessage && $cannotBeSelectedMessage ne "") {
        $self->setErrorMessage($cannotBeSelectedMessage);
        return undef;
    }
    return 1;
}

sub defineUpdateParams {
	my ($self) = @_;

    my $acceleratorOnly = "Valid with $gHostRoleAcceleratorWorker/standby only";
    my $rolesConstraint = "Valid only for installation of SAP HANA components, which support host roles";
    my $studioConstraint = "Valid for Installation of $gProductNameStudio only";

	my $order = scalar keys %{ $self->{params} };
	$self->{paramIds} = undef;    # reset ordered id list cache

    my $tenantCredentialsConstraint = $self->getTenantCredentialsConstraint();

    require SDB::Install::Configuration::GenericServerPlugin;

	my $installParams = {
        'UseSimpleUI'        => GetParamUseSimpleUI($order++, $ini_section_general),
        'SkipHostagentCalls' => $self->getParamSkipHostagentCalls($order++, $ini_section_general),
        'BootstrapFromResident'    => LCM::Configuration::ParametersCreator::GetParamBootstrapFromResident($order++, $ini_section_general),
        'PrepareUpdate'        => $self->getParamPrepareUpdate($order++,  $ini_section_general ),
        'Phase'                => $self->getParamPhase($order++,  $ini_section_general ),
        'SID'            => $self->getParamSID($order++, $ini_section_server, 1),
        'ContinueUpdate' => $self->getParamContinueUpdate($order++, $ini_section_server, 1),
        'CheckOnly'          => $self->getParamCheckOnly( $order++, $ini_section_general, $self->getAction() ),
        'DvdPath'            => GetParamDvdPath( $order++, $ini_section_general, $self->getFlavourProductName() ),
        'ComponentDirs'      => GetParamComponentDirs( $order++, $ini_section_general ),
        'AddComponentsLocation'  => GetParamAddComponentsLocation($order++, $ini_section_general ),
        'ComponentFsRoot'    => GetParamComponentFsRoot( $order++, $ini_section_general ),
        (
            !$isWin
            ? (
                 'RemoteExecution' => $self->getParamRemoteExecution($order++, $ini_section_general, 0),
                 'UseHttp'         => $self->getParamUseHttp        ($order++, $ini_section_general),
                 'SAPControlUseHttp'  => $self->getParamSAPControlUseHttp  ($order++, $ini_section_general)
              )
            : ()
        ),  
        'Target' => { 'hidden' => 1, 'opt' => 'sapmnt', 'set_interactive' => 0, 'order' => $order++, 'type' => 'path'},    # just for internal use
        'Drive'  => { 'hidden' => 1, 'opt' => 'drive',  'set_interactive' => 0, 'order' => $order++, 'type' => 'string'},    # just for internal use
        'InstallHostagent' => $self->getParamInstallHostagent ($order++, $ini_section_server, undef, 1),
        'Scope'          => $self->getParamScope        ($order++, $ini_section_server),
        'AutoInitializeServices'     => $self->getParamAutoInitializeServices($order++, $ini_section_general,undef,$tenantCredentialsConstraint),
        'SystemDBUseSingleDBUserPassword'  => GetParamSystemDBUseSingleDBUserPassword($order++, $ini_section_server),
        'VerifySignature'    => LCM::Configuration::ParametersCreator::GetParamVerifySignature($order++, $ini_section_general),
        'SelectedComponents' => LCM::Configuration::GenericStackConfiguration::getParamSelectedComponents($order++, $ini_section_general),
        'LSSTrustUnsignedServer' => $self->getParamLSSTrustUnsignedServer( $order++, $ini_section_server),
        'ExecutionMode'=> LCM::Configuration::ParametersCreator::GetParamExecutionMode($order++, $ini_section_general, 0, 'update'),
        'HostName'        => $self->getParamHostName( $order++, $ini_section_server ),
        'SecureStore'     => $self->getParamSecureStore($order++, $ini_section_server),
        ( !$isWin ?
            (
             'AddHosts'         => $self->getParamAddHosts ( $order++, $ini_section_addhosts, 1, $rolesConstraint ),
             'AddRoles'         => LCM::Configuration::ParametersCreator::getParamRolesHostmap ( $self, $order++, $ini_section_addhosts, $rolesConstraint ),
             'AutoAddXS2Roles'  => LCM::Configuration::ParametersCreator::GetParamAutoAddXS2Roles($order++, $ini_section_addhosts, $gXSParametersConstraint),
             'InstallSSHKey'    => $self->getParamInstallSSHKey($order++, $ini_section_server),
             'RootUser'         => $self->getParamRootUser     ($order++, $ini_section_server),
             'RootPassword'     => $self->getParamRootPassword ($order++, $ini_section_server,'RootUser'),
            )
            :
            ()
        ),
        'SkipHostagentPw'   => $self->getParamSkipHostagentPw  ($order++, $ini_section_server),
        'HostagentPassword' => $self->getParamHostagentPassword($order++, $ini_section_server),
        ( !$isWin ? 
            (
                'HostsProperties'  => LCM::Configuration::ParametersCreator::GetParamHostsProperties($order++, $ini_section_addhosts),
                'ImportContentXS2' => $self->getParamImportContentXS2 ($order++, $ini_section_xsa, $gXSParametersConstraint),
            )
            :
            ()
        ),
        'SSOCertificate'     => $self->getParamSSOCertificate     ($order++, $ini_section_general),
        'Password'          => $self->getParamPassword( $order++, $ini_section_server ),
        'NoScriptServerRestart' => $self->SDB::Install::Configuration::GenericServerPlugin::getParamNoScriptServerRestart($order++, $ini_section_plugin),
        'SystemUser'        => $self->getParamSystemUser($order++, $ini_section_server),
        'SQLSysPasswd'      => $self->getParamSQLSysPasswd($order++, $ini_section_server, 'passwd', 0 ,undef,undef, 'SystemUser'),
        'SystemDBSQLPassword'  => GetParamSystemDBSQLPasswd($order++, $ini_section_server, 'initial_passwd', 1,'SYSTEM',undef,'SystemUser'),
        'ISCMode'           => $self->getParamISCMode     ($order++, $ini_section_server),
        ( $isWin ?
            (
                'HostagentUserName'=> $self->getParamHostagentUserName($order++, $ini_section_server),
            )
            :
            (
                'ListenInterface'     => $self->getParamListenInterface    ($order++, $ini_section_addhosts, 1, $rolesConstraint),
                'InternalNetwork'     => $self->getParamInternalNetwork    ($order++, $ini_section_addhosts, 1, $rolesConstraint, 1, 0, 1),
                'CertificatesHostmap' => $self->getParamCertificatesHostmap($order++, $ini_section_server),
            )
        ),
        'ClientPath' => {
            'order'             => $order++,
            'opt'               => 'client_path',
            'type'              => 'path',
            'section'           => $ini_section_client,
            'value'             => undef,
            'default'           => $isWin ? '$Drive\usr\sap\$SID\hdbclient' : '$Target/$SID/hdbclient',
            'str'               => 'SAP HANA Database Client Installation Path',
            'constraint'        => "Valid for Installation of $gProductNameClient only",
            'init_with_default' => 1,
            'set_interactive'   => 0,
            'mandatory'         => 1
        },
        'StudioPath' => {
            'order'             => $order++,
            'opt'               => 'studio_path',
            'type'              => 'path',
            'section'           => $ini_section_studio,
            'value'             => undef,
            'default'           => $isWin ? '$Drive\usr\sap\$SID\hdbstudio' : '$Target/$SID/hdbstudio',
            'str'               => "$gProductNameStudio Installation Path",
            'constraint'        => $studioConstraint,
            'init_with_default' => 1,
            'set_interactive'   => 0,
            'mandatory'         => 1
        },
        'StudioRepository' => {
            'order'             => $order++,
            'opt'               => 'studio_repository',
            'type'              => 'boolean',
            'section'           => $ini_section_studio,
            'value'             => undef,
            'default'           => 1,
            'str'               => 'Copy studio repository',
            'constraint'        => $studioConstraint,
            'init_with_default' => 1,
            'set_interactive'   => 0,
            'exit_if_faulty'    => 1
        },
        'TargetRepositoryPath' => $self->getParamTargetRepositoryPath ($order++, $ini_section_studio, $studioConstraint),
        'JavaVm'               => $self->getParamJavaVm ($order++, $ini_section_studio, $studioConstraint),
        'CheckMnt'             => SDB::Install::Configuration::NewServerConfig::getParamCheckMnt($self, $order++, $ini_section_general),
        'Ignore'               => $self->getParamArgIgnore( $order++, $ini_section_general ),
        'AcceleratorUser'      => $self->getParamAcceleratorUser($order++, $ini_section_addhosts, $acceleratorOnly),
        'AcceleratorPassword'  => $self->getParamAcceleratorPassword($order++, $ini_section_addhosts, $acceleratorOnly,undef,'AcceleratorUser'),
        'ReferenceDataPath'    => GetParamReferenceDataPath($order++, $ini_section_refdata),
        ($isWin
                ? ()
                : ('SkipModifySudoers' => $self->getParamSkipModifySudoers($order++, $ini_section_general)),
         ),
        'InstallXSInDefaultTenant' => GetParamInstallXS2InDefaultTenant($order++, $ini_section_xsa, $gXSParametersConstraint),
        'XSTenantDatabaseName',    => GetParamXSTenantDatabaseName($order++, $ini_section_xsa, $gXSParametersConstraint),
        'TenantUser'            => $self->getParamTenantUser    ($order++, $ini_section_general,1,$tenantCredentialsConstraint),
        'SQLTenantUserPassword' => $self->getParamSQLTenantUserPassword($order++, $ini_section_general,1,$tenantCredentialsConstraint,undef,'TenantUser'),
	};


    if (isSidadmin()) {
        $installParams->{SkipHostagentCalls}->{skip} = 1;
    }

	if(exists($installParams->{ImportContentXS2})){
		$installParams->{ImportContentXS2}->{alternative_sections} = [ $ini_section_addhosts ];
	}

	if (exists $installParams->{AddHosts} && exists $installParams->{HostsProperties}) {
		$installParams->{AddHosts}->{custom_input} = \&LCM::App::AddHostsUtils::readInteractiveAddHostsInput;
		$installParams->{HostsProperties}->{custom_input} = \&LCM::App::AddHostsUtils::readHostsPropertiesForUpdate;
	}
	if(exists $installParams->{ContinueUpdate}){
		$installParams->{ContinueUpdate}->{custom_input} = sub {
			my $rc = $self->_handleInitializationOfPendingConfiguration($self->getSID());

			if (!$rc) {
				$self->setSkip('ContinueUpdate', 1);
				return 'n';
			}

			my $pendingConfig = $self->getPendingConfiguration();
			my $summaryTree = $pendingConfig->createSummaryTree();
			LCM::App::ConsoleUtils::printPendingInfo($summaryTree);
			print "\n" . $installParams->{ContinueUpdate}->{str} . ' [y]: ';
			my $input;
			chomp($input = <STDIN>);
			return $input;
		};
	}
	$installParams->{'HostName'}->{'set_interactive'} = 0;
	$installParams->{'HostName'}->{'skip'} = 1;
	$installParams->{'CheckMnt'}->{hidden} = 1;
	
    if (exists $installParams->{Scope}) {
        $installParams->{Scope}->{init_with_default} = 1;
        $installParams->{Scope}->{skip}    = 0;
    }

    if (exists $installParams->{RootUser}) {
        $installParams->{RootUser}->{init_with_default} = 1;
        $installParams->{RootUser}->{set_interactive}   = 1;
    }
    map { $self->{params}->{$_} = $installParams->{$_} } keys %$installParams;
    $order = $self->addPassThroughParameters($self->{params}, $order, 'hdbupd');
    $self->{params}->{'cmd_line_action'} = $self->getParamCmdLineAction($order);

	$installParams->{ListenInterface}->{skip}     = 1;
	$installParams->{InternalNetwork}->{skip}     = 1;
	$installParams->{AcceleratorUser}->{skip}     = 1;
	$installParams->{AcceleratorPassword}->{skip} = 1;
    
    $installParams->{Ignore}->{alternative_sections} = [ $ini_section_studio ]; # Added for backward compatibility

	$self->setDefault('Scope', 'system');
    $self->setHidden('XSTenantDatabaseName', 0);

	return $self;
}

#
# Due to the multiple inheritance of $self, explicitly point to the "getSystemHosts"
# subroutine implementation.
#
sub getSystemHosts {
    my ($self) = shift;
    return $self->SDB::Install::Configuration::AnyConfig::getSystemHosts(@_);
}

sub getIgnoreValues { return ['check_component_dependencies',
							  'check_pending_upgrade',
							  'check_version',
							  'check_resume_hostname',
							  'check_platform',
							  'check_signature_file',
                              'check_busy_files',
                              'verify_signature',
                              'check_diskspace',
                              'check_min_mem']; }



sub _shouldSkipAutoAddXS2Roles {
	my ($self) = @_;
	my $componentManager = $self->getComponentManager();
	my $xs2Component = $componentManager->getComponentByKeyName($gKeynameXS2);
	my $isPendingUpdate = $self->isPendingUpdate();
	my $hasAutoAddXs2RolesPersistingValue = defined $self->getPersistedValue('AutoAddXS2Roles');
	my $shouldInstallXS2Component = defined($xs2Component) && $xs2Component->isComponentSelected() && ! $xs2Component->isUpdate();

	return 0 if $isPendingUpdate && $hasAutoAddXs2RolesPersistingValue;
	return 1 if ! $shouldInstallXS2Component;

	my @hostsForAddition = $self->isSkipped('AddHosts') ? () : split(',', $self->getValue('AddHosts'));
	my $isAddingXsHost = grep { $_ =~ /:role=($gHostRoleXS2Worker|$gHostRoleXS2Standby)/} @hostsForAddition;
	my $isAddingRoles = !$self->isSkipped('AddRoles') && ref($self->getValue('AddRoles')) eq "HASH";

	return 1 if($isAddingXsHost);
	return 0 if(! $isAddingRoles);

	for my $host (@{$self->getOwnInstance()->get_allhosts()}){
		my $additionalRoles = $isAddingRoles ? $self->getValue('AddRoles')->{$host} : '';
		my $isAddingXsRole = grep { $_ =~ /$gHostRoleXS2Worker|$gHostRoleXS2Standby/} split(',', $additionalRoles);

		return 1 if($isAddingXsRole);
	}
	return 0;
}

sub setContinueUpdate {
	my ($self, $value) = @_;

	return 1 if ($self->isSkipped('ContinueUpdate'));
	return undef if($value !~ /^$bool_false_pattern|$bool_true_pattern$/);

	if($value =~ /$bool_true_pattern/){
		$self->{params}->{ContinueUpdate}->{value} = 1;
		return $self->_handlePendingUpdate('Failed to load parameters from previous run.');
	}

	$self->{params}->{ContinueUpdate}->{value} = 0;
	for my $paramId (@{$self->getParamIds()}){
		delete($self->{params}->{$paramId}->{persFileValue});
	}

	return 1;
}

sub _handlePendingUpdate {
	my ($self, $errMsg) = @_;
	if (! $self->handlePendingUpdate($self->getSID())) {
		$self->setNoRetry("ContinueUpdate",1);
		$self->PushError($errMsg);
		return undef;
	}
	return 1;
}

sub handlePendingUpdate {
	my ($self, $value) = @_;
	return $self->getPersistenceManager()->handlePendingUpdate($value);
}

sub setIgnore{
    my($self, $ignoreString) = @_;
    my $rc = $self->SUPER::setIgnore($ignoreString);
    return $rc if (!$rc);
    $self->_addIgnoreCheckBusyFiles();
    return 1;
}

sub InitDefaults {
	my ( $self, $kit ) = @_;

	$self->SUPER::InitDefaults($kit);

	$self->getPersistenceManager()->registerInitialParamsToPersist();

	if ($isWin) {
		my $drive = getSAPDrive ($self->getMsgLst ());
		if ($drive) {
			$self->{params}->{Drive}->{value}           = uc($drive);
			$self->{params}->{Drive}->{hidden}          = 1;
			$self->{params}->{Drive}->{set_interactive} = 0;
		}
	}

	$self->selectAllComponents(0);
	$self->setDefault('ISCMode', 'standard', 1);

	return 1;
}

sub isScopeInstance {
    my ( $self ) = @_;
    my $value = defined $self->{params}->{Scope}->{value}       ? 
                $self->{params}->{Scope}->{value}               :
                defined $self->{params}->{Scope}->{batchValue}  ?
                $self->{params}->{Scope}->{batchValue}          :
                $self->getDefault('Scope');

    return (defined $value && ($value eq 'instance'));
}

# SID listener is notified after setSID subroutine, SID parameter must be set.
sub setSID {
	my ($self, $value) = @_;
	my $rc = $self->checkSID($value);

	return $rc if (!$rc);

	$self->{params}->{SID}->{value} = $value;
	$self->{current_sid} = $value;
    $self->setDatabaseToSystemUserCredentialsParameter($self->{params}->{'SQLSysPasswd'});
    $self->setDatabaseToSystemUserCredentialsParameter($self->{params}->{'SystemUser'});

	return 1;
}

sub _handleInitializationOfPendingConfiguration {
	my ($self, $sid) = @_;
	if(!$self->getIgnore('check_pending_upgrade')){
		my $rc = $self->_initPendingConfiguration($sid);
		if (!defined $rc) {
			$self->addCannotBeSelectedConsoleText();
		} elsif (!$rc) {
			$self->loadComponentDirsPersistedValue($self->getSID());
		}
		return $rc;
	}
	$self->getMsgLst()->addMessage("Ignoring pending update due to --ignore command line switch and starting a new update");
	return undef;
}

sub _initPendingConfiguration {
	my ($self, $sid) = @_;

	return undef if(!existsHdblcmPendingUpdate($sid));
	return 1 if(defined($self->getPendingConfiguration()));
	my $hdblcmPersistedData = $self->_loadHdblcmPersistenceSilently();
	my $class = ref($self);
	my $pendingConfig = $class->new();

	$pendingConfig->_setPendingConfiguration(1);
	$pendingConfig->setIsPendingConfiguration(1);
	$pendingConfig->setMsgLstContext([SDB::Install::MsgLst->new(), $self->getErrMsgLst()]);
	$pendingConfig->InitDefaults();
	$pendingConfig->setValue('SID', $sid);
	$pendingConfig->{step} = 1;
	$pendingConfig->{params}->{SID}->{value} = $sid;
	$pendingConfig->setValue('PrepareUpdate', $self->getValue('PrepareUpdate'));
    $pendingConfig->setTimeouts($hdblcmPersistedData->{'pers_timeouts'});
    $pendingConfig->setOptionIgnore($self->getOptionIgnore());

	$self->_addProvidedComponentsToPendingConfig($pendingConfig);
	$self->_addDefaultDetectedComponentsToPendingConfig($pendingConfig, $hdblcmPersistedData);
	$self->_setPendingConfiguration($pendingConfig);

    my $persistenceManager = $pendingConfig->getPersistenceManager();
    if (! $persistenceManager->validateHdblcmKitVersion($hdblcmPersistedData)) {
        my $errMsg = $pendingConfig->getErrorString();
        printMsg($unableToContinueUpdateMsg);
        printMsg($errMsg, 1);
        $self->setCanContinueUpdate(0);
        return undef;
    }

	for my $parameterId (@{$pendingConfig->getParamIds()}){
		my $parameter = $pendingConfig->{params}->{$parameterId};
		my $persistedValue = $hdblcmPersistedData->{$parameterId};

		next if (!defined($persistedValue));
		next if ($pendingConfig->isHidden($parameterId));
		next if ($pendingConfig->isSkipped($parameterId));
		next if ($parameterId eq 'ComponentDirs' && ($self->existsComponentLocationBatchValue() || !$persistedValue));
		delete($parameter->{shouldUseSetterOnDeserialization}) if($parameterId =~ /RootUser/);

		my $batchValue = $self->{params}->{$parameterId}->{batchValue};
		if (defined($batchValue)) {
			$persistedValue = $batchValue;
		}

		$pendingConfig->resetError();
		$pendingConfig->resetMsgLst();
		my $rc = $pendingConfig->setPersistedValue($parameterId, $parameter, $persistedValue);
		my @warnings = grep( $_->{type} eq 'WRN', @{$pendingConfig->getMsgLst()->getMessages()});
        if ( scalar @warnings > 0 ){
            $self->getMsgLst()->appendMsgLst(@warnings);
        }
        next if ($rc);
		if ($parameterId eq 'ComponentDirs') {
            my $errMsg = $pendingConfig->getErrorString();
            printMsg($unableToContinueUpdateMsg);
            printMsg($errMsg, 1);	
            $self->setCanContinueUpdate(0);
            return undef;
		}
		if ($parameterId eq 'SelectedComponents') {
			my $errMsg = $pendingConfig->getErrorString();
			printMsg($unableToContinueUpdateMsg);
			printMsg($errMsg, 1);
			$self->setCanContinueUpdate(0); 
			return undef;
		}
		if($parameterId eq 'AddHosts'){
		    my $errMsg = $pendingConfig->getErrorString();
		    printMsg($unableToContinueUpdateMsg);
            printMsg($errMsg, 1);
            $self->setCanContinueUpdate(0); 
            return undef;
		}
	}

	for my $cmp (@{$pendingConfig->getComponentManager()->getAllComponents()}) {
		if ($cmp->getIgnorePersistency()) {
			my $msg = $pendingConfig->getPersistenceManager()->_getPendingErrorMessage($cmp->getComponentName(), $cmp->getVersion(), $cmp->getPersistedVersion());
			$pendingConfig->setErrorMessage($msg);
			printMsg($unableToContinueUpdateMsg);
			printMsg($msg, 1);
			$self->setCanContinueUpdate(0); 
			return 0;
		}
	}

	$pendingConfig->setBatchValueOfRequiredParams('ExecutionMode');

    $self->setCanContinueUpdate(1);
	return 1;
}

sub _addProvidedComponentsToPendingConfig {
	my ($self, $pendingConfig) = @_;

	if (!$self->existsComponentLocationBatchValue()) {
		return;
	}

	for my $parameterId (@{['ComponentDirs', 'ComponentFsRoot', 'DvdPath']}) {
		my $value = $self->getBatchValue($parameterId);
		next if (!$value);
		my $parameter = $pendingConfig->{params}->{$parameterId};
		my $rc = $pendingConfig->setPersistedValue($parameterId, $parameter, $value);
	}
}

# Priority of locations during resume of update is: (target > default > persisted).
# If there are provided locations, the persisted locations are ignored.
sub _addDefaultDetectedComponentsToPendingConfig {
	my ($self, $pendingConfig, $hdblcmPersistedData) = @_;
	my $componentManager = $self->getComponentManager();
	my $defaultDetectedComponents = $componentManager->getDefaultDetectedComponents();

	delete $self->{'_modified_persisted_ComponentDirs'};
	delete $self->{'_existsDefaultComponentLocationOverridingPersistedLocation'};

	return if(!defined $defaultDetectedComponents || scalar(@{$defaultDetectedComponents}) == 0);

	if ($self->existsComponentLocationBatchValue()) {
		$self->_mergeProvidedAndDefaultComponentsToPendingConfig($pendingConfig, $defaultDetectedComponents);
	} else {
		$self->_mergePersistedAndDefaultComponentsToPendingConfig($pendingConfig, $hdblcmPersistedData, $defaultDetectedComponents);
	}
}

# Should be called if there are provided component locations.
sub _mergeProvidedAndDefaultComponentsToPendingConfig {
	my ($self, $pendingConfig, $defaultDetectedComponents) = @_;
	my $defaultLocationsToPendingConfig = [];

	for my $defaultDetectedComponent (@{$defaultDetectedComponents}) {
		my $defaultCmpKeyname = $defaultDetectedComponent->getComponentKeyName();
		my $targetComponent = $pendingConfig->getComponentManager()->getComponentByKeyName($defaultCmpKeyname);

		next if(defined $targetComponent);

		my $defaultComponentPath = $defaultDetectedComponent->getPath();
        my $isXSAApplication = (-f $defaultComponentPath && $defaultComponentPath =~ /\.zip$/i) ? 1 : 0;
        $defaultComponentPath = dirname($defaultComponentPath ) if ($isXSAApplication);
		push(@{$defaultLocationsToPendingConfig}, $defaultComponentPath);
	}

	if (scalar(@{$defaultLocationsToPendingConfig}) > 0) {
		my $defaultLocationsToPendingConfigCsv = join(',', @{$defaultLocationsToPendingConfig});
		my $componentDirsParameter = $pendingConfig->{params}->{'ComponentDirs'};
		my $rc = $pendingConfig->setPersistedValue('ComponentDirs', $componentDirsParameter, $defaultLocationsToPendingConfigCsv);
	}
}

# Should be called if there are not provided component locations.
sub _mergePersistedAndDefaultComponentsToPendingConfig {
	my ($self, $pendingConfig, $hdblcmPersistedData, $defaultDetectedComponents) = @_;
	my $persistedComponentDirs = $hdblcmPersistedData->{'ComponentDirs'};

	return if (!$persistedComponentDirs);

    $self->getMsgLst()->addProgressMessage("Processing software locations from pending update...");
	my $tempPersistenceConfiguration = new LCM::Configuration::GenericStackUpdateConfiguration();
	my $tempPersistenceComponentManager = new LCM::ComponentManager::MediumComponentManager($tempPersistenceConfiguration);
	my @persistedComponentDirs = split(',', $persistedComponentDirs);

	return if($tempPersistenceComponentManager->detectComponentsByDirectoryList(\@persistedComponentDirs, undef, 1));

	my $defaultLocationsToPendingConfig = [];
	my $newPersistedComponentDirs = [];

	for my $defaultDetectedComponent (@{$defaultDetectedComponents}) {
		my $defaultCmpKeyname = $defaultDetectedComponent->getComponentKeyName();
		my $persistedComponent = $tempPersistenceComponentManager->getComponentByKeyName($defaultCmpKeyname);
		my $defaultComponentPath = $defaultDetectedComponent->getPath();
        my $isXSAApplication = (-f $defaultComponentPath && $defaultComponentPath =~ /\.zip$/i) ? 1 : 0;
        $defaultComponentPath = dirname($defaultComponentPath ) if ($isXSAApplication);

		push(@{$defaultLocationsToPendingConfig}, $defaultComponentPath);

		if (!defined $persistedComponent){
			next;
		}

		my $persistedComponentPath = $persistedComponent->getPath();
		$newPersistedComponentDirs = removeElementFromArray(\@persistedComponentDirs, $persistedComponentPath);
		$self->setExistsDefaultComponentLocationOverridingPersistedLocation(1);
	}

	if ($self->existsDefaultComponentLocationOverridingPersistedLocation()) {
		my $newPersistedComponentDirsCsv = join(',', @{$newPersistedComponentDirs});
		$hdblcmPersistedData->{'ComponentDirs'} = $newPersistedComponentDirsCsv;
		$self->setModifiedPersistedComponentDirs($newPersistedComponentDirsCsv);
	}

	if (scalar(@{$defaultLocationsToPendingConfig}) > 0) {
		my $defaultLocationsToPendingConfigCsv = join(',', @{$defaultLocationsToPendingConfig});
		my $componentDirsParameter = $pendingConfig->{params}->{'ComponentDirs'};
		my $rc = $pendingConfig->setPersistedValue('ComponentDirs', $componentDirsParameter, $defaultLocationsToPendingConfigCsv);
	}
}

# Improves resume of update interaction in case of persisted newer version of component.
# Provides to user possibility to select the newer version.
sub loadComponentDirsPersistedValue {
	my ($self, $sid) = @_;
	return undef if (!$self->setComponentDirsPersistedValue($sid));
	$self->addCannotBeSelectedConsoleText();
	return 1;
}

sub setComponentDirsPersistedValue {
	my ($self, $sid) = @_;
	if ($self->isPendingConfiguration()) {
		return 1;
	}

	return 1 if(!existsHdblcmPendingUpdate($sid));
	return 1 if($self->existsComponentLocationBatchValue());
	return 1 if($self->existsDefaultComponentLocationOverridingPersistedLocation());

	my $hdblcmPersistedData = $self->_loadHdblcmPersistenceSilently();
	my $cmpDirsPersistedValue = $hdblcmPersistedData->{'ComponentDirs'};

	return 1 if !$cmpDirsPersistedValue;
	return $self->setValue('ComponentDirs', $cmpDirsPersistedValue);
}

sub addCannotBeSelectedConsoleText {
	my ($self) = @_;
	my $cannotBeSelectedMessage = $self->getCannotBeSelectedMessage();

	if (defined $cannotBeSelectedMessage && $cannotBeSelectedMessage ne "") {
		my $consoleText = $self->{params}->{SelectedComponents}->{console_text};
		$self->{params}->{SelectedComponents}->{console_text} = $cannotBeSelectedMessage . "\n" . $consoleText;
		$self->clearRestrictedComponentsMessages();
	}
}

sub checkSID {
    my ( $self, $sid ) = @_;
    my $mcm = $self->getMediumComponentManager();
    my $sidString = $self->{params}->{SID}->{str};
    if(!$self->checkExistingSID($sid, $sidString)) {
        return 0;
    }
    if(!$isWin) {
        my $user = new SDB::Install::NewDBUser($sid);
        my $sidadm = $user->getSidAdmName();
        if(!$user->gid() || $user->userGid() != $user->gid()) { # $user->gid() returns sapsys gid
            $self->PushError ("OS user '$sidadm' exists on the local host already ".
                              "and does not have '$gSapsysGroupName' configured as primary group.");
            $self->{params}->{SID}->{no_retry} = 1;
            return 0;
        }
    }
    my $sapSys = $self->getCollectSAPSystems()->{$sid};
    if (exists $self->{params}->{CheckMnt}) {
        $self->{sapSys}   = $sapSys;
        my $globalSidDir  = $sapSys->get_globalSidDir();
        my $usrSapPattern = join(quotemeta($path_separator), '^', 'usr', 'sap');
        my $sidPattern    = quotemeta($path_separator) . $sid . '$';
        if (($globalSidDir !~ /$usrSapPattern/) && ($globalSidDir =~ /$sidPattern/)) {
            $self->{params}->{CheckMnt}->{value} = $globalSidDir;
        }
    }
    return $self->checkLocalHost($sid, $sapSys);
}

sub checkReferenceDataPath {
	my ($self, $path) = @_;

    if ($path eq "") {
        $self->PushError("Reference data path should not be empty.");
        return 0;
    }
	
	my $uid = getLocalHostSidadmUid($self);
	my $gid = getgrnam ($gSapsysGroupName);

	my $errorMessageList = new SDB::Install::MsgLst();
	if ( ! -e $path ) {
		createDirectory($path, oct(INSTALLATION_PATH_MODE), $uid, $gid, undef, $errorMessageList);
		$self->_addWarningMessageFromMessageList($errorMessageList);
	}

	if ( ! -d $path ) {
		$self->getMsgLst()->addWarning("Provided $path is not a directory");
		return 0;
	}

	$errorMessageList = new SDB::Install::MsgLst();
	my $validator = new LCM::Configuration::Validators::FileModeValidator();
	my $isValid = $validator->isValid($path, $uid, $gid, int(INSTALLATION_PATH_MODE), $errorMessageList);
	$self->_addWarningMessageFromMessageList($errorMessageList);
	return 0 if ! $isValid;

	return $self->checkHanaOptionsPathParam('ReferenceDataPath', $path);
}

sub _addWarningMessageFromMessageList {
	my ($self, $errorMessageList) = @_;
	my $errorMessage = $errorMessageList->getMsgLstString();
	chomp($$errorMessage);
	$self->getMsgLst()->addWarning($$errorMessage);
}

sub isUpdate {
	return 1;
}

sub isPendingUpdate {
	my $self = shift;
	my $isPendingUpdate = LCM::PersistencyUtils::isPendingUpdate($self->getSID(), {});
	my $canContinueUpdate = $self->canContinueUpdate();
	my $isContinueUpdate = $self->getValue('ContinueUpdate') ? 1 : 0;
	my $continueUpdate = defined $canContinueUpdate ? $canContinueUpdate && $isContinueUpdate : 0;
	return $isPendingUpdate && $continueUpdate;
}

sub checkRootUser{
    my ($self, $value) = @_;

	$self->setValue('CheckMnt', $self->getBatchValue('CheckMnt'));
    if ( ! $self->SUPER::checkRootUser( $value ) ) {
        return undef;
    }

    if (! $self->checkPathsExistenceOnHosts()) {
        return 0;
    }

    return 1;
}

sub checkRootPassword {
    my ($self, $value) = @_;

    if ( ! $self->SUPER::checkRootPassword( $value ) ) {
        return undef;
    }

    if (! $self->checkPathsExistenceOnHosts()) {
        return 0;
    }

    return 1;
}

sub getRemoteHostsObjects {
	my ($self) = @_;
	my $remoteHostsAlreadyPartOfLandscape = $self->getRemoteHosts();
	my $additionalRemoteHosts = $self->getAdditionalRemoteHosts();
	my $resultArray = [];

	if(defined $remoteHostsAlreadyPartOfLandscape){
		push @{$resultArray}, $remoteHostsAlreadyPartOfLandscape; 
	}
	if(defined $additionalRemoteHosts){
		push @{$resultArray}, $additionalRemoteHosts; 
	}
	return $resultArray;
}

sub setScope {
    my ($self, $value) = @_;

# Not a clean solution for the problem, but if we want to init remote hosts connections
# we have to know the SID and Scope, also according to Tsetso Tsokov we should do it
# before we set the scope's value
    return 0 if (!$self->_handleMultipleHostParameters($value));

    if ( $self->checkScope($value) ) {
        $self->{params}->{'Scope'}->{value} = $value;
        return $self->SUPER::setScope($value);
    }

    return 0;
}

sub checkScope {
    my ($self, $value) = @_;
    my @validValues = ('instance', 'system');
    return ($value ~~ @validValues);
}

#-------------------------------------------------------------------------------
# Does not collect information of remote HANA hosts because,
# there are no parameters which should be verified.
#
# Changed in May 2014: SSL certificate generation requires CollectHostInfo
# on each HANA host even in upgrade case.
# This makes getRemoteHostsObjectsForCollectHostInfo() obsolete.
#
sub getRemoteHostsObjectsForCollectHostInfo {
    return $_[0]->getRemoteHostsObjects();
}

sub getLocalHanaHost {
    my $self = shift;
    my $instance = $self->getOwnInstance();
    my $hostname = $instance->get_host() if ( defined $instance);
    $hostname    = lc(hostname())        if (!defined $hostname);
    return $hostname;
}

sub _recomputeRequiredRootCredentials {
    my $self = shift;
    my $rootCredentialsRequired;

    if ( $self->isScopeInstance() || $self->isUseSAPHostagent() || $self->isSidadmin($self->getSID())) {
        $rootCredentialsRequired = 0;
    } else {
        my @componentKeys = ($gKeynameEngine, $gKeynameInstaller, $gKeynameClient, $gKeynameXS2, $gKeynameLSS);
        my $systemHosts = $self->getSystemHosts();
        my $addHosts      = $self->getValue('AddHosts');
        my $isAddingHosts = (! $self->isSkipped('AddHosts'))
                            && defined $addHosts && length($addHosts) > 0;
        my $isAssigningRemoteRoles = !$self->isSkipped('AddRoles') && defined($self->getValue('AddRoles')) && $self->_isAssigningAdditionalRemoteRoles();
        my $isAutoAssigningRemoteRoles = !$self->isSkipped('AutoAddXS2Roles') && ($self->getValue('AutoAddXS2Roles') =~ /$bool_true_pattern/) && $self->_isAutoAssigningAdditionalRemoteRoles();
        my $isAddingHostsOrRemoteRoles = $isAddingHosts || $isAssigningRemoteRoles || $isAutoAssigningRemoteRoles;
        my $isRequiredForComponentInstallation = grep { $self->isComponentSelected($_) } @componentKeys;

        $rootCredentialsRequired = ($isRequiredForComponentInstallation && (@$systemHosts > 1) ) || $isAddingHostsOrRemoteRoles;
    }
   	$self->setSkip('RootUser', $rootCredentialsRequired ? 0 : 1);
	$self->setMandatory('RootUser', $rootCredentialsRequired );
	if (!$self->getValue('RootPassword')) {
		$self->setSkip('RootPassword', 1);
		$self->{'params'}->{'RootPassword'}->{'mandatory'} = 1;
	}
}

sub _isAssigningAdditionalRemoteRoles {
    my ($self) = @_;
    my $rolesParameter = $self->{params}->{AddRoles};

    for my $host (@{$rolesParameter->{origin_values}}) {
        my $isLocalHost = LCM::HostsParser::IsLocalHost($host);
        my $value = $rolesParameter->{value}->{$host};

        return 1 if($value && ! $isLocalHost);
    }
    return 0;
}

sub _isAutoAssigningAdditionalRemoteRoles {
	my ($self) = @_;
	my $autoAssignRolesMap = $self->getAutoAssignXs2RolesMap();

	for my $host (keys(%{$autoAssignRolesMap})){
		my $isLocalHost = LCM::HostsParser::IsLocalHost($host);
		return 1 if(!$isLocalHost && length($autoAssignRolesMap->{$host}) > 0);
	}
	return 0;
}

sub _shouldSkipHostagentPassword {
    my ($self) = @_;
    return (!$self->_isAddingHostsViaSHAExecution() && !$self->_shouldInstallSHAOnHosts());
}

sub _isAddingHostsViaSHAExecution {
    my ($self) = @_;
    my $isUseSha = $self->isUseSAPHostagent();
    my $addHosts = $self->getValue('AddHosts');
    my $isAddingHosts = (! $self->isSkipped('AddHosts'))
                        && defined $addHosts && length($addHosts) > 0;
    return ($isUseSha && $isAddingHosts);
}

sub fillCertificatesHostmapDefaultValues {
    my ($self) = @_;

    if ($self->isScopeInstance()) {
       return $self->SUPER::fillCertificatesHostmapDefaultValues("");
    } else {
       return $self->SUPER::fillCertificatesHostmapDefaultValues();
    }
}


sub _recomputeAddHostAddRolesParam {
    my ( $self ) = @_;
    if (!$self->_checkForSuppressedRestartByPassThrough()){
        my $passThroughIdNoStart = $self->getPassThroughParamId('server', 'hdbupd', 'NoStart');
        my $nostartOpt = $self->getOpt ($passThroughIdNoStart);
        my $wrn = "Option cannot be used, if '$nostartOpt' is set.";
        $self->setSkip('AddRoles',1, $wrn);
        $self->setSkip('AddHosts',1, $wrn);
    }
}

sub _recomputeCertificatesHostmapParam {
    my ( $self ) = @_;
    $self->setSkip('CertificatesHostmap', 1);
    # CertificatesHostMap will be recomputed when remote hosts connections are established and a server is selected
    if($self->isRemoteHostAuthenticationSkipped()){
        require LCM::Configuration::ValueChangeListeners::CertificatesSkipValueConfigurator;
        $self->fillCertificatesHostmapDataForLocalHost();
        my $skippingCertificatesHelper = new LCM::Configuration::ValueChangeListeners::CertificatesSkipValueConfigurator();
        $skippingCertificatesHelper->configureSkippingCertificates($self);
    }
}

sub handleSelectionOfServerComponent {
	my $self = shift;
	my $serverComponent = $self->getComponentManager()->getComponentByKeyName($gKeynameEngine);

	if (defined $serverComponent) {
		if ($self->isBootstrapFromResident()) {
			$serverComponent->setCanSelectComponent(0);
			$serverComponent->selectComponent();
		} else {
			$serverComponent->setCanSelectComponent(1);
		}
	}
}

sub checkPassword{
    my ($self, $value) = @_;
    if (!defined $self->{user}){
        return 1;
    }
    my $savedCtx = $self->{user}->setMsgLstContext([$self->getMsgLst()]);
# Do not verify the sidadm password on hosts for addition as it does not exist
    $self->{user}->setVerifyOnAdditionalHosts(0);
    my $rc = $self->{user}->verifyPassword($value, $self);
    $self->{user}->setMsgLstContext($savedCtx);

    if (!defined $rc) {
        $self->AddWarning ("Cannot check password", $self->{user});
        $self->AddMessage ("=> pass check");
        return 1;
    }
    return $rc;
}

sub checkHostagentPassword {
    my ($self, $value) = @_;

    if (!$value || length ($value) == 0) {
        $self->PushError
            ("$self->{params}->{HostagentPassword}->{str} must not be empty.");
        return 0;
    }

    if (!$self->{isSlave} || !$self->isCheckOnly()) {
        my $userName = $self->getValue('HostagentUserName');
        $userName    = 'sapadm' if (!defined $userName);

        require LCM::Configuration::SapAdmUser;

        my $user     = new LCM::Configuration::SapAdmUser($userName);

        if ($user->exists()) {
            my $rc = $user->verifyPassword($value, $self);
            if (!defined $rc){
                $self->getMsgLst()->appendMsgLst ($user->getErrMsgLst());
                $self->getMsgLst()->addMessage ("Ignoring error => password validation skipped.");
                return 1;
            }
            if (!$rc){
                $self->PushError ($self->{params}->{HostagentPassword}->{str} .
                    ': Unknown user password combination');
                return 0;
            }
        }
    }

    if ($self->isUseSAPHostagent() && (! $self->checkPathsExistenceOnHosts())) {
        return 0;
    }

    return 1;
}

sub setAddRoles {
    my ( $self, $hostname, $roles, $overwriteAlreadyDefined ) = @_;

    if ( !$self->checkAddRoles( $hostname, $roles, $overwriteAlreadyDefined ) ) {
        return 0;
    }
    if ($roles eq ""){
        $roles = undef;
    }
    $self->{params}->{AddRoles}->{value}->{ lc($hostname) } = $roles;
    return 1;
}


sub getPassThroughParamId{
    my ($self, $section, $tool, $paramId) = @_;
    my $passThroughPrefix = $self->getPassThroughParamPrefix ($section, $tool);
    my $passThroughId = $passThroughPrefix . $paramId;
    return $passThroughId;
}


sub _checkForSuppressedRestartByPassThrough{
    my ($self, $param) = @_;
    if (!$self->getComponentManager()->isHDBServerComponentSelected()){
        return 1;
    }
    my $passThroughIdNoStart = $self->getPassThroughParamId('server', 'hdbupd', 'NoStart');
    my $nostart = $self->getBatchValue ($passThroughIdNoStart);
    if ($nostart){
        my $nostartOpt = $self->getOpt ($passThroughIdNoStart);
        if (defined $param){
            $self->appendErrorMessage ("Cannot use '$param->{str}', while '$nostartOpt' command line option is set");
        }
        return 0;
    }
    return 1;
}



sub checkAddRoles {
    my ( $self, $hostname, $roles ) = @_;

    if (!$self->_checkForSuppressedRestartByPassThrough($self->{params}->{AddRoles})){
        return 0;
    }

    if ( !defined $roles ) {
        $self->getErrMsgLst()->addError("The roles '$roles' specified for host '$hostname' should not be empty");
        return 0;
    }

    my %originValues = map { $_ => 1 } @ { $self->getParamOriginValues('AddRoles') };
    my $lcHostname   = lc($hostname);
    if ( !defined $originValues{$lcHostname} ) {
        $self->getErrMsgLst()->addError("Host name '$hostname' is not part of the system");
        return 0;
    }

    if ( $hostname !~ /$hostname_regex|$ipv4_regex/ ) {
        $self->getErrMsgLst()->addError("'$hostname' is no valid host name or ipv4 address");
        return 0;
    }

    return 0 if (! $self->_checkRolesForHost($hostname, $roles));

    my $ownInstance = $self->getOwnInstance();
    my $hostsWithRoles = $ownInstance->getHostRolesInfo();
    my $existingHostRoles = $hostsWithRoles->{$hostname};
    if ( !$self->checkHostRoles( $hostname, $roles, $existingHostRoles ) ){
        return 0;
    }

    return 1;
}

sub checkEntriesAddRoles {
    my ($self) = @_;
    my $instance = $self->getOwnInstance();
    my $hasXS2Controller = grep { defined($instance) && $instance->hasXS2Controller($_) } @{$instance->get_allhosts()};

    return 1 if ($hasXS2Controller);
    return 1 if (!defined($instance));
    return 1 if (!$self->isComponentInstalled($gKeynameXS2));

    my $addRoles = $self->getValue('AddRoles');
# Bug: 218796 AddRoles is not a mandatory param and in resume of XSA without
# AutoAddXS2Roles it is not skipped and empty and shouldn't skip OrgManager credentials
    return 1 if (!defined $addRoles);

    my $addedRolesCsv = join(',', values %$addRoles);
    $self->unskipLoadInitialContentParamsIfNecessary($addedRolesCsv);
    return 1;
}

sub _checkRolesForHost {
    my ($self, $hostname, $roles) = @_;
    my $validRoles = GetHostRoleProperties();
    my @rolesForCheck = (   $gHostRoleAcceleratorWorker,
                            $gHostRoleAcceleratorStandby,
                            $gHostRoleEsWorker,
                            $gHostRoleEsStandby);
    my @addHostsWithRole = ();
    for my $role ( split(',', $roles) ) {
        if (!exists($validRoles->{$role}) || $validRoles->{$role}->{isColumnStore}) {
            $self->getErrMsgLst()->addError("Invalid host role '$role' in --add_roles entry for host '$hostname'.");
            return 0;
        }
        @addHostsWithRole = @ { $self->getAddHostsWithRole( $role ) };
        if ( $role ~~ @rolesForCheck && scalar @addHostsWithRole > 0 ) {
            my $humanReadableRole = getHumanReadableRoleByHostRole( $role );
            my @hosts = (@addHostsWithRole, $hostname);
            my $hostsString = join (',', @hosts);

            my $message = "More than one host ($hostsString) has been assigned with the role $humanReadableRole. ";
            $message .= "Only one host with this role can be added to a HANA instance.";
            $self->getErrMsgLst()->addError($message);
            return 0;
        }
    }
    return 1;
}

sub isUsingUDSCommunication { # Unix domain sockets
    my ($self) = @_;
    my $isUDSSupported = $self->getFeatureManager()->isUDSCommunicationSupported();
    my $sysvar = new SDB::Install::SysVars();
    my $isUnix = !$sysvar->isWin();
    my $isAdmin = isAdmin();
    my $isAddingHosts = defined($self->getValue('AddHosts')) ? 1 : 0;
    my $isDistributed = $self->isDistributedSystem();
    return $isUDSSupported && $isUnix && $isAdmin && !$isAddingHosts && !$isDistributed;
}

sub _recomputeRequiredCredentials {
	my $self = shift();
	$self->SUPER::_recomputeRequiredCredentials(@_);
    if ($self->isUsingUDSCommunication()) {
        $self->setSkip('Password');
        return 1;
    }

	my $isAutoAssignEnabled = $self->getValue('AutoAddXS2Roles') && ! $self->isSkipped('AutoAddXS2Roles');
	my $autoAssignedRolesMap = $isAutoAssignEnabled ? $self->getAutoAssignXs2RolesMap() : {};

	for my $host (keys(%{$autoAssignedRolesMap})){
		if(length($autoAssignedRolesMap->{$host}) > 0){
			$self->setSkip('Password', 0);
			return;
		}
	}
}

sub setAutoAddXS2Roles {
	my ($self, $value) = @_;

	return undef if($value !~ /^$bool_false_pattern|$bool_true_pattern$/);

	$value = ($value =~ /^$bool_true_pattern$/) ? 1 : 0;
	$self->{params}->{AutoAddXS2Roles}->{value} = $value;
	$self->_recomputeRequiredRootCredentials(); # Needed due to bug 99557
	$self->_recomputeRequiredCredentials();
	$self->unskipLoadInitialContentParamsIfNecessary($gHostRoleXS2Worker) if($value);

	return 1;
}

# Override
sub unskipLoadInitialContentParamsIfNecessary {
    my ($self, $addRolesOrHostsCsv) = @_;

    if (!$self->isPendingUpdate()){
        $self->SUPER::unskipLoadInitialContentParamsIfNecessary($addRolesOrHostsCsv);
        return;
    }

    return if $self->skipOrgManagerCredentialsIfNecessary($addRolesOrHostsCsv);
    $self->unskipLoadInitialContentParams();
}

sub handleSelectionOfHostRoles {
	my ($self, $value) = @_;

	if(defined($value) && length($value) > 0){
		$self->setSkip('Password', 0);
	}

	$self->setSkip('AutoAddXS2Roles', $self->_shouldSkipAutoAddXS2Roles());

	my $isAcceleratorAvailable = $self->_isComponentAvailable($gKeynameAccelerator);
	my $isAddingAcceleratorRole = ($value =~ /$gHostRoleAcceleratorStandby|$gHostRoleAcceleratorWorker/);

	if ($isAcceleratorAvailable && $isAddingAcceleratorRole){
		$self->setSkip('AcceleratorUser', 0);
		$self->setSkip('AcceleratorPassword', 0);
		$self->setSkip('SystemUser', 0);
		$self->setSkip('SQLSysPasswd', 0);
	}
}

sub checkAddHostsAvailability {
    my ($self) = @_;
    my $param = $self->{params}->{'AddHosts'};
    return $self->_checkForSuppressedRestartByPassThrough($param);
}

sub isPendingAction {
    return $_[0]->isPendingUpdate();
}

sub checkAddHosts{
    my ($self, $value) = @_;

    my $rc = $self->_checkAddHosts($value);
    return $rc if defined $rc;

    return $self->SUPER::checkAddHosts( $value, undef, 1 );
}

sub checkRole {
    my ($self, $role) = @_;
	if ( ! $self->SUPER::checkRole( $role ) ) {
		return undef;
	}

	return ( $role eq $gHostRoleEsWorker && ! $self->canAddEsWorkerHost() ) ? 0 : 1;
}

sub handleAdditionOfHosts {
	my ($self, $value) = @_;
	$self->_recomputeRequiredParameters();
	$self->handleSelectionOfHostRoles($value);
	my $shouldSkipHostagentPassword = $self->_shouldSkipHostagentPassword();
	$self->setSkip('HostagentPassword', $shouldSkipHostagentPassword);
	$self->setSkip('AutoAddXS2Roles', $self->_shouldSkipAutoAddXS2Roles());
}

sub checkFlavourCompatibility {
    my ($self, $sid) = @_;
    my $mcm = $self->getComponentManager();
    my $scm = $self->getSystemComponentManager();
    my $sourceFlavour = $scm ? $scm->getHANAFlavour() : $gFlavourPlatform;
    my $targetFlavour  = $mcm->getHANAFlavour() // $sourceFlavour;

    if($targetFlavour ne $sourceFlavour) {
        my $targetFlavourProductName = SDB::Install::Globals::getFlavourProductName($targetFlavour);
        my $sidString = $self->getString('SID');
        $self->setErrorMessage("$sidString '$sid' is not valid. The system is not $targetFlavourProductName.\n");
        return 0;
    }

    return 1;
}

sub addListeners {
	my ($self,) = @_;
	$self->SUPER::addListeners();

    $self->addParameterListener('RemoteExecution', LCM::Configuration::ValueChangeListeners::Update::RemoteExecutionListener->new() );
	$self->addParameterListener('SID', LCM::Configuration::ValueChangeListeners::Update::SIDListener->new() );
    $self->addParameterListener('SID', LCM::Configuration::ValueChangeListeners::Update::ConsoleWriterSIDListener->new() );
    $self->addParameterListener('SystemDBUseSingleDBUserPassword', LCM::Configuration::ValueChangeListeners::Update::SystemDBUseSingleDBUserPasswordListener->new());
    $self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::Update::DeprecatedComponentsListener->new() );
    $self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::Update::SelectedComponentsListener->new() );
    $self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::SignatureVerificationListenerFactory::getSignatureVerificationListener() );
	$self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::Update::XS::OrgNameDetectionListener->new() );
    $self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::Update::XS::OrgManagerUserDetectionListener->new() );
    $self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::XsInitTenantDatabaseListener->new() );
    $self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::Update::XS::XsTenantUserDetectionListener->new());
    $self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::Update::MetadataSeparationCheckListener->new() );
    $self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::Update::XsTenantInstallationListener->new() );
	$self->addParameterListener('SelectedComponents', SDB::Install::Configuration::LSSUserStringListener->new(['LSSUserID', 'LSSPassword']));
    $self->addParameterListener('SelectedComponents', LCM::Configuration::ValueChangeListeners::InstallUpdate::HandleLSSUserAndGroupListener->new() );
    $self->addParameterListener('LSSUserID', LCM::Configuration::ValueChangeListeners::ReservedIDsHandler->new('LSSUserID', 1));
    $self->addParameterListener('LSSGroupID', LCM::Configuration::ValueChangeListeners::ReservedIDsHandler->new('LSSGroupID', 0));
    $self->addParameterListener('ExecutionMode', LCM::Configuration::ValueChangeListeners::Update::UpdateExecutionModeListener->new() );
    $self->addParameterListener('SystemUser', LCM::Configuration::ValueChangeListeners::Update::SystemUserListener->new() );
    $self->addParameterListener('SystemUser',  LCM::Configuration::ValueChangeListeners::Update::EnableTenantSystemUserCredentials->new('SystemUser'));
    $self->addParameterListener('RootUser', LCM::Configuration::ValueChangeListeners::Update::RootUserListener->new() );
    $self->addParameterListener('RootUser', LCM::Configuration::ValueChangeListeners::SHAVersionListener->new());
    $self->addParameterListener('RootPassword', LCM::Configuration::ValueChangeListeners::SHAVersionListener->new());
    $self->addParameterListener('Scope', LCM::Configuration::ValueChangeListeners::Update::ScopeListener->new() );
    $self->addParameterListener('InstallHostagent', LCM::Configuration::ValueChangeListeners::Update::InstallHostagentListener->new() );
    $self->addParameterListener('HostagentPassword', LCM::Configuration::ValueChangeListeners::Update::NonRootRemoteHostsAuthenticationListener->new() );
    $self->addParameterListener('HostagentPassword', LCM::Configuration::ValueChangeListeners::SHAVersionListener->new() );
    if (!$isWin) {
        $self->addParameterListener('SID', LCM::Configuration::ValueChangeListeners::FillingRolesHostmapListener->new() );   
    }
    $self->addParameterListener('Password', LCM::Configuration::ValueChangeListeners::Update::NonRootRemoteHostsAuthenticationListener->new() );
    $self->addParameterListener('Password', LCM::Configuration::ValueChangeListeners::SHAVersionListener->new());
    $self->addParameterListener('AddHosts', LCM::Configuration::ValueChangeListeners::AddHosts::RemoteCredentialsListener->new() );
    $self->addParameterListener('AddHosts', LCM::Configuration::ValueChangeListeners::Update::AdditionalHostsListener->new() );
    $self->addParameterListener('AddRoles', LCM::Configuration::ValueChangeListeners::Update::AdditionalRolesListener->new() );
    $self->addParameterListener('ContinueUpdate', LCM::Configuration::ValueChangeListeners::Update::ContinueUpdateListener->new() );
    $self->addParameterListener('ProdSpaceName', LCM::Configuration::ValueChangeListeners::InstallUpdate::XSSpaceIsolationStringListener->new());
    $self->addParameterListener('XSSpaceIsolation', LCM::Configuration::ValueChangeListeners::InstallUpdate::RecomputeXs2SpaceIsolationParameters->new());
    $self->addParameterListener('XSSpaceIsolation', LCM::Configuration::ValueChangeListeners::XsSpaceIsolationListener->new());
    $self->addParameterListener('XSSpaceUserIdSAP', LCM::Configuration::ValueChangeListeners::ReservedIDsHandler->new('XSSpaceUserIdSAP', 1) );
    $self->addParameterListener('XSSpaceUserIdProd', LCM::Configuration::ValueChangeListeners::ReservedIDsHandler->new('XSSpaceUserIdProd', 1) );

    my $xs2AppsListener = LCM::Configuration::ValueChangeListeners::InstallUpdate::RecomputeXs2ApplicationParametersListener->new();
    $self->addParameterListener('SelectedComponents', $xs2AppsListener);
    $self->addParameterListener('AutoAddXS2Roles', $xs2AppsListener);
    $self->addParameterListener('AddHosts', $xs2AppsListener);
    $self->addParameterListener('AddRoles', $xs2AppsListener);
    $self->addParameterListener('SQLSysPasswd',  LCM::Configuration::ValueChangeListeners::Update::SystemDBSQLPasswordBatchHandler->new());
    $self->addParameterListener('SQLSysPasswd',  LCM::Configuration::ValueChangeListeners::Update::EnableTenantSystemUserCredentials->new('SQLSysPasswd'));
    $self->addParameterListener('AddHosts',$self->getAutoInitFamilyServicesParameterHanlder('AddHosts'));
    $self->addParameterListener('AddRoles',$self->getAutoInitFamilyServicesParameterHanlder('AddRoles'));
    $self->addParameterListener('InstallXSInDefaultTenant', LCM::Configuration::ValueChangeListeners::InstallUpdate::InstallXSInDefaultTenantListener->new());
    $self->addParameterListener('XSTenantDatabaseName', LCM::Configuration::ValueChangeListeners::Update::XsTenantDatabaseListener->new());
    $self->addParameterListener('TenantUser',SDB::Install::Configuration::TenantUserListener->new());
    $self->addParameterListener('Password', SDB::Install::Configuration::LSSPasswordListener->new());
    $self->addParameterListener('HostagentPassword', SDB::Install::Configuration::LSSPasswordListener->new());
    $self->addParameterListener('RootUser', SDB::Install::Configuration::LSSPasswordListener->new());
    $self->addParameterListener('RootPassword', SDB::Install::Configuration::LSSPasswordListener->new());
    $self->addParameterListener('SelectedComponents', SDB::Install::Configuration::LSSPasswordListener->new());
    $self->addParameterListener('AddComponentsLocation', LCM::Configuration::ValueChangeListeners::AddComponentsLocationListener->new());
    $self->addParameterListener('LSSTrustUnsignedServer', LCM::Configuration::ValueChangeListeners::InstallUpdate::LocalSecureStoreListener->new());
}

sub getAction{
	return UPDATE_ACTION;
}

sub _checkValidTargetCockpitStack {
	my ($self, $sourceCockpitStack, $targetCockpitStack) = @_;
	return undef if ! $sourceCockpitStack || ! $targetCockpitStack;

	my $sourceManifest = $sourceCockpitStack->getManifest();
	my $targetManifest = $targetCockpitStack->getManifest();
	my $msgLst = new SDB::Install::MsgLst();
	if(!$self->canUpgradeComponent($sourceManifest, $targetManifest, $targetCockpitStack->getComponentName(), $msgLst)){
		my $restrictedComponentMessage = $msgLst->getMsgLstString();
		$self->addRestrictedComponentsMessage($targetCockpitStack->getComponentName(), $$restrictedComponentMessage);
		return undef;
	}

	return 1;
}

sub fillValidSelectedComponentValues {
	my ($self) = @_;

	my $rc = $self->SUPER::fillValidSelectedComponentValues();

	my $mcm = $self->getComponentManager();
	my $scm = $mcm->getSystemComponentManager();
	if(defined $scm){
		my $sourceCockpitStack = $scm->getComponentByKeyName($gKeynameCockpitStack);
		if($sourceCockpitStack){
			my $targetCockpitStack = $mcm->getComponentByKeyName($gKeynameCockpitStack);
			return undef if $targetCockpitStack && ! $self->_checkValidTargetCockpitStack($sourceCockpitStack, $targetCockpitStack);
		}
	}

	my $residentInstallerComponent = $mcm->getComponentByKeyName($gKeynameInstaller);
	if(!$rc && $self->getBatchValue('SelectedComponents') eq 'hdblcm' && !$residentInstallerComponent){
		$self->getComponentManager()->_addResidentInstallerComponent();
		$rc = 1;
	}

	return undef if (!$rc);

	$residentInstallerComponent = $mcm->getComponentByKeyName($gKeynameInstaller);
	if (defined $residentInstallerComponent){
		my $residentInstallerCmpBatchKey = $residentInstallerComponent->getComponentBatchKey();
		push( @{ $self->{params}->{SelectedComponents}->{valid_values} }, $residentInstallerCmpBatchKey );
		push( @{ $self->{params}->{SelectedComponents}->{ui_values} }, "" );
		$self->{params}->{SelectedComponents}->{hidden_valid_values} = [$residentInstallerCmpBatchKey];
	}

	return $rc;
}

# must be called after collecting hosts info
sub setupInterServiceCommunicationParameters {
    my $self = shift();
    $self->setSkip('ListenInterface');
    $self->setSkip('InternalNetwork');

    my $currentListenInterface = $self->getListenInterfaceType();
    if ($currentListenInterface =~ /local/ && $self->getValue('AddHosts')) {
        $self->setDefault('ListenInterface', 'global');
        $self->fillInternalNetworkAndListenInterfaceValidValues();
        $self->restrictListenInterfaceMultiHost();
    }
}

sub showCertificatesHostmap {
    my ($self) = @_;

    if ($self->isDistributedSystem() && $self->isScopeInstance()) {
    	$self->getMsgLst()->addProgressMessage( "Note: Cannot retrieve current certificate hostnames for remote hosts, because scope parameter is set to instance." );
        $self->getMsgLst()->addProgressMessage( "      Enter value for 'Certificate Host Name' only if you want to generate certificates for remote hosts.\n" );
    }
}

sub persistValuesForScopeInstance {
    my ($self) = @_;

    my $ownInstance = $self->getOwnInstance();
    my $remoteHosts = $ownInstance->get_hosts();

    my $componentManager = $self->getComponentManager();
    if(!$componentManager->isHDBServerComponentSelected() && $self->isInstallingLSS()){
        $remoteHosts = $self->getRemoteDBHosts();
    }
    for my $host ( @$remoteHosts ) {
        $self->getMsgLst()->addMessage( "Creating status file for host $host..." );
    	if ( ! $self->_persistDataForHost( $host ) ) {
    		return undef;
    	}
    }

    return 1;
}

sub _persistDataForHost {
	my ( $self, $host ) = @_;

	my $ownInstance = $self->getOwnInstance();

	my $hostDir = $ownInstance->get_hostNameDir( $host );
    my $filename = 'hdblcm_update_host';

    my $persistenceFile = File::Spec->catfile( $hostDir, $filename );
    my $data = $self->_getPersistentDataForHost( $host );

    my $rc = $self->pers_store( $persistenceFile, $data );
    if ( $rc ) {
       $self->getMsgLst()->addMessage( "Status file '$persistenceFile' created." );
    } else {
       $self->getMsgLst()->addError( "Creation of status file '$persistenceFile' failed." );
    }
	
	my $user = new SDB::Install::NewDBUser($self->getValue('SID'));
	my $uid = $user->uid();
	my $gid = $user->gid();
	my $fullentry = join($path_separator, $hostDir, $filename);
	$rc = chown($uid, $gid, $fullentry);
	if(!$rc) {
		$self->getMsgLst()->addError( "Could not chown $fullentry: $!" );
		return undef;
	}
	$rc = chmod(0644, $fullentry);
	if(!$rc) {
		$self->getMsgLst()->addError("Could not chmod $fullentry: $!");
		return undef;
	}
    return $rc;
}

sub _getPersistentDataForHost {
    my ( $self, $host ) = @_;

    my $data = {};
    $self->_addCertificatesData( $host, $data );
    return $data;
}

sub _addCertificatesData {
    my ( $self, $host, $data ) = @_;
    my $certificates = $self->{params}->{CertificatesHostmap}->{value};
    $data->{"certificate"} = $certificates->{$host};
}

sub getInstallerCollectOtherHostInfoOperation{
    return $gOperationHdblcmCollectInstallerInfo;
}

sub copyInstallerToOtherHosts{
    my ($self, $remotePath) = @_;
    if ($self->isUseSidadmUserForRemoteHostctrlHosts()) {
       return $self->copyInstallerToOtherHostsThroughSHA($remotePath);
    }
    return $self->SUPER::copyInstallerToOtherHosts($remotePath);
}

sub _getRemotePathForInstaller {
    my $self = shift();
    if ($self->isUseSidadmUserForRemoteHostctrlHosts()) {
        my $tmpDir = $SDB::Install::Configuration::AnyConfig::remoteTmpDir;
        return File::Spec->catfile($self->getValue('Target'), $self->getSID(), $tmpDir);
    }
    return $self->SUPER::_getRemotePathForInstaller(@_);
}

sub copyInstallerToOtherHostsThroughSHA {
    my ( $self, $remotePath ) = @_;
    my $hdblcmDir = dirname(new SDB::Install::Installer()->GetRuntimeDir());
    my $signature_dir = determineSignedManifestRelativePath($hdblcmDir);
    my $suffix = $remotePath;
    $suffix =~ s/^.*hdbinst_remote_check_//;

    my $optionMap = {
        'SAPMNT' => $self->getValue('Target'),
        'SOURCE_DIR' => $hdblcmDir,
        'SID' => $self->getSID(),
        'SUFFIX' => $suffix,
        'SIGNATURE_DIR' => $signature_dir,
    };
    my @localHost = (hostname());
    my $localExecutionHost = new SDB::Install::RemoteHostctrlHosts(@localHost);
    my $password = $self->getValue('Password');

    if (defined $password) {
        $localExecutionHost->useSidadm();
    } else {
        $localExecutionHost->useHostagentUser(); 
    }
    $localExecutionHost->setMsgLstContext($self->getMsgLstContext());
    my $exitCode = $localExecutionHost->executeHostctrlParallel(
                                                    'HdblcmCopyResidentInstaller_v4',
                                                    $self,
                                                    undef, # param IDs
                                                    undef, # password IDs
                                                    undef, # remote ignore
                                                    undef, # remote timeout
                                                    "Copying resident installer...",
                                                    "Resident installer copied.",
                                                    "Copying resident installer failed!",
                                                    $optionMap,
                                                    undef, # host option map
                                                    undef,                    
                                                    undef, # do not fail
                                                    0, # Suppress Console msgs
                                                    );
    if (!defined $exitCode || ($exitCode != 0)) {
      $self->setErrorMessage( "Copying resident installer through SAP Host Agent failed.", $localExecutionHost->getErrMsgLst());
      return undef;
    }
    $self->{_localInstallerDir} = $remotePath;
    return File::Spec->catfile($remotePath, $self->getInstallerExe());
}

sub removeRemoteTmpInstallerThroughSHA {
    my ($self) = @_;

# Very very ugly fix for the problem described with the followint trace (update/install components from the resident installer)
#####
# Use of uninitialized value $_[0] in substitution (s///) at File/Basename.pm line 341.
# Installation of SAP HANA system failed.
#   unhandled exception: fileparse(): need a valid pathname at LCM/Configuration/GenericStackUpdateConfiguration.pm line 722.
#####
    return 1 if(!defined($self->{_localInstallerDir}) || length($self->{_localInstallerDir}) == 0);
# Nothing was copied on the file system, so we have nothing to delete in here. 'copyInstallerToOtherHostsThroughSHA' was not called

    my $suffix = $self->{_localInstallerDir};
    $suffix =~ s/^.*hdbinst_remote_check_//;
    my $optionMap = {
        'SAPMNT' => $self->getValue('Target'),
        'SID'    => $self->getSID(),
        'SUFFIX' => $suffix,
    };
    my @localHost = (hostname());
    my $localExecutionHost = new SDB::Install::RemoteHostctrlHosts(@localHost);
    my $password = $self->getValue('Password');

    if (defined $password) {
        $localExecutionHost->useSidadm();
    } else{
        $localExecutionHost->useHostagentUser(); 
    }

    my $exitCode = $localExecutionHost->executeHostctrlParallel(
                                                    'HdblcmRemoveResidentInstaller',
                                                    $self,
                                                    undef, # param IDs
                                                    undef, # password IDs
                                                    undef, # remote ignore
                                                    undef, # remote timeout
                                                    "Removing temporary resident installer...",
                                                    "Temporary resident installer removed.",
                                                    "Removing temporary resident installer failed!",
                                                    $optionMap,
                                                    undef, # host option map
                                                    undef,
                                                    undef, # do not fail
                                                    0, # Suppress Console msgs
                                                    );
    undef $self->{_remoteInstallerFromKit};  
    undef $self->{_localInstallerDir};
    return 1;
}

sub removeRemoteTmpInstaller{
    my ( $self ) = @_;
    if ($self->isUseSidadmUserForRemoteHostctrlHosts()) {
       return $self->removeRemoteTmpInstallerThroughSHA();
    } else {
       return $self->SUPER::removeRemoteTmpInstaller();     
    } 
} 

sub isCollectOtherHostInfosRequired {
    my $self = shift();

    return 0 if(!$self->isDistributedSystem());
    return 0 if(!$self->getValue('SelectedComponents') eq 'hdblcm');
    return $self->SUPER::isCollectOtherHostInfosRequired(@_);
}

sub setPassword {
    my ($self, $value) = @_;
    my $rc = $self->checkPassword($value);

    return $rc if (!$rc);

    $self->{params}->{Password}->{value} = $value;

    if ($self->isUseSAPHostagent() && $self->isCollectOtherHostInfosRequired()) {
        my $isAddingHosts = $self->hasValue('AddHosts') && !$self->isSkipped('AddHosts');

        if (!$isAddingHosts && !defined $self->CollectOtherHostInfos()) {
            $self->setNoRetry('Password', 1);
            return undef;
        }
    }
    return 1;
}

sub _handleMultipleHostParameters {
	my ($self, $scopeValue) = @_;
	my $instance = $self->getOwnInstance();
	if (not defined $instance) {
		return undef;
	}
	if (!$instance->exists_remote_host() || $scopeValue eq 'instance') {
		return 1;
	}
	if (not $self->initRemoteHosts()) {
		return undef;
	}

	if ($self->isUseSidadmUserForRemoteHostctrlHosts()) {
		$self->enableMultipleHostParameters(1);
	}

	return 1;
}

sub isRemoteHostAuthenticationSkipped{
    my ($self) = @_;	
    return ($self->isSkipped('RootUser') || (isSidadmin($self->getSID()) && $self->isSkipped('Password')));
}

sub getHiddenIgnoreValues{
    return [qw (scan_password_database)];
}

sub _isComponentAvailable{
	my ($self, $componentKeyname) = @_;
	return 1 if($self->SUPER::_isComponentAvailable($componentKeyname));
	return $self->isComponentInstalled($componentKeyname);
}

sub isComponentInstalled {
	my ($self, $componentKeyname) = @_;
	my $scm = $self->getSystemComponentManager();

	return 0 if (!defined($scm));

	for my $component (@{$scm->getDetectedComponents()}){
		if($component->getComponentKeyName() eq $componentKeyname){
			return 1;
		}
	}
	return 0;
}

# Override
sub getPersistenceFileModifiers {
	my ($self) = @_;

	my $user = new SDB::Install::NewDBUser($self->getValue('SID'));
	my $persistenceFileUID = $user->uid();
	my $persistenceFileGID = $user->gid();
	my $persistenceFileMode = 0644;

	return ($persistenceFileUID, $persistenceFileGID, $persistenceFileMode);
}

sub setPersistedSteps{
	my($self,$components) = @_;
    $self->{'pers_update_steps'} = $components;
}

sub getPersistedSteps{
	my ($self) = @_;
	return $self->{'pers_update_steps'};
}

sub setCanContinueUpdate{
    my($self,$value) = @_;
    $self->{'canContinueUpdate'} = $value;
}

sub canContinueUpdate{
    my ($self) = @_;
    return $self->{'canContinueUpdate'};
}

sub pers_filename {
	return $_[0]->getPersistenceManager()->getPersistenceFilename();
}

sub pers_keys {
	my ($self) = @_;

	return [@{$self->SUPER::pers_keys()},
			'pers_update_steps',
			'pers_successfully_prepared_update',
			'pers_convert_to_mdc'];
}

sub pers_getstepname {
	my $self = shift();
	return $self->getStepName();
}

sub getParamPrepareUpdate{
	my ( $self, $order, $section ) = @_;

    my %param = (
            'order'                  => $order,
            'opt'                    => 'prepare_update',
            'type'                   => 'boolean',
            'section'                => $section,
            'value'                  => undef,
            'default'                => 0,
            'set_interactive'        => 0,
            'init_with_default'      => 1,
            'str'                    => 'Stop update before software version switch, resumable',
            'desc'                   => 'Stop update before software version switch, resumable',
            'hidden'                 => 0,
            'mandatory'              => 0
    );

    return \%param; 
}

sub getParamContinueUpdate {
	my ($self, $order, $section) = @_;
	return {
		'order' => $order,
		'opt' => 'continue_update',
		'type' => 'boolean',
		'section' => $section,
		'value' => undef,
		'default' => 1,
		'set_interactive' => 1,
		'init_with_default' => 1,
		'str' => 'Do you want to continue using these parameters? If you choose not to continue, you will be asked for all update parameters again. (y/n)',
		'desc' => 'Continue the pending process with the persisted parameters',
		'hidden' => 0,
		'mandatory' => 1,
		'console_omit_word_Enter' => 1,
		'skip' => 0,
	};
}

sub setPrepareUpdate{
    my ($self, $value) = @_;
    $self->{params}->{PrepareUpdate}->{value} = $value;
    if($value){
    	$self->setValue('Phase',PREPARE_PHASE);
    }
    return 1;
}

sub isPrepareUpdate{
    my ($self) = @_;
    return ($self->{params}->{PrepareUpdate}->{value}) ? 1 : 0;
}

sub getExistingHostsAutoAssignXs2RolesMap {
    my ($self) = @_;
    my $hdbInstance = $self->getOwnInstance();
    my $hostDetector = new LCM::Hosts::HostsDetector(undef, $self->getMsgLstContext(), $hdbInstance);
    my $isAddingRoles = !$self->isSkipped('AddRoles') && ref($self->getValue('AddRoles')) eq "HASH";

    my $resultMap = {};
    for my $host (@{$hdbInstance->get_allhosts()}){
        my $hostRoles = $hostDetector->getHostRoles($host);
        my $additionalRoles = $isAddingRoles ? $self->getValue('AddRoles')->{$host} : '';
        my $hostString = sprintf('%s:role=%s', $host, join(':role=', @{$hostRoles}, split(',', $additionalRoles)));

        $resultMap->{$host} = $self->_determineApplicableXs2Role($hostString);
    }

    return $resultMap;
}

sub getAutoAssignXs2RolesMap {
	my ($self) = @_;
	my $resultMap = $self->getExistingHostsAutoAssignXs2RolesMap();
	return $resultMap if($self->isSkipped('AddHosts'));

	my $additionalHostStrings = [ split(',', $self->getValue('AddHosts')) ];

	for my $hostString (@{$additionalHostStrings}) {
		if($hostString =~ /^([^:]+)/) {
			$resultMap->{$1} = $self->_determineApplicableXs2Role($hostString);
		}
	}
	return $resultMap;
}

sub isXSWorkerHostAvailable {
	my ($self) = @_;
	my $additionalRemoteRoles = $self->hasValue('AddRoles') && !$self->isSkipped('AddRoles') ? $self->getValue('AddRoles') : {};
	my $isAddingXsWorkerRole = grep { $_ =~ /$gHostRoleXS2Worker/ } values(%{$additionalRemoteRoles});
	my $addHostsValue = $self->hasValue('AddHosts') && !$self->isSkipped('AddHosts') ? $self->getValue('AddHosts') : '';
	my $autoAddXS2RolesValue = $self->hasValue('AutoAddXS2Roles') ? $self->getValue('AutoAddXS2Roles') : 1;

	return 1 if($isAddingXsWorkerRole);
	return 1 if($addHostsValue =~ /:role=$gHostRoleXS2Worker/);
	return 1 if(!$self->isSkipped('AutoAddXS2Roles') && $autoAddXS2RolesValue);

	my $hostDetector = new LCM::Hosts::HostsDetector(undef, $self->getMsgLstContext(), $self->getOwnInstance());
	return (scalar(@{$hostDetector->getHostsWithRoles([$gHostRoleXS2Worker])}) > 0) ? 1 : 0;
}

sub _updateReferenceDataPathParam {
    my ($self) = @_;

    my $globalSidDir = $self->getSAPSystem()->get_globalSidDir();
    my $configDir = join ($path_separator, $globalSidDir, 'global', 'hdb', 'custom' ,'config');
    my $defaultValue = join ($path_separator, $globalSidDir, 'global', 'hdb', 'IM' ,'reference_data');
    if(!-f File::Spec->catfile($configDir, 'scriptserver.ini')) {
        $self->setDefault('ReferenceDataPath', $defaultValue);
        return 1;
    }
    
    my $hdbInstance = $self->getOwnInstance();
    my $layeredCfg = $hdbInstance->getLayeredConfig ();
    if (!defined $layeredCfg){
        $self->PushError ('Cannot get layered configuration', $layeredCfg);
        return 0;
    }
    
    $layeredCfg->resetMsgLstContext();
    my $scriptServerIni = $layeredCfg->getIniFile ('scriptserver.ini');

    if (!defined $scriptServerIni){
        $self->PushError ('Cannot get scriptserver.ini', $layeredCfg);
        return 0;
    }

    $scriptServerIni->resetMsgLstContext();

    if (!defined $scriptServerIni->readValues ()){
        $self->PushError ('Cannot read scriptserver.ini', $scriptServerIni);
        return 0;
    }

    my $referenceDataPath = $scriptServerIni->getValue ('adapter_framework','dq_reference_data_path');
    if(! defined $referenceDataPath || !-d $referenceDataPath){
        $self->setDefault('ReferenceDataPath', $defaultValue);
        return 1;
    }

    $referenceDataPath = $1 if $referenceDataPath=~/(.*)\/$/; 
    my @realPathToParentDir = File::Spec->splitdir($referenceDataPath);
    pop(@realPathToParentDir);
    my $realParentPath = File::Spec->catdir(@realPathToParentDir);
    if (!-d $realParentPath){
        $self->PushError ('Cannot get parent directory of stored Reference Data path', $referenceDataPath);
        return 0;
    }
    
    $self->{params}->{'ReferenceDataPath'}->{value} = $realParentPath;
    return 1;
}

sub willInstallOrUpdateSHA {
    my ($self) = @_;
    my $installSHAValue = $self->getValue('InstallHostagent');
    my $isInstallSHA = (!defined $installSHAValue || ($installSHAValue =~ /$bool_true_pattern/)) ? 1 : 0;
    return 0 if (!$isInstallSHA);

    my $isServerSelected = $self->getComponentManager()->isHDBServerComponentSelected();
    my $isAddingHosts = (!$self->isSkipped('AddHosts')) && (length($self->getValue('AddHosts')) > 0);
    return ($isServerSelected || $isAddingHosts);
}

sub getProductName {
    my ($self) = @_;
    my $flavourProductName = $self->getFlavourProductName();
    return "Update $flavourProductName System and Components";
}

sub getDeprecatedComponentsForUninstall {
    my ($self) = @_;

    my $scm = $self->getSystemComponentManager();
	return [] if (!defined $scm);
    my $plugins = $scm->getServerPluginComponents();
    return [ map { $_->isDeprecated($self) ? $_ : () } @{$plugins} ];
}

sub _addDeprecatedPluginsWarnings {
	my ($self) = @_;
    my $mcm = $self->getComponentManager();
	my $serverComponent = $mcm->getComponentByKeyName($gKeynameEngine);
	return if (!defined $serverComponent);
	my $deprecatedPlugins = $serverComponent->getDeprecatedPlugins();
	for my $component (@{$self->getDeprecatedComponentsForUninstall()}) {
		my $componentName = $component->getComponentName();
		my $deprecatedPlugin = $deprecatedPlugins->{$component->getComponentBatchKey()};
		my $message = $deprecatedPlugin->{message};
		my $warning = "Server plugin $componentName is deprecated and will be automatically uninstalled.";
		my $replacedByCaption = $deprecatedPlugin->{replacedByCaption};
		$warning = $warning." It is replaced by ".$replacedByCaption."." if (defined $replacedByCaption);
		$warning = $warning." $message" if (defined $message);
		$self->_addDeprecatedPluginWarning($warning);
	}
}

sub _addDeprecatedPluginWarning {
	my ($self, $warning) = @_;
	$self->addParameterWarning('SelectedComponents', $warning);
	push (@{$self->{warnings}}, $warning);
}

sub validateDeprecatedComponent {
	my ($self, $component) = @_;

    my $mcm = $self->getComponentManager();
	my $serverComponent = $mcm->getComponentByKeyName($gKeynameEngine);
	return 1 if (! defined $serverComponent || ! $serverComponent->isComponentSelected());
	my $deprecatedPlugins = $serverComponent->getDeprecatedPlugins();
	my $deprecatedPlugin = $deprecatedPlugins->{$component->getComponentBatchKey()};
	my $replacedByKey = $deprecatedPlugin->{replacedByKey};
	return 1 if (!defined $replacedByKey);
	my $replacingComponent = $mcm->getComponentByBatchKey($replacedByKey);
	my $componentName = $component->getComponentName();
	my $replacedByMinVersion = $deprecatedPlugin->{replacedByMinVersion};
	my $replacedByCaption = $deprecatedPlugin->{replacedByCaption};
	my $errorMessage = "Already installed server plugin $componentName is deprecated and will be automatically uninstalled. ".
						"It is replaced by $replacedByCaption";
	if (defined $replacedByMinVersion) {
		$errorMessage = $errorMessage . " version $replacedByMinVersion";
	}
	$errorMessage = $errorMessage . ". Plugin $replacedByCaption must be selected for installation.";

	if (!defined $replacingComponent) {
		$self->appendErrorMessage($errorMessage);
		return 0;
	}
	my $replacingComponentName = $replacingComponent->getComponentName();
	if (!$replacingComponent->isComponentSelected()) {
		$self->appendErrorMessage($errorMessage);
		return 0;
	}
	return 1 if (!defined $replacedByMinVersion);
	my $version = $replacingComponent->getManifest()->getVersionObject();
	my $minVersion = new SDB::Install::Version(split('\.', $replacedByMinVersion));
	if ($minVersion->isNewerThan($version)) {
		my $versionString = $version->asString();
		$self->appendErrorMessage("Already installed server plugin $componentName is deprecated and will be automatically uninstalled. ".
									"It is replaced by $replacingComponentName version $replacedByMinVersion. ".
									"The selected version $versionString is lower than the required one.");
		return 0;
	}
	return 1;
}

sub validateComponent {
	my ($self, $component, $manifests, $selectedOnly) = @_;

	return $self->validateDeprecatedComponent($component) if $component->isDeprecated($self);
	return $self->SUPER::validateComponent($component, $manifests, $selectedOnly);
}

sub fillSelectedComponentsDefaultValue {
	my $self = shift();
	$self->SUPER::fillSelectedComponentsDefaultValue(@_);
	$self->addReplacementsOfDeprecatedPluginsDefaultValues();
}

sub addReplacementsOfDeprecatedPluginsDefaultValues {
	my $self = shift();
	my $defaultSelection = $self->getDefault('SelectedComponents');
	return if( $defaultSelection eq 'all' );
	return if ($defaultSelection !~ /server/);
    my $mcm = $self->getComponentManager();
	my $serverComponent = $mcm->getComponentByKeyName($gKeynameEngine);
	my $deprecatedPlugins = $serverComponent->getDeprecatedPlugins();
	for my $component (@{$self->getDeprecatedComponentsForUninstall()}) {
		my $deprecatedPlugin = $deprecatedPlugins->{$component->getComponentBatchKey()};
		my $replacedByKey = $deprecatedPlugin->{replacedByKey};
		next if (!defined $replacedByKey);
		my $replacingComponent = $mcm->getComponentByBatchKey($replacedByKey);
		next if (!defined $replacingComponent);
		my $batchKey = $replacingComponent->getComponentBatchKey();
		if ( (scalar $defaultSelection) ) {
   			$defaultSelection .= ",";
   		}
		$defaultSelection .= $batchKey;
	}
    $self->setDefault('SelectedComponents', $defaultSelection);
}

sub isConvertToMultiDbRequired {
    my $self = shift;
    my $isServerUpdate = $self->isComponentSelected($gKeynameEngine);
    my $isSingleDb = $self->isSingleTenantSystem();
    return ($isSingleDb && $isServerUpdate) ? 1 : 0;
}

sub hasStreaming{
    my($self) = @_;
    my $scm = $self->getSystemComponentManager();
    if ( defined($scm) && $scm->isComponentAvailable($gKeynameStreaming)) {
        return 1;
    }
    return 0;
}

sub isServerNoStart{
    my $self = shift;
    if (!$self->getComponentManager()->isHDBServerComponentSelected()){
        return 0;
    }
    my $passThroughPrefix = $self->getPassThroughParamPrefix($ini_section_server, 'hdbupd');
    my $passThroughParamIdNoStart = sprintf('%sNoStart', $passThroughPrefix);
    return $self->getValue($passThroughParamIdNoStart) // $self->getBatchValue($passThroughParamIdNoStart);
}

sub storeConvertToMDCState{
 my($self,$value) = @_;
 $self->{'pers_convert_to_mdc'} = $value;
}

sub setSystemDBSQLPasswordSetByUser{
    my($self,$value) = @_;
    $self->{'SystemDBSQLPasswordSetByUser'} = $value;
}

sub isSystemDBSQLPasswordSetByUser{
    my($self) = @_;
    return $self->{'SystemDBSQLPasswordSetByUser'} // 1;
}
sub isSystemInCompatibilityMode{
   my $self = shift;
   return 1 if($self->SUPER::isSystemInCompatibilityMode()) ;
   return $self->isConvertToMultiDbRequired();
}

sub isAutoInitializeServicesRequired {
     my ($self) = @_;
     return 1 if($self->isConvertToMultiDbRequired());
     my $isMDC = $self->getOwnInstance()->isMultiDb();
     my $isInCompatibilityMode = $self->isSystemInCompatibilityMode($self);
     return $isMDC && $isInCompatibilityMode;
}

sub _addIgnoreCheckBusyFiles {
    my ($self) = @_;
    my $ignoreCheckVersion = $self->getIgnore('check_version');
    my $ignoreCheckBusyFilesVersion = $self->getIgnore('check_busy_files');
    return if (!$ignoreCheckVersion || $ignoreCheckBusyFilesVersion);
    my $optionIgnore = $self->getOptionIgnore();
    $optionIgnore->parseArg('check_busy_files', undef, '--ignore');
}

sub isConvertToMultiDbResumed{
    my ($self) = @_;
    return $self->{_convertToMultiDBResumed};
}
sub setConvertToMultiDbResumed{
    my ($self,$value) = @_;
    if(!$value){
        $self->{_convertToMultiDBResumed} = 0;
    } else{
            $self->{_convertToMultiDBResumed} = 1;
    }
}

sub validatePersistency {
    my ($self, $persistedData) = @_;
    my $rc =$self->SUPER::validatePersistency($persistedData);
    if($rc && $persistedData->{'pers_convert_to_mdc'}){
        my $systemComponentManager = $self->getSystemComponentManager();
        my $server = $systemComponentManager->getComponentByKeyName($gKeynameEngine);
        $server->setConvertToMultiDBState(LCM::Component::Installed::HDBServer::CONVERT_MDC_STARTED);
        $self->setType('SystemDBSQLPassword','passwd');
        $self->setSkip('SystemDBSQLPassword',0);
        my $sid = $self->getValue('SID');
        $self->{params}->{'SystemUser'}->{str} =  "Initial Tenant '$sid' User Name";
        $self->{params}->{'SQLSysPasswd'}->{str} = "Initial Tenant '$sid' User Password";
        $self->{params}->{'SQLSysPasswd'}->{str_templ} = "Initial Tenant '$sid' User (%s) Password";
        $self->setConvertToMultiDbResumed(1);
    }
    return $rc;
}

sub checkSystemDBSQLPassword {
    my ($self, $value) = @_;
    return 1 if $value eq 'manager';
    return 1 if !$self->isSystemDBSQLPasswordSetByUser();

    my $msglst = SDB::Install::MsgLst->new();
    my $paramType = $self->getType('SystemDBSQLPassword');
    my $rc = $paramType eq 'initial_passwd' ? $self->complySqlPasswordPolicy($value, 1, $msglst, 'SystemDBSQLPassword')
                                            : $self->checkSQLSysPasswd($value, undef, 'SystemDBSQLPassword');
    if (!$rc) {
        $self->getErrMsgLst()->appendMsgLst($msglst);
    }
    return $rc;
}

sub checkSQLSysPasswd{
    my (
        $self,
        $value,
        $msglst_arg,
        $paramId,
        $outSqlConnection,
        $failIfNoConnection,
        $systemUser,
        $service
    ) = @_;
    my $isSQLSysPasswd = !defined $paramId || $paramId eq 'SQLSysPasswd';
    if($isSQLSysPasswd && $self->isConvertToMultiDbResumed() && !$self->_isServerUpdateInOnlinePhase()){
        return $self->SUPER::checkSQLSysPasswd( $value,$msglst_arg,$paramId,$outSqlConnection,$failIfNoConnection,$systemUser,"IndexServer");
    }
    return $self->SUPER::checkSQLSysPasswd( $value,$msglst_arg,$paramId,$outSqlConnection,$failIfNoConnection,$systemUser,$service);
 
}

sub _isServerUpdateInOnlinePhase{
    my $self = shift;
    my $isUpdateInOnlinePhase = 0;
    my $componentManager = $self->getComponentManager();
    if(defined $componentManager){
       my $server = $componentManager->getComponentByKeyName($gKeynameEngine);
       $isUpdateInOnlinePhase = defined $server && $server->getPhase() eq ONLINE_PHASE;
    }
    return $isUpdateInOnlinePhase;
}

sub checkSQLTenantUserPassword{
    my ($self, $value, $msglst_arg) = @_;
    if($self->isConvertToMultiDbRequired()){
        return 1; # no tenant exists at the moment
    }
    my $tenantDB = $self->getValue('XSTenantDatabaseName') // $self->getSID();
    return $self->SUPER::checkSQLTenantUserPassword($value, $msglst_arg, $tenantDB);
}

sub checkXSTenantDatabaseName {
    my ($self, $value) = @_;
    return 1 if !$value; # XSA goes into SYSTEMDB
    return $self->SUPER::checkXSTenantDatabaseName($value) if !$self->isConvertToMultiDbRequired();
# The database <SID> will be created during the convert to MDC
    my $sid = $self->getSID();
    if ($value ne $sid) {
        $self->setErrorMessage("Database name '$value' is invalid as the only existing database after the convert will be '$sid'.");
        return 0;
    }
    return 1;
}

sub handleVerifySignatureDefaultValue {
    my ($self) = @_;
    my $scm = $self->getSystemComponentManager();
    my $installer = $scm ? $scm->getComponentByKeyName($gKeynameInstaller) : undef;
    if (!defined($installer)) {
        $self->getErrMsgLst()->addError("Failed to detect installed '$gProductNameInstaller'.");
        return 0;
    }

    my $sapdsigner = LCM::SAPDSigner->getInstance();
    if (!defined($sapdsigner)) {
        $self->getMsgLst()->addMessage("No installed $gProductNameSHA was found on the host. Signature verification will be disabled by default.");
        return 1;
    }

    if ($sapdsigner->authenticateInstaller($installer)) {
        $self->setDefault('VerifySignature', 1, 1);
    } else {
        $self->getMsgLst()->addMessage("The installed system doesn't have an authentic signature. Authenticity verification of the software packages will be switched off. For further information see SAP note 2577617.");
        $self->setDefault('VerifySignature', 0, 1);
    }
    $sapdsigner->resetMsgLstContext();
    return 1;
}

sub getMandatoryServerSelectionMessage {
    my ($self) = @_;
    my $detectedServer = $self->getComponentManager()->getComponentByKeyName($gKeynameEngine);
    my $installedServer = $self->getSystemComponentManager()->getComponentByKeyName($gKeynameEngine);
    my $msgTemplate = "%s will be updated from version '%s' to version '%s'.";
    return sprintf($msgTemplate, $installedServer->getComponentName(), $installedServer->getVersion(), $detectedServer->getVersion());
}

sub initializeSelectedComponents {
    my ($self) = @_;

    return undef if !$self->SUPER::initializeSelectedComponents();

    my $consoleText;
    if ($self->isBootstrapFromResident()) {
        my $serverMsg = $self->getMandatoryServerSelectionMessage();
        $consoleText = "$serverMsg\n\nSelect additional components for update:\n";
    } else {
        $consoleText = "Choose components to be installed or updated:\n";
    }
    $self->{params}->{SelectedComponents}->{console_text} = $consoleText;
    return 1;
}

sub shallInitializeSystemUserDuringConvertToMDC{
    my $self = shift;
    return 0 if !$self->isConvertToMultiDbRequired();
    my $isReplicationSystem = LCM::Utils::CommonUtils::isSystemReplication($self);
    if(!$isReplicationSystem){
        my $sytemUserPassword = $self->getValue ('SystemDBSQLPassword');

        if (defined $sytemUserPassword && $sytemUserPassword ne 'manager'){
            return 1;
        }
    }
    return 0;
}

sub useNonDefaultTenantForXSA {
    my ($self, $tenantDBName) = @_;
    return 0 if $self->isUpdatingComponent($gKeynameXS2);

    my $installXSInDefaultTenant = $self->getValue('InstallXSInDefaultTenant');
    my $xsTenantDB = $tenantDBName // $self->getValue('XSTenantDatabaseName');
# Check for the value of 'InstallXSInDefaultTenant' is necessary because 'XSTenantDatabaseName' might also have value in the
# case of singletenant system, in which case XSA will go into the default tenant
    return $xsTenantDB && !$installXSInDefaultTenant;
}

# Override
sub shouldUseOptimizedExecution {
    my ($self) = @_;
    if ($self->SUPER::shouldUseOptimizedExecution()) {
        return 1;
    }
    return $self->isConvertToMultiDbRequired();
}

sub isResidentUpdate {
    return 0;
}

sub canMigrateToLSS {
    my ($self) = @_;

    my $instance = $self->getOwnInstance();
    my $isSecondary = $instance->isSecondarySystem($self->getMsgLst(), $self->getErrMsgLst());
    if (!defined $isSecondary) {
        return 0;
    }
    my $topologyFile = $instance->getTopologyFilename();
    if ($isSecondary && !File::stat::stat($topologyFile) && $self->isServerNoStart()) {
        my $err = "Cannot enable secure store LSS without starting the $gProductNameEngine on secondary systems when the '$topologyFile' is missing";
        $self->appendErrorMessage($err);
        return 0;
    }
    return 1;
}

1;

