package LCM::Configuration::GenericStackUpdateResidentConfiguration;

use strict;
use File::Spec;
use SDB::Install::SysVars;
use SDB::Install::Globals qw($gKeynameEngine $gProductNameStudio $gShortProductNameXS2 $gProductNamePlatform $gFlavourPlatform $gXSAAppsOnlyDetectedMessage);
use SDB::Install::System qw (isSidadmin);
use LCM::ComponentManager::ResidentMediumComponentManager;
use LCM::Installer;
use LCM::Configuration::GenericStackAny qw(UPDATE_COMPONENTS_ACTION $ini_section_xsa UPDATE_ACTION );
use LCM::Configuration::ValueChangeListeners::Update::ConsoleWriterSIDListener;
use LCM::PersistencyUtils qw (existsHdblcmPendingUpdate);
use LCM::Component qw($componentKeynameToBatchKey);

use base qw(LCM::Configuration::GenericStackUpdateConfiguration);

sub new{
	my $class = shift;
	my $self = $class->SUPER::new(@_);

	for my $paramId ('SID', 'CertificatesHostmap', 'HostName', 'UseSimpleUI'){
		if(exists($self->{params}->{$paramId})){
			$self->{params}->{$paramId}->{skip} = 1;
			$self->{params}->{$paramId}->{hidden} = 1;
		}
	}

	my $sid = new LCM::Installer()->getSid ();
	$self->_setSID($sid);
	$self->{params}->{SelectedComponents}->{valid_values} = [];
	$self->{params}->{ISCMode}->{section} = $ini_section_xsa;

	my $sapSys = $self->getCollectSAPSystems()->{$sid};
	my $instance;
	$instance = $sapSys->getNewDBInstances()->[0] if defined $sapSys;

	if (defined $instance && !$instance->exists_remote_host()) {
		$self->{params}->{Scope}->{skip} = 1;
	}
	if (isSidadmin($sid)) {
		$self->setUseSidadmUserForRemoteHostctrlHosts(1);
	}

	return $self;
}

sub InitDefaults {
	my ($self, $kit) = @_;

	my $sid = new LCM::Installer()->getSid ();
	return undef if (!$self->detectInstalledComponents($sid));
	return undef if (!$self->handleVerifySignatureDefaultValue());

	if (!existsHdblcmPendingUpdate($sid)) {
		$self->setSkip('ContinueUpdate', 1);
	}

	$self->SUPER::InitDefaults($kit);

	my $target = $self->getValue('Target') || $self->getDefault('Target');
	$self->setDefault('XsEaDataPath', File::Spec->catfile($target, $sid, 'xs', 'app_working'));

	return 1;
}

sub defineUpdateParams {
	my ($self) = @_;

    my $rc = $self->SUPER::defineUpdateParams();
    
    my $constraint = "Valid only for installation of SAP HANA components, which support host roles";
    if (exists $self->{params}->{InstallHostagent}) {
   		$self->{params}->{InstallHostagent}->{constraint} = $constraint;
    }
    $self->setInteractive('ComponentFsRoot', 1);
    $self->setSkip('AddComponentsLocation', 1);
    return $rc;
}

sub _handleInitializationOfPendingConfiguration {
    my ($self) = shift();
    return undef if(!$self->_shouldHandlePendingConfiguration());
    return $self->SUPER::_handleInitializationOfPendingConfiguration(@_);
}

sub _shouldHandlePendingConfiguration{
    my ($self) = @_;
    my $persistedAction = $self->getPersistedValue('pers_action');
    return 1 if($persistedAction eq $self->getAction());

    my $rc;
    if ($persistedAction eq UPDATE_ACTION){
        my $server = $componentKeynameToBatchKey->{$gKeynameEngine};
        my $isServerSelected = $self->getPersistedValue('SelectedComponents') =~ /$server/ ? 1 : 0;
        $rc = !$isServerSelected;
    }
    return $rc;
}

sub _setSID{
	my ($self, $sid) = @_;
# Do not use setValue -> see Bug: 58902
	$self->{params}->{SID}->{value} = $sid;
	$self->{current_sid} = $sid;
# Inherited SID listeners are not called, init here origin_values of AddRoles
	my $systemHosts = $self->getSystemHosts();
	if( defined $self->{params}->{AddRoles} ){
	   $self->{params}->{AddRoles}->{default_map} = undef;
	   $self->{params}->{AddRoles}->{origin_values} = $systemHosts;
	}
    $self->setDatabaseToSystemUserCredentialsParameter($self->{params}->{'SQLSysPasswd'});
    $self->setDatabaseToSystemUserCredentialsParameter($self->{params}->{'SystemUser'});
    $self->setDatabaseToTenantUserCredentialsParameter($self->{params}->{'TenantUser'});
    $self->setDatabaseToTenantUserCredentialsParameter($self->{params}->{'SQLTenantUserPassword'});
}

sub CheckParams{
	my ($self, $batchMode) = @_;
	my $sid = $self->getValue('SID');
	my $rc = $self->SUPER::CheckParams( $batchMode );
	return undef if (! $rc);
	my $componentManager = $self->getComponentManager();

	my $rcHandlePendingConfig = $self->_handleInitializationOfPendingConfiguration($sid);
	if (!$rcHandlePendingConfig) {
		$self->setSkip('ContinueUpdate', 1);
	}

	return 1;
}

sub _addNoSelectableComponentsErrorMsg {
	my ($self) = @_;
	my $message = "There are no $gProductNamePlatform components that could be installed or updated.";
	my $xsComponents = $self->getComponentManager()->getXs2ApplicationComponents();

	if(scalar(@{$xsComponents}) > 0){
		$message .= " $gXSAAppsOnlyDetectedMessage";
	} else {
		$message .= " Use the command line options 'component_medium', 'component_root' or 'component_dirs' to specify component locations";
	}

	$self->getErrMsgLst()->addError($message);
}

sub setContinueUpdate {
	my ($self, $value) = @_;
	return undef if (!$self->SUPER::setContinueUpdate($value));
	
	if ($self->getValue('ContinueUpdate') && !$self->getIgnore ('check_pending_upgrade')) {
		return 1;
	}

	my $componentKeynameToAction = $self->getComponentKeynameToAction();
	if (!$self->_hasSelectableComponents(keys(%{$componentKeynameToAction}))) {
		$self->_addNoSelectableComponentsErrorMsg();
		$self->{params}->{ContinueUpdate}->{no_retry} = 1;
		return undef;
	}

	return 1;
}

sub addListeners {
	my ($self) = @_;
	$self->SUPER::addListeners();
	$self->addParameterListener( 'DvdPath', LCM::Configuration::ValueChangeListeners::ResidentUpdateComponentDetectionListener->new('DvdPath') );
	$self->addParameterListener( 'ComponentDirs', LCM::Configuration::ValueChangeListeners::ResidentUpdateComponentDetectionListener->new('ComponentDirs') );
	$self->addParameterListener( 'ComponentFsRoot', LCM::Configuration::ValueChangeListeners::ResidentUpdateComponentDetectionListener->new('ComponentFsRoot') );
# Bug: 99526 - in this scenario setSID is never called (see bug 58902), therefore the listener is atached to parameter VerifySignature
	$self->addParameterListener( 'VerifySignature', LCM::Configuration::ValueChangeListeners::Update::ConsoleWriterSIDListener->new() );
}

sub getStaticDialogs {
    my ($self) = @_;
    my @dialogs          = ('LCM::Gui::Dialogs::SummaryDialog', 'LCM::Gui::Dialogs::InstallDialog');
    my $componentManager = $self->getComponentManager();
    my $curFlavour       = $componentManager->getHANAFlavour();
	if ( $curFlavour eq $gFlavourPlatform ) {
	   push @dialogs, 'LCM::Gui::Dialogs::FinalizeDialog';
	} else {
       push @dialogs, 'LCM::Gui::Dialogs::FinalizeCockpitInstallationDialog';
    }
    return \@dialogs;
}

sub getConfigureDialogs {
	my ($self, $wizard) = @_;
	my $sid = $self->getValue('SID');
	if(existsHdblcmPendingUpdate($sid) && $self->canContinueUpdate()){
		require LCM::Gui::Dialogs::PendingSummaryDialog;
		return [new LCM::Gui::Dialogs::PendingSummaryDialog($wizard)];
	}
	require LCM::Gui::Dialogs::SelectComponentSourceResidentDialog;
	require LCM::Gui::Dialogs::SelectComponentsForResidentUpdateDialog;
	return [
	   new LCM::Gui::Dialogs::SelectComponentSourceResidentDialog($wizard),
	   new LCM::Gui::Dialogs::SelectComponentsForResidentUpdateDialog($wizard),
	];
}

sub getComponentManager {
	my ($self) = @_;
	if (!defined($self->{componentManager})) {
        $self->{componentManager} = new LCM::ComponentManager::ResidentMediumComponentManager($self);
        $self->{componentManager}->setMsgLstContext( [ $self->getMsgLst() ] );
    }
	return $self->{componentManager};
}

sub handleSelectionOfServerComponent {
	my $self = shift;
	my $serverComponent = $self->{componentManager}->getComponentByKeyName($gKeynameEngine);	
	if (defined $serverComponent) {
		$serverComponent->setCanSelectComponent(0);		
	}
}

sub getRemoteInstallerExecutable{
    my ($self,$actionProgressive, $remoteTool) = @_;
    return $self->getRemoteExecutable($actionProgressive, $remoteTool);
}

sub getAction{
	return UPDATE_COMPONENTS_ACTION;
}

sub getIgnoreValues {
	return [
		'check_component_dependencies',
		'check_platform',
		'check_pending_upgrade',
		'check_resume_hostname',
		'check_version',
		'check_busy_files',
		'verify_signature',
		'check_diskspace'
	];
}

sub getProductName {
    my ($self) = @_;
    return "Install or Update Additional Components";
}

sub isResidentUpdate {
	return 1;
}

1;
