package LCM::Configuration::Hosts::AddHosts::AddLocalAndRemoteHostsConfiguration;

use strict;

use LCM::Configuration::Hosts::AddHosts::AddHostsBaseConfiguration;
use LCM::Configuration::Hosts::AddHosts::AddHostConfiguration;
use SDB::Install::RemoteHosts;
use SDB::Install::SAPInstance::TrexInstance;
use SDB::Install::Globals qw($gSapsysGroupName $gHostRoleWorker $gHostRoleStandby);
use SDB::Install::SysVars qw($path_separator $isWin);
use SDB::Install::Configuration qw( $bool_false_pattern $bool_true_pattern );
use LCM::HostsParser qw( ExtractLocalAndRemoteHosts ParseHostStringToMap );
use LCM::HostsUtility qw( GetInstanceDirOnHostNotPartOfSystem IsHostPartOfSystem );
use LCM::Utils::CommonUtils qw ( getSidadmName getSystemSidadmUid getSystemSidadmGid getLocalHostSidadmUid getLocalHostSidadmGid);
use SDB::Install::Saphostagent qw (getSaphostexecPath);
use SAPDB::Install::Hostname;
use LCM::Configuration::NewDBUser::User;
use File::Basename qw(dirname);
use LCM::App::ApplicationContext;

use base qw (LCM::Configuration::Hosts::AddHosts::AddHostsBaseConfiguration);

sub new{
    my $self  = shift->SUPER::new (@_);

	$self->_handleParameters();
	$self->{localHostConfiguration} = new LCM::Configuration::Hosts::AddHosts::AddHostConfiguration();
	return $self;
}

sub InitDefaults {
	my $self = shift();
	my $result = $self->SUPER::InitDefaults(@_);

	if($result){
		my $localHostConfiguration = $self->{localHostConfiguration};
		$localHostConfiguration->{sapSys} = $self->getSAPSystem();
	}

	return $result;
}

sub _handleParameters {
	my ($self) = @_;

	my $params = $self->{params};

    $params->{RootPassword}->{skip} = 0;
	$params->{Password}->{skip} = 0;
	$params->{NoStart}->{skip} = 0;
	$params->{ListenInterface}->{skip} = 1;
	$params->{ListenInterface}->{hidden} = 1;
	$params->{InternalNetwork}->{skip} = 1;
	$params->{InternalNetwork}->{hidden} = 1;
}

sub checkAddHosts {
	my ($self, $value) = @_;

	my $rc = $self->_checkExternalHosts( $value );

	if ( $rc ) {
		$rc =  $self->_checkLocalHost( $value );
	}
	if ( $rc ) {
		$rc =  $self->_checkLocalAndRemoteRoles( $value );
	}
	if ( $rc ) {
		$rc =  $self->_checkLocalSidadmUid();
	}
	if ( $rc ) {
		$rc =  $self->_checkLocalSidadmGid();
	}
	if ( $rc ){
		$rc = $self->_checkLocalSapsysGid();
	}
	return $rc;
}

sub _prepareCertificatesHostmap {
	my $self = shift();
	$self->fillCertificatesHostmapDataForLocalHost();
	$self->SUPER::_prepareCertificatesHostmap(@_);
}

sub _checkExternalHosts{
	my ($self, $value) = @_;

	my ($localHostInputString, $remoteHostsInputString) = ExtractLocalAndRemoteHosts($value, $self);
	if ( ! defined $remoteHostsInputString ) {
		return 1;
	}

	my $notHandleListenInterfaceParam = 1;
	return $self->SUPER::checkAddHosts( $remoteHostsInputString, undef, $notHandleListenInterfaceParam );
}

sub _checkLocalHost {
	my ( $self, $value ) = @_;

	if (!$value){
		$self->PushError ("Host name is emtpy");
		return 0;
	}

    my $flavourProductName = LCM::App::ApplicationContext::getFlavourProductName();
	my ($localHostInputString, $remoteHostsInputString) = ExtractLocalAndRemoteHosts($value, $self);
	my $localHostConfiguration = $self->{localHostConfiguration};

	if (!$localHostInputString){
		$self->PushError ("Local host is not specified. Since the current host is not part of the $flavourProductName System, it must be defined in the list of additional hosts.");
		return 0;
	}

	if (!defined $remoteHostsInputString){
		$self->{params}->{RootUser}->{skip} = 1;
		$self->{params}->{RootPassword}->{skip} = 1;
	}

	# For interactive AddHosts CLI input (i.e. no batch value provided) we first enter hostnames only
	# and a preliminary checkAddHosts is called with that hostname list. There we shouldn't assume that
	# the default worker role is intended because it can cause problem with subsequent checks.
	my $ignoreDefaultRoleValue = (defined $self->getBatchValue('AddHosts')) ? 0 : 1;
	my $hostsParser = new LCM::HostsParser(ignoreDefaultRoleValue => $ignoreDefaultRoleValue);
	my ($rc, $parserErrMsgLst) = $hostsParser->parse($localHostInputString);
	if (!defined $rc) {
		$self->PushError( "Error parsing string: $localHostInputString\n", $parserErrMsgLst );
		return 0;
	}
	$self->setLocalHostParser($hostsParser);

	return 0 if (!$self->_setLocalHostConfigurationParams());
	$self->handleHostagentPassword();

	return 1;
}

sub _setLocalHostConfigurationParams {
	my ($self) = @_;

	my $localHostConfiguration = $self->{localHostConfiguration};
	my $hostsParser = $self->getLocalHostParser();
	my $hosts = $hostsParser->getHosts();
	my $localHost = $hosts->[0];

	my $rc = $localHostConfiguration->setValue( "HostName", $localHost );

	my $localHostStoragePartition = $hostsParser->getStoragePartition($localHost);
	$rc &&= $localHostConfiguration->setValue( "StoragePartitionNumber", $localHostStoragePartition ) if defined $localHostStoragePartition;

	my $localHostRolesArr = $hostsParser->getRoles($localHost) // [];
	my $localRoleList = join(',', @{$localHostRolesArr});
	$rc &&= $localHostConfiguration->setValue( "HostRoles", $localRoleList ) if defined $localRoleList;

	my $localHostGroup = $hostsParser->getGroup($localHost);
	$rc &&= $localHostConfiguration->setValue( "HostFailoverGroup", $localHostGroup ) if defined $localHostGroup;

	my $localWorkerGroup = $hostsParser->getWorkerGroups ($localHost);
	$rc &&= $localHostConfiguration->setValue( "WorkerGroup", $localWorkerGroup ) if defined $localWorkerGroup;

	if (!$rc){
		my $errorStr = $localHostConfiguration->GetErrorString ();
		$self->PushError ($errorStr);
		return 0;
	}
	return 1;
}

sub _checkLocalAndRemoteRoles {
	my ( $self, $value ) = @_;

	my ($localHostInputString, $remoteHostsInputString) = ExtractLocalAndRemoteHosts($value, $self);
	if ( ! defined $localHostInputString || ! defined $remoteHostsInputString ) {
		return 1;
	}

	my $localHostParser = new LCM::HostsParser();
	my ($rc, $parserErrMsgLst) = $localHostParser->parse($localHostInputString);
	if ( ! defined $rc ) {
		$self->PushError( "Error parsing string: $localHostInputString\n", $parserErrMsgLst );
		return 0;
	}

	my $localHost = $localHostParser->getHosts()->[0];
	my $localHostRoles = $localHostParser->getRoles($localHost);
	my @allRoles = ();
	push @allRoles , @$localHostRoles;

	my $remoteHostsParser = new SDB::Install::Configuration::AddHostsParser ();
    if ( ! defined $remoteHostsParser->parse( $remoteHostsInputString ) ) {
        $self->PushError (undef, $remoteHostsParser);
    }

    my @remoteRoles = ();
    for my $host ( @{$remoteHostsParser->getHosts()} ) {
		my $roles = $remoteHostsParser->getRoles($host);
		push @remoteRoles, @$roles;
    }

	push @allRoles, @remoteRoles;

	return $self->checkNewRolesCompatibility(\@allRoles);
}

sub _checkLocalSidadmUid {
	my ($self, ) = @_;

    my $flavourProductName = LCM::App::ApplicationContext::getFlavourProductName();
	my $systemHostSidadmUid = getSystemSidadmUid($self);
	my $localHostSidadmUid = getLocalHostSidadmUid($self);
	my $sidadmName = getSidadmName($self);
	if ( defined $localHostSidadmUid && $localHostSidadmUid ne $systemHostSidadmUid ) {
		$self->setErrorMessage("The user id '$localHostSidadmUid' of '$sidadmName' on local host does not match the user id " .
								"'$systemHostSidadmUid' of the $flavourProductName system");
		return undef;
	}

	my $localUsernameWithSameUid = getpwuid($systemHostSidadmUid);
	if ( defined $localUsernameWithSameUid && $localUsernameWithSameUid ne $sidadmName ) {
		$self->setErrorMessage("User id '$systemHostSidadmUid' is already in use by user '$localUsernameWithSameUid'");
		return undef;
	}
	return 1;
}

sub _checkLocalSidadmGid {
	my ($self, ) = @_;

    my $flavourProductName = LCM::App::ApplicationContext::getFlavourProductName();
	my $systemHostSidadmGid = getSystemSidadmGid($self);
	my $localHostSidadmGid = getLocalHostSidadmGid($self);
	my $sidadmName = getSidadmName($self);
	if ( defined $localHostSidadmGid && $localHostSidadmGid ne $systemHostSidadmGid ) {
		$self->setErrorMessage("The group id '$localHostSidadmGid' of '$sidadmName' on local host does not match the '$sidadmName' group id " .
								"'$systemHostSidadmGid' of the $flavourProductName system");
		return undef;
	}
	return 1;
}

sub _checkLocalSapsysGid {
	my ($self, ) = @_;
	my $localGID = getgrnam($gSapsysGroupName);
	my $expectedGID = getSystemSidadmGid($self);

	if (defined($localGID) && $localGID != $expectedGID) {
		$self->setErrorMessage("Existing group '$gSapsysGroupName' has wrong GID on the local host (expected '$expectedGID', found '$localGID')");
		return undef;
	}
	return 1;
}

sub checkSID {
	my ($self, $value) = @_;

	if (IsHostPartOfSystem()) {
		return $self->_checkExternalSID( $value );
	}

	return $self->_checkLocalSID( $value );
}

sub _checkExternalSID {
	my ($self, $value) = @_;

	return $self->SUPER::checkSID ($value);
}

sub _checkLocalSID {
	my ($self, $value) = @_;
	my $rc;
	my $localHostConfiguration = $self->{localHostConfiguration};
	$rc = $localHostConfiguration->setValue( "SID", $value );

	if (!$rc){
		my $errorStr = $localHostConfiguration->GetErrorString ();
		$self->PushError ($errorStr);
		return 0;
	}

	return 1;
}

sub checkTarget{
	my ($self, $value) = @_;

	if (IsHostPartOfSystem()) {
		return $self->_checkExternalTarget( $value );
	}
	return $self->_checkLocalTarget( $value );
}

sub _checkLocalTarget{
	my ($self, $value) = @_;
	my $localHostConfiguration = $self->{localHostConfiguration};
	my $sid = $self->getDefault("SID");
	my $rc = $localHostConfiguration->setValue( "SID", $sid );
	$rc &= $localHostConfiguration->setValue( "Target", $value );

	if (!$rc){
		my $errorStr = $localHostConfiguration->GetErrorString ();
		$self->PushError ($errorStr);
		return 0;
	}
	return 1;
}

sub _checkExternalTarget {
	my ($self, $value) = @_;
	return $self->SUPER::checkTarget( $value );
}

sub checkPassword{
	my ($self, $value) = @_;

	if (!IsHostPartOfSystem()){
        return 0 if(!$self->_checkLocalPassword( $value ));
        $self->_prepareCertificatesHostmap($self->getValue('AddHosts'));
        my $sidadmUser = LCM::Configuration::NewDBUser::User->new(getSidadmName($self));
        $sidadmUser->setVerifyOnAdditionalHosts(0);
        return $sidadmUser->verifyPasswordSHA($value, $self);
	}
	return 1;
}

sub _checkLocalPassword{
	my ($self, $value) = @_;
	my $localHostConfiguration = $self->{localHostConfiguration};
	my $rc = $localHostConfiguration->setValue ("Password", $value);

	if (!$rc){
		my $errorStr = $localHostConfiguration->GetErrorString();
		$self->PushError( $errorStr );
		return 0;
	}
	return 1;
}

sub setNoStart{
	my ($self, $value) = @_;

	if ($value =~ /$bool_false_pattern/i){
    	$value = 0;
    } elsif ($value =~ /$bool_true_pattern/i){
    	$value = 1;
   	} else {
   		$value = 0;
   	}

    $self->{params}->{NoStart}->{value} = $value;

	my $localHostConfig = $self->{localHostConfiguration};
	$localHostConfig->{params}->{NoStart}->{value} = $value;

	return 1;
}

sub checkInstallHostagent {
	my ($self, $value) = @_;
	if ($value =~ /$bool_false_pattern/i) {
		my $isSkipped = $self->isSkipped('HostagentPassword');
		my $isUseSHA = $self->isUseSAPHostagent();
		$self->setSkip('HostagentPassword', (!$isSkipped && $isUseSHA ? 0 : 1));
		$self->setSkip('SkipHostagentPw', (!$isSkipped && $isUseSHA ? 0 : 1));
		return 1;
	} elsif ($value =~ /$bool_true_pattern/i) {
		my $isSkipped = $self->isSkipped('HostagentPassword');
		my $saphostexecPath = getSaphostexecPath();
		$self->setSkip('HostagentPassword', ($isSkipped && defined($saphostexecPath) ? 1 : 0));
		return 1;
	}
	return undef;
}

sub setInstallHostagent{
	my ($self, $value) = @_;

	my $rc = $self->checkInstallHostagent($value);

	return $rc unless($rc);

	my $installHostagentParam = $self->{params}->{InstallHostagent};
	$installHostagentParam->{value} = $value;

	my $localConfigInstallHostagentParam = $self->{localHostConfiguration}->{params}->{InstallHostagent};
	$localConfigInstallHostagentParam->{value} = $value;
	$localConfigInstallHostagentParam->{skip} = $installHostagentParam->{skip};

	return 1;
}

sub setHostagentPassword {
	my ($self, $value) = @_;
	my $rc = $self->SUPER::setHostagentPassword ($value);

	return 0 if (!$rc);

	my $hostagentPasswordParam = $self->{params}->{HostagentPassword};
	$hostagentPasswordParam->{value} = $value;

	my $localConfigHostagentPasswordParam = $self->{localHostConfiguration}->{params}->{HostagentPassword};
	my $skipLocalHostagentPassword = $self->isHostagentInstalled();
	$localConfigHostagentPasswordParam->{skip} = $skipLocalHostagentPassword;

	if (not $skipLocalHostagentPassword) {
		$localConfigHostagentPasswordParam->{value} = $value;
	}

	return 1;
}

sub setHostagentUserName{
	my ($self, $value) = @_;
	my $rc = $self->SUPER::checkHostagentUserName ($value);

	if (!$rc){
		return 0;
	}

	$self->{params}->{HostagentUserName}->{value} = $value;
	my $localHostConfig = $self->{localHostConfiguration};
	$localHostConfig->{params}->{HostagentUserName}->{value} = $value;

	return 1;
}

sub getStaticDialogs {
    return ["LCM::Gui::Dialogs::SummaryDialog", "LCM::Gui::Dialogs::ConfigurationExecutionDialog", "LCM::Gui::Dialogs::FinalizeDialog"];
}

sub getConfigureDialogs {
	my ($self, $wizard) = @_;

	require LCM::Gui::Dialogs::AddHosts::LocalHostPropertiesDialog;
	require LCM::Gui::Dialogs::AddHosts::AdditionalHostsPropertiesDialog;
	require LCM::Gui::Dialogs::SystemUserPropertiesDialog;
	require LCM::Gui::Dialogs::AddHosts::AcceleratorPropertiesDialog;
	require LCM::Gui::Dialogs::OrgManagerCredentialsDialog;
	require LCM::Gui::Dialogs::AddHosts::CertificatesDialog;
	require LCM::Gui::Dialogs::HostagentPasswordDialog;

	return [
        new LCM::Gui::Dialogs::AddHosts::LocalHostPropertiesDialog ($wizard),
		new LCM::Gui::Dialogs::AddHosts::AdditionalHostsPropertiesDialog ($wizard),
		new LCM::Gui::Dialogs::SystemUserPropertiesDialog($wizard),
		new LCM::Gui::Dialogs::AddHosts::AcceleratorPropertiesDialog ($wizard),
		new LCM::Gui::Dialogs::OrgManagerCredentialsDialog($wizard),
		new LCM::Gui::Dialogs::AddHosts::CertificatesDialog ($wizard),
		new LCM::Gui::Dialogs::HostagentPasswordDialog ($wizard),
	];
}

sub getProductName {
    return LCM::Configuration::Hosts::AddHosts::AddHostsBaseConfiguration::getProductName();
}

sub handleHostagentPassword {
	my ($self,$hanldeRemoteHosts) = @_;

	$self->SUPER::handleHostagentPassword($hanldeRemoteHosts);
	my $isHostagentPasswordSkipped = $self->isSkipped('HostagentPassword') && $self->isSkipped('InitialHostagentPassword');

	if (not $isHostagentPasswordSkipped) {
		# Executor provides option '--skip_hostagent_password=0' to hdbmodify
		$self->setValue('SkipHostagentPw', 0);
		return;
	}

	my $installHostagentValue = (defined $self->getBatchValue("InstallHostagent")) ?
								$self->getBatchValue("InstallHostagent") : $self->getDefault("InstallHostagent");

	if (($installHostagentValue) and (not $self->isHostagentInstalled())) {
		# Executor provides option '--skip_hostagent_password=0' to hdbmodify
		$self->setValue('SkipHostagentPw', 0);
		$self->{params}->{HostagentPassword}->{skip} = 0;
		return;
	}

	if ($installHostagentValue) {
		# Executor provides option '--skip_hostagent_password=1' to hdbmodify
		$self->setValue('SkipHostagentPw', 1);
	}

	$self->{params}->{HostagentPassword}->{skip} = 1;
}

# clears _remoteHosts hash which is populated by previous call of checkAddHosts()
sub clearRemoteHosts{
	my ($self) = @_;

	my $newHosts = $self->{params}->{AddHosts}->{value};
	my ($localHostInputString, $remoteHostsInputString) = ExtractLocalAndRemoteHosts($newHosts, $self) if defined $newHosts;
	my $newHostsMap = ParseHostStringToMap( $remoteHostsInputString ) if defined $remoteHostsInputString;
	my $currentRemoteHosts = $self->{_remoteHosts}->{_hosts};

	if (not defined $newHostsMap || not defined $currentRemoteHosts){
		return;
	}

	for my $newHost (keys %$newHostsMap){
		if (grep { $_ eq $newHost } (@$currentRemoteHosts)){
			$self->_deleteElementFromArray($currentRemoteHosts, $newHost);
		}
	}
}

sub _deleteElementFromArray{
	my ($self, $currentHosts, $elementForDeleting) = @_;

	my $index=0;
	$index++ until @$currentHosts[$index] eq $elementForDeleting;
	splice(@$currentHosts, $index, 1);
}

sub establishRemoteHostConnections{
	my ($self) = @_;

	my $rc = $self->SUPER::establishRemoteHostConnections();
	if ($rc){
		$rc = $self->_handleRemoteHosts();
	}

	return $rc;
}

sub _handleRemoteHosts{
	my ($self) = @_;

	my $target = $self->getDefault("Target");
	my $sid = $self->getDefault("SID");
	my $sapSidDir = $target . $path_separator . $sid;
	my $instanceDir = LCM::HostsUtility::GetInstanceDirOnHostNotPartOfSystem( $self, $sapSidDir );

	if (!defined $instanceDir){
		return undef;
	}

	my $instance = new SDB::Install::SAPInstance::TrexInstance( $instanceDir, $sid );
	my $instanceRemoteHostnames = $instance->get_hosts();

	eval {
		require SDB::Install::RemoteHosts;
    };

	if ($@){
		$self->setErrorMessage ("Cannot get SSH connections: $@");
		return undef;
	}

	my $instanceRemoteHosts = new SDB::Install::RemoteHosts(@$instanceRemoteHostnames);
    $instanceRemoteHosts->SetProgressHandler($self->GetProgressHandler());

    if ($instanceRemoteHosts->connect () != 0){
		$self->AddError (undef, $instanceRemoteHosts);
		return 0;
    }

	if (defined $self->{_remoteHosts}){
		$self->_mergeRemoteHosts( $instanceRemoteHosts );
	} else {
		$self->{_remoteHosts} = $instanceRemoteHosts
	}
}

sub _mergeRemoteHosts{
	my ($self, $instanceRemoteHosts) = @_;

	my $remoteHosts = $self->{_remoteHosts};

	foreach my $i (0 .. (scalar @{$instanceRemoteHosts->{_hosts}} -1)) {
		my $instanceRemoteHostname = $instanceRemoteHosts->{_hosts}->[$i];
		my $instanceRemoteConnection = $instanceRemoteHosts->{_connections}->[$i];

		if( grep { $_ eq $instanceRemoteHostname} @{$remoteHosts->{_hosts}} ) {
			next;
		}

		push @{$remoteHosts->{_hosts}},       $instanceRemoteHostname;
		push @{$remoteHosts->{_connections}}, $instanceRemoteConnection;
	}
}

sub getNumberOfExpectedOutputLines {
    return 50;
}

sub getLocalHanaHost {
	my ($self) = @_;
	my $hostname;

	if (defined $self->{localHostConfiguration}) {
		# Validated local hostname is returned
		$hostname = $self->{localHostConfiguration}->getValue('HostName');
	}

	if (not defined $hostname) {
		$hostname = lc(hostname());
	}

	return $hostname;
}

# Setter here is needed to copy over the value of
# 'ImportContentXS2' to the local configuration
sub setImportContentXS2 {
	my ($self, $value) = @_;
	$self->{params}->{ImportContentXS2}->{value} = $value;
	$self->{localHostConfiguration}->{params}->{ImportContentXS2}->{value} = $value;
	return 1;
}

sub checkSystemRequirements {
	my ($self, $sysinfo) = @_;

	require SDB::Install::Configuration::NewDB;
	my $handler = new SDB::Install::Configuration::NewDB();
	$handler->{isWin} = $self->{isWin};
	$handler->{optionIgnore} = $self->getOptionIgnore();
	my $oldcontext = $handler->setMsgLstContext($self->getMsgLstContext());
	my $result = $handler->checkSystemRequirements($sysinfo);
	$handler->setMsgLstContext($oldcontext);

    if(defined $handler->{'warnings'}) {
        if(!defined $self->{'warnings'}) {
            $self->{'warnings'} = [];
        }
        push @{$self->{'warnings'}}, @{$handler->{'warnings'}};
    }
	if (defined $result) {
		return $result;
	}

	if (defined $handler->errorState()) {
		$self->appendErrorMessage("Checking system requirements failed");
		my $errlst = $handler->getErrMsgLst();
		$self->appendErrorMessage (undef, $errlst);
	}

	return undef;
}

sub setInteractiveAddHostInput {
    my($self, $interactiveInput) = @_;
    $self->SUPER::setInteractiveAddHostInput($interactiveInput);
    $self->{localHostConfiguration}->setInteractiveAddHostInput($interactiveInput);
}

sub getLssPath {
    my $self = shift;
    my $lssInstance = $self->getLssInstance();
    return undef if (!defined $lssInstance);
    #Handles the case when the host is not part of the sysem and the lss is installed on the master host but not mounted on the local as the local will not have a DB Role
    if(!$lssInstance->isInstalled()){
        return SDB::Install::System::readLink($lssInstance->getLssLink());
    }
    return $lssInstance->getLssSidDir();
}

sub checkPathExistenceOnHost {
    my ($self, $pathExistenceMap, $path, $host) = @_;
    if( LCM::HostsParser::IsLocalHost($host) && $self->usesNonSharedStorage() ) {
        $self->{localHostConfiguration}->setUsesNonSharedStorage(1);
    }
    return $self->SUPER::checkPathExistenceOnHost($pathExistenceMap, $path, $host);
}

sub checkDataAndLogPathExistenceOnHost {
    my ($self, $host) = @_;
    return 0 if(!$self->SUPER::checkDataAndLogPathExistenceOnHost($host));
    if(LCM::HostsParser::IsLocalHost($host)){
        my $config = $self->{localHostConfiguration};
        if(!$config->initVolumePaths($config->getOwnInstance(), $config->getSAPSystem())){
            my $error = $config->getErrorString();
            $error =~ s/\n/\./g;
            $self->setErrorMessage($error);
            return 0;
        }
    }
    return 1;
}

sub checkRole {
    my ($self, $role, $host) = @_;
    return 0 if !$self->SUPER::checkRole($role, $host);
    my $rc = 1;
    if(LCM::HostsParser::IsLocalHost($host) && $role =~ /^$gHostRoleWorker|^$gHostRoleStandby/){
        my $localHostConfiguration = $self->{localHostConfiguration};
        my $saveContext = $localHostConfiguration->setMsgLstContext($self->getMsgLstContext());
        $rc = $localHostConfiguration->checkSharedVolumePaths();
        $localHostConfiguration->setMsgLstContext($saveContext) if defined $saveContext;
    }
    return $rc;
}

1;
