#!perl 
#===============================================================================
#
# Copyright (C) 2000 - 2013 by SAP AG,  http://www.sap.com.
# All rights reserved.
#
# This software is the confidential and proprietary information of SAP AG.
# You shall not disclose such confidential information and shall use it only
# in accordance with the terms of the license agreement you entered into with SAP.
#
#       AUTHOR: i072928
#

=head1 NAME

RemoveHostsConfiguration

=head1 DESCRIPTION

Configuration class for the 'remove hosts' scenario.
=cut

#===============================================================================
package LCM::Configuration::Hosts::RemoveHosts::RemoveHostsConfiguration;

use strict;
use base qw ( SDB::Install::Configuration::AnyModifyConfig LCM::Configuration::AutoInitializeFamilyServicesConfiguration);

use SDB::Install::Configuration qw($bool_false_pattern);
use SDB::Install::Globals qw( GetHostRoleProperties $gProductNameEngine $gProductNameSystem $gXSParametersRemoveHostOrRolesConstraint $gLSSParametersRemoveHostOrRolesConstraint);

use SDB::Install::Globals qw ($gHostRoleAcceleratorStandby
                              $gHostRoleAcceleratorWorker
                              $gHostRoleWorker
                              $gProductNameSystem);
use SDB::Install::SysVars qw($path_separator $isWin);
use LCM::ResidentInstaller qw($installedPattern);
use SDB::Install::Configuration::AnyConfig;
use LCM::App::ApplicationContext;
use LCM::Hosts::HostsDetector;
use Exporter;
use LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::SkippedComponentsManager qw ( $SCENARIO_ADD_REMOVE_HOSTS );
use LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::LocalComponentsManager;
use LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::RemoteComponentsManager;
use SDB::Install::System qw (isSidadmin);
use LCM::Configuration::ParametersCreator;
use LCM::Task qw($gFlavourProductName);
use LCM::Hosts::RemoveHostCheck;
use LCM::HostsParser qw(ExtractLocalAndRemoteHosts);
use SDB::Install::Configuration::TenantUserListener;
use LCM::Configuration::ValueChangeListeners::RemoveHosts::LSSParameterListener;
use SDB::Install::Configuration::ValueChangeListeners::KeepLSSUserGroupSkipListener;

our $section = 'RemoveHosts';
my $TITLE = "Remove Hosts from $gFlavourProductName System";
our $SYSTEM_REMOVE_ACTION = 'remove_hosts';
our @EXPORT = qw ($SYSTEM_REMOVE_ACTION);


sub new {
	my ( $self, $order ) = shift->SUPER::new ( @_ );

    my $tenantCredentialsConstraint = $self->getTenantCredentialsConstraint();
	$self->{params} = {
		(!$isWin ? ('RemoteExecution' => $self->getParamRemoteExecution($order++, $section, 0),
		            'UseHttp'         => $self->getParamUseHttp        ($order++, $section),
		           ) : ()),
        'Target'             => $self->getParamTarget            ($order++, $section),
        'SID'                => $self->getParamSID               ($order++, $section),
        'AutoInitializeServices'     => $self->getParamAutoInitializeServices($order++, $section,undef,$tenantCredentialsConstraint),
        'RemoveHosts'        => LCM::Configuration::ParametersCreator::GetParamRemoveHosts($order++, $section),
        'KeepLSSUser'        => $self->getParamKeepLSSUser ($order++, $section, $gLSSParametersRemoveHostOrRolesConstraint),
        'KeepLSSUserGroup'    => $self->getParamKeepLSSUserGroup  ($order++, $section, $gLSSParametersRemoveHostOrRolesConstraint),
        'InstallSSHKey'      => $self->getParamInstallSSHKey     ($order++, $section),
        'RootUser'           => $self->SDB::Install::Configuration::AnyConfig::getParamRootUser($order++, $section),
        'RootPassword'       => $self->getParamRootPassword      ($order++, $section),
        'HostagentPassword'  => $self->getParamHostagentPassword ($order++, $section),
        'KeepUser'           => $self->getParamKeepUser          ($order++, $section),
        'KeepUserHomeDir'    => $self->getParamKeepUserHomeDir   ($order++, $section),
        'KeepXsUsers'        => $self->getParamKeepXsUsers        ($order++, $section, $gXSParametersRemoveHostOrRolesConstraint),
        'SkipModifySudoers'  => $self->getParamSkipModifySudoers  ($order++, $section),
        'SystemUser'         => $self->getParamSystemUser         ($order++, $section, 1),
        'SQLSysPasswd'       => $self->getParamSQLSysPasswd       ($order++, $section, 'passwd', 1),
        'AcceleratorUser'    => $self->getParamAcceleratorUser    ($order++, $section),
        'AcceleratorPassword'=> $self->getParamAcceleratorPassword($order++, $section, undef, 'passwd'),
        'TenantUser'     => $self->getParamTenantUser    ($order++, $section,1,$tenantCredentialsConstraint),
        'SQLTenantUserPassword' => $self->getParamSQLTenantUserPassword($order++, $section,1,$tenantCredentialsConstraint),
    };
    
    $self->{params}->{HostagentPassword}->{type} = 'passwd';
    $self->{params}->{HostagentPassword}->{opt_arg} = '<password>';
    $self->{params}->{HostagentPassword}->{log_value} = sub {'***';};
    
	$self->_reconfigureParameters();
    $self->addListeners();
    $self->{isOnlyLocalHostMarked} = 0;
	return $self;
}

sub _reconfigureParameters {
    my ($self,) = @_;
    	
	$self->{params}->{RemoveHosts}->{set_interactive}= 1;
	$self->{params}->{RemoveHosts}->{mandatory}      = 1;
	$self->{params}->{RemoveHosts}->{skip}           = 0;
	$self->{params}->{RemoveHosts}->{hidden}         = 0;
	$self->{params}->{HostagentPassword}->{skip}     = 1;
	$self->{params}->{RootUser}->{skip}              = 1;
    $self->{params}->{RootUser}->{init_with_default} = 1;
    $self->{params}->{RootUser}->{set_interactive}   = 1;
    $self->{params}->{RootPassword}->{skip}   = 1;
    $self->{params}->{RootPassword}->{mandatory}   = 0;
    $self->{params}->{KeepUser}->{set_interactive}   = 1;
    $self->{params}->{KeepUser}->{skip}              = 0;
    $self->{params}->{KeepUser}->{console_omit_word_Enter} = 1;
    $self->{params}->{KeepUserHomeDir}->{set_interactive}   = 1;
    $self->{params}->{KeepUserHomeDir}->{skip}       = 0;
    $self->{params}->{KeepUserHomeDir}->{console_omit_word_Enter} = 1;

    # Target and SID are set by SBD::Install::Installer::getInstallation
    $self->{params}->{Target}->{hidden} = 1;
    $self->{params}->{SID}->{hidden} = 1;
}

sub addListeners {
	my ( $self, ) = @_;
	
	require LCM::Configuration::ValueChangeListeners::RemoveHostRemoveHostsListener;
    $self->addParameterListener( 'RemoveHosts', LCM::Configuration::ValueChangeListeners::RemoveHostRemoveHostsListener->new());
    $self->addParameterListener( 'RemoveHosts',$self->getAutoInitFamilyServicesParameterHanlder('RemoveHosts'));
    $self->addParameterListener( 'TenantUser',SDB::Install::Configuration::TenantUserListener->new());
    $self->addParameterListener( 'RemoveHosts', LCM::Configuration::ValueChangeListeners::RemoveHosts::LSSParameterListener->new());
    $self->addParameterListener('KeepLSSUser', SDB::Install::Configuration::ValueChangeListeners::KeepLSSUserGroupSkipListener->new());
}

sub setKeepUser {
    my ( $self, $keepUser ) = @_;
    
    if ( ! defined $keepUser ) {
    	$keepUser = 0;
    }
    
    $self->{params}->{KeepUser}->{value} = $keepUser;
    if ( $keepUser ) {
    	$self->{params}->{KeepUserHomeDir}->{value} = 1;
    	$self->{params}->{KeepUserHomeDir}->{set_interactive} = 0;
    }
    
    return 1;
}

sub setRootUser {
	my ($self, $rootUser) = @_;
    
    my $sub = $self->can("checkRootUser");
    my $rc = 1;
    if ( $sub ) {
    	$rc = &$sub($self, $rootUser);
    }
    
    if ( $rc ) {
    	$self->{params}->{RootUser}->{value} = $rootUser;
    }
    
    return $rc;
}

sub checkRootUser {
    my ($self, $rootUser) = @_;
    my $remoteHosts = $self->{_remoteHosts};
    
    if(defined $remoteHosts){
        $remoteHosts->destroy();
        $remoteHosts->connect();
    }
    
    my $returnCode = $self->SUPER::checkRootUser($rootUser);
    $remoteHosts = $self->{_remoteHosts};
    
    if($returnCode && defined($remoteHosts) && !$remoteHosts->isHostctrl()){
        my $areHostsAuthenticated = $remoteHosts->areHostsAuthenticated();
        $self->setSkip('RootPassword', $areHostsAuthenticated); 	   # if authentication is OK, there is no need of password
        $self->setMandatory('RootPassword', !$areHostsAuthenticated); # if authentication is not OK, the password becomes mandatory
    }
    
    return $returnCode;
}

sub setRootPassword {
    my ($self, $rootPassword) = @_;

    my $sub = $self->can("checkRootPassword");
    my $rc = 1;
    if ( $sub ) {
        $rc = &$sub($self, $rootPassword);
    }
    
    if ( $rc ) {
        $self->{params}->{RootPassword}->{value} = $rootPassword;
    }
    
    return $rc;
}

sub setRemoveHosts {
    my ($self, $removeHosts) = @_;
    $removeHosts =~ s/\s+//g;

    my $sub = $self->can("checkRemoveHosts");
    my $rc = 1;
    if ( $sub ) {
        $rc = &$sub($self, $removeHosts);
    }
    
    if ( $rc ) {
        $self->{params}->{RemoveHosts}->{value} = $removeHosts;
    }
    return $rc;
}

# Override
sub InitDefaults {
	my ( $self, $kit ) = @_;
	$self->SUPER::InitDefaults ( $kit );
	my $instance = $self->getOwnInstance();
	my $validValues = $instance->get_allhosts();
	my $displayValues = $self->_getRemoveHostsDisplayValues($validValues); # aka host roles
	$self->{params}->{RemoveHosts}->{valid_values} = $validValues;
	$self->{params}->{RemoveHosts}->{ui_values} = $displayValues;
	return 1;
}

sub _getRemoveHostsDisplayValues {
	my ($self, $removeHosts) = @_;
	my $hostRoleProperties = GetHostRoleProperties();
	my @hostRoles = ();

	my $instance = $self->getOwnInstance();
	my $hostRolesInfo = $instance->getHostRolesInfo();

	foreach my $hostname (@{$removeHosts}) {
		my $roles = $hostRolesInfo->{$hostname};
		$roles =~ s/\s+/, /g;
		$roles =~ s/([^\s,]+)/$hostRoleProperties->{$1}->{str}/g;
		push(@hostRoles, $roles);
	}
	return \@hostRoles;
}

sub isLocalhostMarkedForRemoval {
    my ($self) = @_;
    return $self->getRemoveHostsDetector()->userInputContainsLocalhost($self->getValue('RemoveHosts'));
}

sub isRemoteHostsMarkedForRemoval {
    my ($self) = @_;
    return $self->getRemoveHostsDetector()->userInputContainsRemoteHosts($self->getValue('RemoveHosts'));
}

sub getRemoteHostsMarkedForRemoval {
    my ($self) = @_;
    return $self->getRemoveHostsDetector()->getSelectedRemoteHosts($self->getValue('RemoveHosts'));
}

sub getRemoveHostsDetector {
    my $self = shift();
    unless ( defined $self->{_scenarioDetector} ) {
        $self->{_scenarioDetector} = LCM::Hosts::HostsDetector->new($self->getInstaller(), [$self->getMsgLst()]);
        if($self->{_scenarioDetector}->hasParsingErrors()){
            $self->getMsgLst()->addWarning("Could not retrieve host roles from file 'nameserver.ini'.");
        }
    }
    return $self->{_scenarioDetector};
}

sub shallUnregisterComponents{
    my ($self) = @_;
    my $rc = 0;
    my $componentsRegistrationMgr;
    if($self->isRemoteHostsMarkedForRemoval()){
        $componentsRegistrationMgr = $self->_getRemoteComponentsRegistrationHandler(); 
        $rc = $componentsRegistrationMgr->existRegistrableComponents();
    } 
    if($self->isLocalhostMarkedForRemoval()) {
        $componentsRegistrationMgr = $self->_getLocalComponentsRegistrationHandler();
        $rc = $rc && $componentsRegistrationMgr->existRegistrableComponents();
    }
    return $rc;
}

#-------------------------------------------------------------------------------
# Checks the host roles and enables the system user and accelerator user.

sub checkHostRoles {
    my ($self, $roleList) = @_;
    my $shouldSkipAcceleratorParameters = 1;

    foreach my $currRole (split (',', $roleList)) {
        if ($currRole =~ /$gHostRoleAcceleratorStandby|$gHostRoleAcceleratorWorker/){
            $shouldSkipAcceleratorParameters = 0;
            last;
        }
    }

    $self->setSkip('SystemUser', $shouldSkipAcceleratorParameters);
    $self->setSkip('SQLSysPasswd', $shouldSkipAcceleratorParameters);
    $self->setSkip('AcceleratorUser', $shouldSkipAcceleratorParameters);
    $self->setSkip('AcceleratorPassword', $shouldSkipAcceleratorParameters);

    return 1;
}

sub checkRemoveHosts {
    my ($self, $value) = @_;
    my $flavourProductName = LCM::App::ApplicationContext::getFlavourProductName();
    my $detector = $self->getRemoveHostsDetector();
    my $error = "Current $flavourProductName System landscape does not allow removal of the provided hosts";

    $self->clearParameterWarnings('RemoveHosts');
    if($detector->hasParsingErrors()){
        $self->addParameterWarning('RemoveHosts',"Could not retrieve host roles from file 'nameserver.ini'.");
    }
    if($self->getOwnInstance()->isMultiDb() && $self->getValue('AutoInitializeServices') =~ /$bool_false_pattern/){
        $self->addParameterWarning('RemoveHosts',"Please make sure that all tenant-specific services are removed from $value.");
    }
    my $check = LCM::Hosts::RemoveHostCheck->new();
    my $result = $check->check($self,$detector,$value);
    $self->getMsgLst()->appendMsgLst($check->getMsgLst());
    if ($detector->userInputContainsLocalhost($value)) {
       if (! $result) {
            $error = $self->isCurrentUserSidadm() ? "Removal of the local host as $flavourProductName System administrator user is not supported. The tool must be started as root." : $error;
            $self->PushError($error,$check->getErrMsgLst());
            return undef;
       }
       $self->AddMessage("Localhost is choosen for removal.");
       $self->{isOnlyLocalHostMarked} = 1;
    }

    if($detector->userInputContainsRemoteHosts($value)) {
       if (! $result) {
            $self->PushError($error,$check->getErrMsgLst());
            return undef;
        }
        my $isUseSHA = $self->isUseSAPHostagent();
        $self->setSkip('HostagentPassword', $isUseSHA ? 0 : 1);
        $self->setSkip('InstallSSHKey', $isUseSHA ? 1 : 0);
        $self->setSkip('RootUser', $isUseSHA ? 1 : 0);
# RootPassword will be unskipped in tryRemtoeKeyAuthorization only if needed
        $self->setSkip('RootPassword', 1);
        $self->AddMessage("Remote hosts are choosen for removal.");
        $self->{isOnlyLocalHostMarked} = 0;
    }
    if(!$self->{isOnlyLocalHostMarked} && !$self->initRemoteHostsForInstance()){
        return undef;
    }

    return $self->SUPER::checkRemoveHosts($value, undef, 1);
}

sub getInstaller {
	my $self = shift();
	unless ( defined $self->{_installer} ) {
		$self->{_installer} = LCM::ResidentInstaller->new ();
	}
	return $self->{_installer};
}

sub getAction {
	return $SYSTEM_REMOVE_ACTION;
}

sub getProductName {
    my $flavourProductName = LCM::App::ApplicationContext::getFlavourProductName();
    $TITLE =~ s/\Q$gFlavourProductName\E/$flavourProductName/g;
    return $TITLE;
}

sub createSummaryTree {
	my($self) = @_;
	if($self->{isOnlyLocalHostMarked}){
		$self->setSkip('RemoteExecution',1);
	}
	return LCM::SummaryTreeBuilder::buildSummaryTree ( $self );
}

sub getSummaryTitle {
    return 'Remove Hosts Parameters';
}

sub getStaticDialogs {
    return ["LCM::Gui::Dialogs::SummaryDialog","LCM::Gui::Dialogs::ConfigurationExecutionDialog", "LCM::Gui::Dialogs::FinalizeDialog"];
}

sub getConfigureDialogs {
	my ($self, $wizard) = @_;
	   
	require LCM::Gui::Dialogs::RemoveHosts::SelectRemoveHostsDialog;
	require LCM::Gui::Dialogs::RemoveHosts::PasswordsPropertiesDialog;
	require LCM::Gui::Dialogs::SystemUserPropertiesDialog;
	require LCM::Gui::Dialogs::AddHosts::AcceleratorPropertiesDialog;
	
	return [
	   new LCM::Gui::Dialogs::RemoveHosts::SelectRemoveHostsDialog($wizard),
	   new LCM::Gui::Dialogs::RemoveHosts::PasswordsPropertiesDialog($wizard),
	   new LCM::Gui::Dialogs::SystemUserPropertiesDialog($wizard),
	   new LCM::Gui::Dialogs::AddHosts::AcceleratorPropertiesDialog ($wizard)
	];
}

sub getNumberOfExpectedOutputLines {
	return 40;
}

sub getSystemComponentManager {
    my ($self) = @_;
    my $installer = getInstaller();
    $installer->setMsgLstContext([$self->getMsgLst()]);
    my $scm = $installer->getOwnSystemComponentManager();

    if (!defined $scm){
        $self->setErrorMessage("Cannot determine the installed '$gProductNameEngine'. Start the tool from its original location inside the database.");
        return undef;
     }
    return $scm;
}

sub setRequiredParamsValue {
	my ($self) = @_;

	return $self->setBatchValueOfRequiredParams('RemoveHosts');
}

sub isScopeInstance{
	return 0;
}

sub getSkippedComponentsForRegistration{
    my ($self, $isRemote) = @_;
    my $componentsRegistrationMgr;
    if($isRemote){
        $componentsRegistrationMgr = $self->_getRemoteComponentsRegistrationHandler(); 
    }else{
        $componentsRegistrationMgr = $self->_getLocalComponentsRegistrationHandler();
    }
    return $componentsRegistrationMgr->getSkippedComponents();
}
  
sub _getRemoteComponentsRegistrationHandler{
    my($self) = @_;
    if(not defined $self->{_remoteComponentsRegistrationHandler}){
        my $componentManager = $self->getSystemComponentManager();
        my $installedComponents = $componentManager->getAllComponents();
        $self->{_remoteComponentsRegistrationHandler} = new LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::RemoteComponentsManager($installedComponents,$installedComponents,$SCENARIO_ADD_REMOVE_HOSTS);
        $self->{_remoteComponentsRegistrationHandler}->detectComponents();
    
    }
    return $self->{_remoteComponentsRegistrationHandler};
}  

sub _getLocalComponentsRegistrationHandler{
    my($self) = @_;
    if(not defined $self->{_localComponentsRegistrationHandler}){
        my $componentManager = $self->getSystemComponentManager();
        my $installedComponents = $componentManager->getAllComponents();
        $self->{_localComponentsRegistrationHandler} = new LCM::Configuration::Hosts::UpdateHosts::ComponentsRegistration::LocalComponentsManager($installedComponents,$installedComponents,$SCENARIO_ADD_REMOVE_HOSTS);
        $self->{_localComponentsRegistrationHandler}->detectComponents(); 
    }
    return $self->{_localComponentsRegistrationHandler};
}

sub checkSID{
    my ($self, $sid, $msglst_arg) = @_;
    if(!$self->isValidSID($sid)){
    	return 0;
    }
	return 1;
}

sub getRemoteHosts{
	my($self) = @_;
	if($self->{isOnlyLocalHostMarked}){
		my @remoteHosts = ();
		return \@remoteHosts;
	} else {
		return $self->SUPER::getRemoteHosts();
	}
}

sub checkHostagentPassword {
	my ($self, $password) = @_;

	return 1 if (! $self->isUseSAPHostagent());
	return 1 if (! $self->isRemoteHostsMarkedForRemoval());

	my $targetHosts = $self->getRemoteHostsMarkedForRemoval();

	return 1 if (scalar(@{$targetHosts}) == 0);

	my $sapAdmUser = $self->_createSapAdmUser();
	my $rc = $sapAdmUser->verifyPasswordSHA($password, $self, $targetHosts);
	if (! defined $rc) {
		$self->{params}->{HostagentPassword}->{no_retry} = 1;
	}

	return $rc;
}

sub _createSapAdmUser {
	my ($self) = @_;

	my $userName = 'sapadm';

	require LCM::Configuration::SapAdmUser;

	return new LCM::Configuration::SapAdmUser($userName);
}

sub getAutoInitializeServicesAction {
    return "removing";
}

sub isCollectOtherHostInfosRequired {
	return 0;
}

sub isCurrentUserSidadm{
    return isSidadmin();
}

sub isAutoInitializeServicesApplicableParameterCallback{
    my $self = shift;
    return $self->isRemoveHostsValueApplicableForAutoInitialize(@_);
}
1;
