package LCM::Configuration::Hosts::UpdateHosts::UpdateLocalHostConfiguration;

use base qw ( LCM::Configuration::ContinueConfiguration LCM::Configuration::CertifiedHostConfig);
use LCM::Configuration::GenericStackAny qw ( $ini_section_general $ini_section_hlm $ini_section_client );
use SDB::Install::Configuration qw($bool_true_pattern $bool_false_pattern);
use SDB::Install::Globals qw ( $gProductNameEngine $gProductNameClient $gProductName);
use SDB::Install::SysVars;
use LCM::ComponentManager::SystemComponentManager;
use LCM::Component;
use LCM::Configuration::ParametersCreator qw(GetParamCertificatesHostmap GetParamComponentsRegistration GetParamDeploySHAConfigurations GetParamCertificateHostname GetParamUpdateComponentListLocation);
use SAPDB::Install::Hostname;
use SDB::Common::Utils qw(getSidcryptName);
use strict;

my $componentsMapping = {
	$gProductNameClient => "LCM::Component::Installable::HDBClient::HDBClientRoot",
	$gProductNameEngine => "LCM::Component::Installable::HDBServer::HDBServerRoot",
};

our $PROGRESS_MSG_FILTER = "Registering.*|Unregistering.*|Regenerating.*|Deploying.*|Updating component list location.*|Updating.*Instance Integration.*|supported: SLES|Minimal SLES release";
our $UPDATE_LOCAL_HOST_STEP_MSG = "Updating $gProductName Instance Integration on Local Host";
our $PROGRESS_MSG_FILTER_REMOTE = "Updating $gProductName Instance Integration.*|Registering.*|supported: SLES|Minimal SLES release";
our $UPDATE_REMOTE_HOST_STEP_MSG = "Updating $gProductName Instance Integration on Remote Hosts";

our $PARAMETER_TO_OPTION_MAPPING = {
                                        'DeploySHAConfigurations' => 'DHC', 
                                        'UpdateComponentListLocation' => 'UCLL', 
                                        'ComponentsRegistration' => 'CR',
                                        'CertificateHostname' => 'CH',
                                        'SkippedComponents' => 'SKIP',
                                        'GenerateCertificates' => 'GEN_CERT',
                                        'KeepXsUsers'       => 'KXU',
                                        'SkipModifySudoers' => 'SMS',
                                        'XsEaDataPath'      => 'XEDP',
};

our @EXPORT_OK = qw(
    $PARAMETER_TO_OPTION_MAPPING
    $PROGRESS_MSG_FILTER
    $UPDATE_LOCAL_HOST_STEP_MSG
    $PROGRESS_MSG_FILTER_REMOTE
    $UPDATE_REMOTE_HOST_STEP_MSG
);
    
sub new {
    my $self = shift->SUPER::new (@_);
    $self->_setPersistenceFileName('hdblcm_update_host');
    $self->{persistenedParams} = {
        'SelectedComponents' => $self->getPersistenedParamSelectedComponents(),
    };
    $self->{params} = $self->getUpdateHostParameters(0);
    my $doCertificatesExist = $self->_existsCertificatesOnLocalHost() ? 1 : 0;
    $self->setSkip('Password', $doCertificatesExist);
    return $self;
}

sub getUpdateHostParameters{
	my($self,$order)= @_;
	my $params = {
        'SID'                           => $self->getParamSID( $order++, $ini_section_general, 1),
        'CollectUpdateHostCredentials'  => $self->getParamCollectUpdateHostCredentials($order++, $ini_section_general),
        'SkipHostagentCalls' => $self->getParamSkipHostagentCalls($order++, $ini_section_general),
        ( !$isWin ? ( 'RemoteExecution' => $self->getParamRemoteExecution($order++, $ini_section_general),
                      'UseHttp'         => $self->getParamUseHttp        ($order++, $ini_section_general)
                    ) : () ),
        'ComponentsRegistration'        => LCM::Configuration::ParametersCreator::GetParamComponentsRegistration($order++, $ini_section_general),
        'GenerateCertificates'          => $self->_getParamGenerateCertificates($order++, $ini_section_general),
        'DeploySHAConfigurations'       => LCM::Configuration::ParametersCreator::GetParamDeploySHAConfigurations($order++, $ini_section_general),
        'SSOCertificate'                => $self->getParamSSOCertificate ($order++, $ini_section_general),
        'UpdateComponentListLocation'   => LCM::Configuration::ParametersCreator::GetParamUpdateComponentListLocation($order++, $ini_section_general),
        'KeepXsUsers'                   => $self->getParamKeepXsUsers($order++, $ini_section_general),
        'SkipModifySudoers'             => $self->getParamSkipModifySudoers($order++, $ini_section_general),
        'XsEaDataPath'                  => $self->getParamXsEaDataPath($order++, $ini_section_general),
        'CertificatesHostmap'           => $self->getParamCertificatesHostmap($order++),
        'CertificateHostname'           => LCM::Configuration::ParametersCreator::GetParamCertificateHostname($order++, $ini_section_general),
        'Password'                      => $self->getParamPassword($order++, $ini_section_general),
        'SystemUser'                    => $self->getParamSystemUser($order++, $ini_section_general),
        'SQLSysPasswd'                  => $self->getParamSQLSysPasswd($order++, $ini_section_general),
        'LSSPassword'                   => $self->getParamLSSPassword( $order++, $ini_section_general),
        'SkippedComponents'     => {          
            'order'             => $order++,
            'opt'               => 'skipped_components',
            'type'              => 'csv',
            'section'           => $ini_section_general,
            'value'             => undef,
            'str'               => 'Skipped Components',
            'desc'              => 'Skipped Components',
            'skip'              => 0,
            'hidden'            => 0,
            'mandatory'         => 0,
            'set_interactive'   => 0
        },
    };
    $params->{CertificatesHostmap}->{hidden} = 1;
    $params->{RemoteExecution}->{hidden} = 1;
    $params->{SID}->{hidden} = 1;
    return $params;
}

sub setGenerateCertificates {
	my ($self, $value) = @_;
	if($value =~ /$bool_false_pattern/){
		$self->{params}->{GenerateCertificates}->{value} = $value;
		$self->setSkip('CertificatesHostmap', 1);
		$self->setSkip('CertificateHostname', 1);
		$self->setSkip('Password', 1);
		return 1;
	} elsif ($value =~ /$bool_true_pattern/) {
		$self->{params}->{GenerateCertificates}->{value} = $value;
		return 1;
	}
	return 0;
}

sub _getParamGenerateCertificates{
	my ($self, $order, $section) = @_;
	return {
		'order'   		   => $order,
        'opt'     		   => 'generate_certificates',
        'type'    		   => 'boolean',
        'section' 		   => $section,
		'value'   		   => undef,
		'default' 		   => 1,
        'str'     		   => 'Enable generation of SAP Host Agent SSL certificates',
        'desc'			   => 'Enables generation of SAP Host Agent SSL certificates',
        'init_with_default'=> 1,
        'set_interactive'  => 0,
	};
}

sub checkComponentsRegistration {
	my ($self, $value) = @_;
	my @validValues = @{$self->{params}->{ComponentsRegistration}->{valid_values}};
	my $isValueValid = (scalar(grep {$_ eq $value} @validValues) == 1);
	return $isValueValid;
}

sub checkDeploySHAConfigurations{
	my ($self, $value) = @_;
	
	if($value =~ /$bool_true_pattern/){
# Password is mandatory if we want to deploy SHA operations as non-root user
		if ($>){
			$self->setSkip('Password', 0);
		}
		return 1;
	}
	if($value =~ /$bool_false_pattern/){
		return 1;
	}
	return 0;
}

sub setCertificateHostname {
	my ($self, $value) = @_;
	if(!defined $self->{localHostname}){
	   $self->{localHostname} = hostname();
	}
	$self->{params}->{CertificatesHostmap}->{origin_values} = [$self->{localHostname}];
	$self->{params}->{Password}->{skip} = 0;
	my $rc = $self->setCertificatesHostmap($self->{localHostname}, $value, 1);
	if($rc){
		$self->{params}->{'CertificateHostname'}->{value} = $value;
	}
	return $rc;
}

sub getSystemComponentManager {
	my $self = shift;
	if(!defined $self->{systemComponentManager} ){
	   $self->{systemComponentManager} = new LCM::ComponentManager::SystemComponentManager( $self );
	}
    return $self->{systemComponentManager}; 
}

sub detectSystemComponents {
    my ($self) = @_;
# getSID will return undef in the case of standalone 'update_host' scenario (when not part of other scenario),
# because otherwise the SID is explicitly set (see UpdateHostConfigurationFactory::createLocalConfiguration for example).
# In this case we can get the default value of the SID as it is already set inside the InitDefaults of AnyConfig.
    my $sid = $self->getSID() // $self->getDefault('SID');
    my $scm = $self->getComponentManager();
    $scm->clearDetectedComponents();
    $scm->detectComponentsBySid($sid);

    my $sortedComponents = $scm->getSortedComponentKeynames();
    my $byKeyname = $scm->{components};

    my $installedComponents = [];
    for ( @$sortedComponents ) {
        if ( exists $byKeyname->{$_} ) {
            my $component = $scm->getComponentByKeyName($_);
            push ( @$installedComponents, $component );
        }
    }

    $self->{installedComponents} = $installedComponents;
}

sub getInstalledComponents {
	my ( $self,  ) = @_;
	return $self->{installedComponents};
}

sub getParamCertificatesHostmap {
    my ( $self, $order ) = @_;
    my $paramCertificatesHostmap = LCM::Configuration::ParametersCreator::GetParamCertificatesHostmap( $order );
    $paramCertificatesHostmap->{'skip'} = $self->_existsCertificatesOnLocalHost();
    $paramCertificatesHostmap->{'set_interactive'} = 0;
    $paramCertificatesHostmap->{'hidden'} = 0;
    return $paramCertificatesHostmap;
}

sub initParams {
	my ( $self, ) = @_;
    
    $self->_initComponents(); 
    $self->_initCertificate();
    $self->fillCertificatesHostmapDefaultValuesAndLocalData();
    
    require SDB::Install::NewDBUser;
	my $instance = $self->getHdbOwnInstance();
    my $sid = $instance->get_sid();
	my $user = new SDB::Install::NewDBUser($sid);
	$self->{user} = $user->getSidAdm();
	$self->{username} = $user->getSidAdmName();
	$self->{params}->{Password}->{'str'} = "System Administrator ($self->{username}) Password";

    my $lssInstance = $instance->getLssInstance(1);
    my $skip = !$lssInstance->isInstalled() || $lssInstance->isConfiguredOnHost()  ? 1 : 0;
    $self->setSkip("LSSPassword", $skip);
    return 1;
}

sub checkLSSPassword {
    my ($self, $value) = @_;
    my $sidcryptUser = SDB::Install::LSS::LssSidcryptUser->new(getSidcryptName($self->getValue('SID')));
    $sidcryptUser->setMsgLstContext([$self->getMsgLst(), $self->getErrMsgLst()]);
    return $sidcryptUser->verifyExistingUserPassword($value, $self);
}

sub checkPassword{
    my ($self, $value) = @_;

    if (!defined $self->{user}){
        return 1;
    }

    my $rc = $self->{user}->verifyPassword($value);
    if (!defined $rc){
        $self->AddWarning ("Cannot check password", $self->{user});
        $self->AddMessage ("=> pass check");
        return 1;
    }

    if (!$rc){
        $self->AddError ('Unknown user password combination');
        return 0;
    }

    return 1;
}

sub checkSSOCertificate{
    my ($self, $value) = @_;
    $self->setSkip('Password');
    return 1;
}


sub _initComponents {
    my ( $self, ) = @_;
    my $componentNames = $self->_getComponentsName();;
    
    for ( @$componentNames ) {
    	if ( $componentsMapping->{$_} ) {
    		push @{$self->{persistenedParams}->{SelectedComponents}->{modules}}, $componentsMapping->{$_};
    	}
    }
}

sub _getComponentsName {
	my ( $self, $parsedInformation) = @_;
	
    $self->detectSystemComponents();
    my $installedComponents = $self->getInstalledComponents();
	my $componentsName = [];
	
	for my $component ( @$installedComponents ) {
		my $name = $component->getComponentName();
		push ( @$componentsName, $name );
	}
	
	return $componentsName;
}

sub _initCertificate {
    my ( $self, ) = @_;
    
    my $existsOnLocalHost = $self->_existsCertificatesOnLocalHost();
    my $existsInFile = $self->_existsCertificateInPersistenceFile();
    my $certificatesHostmap = $self->{params}->{CertificatesHostmap};
    my $localHost = $self->{localHostname};
    my $isPasswordSkipped = 1;
    if ( $existsOnLocalHost && $existsInFile ) {
    	$certificatesHostmap->{value}->{$localHost} = $self->_getCertificateFromPersistenceFile();
    	$isPasswordSkipped = 0;
    } elsif ( ! $existsOnLocalHost && ! $existsInFile ) {
    	$certificatesHostmap->{value}->{$localHost} = $localHost;
    	$isPasswordSkipped = 0;
    } elsif ( ! $existsOnLocalHost && $existsInFile ) {
    	$certificatesHostmap->{value}->{$localHost} = $self->_getCertificateFromPersistenceFile();
    	$isPasswordSkipped = 0;
    }
    $self->{params}->{Password}->{skip} = $isPasswordSkipped;
}

sub _existsCertificatesOnLocalHost {
    require LCM::RegenerateCertificates;
    my $localCertificateHostname = LCM::RegenerateCertificates::getCertOwner();
    $localCertificateHostname =~ s/CN=//;
    return ( $localCertificateHostname ne '' );
}

sub _existsCertificateInPersistenceFile {
    my ( $self, ) = @_;
    my $parsedInformation = $self->{parsedInformation};
    return ( defined $parsedInformation && defined $parsedInformation->{certificate} && $parsedInformation->{certificate} ne '' );
}

sub _getCertificateFromPersistenceFile {
	my ( $self, ) = @_;
    
    my $parsedInformation = $self->{parsedInformation};
    my $localHostCertificate = $parsedInformation->{'certificate'};
    return $localHostCertificate;
}

sub getPersistenedParamSelectedComponents {
	return { "modules" => [] };
}

sub getProductName { return "Update SAP HANA Host"; }

sub getAction { return "update_host"; }

sub createSummaryTree {
    my ($self) = @_;
    return LCM::SummaryTreeBuilder::buildSummaryTree($self); 
}

sub getLocalHanaHost {
    my ( $self ) = @_;
    my $hostname = $self->getHdbOwnInstance()->get_host();
    $hostname    = lc(hostname()) if (!defined $hostname);
    return $hostname;
}

sub getComponentManager{
	my($self) = @_;
	return $self->getSystemComponentManager();
}

sub setCertificatesHostmap{
	my ($self, $hostname, $certificateHostname, $overwriteAlreadyDefined) = @_;
	my $rc = $self->SUPER::setCertificatesHostmap( $hostname, $certificateHostname, $overwriteAlreadyDefined);
	if($rc){
	   $self->{params}->{Password}->{skip} = 0;	
	}
	return $rc;
}

sub setRemoteExecution{
	my ($self, $value) = @_;
    my $rc = $self->SUPER::setRemoteExecution($value);
    if($rc && $value eq 'saphostagent'){
        $self->{params}->{Password}->{skip} = 0;	
    }
    return $rc;
}

sub getHdbOwnInstance{
	my( $self ) = @_;
	if( !defined $self->{hdbInstance}){
        my $hdbInstance = $self->getOwnInstance(1);
        $self->{hdbInstance} = $hdbInstance;
	}
	return $self->{hdbInstance};
}

sub setMsgLstContext{
	my ($self,$context, $keepProgressHandler) = @_;
	$self->SUPER::setMsgLstContext($context, $keepProgressHandler);
	if(defined $self->getComponentManager()){
	   $self->getComponentManager()->setMsgLstContext($context, $keepProgressHandler);
	}
}

sub setHostagentPassword {
	my ($self, $value) = @_;

	if (! $self->checkHostagentPassword($value)) {
		return 0;
	}

	$self->{params}->{HostagentPassword}->{value} = $value;
	return 1;
}

sub shallRegisterComponents{
    my($self) = @_;
    my $componentsRegistration = $self->getValue('ComponentsRegistration');
    return 0 if($componentsRegistration !~ /^register$|^unregister$/);
    return 1;
}

sub shallRegenerateCertificates{
    my($self) = @_;
    my $skipSHACalls = $self->getValue('SkipHostagentCalls');
    my $generateCertificates = $self->getValue('GenerateCertificates');
    return ($skipSHACalls || !$generateCertificates) ? 0 : 1;
}

sub shallUpdateSLD{
    my($self) = @_;
    my $isUpdate = $self->getValue('UpdateComponentListLocation');
    return $isUpdate =~ /$bool_true_pattern/
}

sub shallDeployConfigurations{
    my($self) = @_;
    my $skipSHACalls = $self->getValue('SkipHostagentCalls');
    my $deployShaConfigurations = $self->getValue('DeploySHAConfigurations');
    return ($skipSHACalls || !$deployShaConfigurations) ? 0 : 1;
}

sub shallSetDatabaseProperty{
    my ($self) = @_;
    return ($self->getValue('SkipHostagentCalls') || !$self->getValue('CollectUpdateHostCredentials')) ? 0 : 1;
}

sub checkSQLSysPasswd {
    my ($self, $value) = @_;
# When called as part of the installation process task
# for example this configuration won't be able to check
# the password, because it doesn't have an instance
    return 1 if (! defined $self->getOwnInstance(1));
    return $self->SUPER::checkSQLSysPasswd($value);
}

1;
