package LCM::Configuration::ValueChangeListeners::InstallUpdate::SignatureVerificationListenerBase;

use strict;
use warnings;
use Exporter;
use LCM::SAPDSigner;
use SDB::Install::Globals qw($gProductNameSHA $gFailedSignatureVerificationMessage $gSAPDSignerNotDetectedMessage);
use LCM::VerificationError qw(getNamedPipeErrorMessage);
use SDB::Install::MsgLst;
use LCM::Component::Installable::InstallationKitChecker;

sub new {
    return bless({}, shift());
}

sub onValueChange {
    my ($self, $selectedCmpsValue, $instconfig) = @_;
    return 1 if !$instconfig->getValue('VerifySignature');
    $self->resetVerificationErrors();

    if (!$self->checkSelectedComponentsFiles($instconfig)){
        $instconfig->getErrMsgLst()->addError("\nFor security reasons pipe files are not allowed.");
        $instconfig->setNoRetry("SelectedComponents", 1);
        return 0;
    }

    my $filePermissionsVerificationSuccess = $self->verifyComponentFilesPermissions($instconfig);
    return 1 if ($self->verifyComponentAuthenticity($instconfig) && $filePermissionsVerificationSuccess); 

    if ($instconfig->getIgnore('verify_signature')) {
        $instconfig->getMsgLst()->addMessage("Ignoring signature check errors due to --ignore command line switch");
        return 1;
    }

    $self->handleErrorMessages($instconfig);
    return $self->handleVerificationFailure($instconfig);
}

sub verifyComponentAuthenticity {
    my ($self, $instconfig) = @_;
    my $sapdsigner = LCM::SAPDSigner->getInstance();
    if (!defined $sapdsigner) {
        $self->addVerificationError(LCM::VerificationError->new(undef, $gSAPDSignerNotDetectedMessage, 0));
        return 0;
    }
    $self->handleSapdsignerLogging($sapdsigner, $instconfig);

    my $componentManager = $instconfig->getComponentManager();
    my $selectedComponents = $componentManager->getSelectedComponents();
    my @componentSetToAuthenticate = grep {$_->isSigned()} @$selectedComponents;
    return 1 if (!@componentSetToAuthenticate);
    my @componentNames = map {$_->getComponentName()} @componentSetToAuthenticate;
    $instconfig->getMsgLst()->addMessage("Verifying authenticity of components: " . join (', ', @componentNames));

    my $rc = $sapdsigner->authenticateComponentSet(\@componentSetToAuthenticate);
    $self->addVerificationErrorList($sapdsigner->getAuthenticationErrors());
    return $rc;
}

sub checkSelectedComponentsFiles {
    my ($self, $instconfig) = @_;
    my $componentManager = $instconfig->getComponentManager();
    my $selectedComponents = $componentManager->getSelectedComponents();
    $instconfig->getMsgLst()->addProgressMessage("Verifying files...");

    my $rc = 1;
    my $errorList = SDB::Install::MsgLst->new();
    foreach my $component (@{$selectedComponents}){
        if (!$component->checkFilesForPipes($errorList)) {
            my $cmpName = $component->getComponentName();
            $instconfig->getErrMsgLst()->addError("$cmpName:", $errorList);
            $rc = 0;
        }
    }
    return $rc;
}

sub handleSapdsignerLogging {
    my ($self, $sapdsigner, $instconfig) = @_;
    $sapdsigner->setMsgLstContext($instconfig->getMsgLstContext());
}

sub handleVerificationFailure {
    my ($self, $instconfig) = @_;
    ...
}

sub onErrorMessage {
    my ($self, $configuration) = @_;
    ...
}

sub handleErrorMessages {
    my ($self, $instconfig) = @_;
    my $authenticationFailures = $self->getVerificationErrors();
    my $hasAuthenticityErrors = grep{ $_->isVerificationAuthenticityError()} @{$authenticationFailures};
    my $failedComponents = scalar(@{$authenticationFailures});
    $self->clearWarningMessages();

    if ($failedComponents > 1) {
        my $msg = $hasAuthenticityErrors ? "Authenticity verification of the following components failed:" : "Verification of the following components failed:";
        $self->onErrorMessage($msg, $instconfig);
        $self->onErrorMessage((' ' x 2).$_->getMultipleFailedComponentsMessage(), $instconfig) for @{$authenticationFailures};
    } elsif($failedComponents == 1) {
        $self->onErrorMessage($authenticationFailures->[0]->getSingleFailedComponentMessage(), $instconfig);
    }
}

sub getVerificationErrors{
    my ($self,$sapdsigner) = @_;
    return $self->{errors} // [];
}

sub verifyComponentFilesPermissions{
    my($self,$instconfig) = @_;
    my $componentManager = $instconfig->getComponentManager();
    my $selectedComponents = $componentManager->getSelectedComponents();

    my $hasError = 0;
    foreach my $component(@$selectedComponents){
        my $verificationMsgLst = new SDB::Install::MsgLst();
        if(!$component->verifyFilesPermissions($verificationMsgLst)){
            chomp(my $error = ${$verificationMsgLst->getMsgLstString('')});
            $self->addVerificationError(LCM::VerificationError->new($component->getVerificationString(), $error, 0));
            $hasError = 1;
        }
    }
    return !$hasError;
}

sub addVerificationError{
    my ($self,$error) = @_;
    $self->{errors} = [] if (!defined $self->{errors});
    push(@{$self->{errors}}, $error);
}

sub addVerificationErrorList{
    my ($self,$errorList) = @_;
    $self->{errors} = [] if (!defined $self->{errors});
    push(@{$self->{errors}}, @$errorList);
}

sub resetVerificationErrors{
    my $self  = shift;
    $self->{errors} = [];
    my $sapdsigner = LCM::SAPDSigner->getInstance();
    $sapdsigner->clearAuthenticationErrors() if defined $sapdsigner;
}

# The following 3 methods are used by the SLP and GUI listeners only.
# They are in the base class as code duplication is thus avoided
sub getFailedVerificationMessage {
    my ($self) = @_;
    my $msg = join("\n", @{$self->{warningMessages} // []});
    $msg .= "\n\n$gFailedSignatureVerificationMessage";

    return $msg;
}

sub addWarningMessage {
    my ($self, $msg) = @_;
    $self->{warningMessages} //= [];
    push @{$self->{warningMessages}}, $msg;
}

sub clearWarningMessages {
    my ($self) = @_;
    $self->{warningMessages} = [];
}

1;
