package LCM::DownloadComponents::DownloadableArchive;

use strict;
use parent 'SDB::Install::Base';
use File::stat;
use File::Spec;
use File::Copy;
use LCM::FileUtils qw(readFile deleteFile);
use IO::File;
use Carp 'croak';

my $MAX_RETRIES = 10;
# How many seconds to wait between the download retries
my $WAIT_TIME = 10;
# Mapping between error and possibility to retry the download
my %ACCEPTABLE_ERRORS = (
    'ERR_FILENAME_UNDEFINED'            => 0,
    'ERR_RESUME_NEEDED'                 => 1,
    'ERR_PART_FILE_OPEN'                => 0,
    'ERR_META_FILE_OPEN'                => 0,
    'ERR_PART_FILE_WRITE'               => 0,
    'ERR_PARTIAL_DOWNLOAD'              => 1,
    'ERR_CLEANUP_FAILED'                => 0,
    'INVALID_ARCHIVE_CLEANUP_FAILED'    => 0,
);
# Mapping between error and message
my %ERROR_MESSAGES = (
    'ERR_FILENAME_UNDEFINED'            => "Couldn't get the archive filename neither from the SWDC, nor from the request header",
    'ERR_RESUME_NEEDED'                 => "Resume files found for '%s' after fetching the name from the request.",
    'ERR_PART_FILE_OPEN'                => "Couldn't open .PART file for writing: %s",
    'ERR_META_FILE_OPEN'                => "Couldn't create .META file: %s",
    'ERR_PART_FILE_WRITE'               => "Couldn't write data to the .PART file: %s",
    'ERR_PARTIAL_DOWNLOAD'              => "%s was downloaded partially, retrying...",
    'ERR_CLEANUP_FAILED'                => "Failed to cleanup invalid resumes file for '%s'",
    'INVALID_ARCHIVE_CLEANUP_FAILED'    => "Couldn't delete invalid archive '%s' from the download directory.",
);
# Separate error message for the case when at first the archive name is unknown, but at the time of the request
# when we fetch the name from it, it turns out that the archive already exists
my $ARCHIVE_EXISTS_ERROR = "ARCHIVE_EXISTS";


sub new {
    my($class, $fileHash, $configuration) = @_;
    my $self = bless {}, $class;
    my $downloadDir = $configuration->getValue('DownloadDir');

    $self->setName($fileHash->{'fileName'});
    $self->setSize($fileHash->{'fileSize'}, 1000000); # SWDC gives us the size in MB, convert it to B
    $self->setUrl($fileHash->{'downloadUrl'});
    $self->setDownloadDir($downloadDir);
    $self->_setConfiguration($configuration);

    return $self;
}

sub setUserAgent {
    my ($self, $userAgent) = @_;
    $self->{userAgent} = $userAgent ? $userAgent : undef;
}

sub getUserAgent {
    my ($self) = @_;
    return $self->{userAgent};
}

sub setDownloadDir {
    my ($self, $downloadDir) = @_;
    $self->{downloadDir} = $downloadDir ? $downloadDir : undef;
}

sub getDownloadDir {
    my ($self) = @_;
    return $self->{downloadDir};
}

sub setName {
    my ($self, $name) = @_;
    $self->{filename} = $name ? $name : undef;
}

sub getName {
    my ($self) = @_;
    return $self->{filename};
}

sub getPartFilename {
    my ($self) = @_;
    my $filename = $self->getName();
    return (defined($filename) ? sprintf('%s.PART', $filename) : undef);
}

sub getMetaFilename {
    my ($self) = @_;
    my $filename = $self->getName();
    return (defined($filename) ? sprintf('%s.META', $filename) : undef);
}

sub getMetaFile {
    my ($self) = @_;
    my $metaFileName = $self->getMetaFilename();
    if ($metaFileName) {
        my $downloadDir = $self->getDownloadDir();
        my $metaFilePath = File::Spec->catfile($downloadDir, $metaFileName);
        return File::stat::stat($metaFilePath);
    }
    return undef;
}

sub getPartFile {
    my ($self) = @_;
    my $partFileName = $self->getPartFilename();
    if ($partFileName) {
        my $downloadDir = $self->getDownloadDir();
        my $partFilePath = File::Spec->catfile($downloadDir, $partFileName);
        return File::stat::stat($partFilePath);
    }
    return undef;
}

sub hasPartFile {
    my ($self) = @_;
    return defined($self->getPartFile()) ? 1 : 0;
}

sub hasMetaFile {
    my ($self) = @_;
    my $metaFile = $self->getMetaFile();
    return (defined($metaFile) && -r $metaFile) ? 1 : 0; # Unreadable .META file is useless
}

sub hasResumeData {
    my ($self) = @_;
    return $self->hasMetaFile() && $self->hasPartFile();
}

sub getArchive {
    my ($self) = @_;
    my $name = $self->getName();
    if ($name) {
        my $downloadDir = $self->getDownloadDir();
        my $archiveFilename = File::Spec->catfile($downloadDir, $name);
        return File::stat::stat($archiveFilename);
    }
    return undef;
}

sub archiveExists {
    my ($self) = @_;
    my $archiveFile = $self->getArchive();
    return defined($archiveFile) ? 1 : 0;
}

sub getDownloadedBytes {
    my ($self) = @_;
    my $partFile = $self->getPartFile();
    return (defined($partFile) ? int($partFile->size()) : 0);
}

sub setSize {
    my ($self, $size, $multiplier) = @_;
    $self->{filesize} = $size ? int($size) : undef;
    if ($multiplier) {
        $self->{filesize} *= $multiplier;
    }
}

sub getSize {
    my ($self) = @_;
    return $self->{filesize};
}

sub setUrl {
    my ($self, $url) = @_;
    $self->{url} = $url ? $url : undef;
}

sub getUrl {
    my ($self) = @_;
    return $self->{url};
}

sub download {
    my ($self, $responseDataCallback) = @_;
    my $name = $self->getName();
    $self->_setResponseDataCallback($responseDataCallback);

    if ( $self->archiveExists() ) {
        if ( $self->_isValidArchive() ) {
            $self->getMsgLst()->addMessage("Archive '$name' already exists in the download directory. Skipping download...");
            return 1;
        } elsif ( !$self->_deleteArchive() ) {
            $self->getErrMsgLst()->addError("Couldn't delete invalid archive '$name' from the download directory.");
            return 0;
        }
    }

    my $downloadDir = $self->getDownloadDir();
    my $willResume = $self->_hasValidResumeData();
    my $downloadMsg = $willResume ? "Valid resume files for '$name' are present. Resuming download..." : "Downloading '$name' in '$downloadDir' ...";
    my $msg = $name ? $downloadMsg : "Couldn't fetch the name from SWDC. Will try to get it from the request headers.";

    if (!$willResume && !$self->_cleanupResumeData()) {
        $self->getErrMsgLst()->addError("Cleanup of invalid resume files failed.");
        return undef;
    }

    $self->getMsgLst()->addProgressMessage($msg);
    return $self->_downloadArchive();
}

# -------------------------------------------------------
# Only private subs below this line
# -------------------------------------------------------

sub _downloadArchive {
    my ($self) = @_;
    my $currentTry = 1;

    while ($currentTry <= $MAX_RETRIES && $self->_canRetry()) {
        my ($errorString, $response) = $self->_getContent();
        return 1 if $errorString =~ /$ARCHIVE_EXISTS_ERROR/;

        $self->_closePartFile();

        my $isSuccess = (defined($response) && $response->is_success() && !$errorString);
        if ($isSuccess) {
            $self->_handleSuccessfulDownload();
            $self->getMsgLst()->addMessage("Archive '".($self->getName())."' downloaded successfully.");
            return 1;
        }

        $self->_handleError($errorString, $response);

        sleep($WAIT_TIME);
        $currentTry++;
    }

    $self->getErrMsgLst()->addError("Download of '".($self->getName())."' failed.");
    return 0;
}

sub _isValidArchive {
    my ($self) = @_;
    my $archiveFile = $self->getArchive();
    my $headers = $self->_getHeaders();
    return 0 if !$archiveFile;
    return 1 if !$headers;

    if (defined $archiveFile) {
        return (int($archiveFile->size()) == int($headers->header("Content-Length")));
    }

    return 0;
}

sub _handleSuccessfulDownload {
    my ($self) = @_;
    my $downloadDir = $self->getDownloadDir();
    my $filePath = File::Spec->catfile($downloadDir, $self->getName());
    my $partFilePath = File::Spec->catfile($downloadDir, $self->getPartFilename());
    my $metaFilePath = File::Spec->catfile($downloadDir, $self->getMetaFilename());

    if(!File::Copy::move($partFilePath, $filePath)) {
        $self->getMsgLst()->addWarning("Failed to rename '$partFilePath' to '$filePath'.");
    }

    if(!deleteFile($metaFilePath)) {
        $self->getMsgLst()->addWarning("Failed to delete file '$metaFilePath'.");
    }

    return 1;
}

sub _hasValidResumeData {
    my ($self) = @_;
    my $name = $self->getName();
    return 1 if !defined($name); # If the name is not defined, the resume data will be validated later

    if ($self->hasResumeData()) {
        $self->getMsgLst()->addMessage("Validating resume data for '$name'...");
        return $self->_validateMetaFile();
    }

    return 0;
}

sub _cleanupResumeData {
    my ($self) = @_;
    my $name = $self->getName();
    my $downloadDir = $self->getDownloadDir();
    my $metaFilePath = File::Spec->catfile($downloadDir, $self->getMetaFilename());
    my $partFilePath = File::Spec->catfile($downloadDir, $self->getPartFilename());
    my $rc = 1;

    $rc &&= deleteFile($metaFilePath) if $self->hasMetaFile();
    $rc &&= deleteFile($partFilePath) if $self->hasPartFile();

    return $rc;
}

sub _deleteArchive {
    my ($self) = @_;
    my $name = $self->getName();
    my $downloadDir = $self->getDownloadDir();
    my $archivePath = File::Spec->catfile($downloadDir, $name);

    return $self->archiveExists() ? deleteFile($archivePath) : 1;
}

sub _canRetry {
    my ($self) = @_;
    my $canRetry = defined($self->{canRetry}) ? $self->{canRetry} : 1;
    return $canRetry;
}

sub _setCanRetry {
    my ($self, $canRetry) = @_;
    $self->{canRetry} = $canRetry;
}

sub _setResponseDataCallback {
    my ($self, $callback) = @_;
    $callback = (defined($callback) && (ref($callback) eq 'CODE')) ? $callback : undef;
    $self->{responseDataCallback} = $callback;
}

sub _getResponseDataCallback {
    my ($self) = @_;
    return $self->{responseDataCallback};
}

sub _setConfiguration {
    my ($self, $configuration) = @_;
    $self->{instconfig} = $configuration ? $configuration : undef;
}

sub _getConfiguration {
    my ($self) = @_;
    return $self->{instconfig};
}

sub _handleError {
    my ($self, $errorString, $response) = @_;
    chomp($errorString);
    if ($errorString) {
        my @parts = split(":", $errorString);
        my $error = shift(@parts);
        my $shouldRetry = exists($ACCEPTABLE_ERRORS{$error}) ? $ACCEPTABLE_ERRORS{$error} : 0;
        my $message = $ERROR_MESSAGES{$error} // "Unexpected error occured: $error";

        my $formattedMessage = sprintf($message, @parts);
        if ($shouldRetry) {
            $self->getMsgLst()->addWarning($formattedMessage);
        } else {
            $self->getErrMsgLst()->addError($formattedMessage);
        }

        $self->_setCanRetry($shouldRetry);
    } elsif (defined($response) && !$response->is_success()) {
        my $responseCode = $response->code();
        my $statusLine = $response->status_line();
        $self->getErrMsgLst()->addError(sprintf("HTTP request failed (%d): %s", $responseCode, $statusLine));
        $self->_setCanRetry(0);
    }
}

sub _validateMetaFile {
    my ($self) = @_;
    my $metaFileName = $self->getMetaFilename();
    my $headers = $self->_getHeaders();
    if (!$headers) {
        $self->getMsgLst()->addWarning("Couldn't validate the '$metaFileName'.");
        return 1;
    }
    return (int($headers->header('Content-Length')) == $self->_getPersistedSizeFromMeta()) ? 1 : 0;
}

sub _getPersistedSizeFromMeta {
    my ($self) = @_;
    my $lines = [ -1 ];

    if ($self->getMetaFile()) {
        my $downloadDir = $self->getDownloadDir();
        my $metaFilePath = File::Spec->catfile($downloadDir, $self->getMetaFilename());
        $lines = readFile($metaFilePath, undef) || [ -1 ];
    }

    return int($lines->[0]);
}

sub _getRefererHeader {
    my ($self) = @_;
    my $configuration = $self->_getConfiguration();
    my $trexInstance = $configuration->getOwnInstance();
    my $hanaHost = $trexInstance->get_host();
    my $sid = $trexInstance->get_sid();
    my $refererURI = sprintf('https://%s:1129/lmsl/HDBLCM/%s/index.html', $hanaHost, $sid);

    return [ 'Referer' => $refererURI ];
}

sub _getRangeHeader {
    my ($self) = @_;
    my $partFile = $self->getPartFile();
    my $startByte = defined($partFile) ? $partFile->size() : 0;

    return $startByte > 0 ? [ 'Range' => sprintf('bytes=%d-', $startByte) ] : [];
}

sub _extractFilename {
    my ($self, $contentDsiposition) = @_;
    my @parts = split("filename=", $contentDsiposition);
    my $fileName = ($parts[1] =~ s/\s*('|")(.+)\1\s*/$2/r);
    return $fileName;
}

sub _getFilehandle {
    my ($self) = @_;
    return $self->{fhOut};
}

sub _openPartFile {
    my ($self) = @_;
    my $downloadDir = $self->getDownloadDir();
    my $partFilePath = File::Spec->catfile($downloadDir, $self->getPartFilename());
    my $fileHandle = new IO::File(">>$partFilePath");

    if (!$fileHandle || !$fileHandle->binmode()) {
        return undef;
    }

    $self->{fhOut} = $fileHandle;
    return $fileHandle;
}

sub _closePartFile {
    my ($self) = @_;
    my $fhOut = $self->_getFilehandle();
    if (defined($fhOut)) {
        $fhOut->close();
        $self->{fhOut} = undef;
    }
}

sub _createMetaFile {
    my ($self, $targetLength) = @_;
    my $downloadDir = $self->getDownloadDir();
    my $metaFilePath = File::Spec->catfile($downloadDir, $self->getMetaFilename());
    my $fileHandle = new IO::File(">$metaFilePath");

    if(!$fileHandle) {
        return undef;
    }

    $fileHandle->print($targetLength) or return undef;

    $fileHandle->flush();
    $fileHandle->close();
}

sub _checkExistingFiles {
    my($self) = @_;
    my $targetLength = $self->_getHeaders()->header("Content-Length");
    my $name = $self->getName();

    if ($self->archiveExists()) {
        if (int($self->getArchive()->size()) == $targetLength) {
            $self->getMsgLst()->addMessage("Archive '$name' already exists in the download directory. Skipping download...");
            die("$ARCHIVE_EXISTS_ERROR\n");
        } elsif (!$self->_deleteArchive()) {
            die("INVALID_ARCHIVE_CLEANUP_FAILED:$name\n");
        }
    } elsif ($self->hasResumeData() && ($self->getDownloadedBytes() > 0)) {
        my $shouldResume = $self->_validateMetaFile();
        die("ERR_RESUME_NEEDED:$name\n") if $shouldResume;
        $self->_cleanupResumeData() or die("ERR_CLEANUP_FAILED:$name\n");
    }
}

sub _createResponseDataHandler {
    my ($self) = @_;
    my $downloadedBytes = $self->getDownloadedBytes();
    my $fhOut = $self->_getFilehandle();

    return sub {
        my($response, $userAgent, $handler, $data) = @_;
        my $targetLength = int($response->header('Content-Length'));

        if (!defined($self->getName())) {
            my $contentDsiposition = $response->header("Content-Disposition");
            my $newFilename = $self->_extractFilename($contentDsiposition);
            if (!$contentDsiposition || !$newFilename) {
                die ("ERR_FILENAME_UNDEFINED\n");
            }

            $self->setName($newFilename);
            $self->{headers} = $response->headers() if !$self->{headers};

            $self->_checkExistingFiles();
        }

        if (!$fhOut) {
            $fhOut = $self->_openPartFile();
            die("ERR_PART_FILE_OPEN:$!\n") if !$fhOut;
        }

        if(!$self->hasMetaFile()){
            if (!$self->_createMetaFile($targetLength)) {
                die("ERR_META_FILE_OPEN:$!\n")
            }
        }

        $fhOut->print($data) or die("ERR_PART_FILE_WRITE:$!\n");
        $fhOut->flush();

        $downloadedBytes += length($data);

        my $responseCallback = $self->_getResponseDataCallback();
        $responseCallback->($downloadedBytes) if defined($responseCallback);

        return 1;
    };
}

sub _createPostDownloadHandler {
    my ($self) = @_;

    return sub {
        my($response, $userAgent, $handler) = @_;
        my $name = $self->getName();
        my $downloadedBytes = $self->getDownloadedBytes();
        my $targetSize = $self->_getPersistedSizeFromMeta();
        my $isSuccess = $response->is_success() && !$response->header('x-died');

        if ($isSuccess && ($downloadedBytes != $targetSize)) {
            die("ERR_PARTIAL_DOWNLOAD:$name\n");
        }

        return 1;
    }
}

sub _getHeaders {
    my ($self, $noCache) = @_;
    if (!$self->{headers} || $noCache) {
        my $userAgent = $self->getUserAgent();
        my $url = $self->getUrl();

        my $response = undef; # Will contain only the headers
        eval {
            local $SIG{__DIE__} = 'DEFAULT';
            local $SIG{__WARN__} = 'DEFAULT';

            $userAgent->set_my_handler('response_data', sub { croak(); });  # We dont need the data, only the headers, so abort the request
            $response = $userAgent->get($url, @{$self->_getRefererHeader()});
        };
        my $isSuccess = !$@ && defined($response) && $response->is_success();
        $self->{headers} = $isSuccess ? $response->headers() : undef;
    }
    return $self->{headers};
}

sub _getContent {
    my ($self) = @_;
    my $userAgent = $self->getUserAgent();
    my $url = $self->getUrl();
    my @requestHeaders = (
        @{$self->_getRefererHeader()},
        @{$self->_getRangeHeader()},
    );

    my $response = undef;
    eval {
        local $SIG{__DIE__} = 'DEFAULT';
        local $SIG{__WARN__} = 'DEFAULT';

        $userAgent->set_my_handler('response_data', $self->_createResponseDataHandler());
        $userAgent->set_my_handler('response_done', $self->_createPostDownloadHandler());

        $response = $userAgent->get($url, @requestHeaders);
    };
    my $error = $@ ? $@ : $response->header('x-died');
    return ($error, $response);
}

1;
