package LCM::Gui::Dialogs::AddHosts::LocalHostPropertiesDialog;

use strict;
use LCM::Gui::Dialogs::AdditionalHostPropsDialog;
use base qw (SAPDB::Install::Wizard::Dialog::Base);
use Wx qw ( wxVERTICAL wxEXPAND wxALL );
use Wx::Event qw (EVT_CHECKBOX EVT_TEXT EVT_COMBOBOX);
use LCM::Gui::Dialogs::HostsPanel;
use SDB::Install::Configuration::AnyMultiHostConfig qw ($validHostRoles);
use SDB::Install::Globals qw (getHostRoleByHumanReadableRole getHumanReadableRoleByHostRole GetHostRoleProperties);
use SDB::Install::Configuration qw($bool_false_pattern);
use LCM::Gui::Dialogs::UiComponentsFactory;

sub new {
	my ($self) = shift->SUPER::new(@_);

	$self->{id}   = 'LocalHostProperties';
	$self->{name} = "Enter Local Host Properties";

	return $self;
}

sub createControlActions {
	my ( $self, ) = @_;

	EVT_CHECKBOX( $self, $self->{defineSPManually}->GetId(), \&_onChkBoxDefineSPManually );
	EVT_TEXT( $self, $self->{Hostname}->GetId(),         \&_updateButtonsState );
	EVT_TEXT( $self, $self->{StoragePartition}->GetId(), \&_updateButtonsState );
	EVT_COMBOBOX( $self, $self->{Role}->GetId(), \&_onCbRoleChanged );
}

sub init {
    my ( $self, $config ) = @_;

    return {} if ($self->{initiliazed} );

	my $app = Wx::wxTheApp;
	my $instconfig = $app->{instconfig};  
	my $outer_sizer = new Wx::BoxSizer(wxVERTICAL);

	$self->SetSizer($outer_sizer);

	my $sizer = new Wx::BoxSizer(wxVERTICAL);
	$outer_sizer->Add( $sizer, 1, wxALL | wxEXPAND, 20 );

	my $is_gtk1 = Wx::wxGTK1();
	my $title = CreateTitle($self, 'Local Host Properties', $is_gtk1);
	$sizer->Add( $title, 0 );
	$sizer->AddSpacer(20);

	my $componentsSizer = $self->LCM::Gui::Dialogs::AdditionalHostPropsDialog::CreateInputFormSizer(1);

	$sizer->Add( $componentsSizer, 1, wxEXPAND );
	$self->createControlActions();
	$self->SetAutoLayout(1);
	if(!$instconfig->setBatchValueOfRequiredParams('AutoAddXS2Roles')) {
		$self->showError();
	}
	$self->_setLocalAddHostsValues($app, $instconfig);
	$self->{initiliazed} = 1;

    return {};
}

sub showError {
    my ( $self ) = @_;
    my $app = Wx::wxTheApp;
    my $instconfig = $app->{instconfig};
    my $errlst = $instconfig->getErrMsgLst();
    $app->ShowErrorMsg ('Configuration error', $errlst) if (!$errlst->isEmpty());
}

sub _setLocalAddHostsValues {
    my ($self, $app, $instconfig) = @_;
    
    my $addHostsBatchValue =  $instconfig->{'params'}->{'AddHosts'}->{'batchValue'};
    if ( ! $addHostsBatchValue ) {
        $self->_setDefaultValues();
        return;
    }
    my $localHostConfiguration = _getLocalHostConfiguration();
    my $localHostname = $localHostConfiguration->getDefault('HostName');
    if (! $localHostname ) {
        $self->_setDefaultValues();
        return;
    }
    $self->_setLocalAddHostsBatchValues($app, $instconfig, $addHostsBatchValue, $localHostname);
}

sub _setLocalAddHostsBatchValues {
    my ($self, $app, $instconfig, $addHostsBatchValue, $localHostname) = @_;
    require SDB::Install::Configuration::AddHostsParser;
    my $parser = new SDB::Install::Configuration::AddHostsParser();
    $parser->setMsgLstContext([$app->getMsgLst()]);

    if ( ! defined $parser->parse ($addHostsBatchValue) ) {
        $app->ShowErrorMsg ("Error parsing command line option " . $instconfig->getOpt('AddHosts'), $parser->getErrMsgLst());
        $self->_setDefaultValues();
        return;
    }

    for my $host ( @{$parser->getHosts()} ) {
        if(! $host or $host ne $localHostname) {
            next;
        }

        $self->{Hostname}->SetValue($host);

        my @roles = @ { $parser->getRoles($host) };

        my $csvRoles = join(",", @roles);
        if (scalar(@roles) == 1) {
            $self->{Role}->SetValue(getHumanReadableRoleByHostRole($roles[0]));
            if ($roles[0] eq 'standby') {
                # Disable checkbox to add storage partition
                $self->{defineSPManually}->Enable(0);
            }
        }
        else {
            $self->_setDefaultValue('HostRoles');
        }

        my $group = $parser->getValue($host, 'group');
        if ($group) {
            $self->{HostGroup}->SetValue($group);
        } elsif (! $group && $csvRoles) {
            $self->{HostGroup}->SetValue($self->_getDefaultGroupValueAccordingToRole($csvRoles));
        } else {
            $self->_setDefaultValue('HostFailoverGroup');
        }

        my $sp = $parser->getValue($host, 'storage_partition');
        if ($sp) {
            $self->{defineSPManually}->Enable(1);
            $self->{defineSPManually}->SetValue(1);
            $self->{StoragePartition}->Enable(1);
            $self->{StoragePartition}->SetValue($sp);
        } else {
            $self->_setDefaultValue('StoragePartitionNumber');
        }
    } 
}

sub _setDefaultValues {
    my ( $self, ) = @_;
    $self->_setDefaultValue('HostName');
    $self->_setDefaultValue('HostRoles');
    $self->_setDefaultValue('HostFailoverGroup');
    $self->_setDefaultValue('StoragePartitionNumber');
    $self->_setDefaultValue('WorkerGroup');
}

sub _setDefaultValue {
    my ($self, $paramName) = @_;
    my $localHostConfiguration = _getLocalHostConfiguration();
    my $guiControl;
    if ($paramName eq 'HostName') {
        $guiControl = 'Hostname';
    } elsif ($paramName eq 'HostFailoverGroup') {
        $guiControl = 'HostGroup';
    } elsif ($paramName eq 'StoragePartitionNumber') {
        $guiControl = 'StoragePartition';
    } elsif ($paramName eq 'WorkerGroup') {
        $guiControl = 'WorkerGroup';
    } elsif ($paramName eq 'HostRoles') {
        $guiControl = 'Role';
    } else {
        return;
    }

    my $defaultValue = $localHostConfiguration->getDefault($paramName);
    if ($defaultValue) {
        if ( $paramName eq 'StoragePartitionNumber') {
            $self->{defineSPManually}->Enable(1);
        }
        if ($guiControl eq 'Role') {
            $self->{$guiControl}->SetValue(getHumanReadableRoleByHostRole($defaultValue));
        } else {
            $self->{$guiControl}->SetValue($defaultValue);
        }
    }
}

our $hostRoleProperties = GetHostRoleProperties ();

sub _onCbRoleChanged ($){
	my ($self) = @_;
	my $humanReadableRole = $self->{Role}->GetValue();
	my $enableStoragePartition = 1;
	my $hostRole = getHostRoleByHumanReadableRole($humanReadableRole);
	my $localHostConfiguration = _getLocalHostConfiguration();

	if ("standby" eq $hostRole) {
		$self->{defineSPManually}->SetValue(0);
		$enableStoragePartition = 0;
	}
	if($self->_shouldDisplayWorkerGroup($hostRole)){
		$self->{WorkerGroup}->Enable(1);
		$self->{WorkerGroupLabel}->Enable(1);
	} else {
		my $defaultWorkerGroup = $localHostConfiguration->getDefault('WorkerGroup');
		$self->{WorkerGroup}->SetValue($defaultWorkerGroup);
		$self->{WorkerGroup}->Enable(0);
		$self->{WorkerGroupLabel}->Enable(0);
	}

	$self->{WorkerGroup}->Enable ($hostRoleProperties->{$hostRole}->{hasWorkerGroup});

	$self->{defineSPManually}->Enable($enableStoragePartition);

	my $group = $self->_getDefaultGroupValueAccordingToRole($hostRole);
	
	$self->{HostGroup}->SetValue($group);
	$self->_onChkBoxDefineSPManually();
}

sub _shouldDisplayWorkerGroup {
    my ($self, $role) = @_;
    return exists($validHostRoles->{$role}) && $validHostRoles->{$role}->{hasWorkerGroup};
}

sub _getDefaultGroupValueAccordingToRole {
	my ($self, $hostRole) = @_;
	if(exists($validHostRoles->{$hostRole})){
		return $validHostRoles->{$hostRole}->{defaultGroup};
	}
	return 'default';
}

sub _onChkBoxDefineSPManually {
	my ($self)    = @_;
	my $isChecked = $self->{defineSPManually}->IsChecked();
	my $isEnabled = $self->{defineSPManually}->IsEnabled();

	$self->{StoragePartition}->Enable($isChecked);
	$self->{StoragePartition}->SetValue( $isEnabled ? ( $isChecked ? "" : $LCM::AdditionalHostHelper::SP_AUTO_ASSIGN ) : $LCM::AdditionalHostHelper::SP_NA );
	$self->_updateButtonsState();
}

sub _updateButtonsState ($) {
	my ($self) = @_;
	my $okState = 1;

	if ( !$self->{Hostname}->GetValue() ) {
		$okState = 0;
	}

	if ( $okState and $self->{defineSPManually}->IsChecked() and !$self->{StoragePartition}->GetValue() ) {
		$okState = 0;
	}
}

sub updateHostname ($);
*updateHostname = \&_updateButtonsState;

sub updateRole ($);
*updateRole = \&_onCbRoleChanged;

sub updateStoragePartition ($);
*updateStoragePartition = \&_updateButtonsState;

sub Process {
	my ( $self, ) = @_;
	
	my $localHostConfiguration = _getLocalHostConfiguration();
	$localHostConfiguration->ResetError();

	my $ctrl = $self->{Hostname};
	my $hostnameValue = $ctrl->GetValue();
	if ( ! $localHostConfiguration->setValue('HostName', $hostnameValue) ) {
		return _handleError( $ctrl, $localHostConfiguration );
	}

	my $localAndRemoteHostsConfiguration = $self->LCM::Gui::DialogUtilities::GetAppConfig();
	$localAndRemoteHostsConfiguration->ResetError();

	$ctrl = $self->{Password};
	my $systemAdminPasswordValue = $ctrl->GetValue();
	if ( ! $localAndRemoteHostsConfiguration->setValue('Password', $systemAdminPasswordValue) ) {
		return _handleError( $ctrl, $localAndRemoteHostsConfiguration );
	}

	$ctrl = $self->{Role};
	my $humanReadableRole = $ctrl->GetValue();
	if (defined $humanReadableRole){
		my $hostRole = getHostRoleByHumanReadableRole($humanReadableRole);
		if ( ! $localHostConfiguration->setValue( 'HostRoles', $hostRole ) ) {
			return _handleError( $ctrl, $localHostConfiguration );
		}
	}

	if ( $self->{defineSPManually}->IsChecked() ) {
		$ctrl = $self->{StoragePartition};
		my $storagePartitionValue = $ctrl->GetValue();
		if ( ! $localHostConfiguration->setValue( 'StoragePartitionNumber', $storagePartitionValue ) ) {
			return _handleError( $ctrl, $localHostConfiguration );
		}
	} else {
        $localHostConfiguration->resetParam('StoragePartitionNumber');
	}

	$ctrl = $self->{WorkerGroup};
	my $workergroup = $ctrl->GetValue();
	if (defined $workergroup){
		if ( ! $localHostConfiguration->setValue( 'WorkerGroup', $workergroup ) ) {
			return _handleError( $ctrl, $localHostConfiguration );
		}
	}

	$ctrl = $self->{Password};
	my $group = $self->{HostGroup}->GetValue();
	if ( ! $localHostConfiguration->setValue( 'HostFailoverGroup', $group ) ) {
		return _handleError( $ctrl, $localHostConfiguration );
	}

	return {};
}

sub _getLocalHostConfiguration{
	my $localAndExternalHostsConfiguration = LCM::Gui::DialogUtilities::GetAppConfig();
	
	return $localAndExternalHostsConfiguration->{localHostConfiguration};
}

sub _handleError {
	my ( $ctrl, $config )     = @_;
	$ctrl->SetFocus();
	my $app = Wx::wxTheApp;
	$app->ShowErrorMsg( 'Configuration error', $config->getErrMsgLst() );
	return undef;
}

1;
