package LCM::Landscape::Product;

use strict;
use LCM::Landscape::ProductInstance;

my %PD_XML_TO_LM_STRUCTURE_MAPPING = (
	'name' => 'name',
	'vendor' => 'vendor',
	'caption' => 'caption',
	'version' => 'version',
	'PPMS-ID' => 'identifyingNumber',
);

sub new {
	require XML::LibXML;

	my ($class, $systemComponentManager, $productVersionNode, $context) = @_;
	my $self = bless({}, $class);

	eval { $self->_detectProductTags($productVersionNode, $context); };
	$self->_detectProductInstances($systemComponentManager, $productVersionNode, $context);

	return $self;
}

sub getName {
	my ($self) = @_;
	my $productTags = $self->_getProductTags();

	return $productTags->{name};
}

sub getVendor {
	my ($self) = @_;
	my $productTags = $self->_getProductTags();

	return $productTags->{vendor};
}

sub getCaption {
	my ($self) = @_;
	my $productTags = $self->_getProductTags();

	return $productTags->{caption};
}

sub getVersion {
	my ($self) = @_;
	my $productTags = $self->_getProductTags();

	return $productTags->{version};
}

sub getPPMSId {
	my ($self) = @_;
	my $productTags = $self->_getProductTags();

	return $productTags->{identifyingNumber};
}

sub getStackId {
	my ($self) = @_;
	my $productTags = $self->_getProductTags();

	return $productTags->{stackId};
}

sub getStackName {
	my ($self) = @_;
	my $productTags = $self->_getProductTags();

	return $productTags->{stackName};
}

sub getStackCaption {
	my ($self) = @_;
	my $productTags = $self->_getProductTags();

	return $productTags->{stackCaption};
}

sub isEmpty {
	my ($self) = @_;
	my $productInstances = $self->getProductInstances();

	return scalar(@{$productInstances}) == 0;
}

sub hasProductInstance {
	my ($self, $productInstance) = @_;
	my $productInstances = $self->getProductInstanceByID($productInstance->getID());

	return defined($productInstances);
}

sub getID {
	my ($self) = @_;
	my %tags = %{$self->_getProductTags()};
	return sprintf("ID<%s>Name<%s>Vendor<%s>Version<%s>", @tags{qw(identifyingNumber name vendor version)});
}

sub addProductInstances {
	my ($self, $productInstances) = @_;
	for my $productInstance( @$productInstances){
		my $id = $productInstance->getID();
		if(!$self->hasProductInstance($productInstance) && !$productInstance->isEmpty()){
			$self->{productInstances}->{$id} = $productInstance;
		} elsif ($self->hasProductInstance($productInstance)) {
			$self->getProductInstanceByID($id)->addSoftwareComponents($productInstance->getSoftwareComponents());
		}
	}
}

sub getProductInstanceByID {
	my ($self, $id) = @_;
	return $self->{productInstances}->{$id};
}

sub getProductInstances {
	my ($self) = @_;
	return [ values(%{$self->{productInstances}}) ];
}

sub createXmlElement {
	my ($self, $shallAddSoftwareComponentsData) = @_;
	my $element = $self->_createXmlElement($self->_getProductTags());
	my $featuresNode = $element->findnodes('collectedSoftwareFeatures')->get_node(1);

	while(my ($key, $productInstance) = each %{$self->{productInstances}}) {
		my $productInstanceElement = $productInstance->createElement($shallAddSoftwareComponentsData);
		$featuresNode->appendChild($productInstanceElement);
	}
	return $element;
}

sub _detectProductInstances {
	my ($self, $systemComponentManager, $productVersionNode, $context) = @_;

	_addMessage($context, "Reading product instances information...");

	my $productInstanceNodes = $productVersionNode->getElementsByTagName('product-instance') || [];
	my @detectedProductInstances = map { new LCM::Landscape::ProductInstance($systemComponentManager, $_, $context) } @{$productInstanceNodes};
	my @nonEmptyProductInstances = grep { !$_->isEmpty() } @detectedProductInstances;
	my %productInstancesMap = map { $_->getID() => $_ } @nonEmptyProductInstances;

	$self->_setProductInstances(\%productInstancesMap);
}

sub _setProductInstances {
	my ($self, $productInstances) = @_;
	$self->{productInstances} = $productInstances;
}

sub _getProductTags{
	my ($self) = @_;
	return $self->{productTags};
}

sub _setProductTags{
	my ($self, $productTags) = @_;
	$self->{productTags} = $productTags;
}

sub _createXmlElement {
	my ($self, $productTags) = @_;
	my $installedProductVersion = XML::LibXML::Element->new('installedProductVersion');
	my @orderedTags = ('identifyingNumber', 'name', 'vendor', 'version', 'caption');

	for my $name (@orderedTags) {
		my $tag = XML::LibXML::Element->new($name);
		my $value = $productTags->{$name};
		$tag->appendTextNode($value);
		$installedProductVersion->appendChild($tag);
	}

	my $collectedSoftwareFeatures = XML::LibXML::Element->new('collectedSoftwareFeatures');
	$installedProductVersion->appendChild($collectedSoftwareFeatures);
	return $installedProductVersion;
}

sub _detectProductTags {
	my ($self, $productVersionNode, $context) = @_;
	my $tags = {};
	my $productVersionKeyNode = $productVersionNode->getElementsByTagName('product-version-key')->[0];
	my $spStackNode = $productVersionNode->getElementsByTagName('sp-stack')->[0];
	my $spKeyNode = $spStackNode ? $spStackNode->getElementsByTagName('sp-key')->[0] : undef;
	my $stackIdElement = $spKeyNode ? $spKeyNode->getElementsByTagName('PPMS-ID')->[0] : undef;
	my $stackNameElement = $spKeyNode ? $spKeyNode->getElementsByTagName('name')->[0] : undef;
	my $stackCaptionElement = $spStackNode ? $spStackNode->getElementsByTagName('sp-stack-caption')->[0] : undef;

	$tags->{stackId} = defined($stackIdElement) ? $stackIdElement->textContent() : undef;
	$tags->{stackName} = defined($stackNameElement) ? $stackNameElement->textContent() : undef;
	$tags->{stackCaption} = defined($stackCaptionElement) ? $stackCaptionElement->textContent() : undef;

	for my $name (keys %PD_XML_TO_LM_STRUCTURE_MAPPING){
		my $tagsKey = $PD_XML_TO_LM_STRUCTURE_MAPPING{$name};
		my $element;
		if($name eq 'caption'){
			# BECAUSE PD.XML FORMAT IS STRANGE
			$element = $productVersionNode->getElementsByTagName($name)->[0];
		} else {
			$element = $productVersionKeyNode->getElementsByTagName($name)->[0];
		}
		$tags->{$tagsKey} = $element ? $element->textContent() : '';
	}

	$self->_setProductTags($tags);
}

sub _addMessage {
	my ($context, $message) = @_;
	$context->getMsgLst()->addMessage($message) if($context);
}

1;